/********************************************************************** 
 * Copyright (c) 2005, 2006 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: ProbeBundler.java,v 1.2 2006/02/06 20:18:16 nmehrega Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 

package org.eclipse.tptp.platform.probekit.builder;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.probekit.CompilerFactory;
import org.eclipse.hyades.probekit.IProbeCompiler;
import org.eclipse.hyades.probekit.ProbekitPlugin;
import org.eclipse.tptp.platform.probekit.registry.ProbeRegistry;
import org.eclipse.tptp.platform.probekit.util.InaccessibleProbeRsrcException;
import org.eclipse.tptp.platform.probekit.util.ProbeResourceBundle;
import org.eclipse.tptp.platform.probekit.util.ProbekitConstants;


/**
 * @author apratt
 *
 * To change the template for this generated type comment go to
 * Window - Preferences - Java - Code Generation - Code and Comments
 */
public class ProbeBundler extends IncrementalProjectBuilder {
	
	static final ProbeRegistry registry = ProbeRegistry.getRegistry();
	/**
	 * The number of non-class files in a logical probe bundle: probe source, 
	 * probeinfo, and probescript.
	 */
	private static final int NUM_NON_SUPPORT_FILES = 3;
	
	 /**
	  * This class wraps a Map with appropriate casting and operations
	  * for a map of Vectors, keyed by Strings. 
	  */
	 static class MapOfVectors extends HashMap {
        private static final long serialVersionUID = 3257849870223358261L;

        public Vector getVector(String candidate) {
	 		return (Vector)get(candidate);
	 	}
	 	
	 	public void putResource(String key, IResource value) {
	 		Vector v = (Vector)get(key);
	 		if (v == null) {
	 			v = new Vector();
	 			put(key, v);
	 		}
 			v.add(value);
	 	}
	 }

	/* (non-Javadoc)
	 * @see org.eclipse.core.internal.events.InternalBuilder#build(int, java.util.Map, org.eclipse.core.runtime.IProgressMonitor)
	 */
	protected IProject[] build(int kind, Map args, IProgressMonitor monitor)
			throws CoreException 
	{
		BuilderUtils.trace("---bundler---");

		ProbeBundlerVisitor visitor = new ProbeBundlerVisitor(monitor);
		
		if (kind == IncrementalProjectBuilder.FULL_BUILD) {
			getProject().accept(visitor);
		}
		else {
			// Incremental build
			IResourceDelta delta = getDelta(getProject());
			if (delta == null) {
				// Run the visitor as a visitor, not a delta visitor.
				getProject().accept(visitor);
			}
			else {
				// Run the visitor as a delta visitor
				delta.accept(visitor);
			}
		}

		// Now we should have seen everything we're going to see
		MapOfVectors mb = visitor.getMemoryBuckets();
		Iterator i = mb.entrySet().iterator();
		while (i.hasNext()) {
			Map.Entry e = (Map.Entry)i.next();
			Vector value = (Vector)e.getValue();
			ProbeResourceBundle bundle = new ProbeResourceBundle();

			if (value.size() < NUM_NON_SUPPORT_FILES) {
				// Can't possibly be a good probekit file set because it 
				// doesn't include enough files; ignore it.
			}
			else {
				boolean anyFileListProblems = false;
				IResource source = null;
				IResource engineScript = null;
				Iterator vi = value.iterator();
				
				// Loop through the resources in the memory bucket.
				// Remember, these were identified as having a common prefix 
				// (based on the project and directory name, and the standard 
				// Probekit filename-to-classname mapping). A file set should 
				// be handed off to the probe registry if it includes all the
				// the expected parts.
				while (vi.hasNext()) {
					IResource r = (IResource)vi.next();
					try {
						// bundle code will check for general validity of what
						// we hand it and categorize the input (probescript,
						// probeinfo, class file, etc.)
						bundle.addResource(r);
					} catch (InaccessibleProbeRsrcException ex) {
						// this resource can't be part of a bundle
						anyFileListProblems = true;
						break;
					}
				} // end for each IResource in the memoryBucket
				
				IResource[] supportFiles = bundle.getSupporting();
				if (supportFiles.length != (value.size() - NUM_NON_SUPPORT_FILES)) {
					// Whoops! There are too many files that aren't the source
					// or the engine script file. (Or not enough of them).
					// Guess this isn't a probe file set after all.
					anyFileListProblems = true;
				}
				
				if ( !anyFileListProblems && bundle.isComplete() ) {
					// Success! The same-prefix file set looks like a probekit file set.
					// Give it to the Probe Registry.
					try {
						bundle.createResourceAssociations();
						registry.add(bundle);
						BuilderUtils.trace("Registered file set for " + (String)e.getKey());	//$NON-NLS-1$
					} catch (Exception ex) {
						IStatus s = new Status(IStatus.ERROR, 
								"org.eclipse.tptp.platform.probekit", //$NON-NLS-1$
								IStatus.OK,
								ProbekitPlugin.getResourceString("Bundler.registryError"), //$NON-NLS-1$ 
								ex);
						throw new CoreException(s);
					}
				} else {
					// Not a well-formed resource bundle.
					// Guess this isn't a probekit file set after all.
				}
			}
		} // end for each memory bucket
		BuilderUtils.trace("---bundler end---"); 	//$NON-NLS-1$
		return null;
	}
	
	/**
	 * Helper function that returns the basename (minus the extension)
	 * for a file name.
	 * @param name The input file name to chop down to its basename. Must not have any path info.
	 * @return the part of the input name before the last period character in it.
	 */
	static private String getBaseName(String name) {
		if (name == null) return null;
		int lastdot = name.lastIndexOf('.');
		if (lastdot == -1) return name;
		return name.substring(0, lastdot);
     }
	
	/**
	 * ProbeBundlerVisitor: an IResourceDeltaVisitor for collecting up
	 * all the generated files after a probe changed.
	 * 
	 * Here's what happens here:
	 *   The visit() method of this class is called for every resource that changed.
	 *   We collect every file whose name matches certain patterns:
	 * 		*.probe
	 * 		*.probescript
	 * 		*_probe.class
	 * 		*_probe$*.class
	 * 
	 * We keep a "memory bucket" for each different prefix - the pattern matched
	 * by the initial "*" in the patterns above. Also, two files in different projects
	 * will never be in the same "memory bucket." 
	 * 
	 * When all is said and done, we hand the completed buckets back to our caller.
	 * 
	 * The "buckets" are entries in a PrefixKeyedMap: the key is a string formed from
	 * the project name and the prefix (the part matching "*"),
	 * and the value is a Vector of IResource objects matching that key.
	 * 
	 * TODO: This code doesn't really use the project name as a prefix yet.
	 * 
	 * TODO: Attach the list of derived objects to the original source file as a persistent property.
	 * If the source changes and the builder fails to compile, we'll be able to delete the
	 * fruits of a past, successful compile. Ditto if the source is deleted.
	 * 
	 * TODO: handle the case where the user changes the ID inside the file.
	 * In this case we also want to delete the past fruits. We can tell it happened
	 * because the ID attached to the file now is different from the previous one.
	 * 
	 * This means the persistent property on the resource needs to be the id and
	 * list of files produced from the last successful compile. If a subsequent
	 * compile fails and we remove those, we'll remove them from the persistent property, too.
	 *  
	 */
	static private class ProbeBundlerVisitor implements IResourceDeltaVisitor, IResourceVisitor {
		 private IProgressMonitor monitor;
		 private IProbeCompiler probeCompiler;
		 public ProbeBundlerVisitor(IProgressMonitor m) {
		 	monitor = m;
		 	
		 	// Create a probe compiler instance so we can use the utility functions,
		 	// like getClassSuffix and makeValidJavaIdentifier.
		 	// This is not the same instance that actually compiled anything in the past.
		 	probeCompiler = CompilerFactory.INSTANCE.createCompiler();
		 }

		 /**
		 * @return the memory buckets
		 */
		public MapOfVectors getMemoryBuckets() {
			return memoryBuckets;
		}

		/**
		  * This is the map of resource name prefixes we're remembering and 
		  * the Vector of IResources for each one.
		  */
		MapOfVectors memoryBuckets = new MapOfVectors();

		public void dumpMemoryBuckets() {
			Iterator i = memoryBuckets.keySet().iterator();
			while (i.hasNext()) {
				String s = (String)i.next();
				Vector v = memoryBuckets.getVector(s);
				System.out.println(s + " " + v.size()); //$NON-NLS-1$
				Iterator j = v.iterator();
				while (j.hasNext()) {
					IResource r = (IResource)j.next();
					System.out.println("    " + r.getLocation()); //$NON-NLS-1$
				}
			}
		}

	     public boolean visit(IResourceDelta delta) throws CoreException {
	     	IResource r = delta.getResource();
	     	
	     	if ( r.getType() != IResource.FILE ) {
	     		// The only resource we care about right now are files
	     		return true;
	     	}
	     	
	     	// TODO: handle delete as well as add/change
	     	// TODO: attach the list of derived files to the source file, 
	     	// so they can all be removed if the source file goes away or changes.
	     	// This way a change which results in a build error will still wipe out derived files. 
	     	switch (delta.getKind()) {
	     		case IResourceDelta.ADDED :
	     		case IResourceDelta.CHANGED :
	     			// handle added and changed resources
	     			visit(r);
	     			break;
	     		case IResourceDelta.REMOVED :
	     			// TODO: handle removed resources here. Especially removed source files.
	     			// System.out.println("BUNDLER REMOVED delta not implemented, resource name: " + delta.getResource().getName());
	     			if (r.getName().endsWith(ProbekitConstants.PROBE_SRC_EXT) &&
	     					BuilderUtils.isInSrcDir(r)) {
	     				registry.remove(r);
	     			}
	     			break;
	     		default:
	     			// A change type we don't notice
	     			// System.out.println("BUNDLER Delta of unhandled type " + kind + "on resource " + delta.getResource().getName());
	     			break;
	     	}
	     	return true;
	     }

		/**
		 * Return the "key" string for this file if the file name 
		 * matches any of our interesting patterns.
		 * The patterns are for probe source files, generated class files, and
		 * generated probescript files.
		 * 
		 * The key includes the project name, so identically-named probe
		 * source files in different projects will work just fine.
		 * 
		 * But two source files in a single project (in different directories)
		 * with the same name will probably trip you up, because the key
		 * does not include the subdirectory structure leading to the file.
		 * If it did, we'd get confused by having one path to the source
		 * file and another to the binary files - if your Java project
		 * is the kind that puts binaries in a separate folder.
		 * 
		 * @param r the resource to get the key for
		 * @return the string to use as the key - it'll be the project name
		 * plus the (possibly converted) file basename.
		 */
		private String getFilesetKeyIfInteresting(IResource r) {
			String leafname = r.getName();
			String basename;
			
			if (leafname.endsWith(ProbekitConstants.PROBE_SRC_EXT)) {
				if (BuilderUtils.isInSrcDir(r)) {
					basename = leafname.substring(0, leafname.lastIndexOf(ProbekitConstants.PROBE_SRC_EXT));
					basename = probeCompiler.makeValidJavaIdentifier(basename);
				} else {
					// It is probably the .probe file copied by the java builder
					// from source to output dir. Ignore it.
					return null;
				}
			}
			else if (leafname.endsWith(ProbekitConstants.PROBE_SCRIPT_EXT)) {
				
				// Probescript file is generated into the same directory
				// as the source. The basename is already mangled.
				if ( BuilderUtils.isInSrcDir(r)) {
					basename = leafname.substring(0, leafname.lastIndexOf(ProbekitConstants.PROBE_SCRIPT_EXT));
				} else {
					// This is the copy made by the java builder. Ignore it.
					return null;
				}
			} else if (leafname.endsWith(ProbekitConstants.PROBE_INFO_EXT)) {
				// Probeinfo file is generated into the same directory as 
				// the source. The basename is already mangled.
				if ( BuilderUtils.isInSrcDir(r)) {
					basename = leafname.substring(0, leafname.lastIndexOf(ProbekitConstants.PROBE_INFO_EXT));
				} else {
					// This is the copy made by the java builder. Ignore it.
					return null;
				}
			}
			else {
				String classNameSuffix = probeCompiler.getClassSuffix();
				String classFileSuffix = classNameSuffix + ".class"; //$NON-NLS-1$
				String classInnerPattern = classNameSuffix + "$"; //$NON-NLS-1$
				if (leafname.endsWith(classFileSuffix)) {
					basename = leafname.substring(0, leafname.lastIndexOf(classFileSuffix));
				}
				else if (leafname.indexOf(classInnerPattern) != -1 && leafname.endsWith(".class")) { //$NON-NLS-1$
					basename = leafname.substring(0, leafname.indexOf(classInnerPattern));
				}
				else {
					// none of the above: the input file name is not interesting.
					return null;
				}
			}
			String key = r.getProject().getName() + "/" + basename;
//			System.out.println("key for " + r.getFullPath() + " is " + key);
			return key;
		}

		/* This is called for every changed resource we want to notice.
		 * @see org.eclipse.core.resources.IResourceVisitor#visit(org.eclipse.core.resources.IResource)
		 */
		public boolean visit(IResource resource) throws CoreException {
 			String fullname = resource.getName();
 			String key;
 			if (resource.getType() == IResource.FILE &&
	 				(key = getFilesetKeyIfInteresting(resource)) != null)
 			{
 				// Remember this file for later
 				memoryBuckets.putResource(key, resource);
	 		} else {
 				// This is a resource we don't care about
 			}
 			return true;
		}
	}
	
}
