/********************************************************************** 
 * Copyright (c) 2005, 2006 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: ProbeNature.java,v 1.2 2006/02/06 20:18:16 nmehrega Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.tptp.platform.probekit.builder;

import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IProjectNature;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;

/**
 * This is the probe nature class. When applied to a project, it
 * puts the ProbeBuilder command at the start of the project's build commands,
 * and the ProbeBundler command at the end. Presumably in the middle is
 * the Java build command, since this nature requires the Java nature.
 */
public class ProbeNature implements IProjectNature {
	private IProject theProject;
	
	public static final String NATURE_NAME =
		"org.eclipse.hyades.probekit.probekitnature";
	private static final String BUILDER_NAME =
		"org.eclipse.hyades.probekit.probekitbuilder";
	private static final String BUNDLER_NAME =
		"org.eclipse.hyades.probekit.probekitbundler";

	/* (non-Javadoc)
	 * @see org.eclipse.core.resources.IProjectNature#configure()
	 */
	public void configure() throws CoreException {
		// Scan the build commands to see if our guys are already there.
		// TODO: this logic should remove our builders if they exist,
		// and then add them again - to repair improper ordering.
		IProjectDescription desc = theProject.getDescription();
		ICommand[] oldCommands = desc.getBuildSpec();
		boolean found_builder = false;
		boolean found_bundler = false;
		for (int i = 0; i < oldCommands.length; i++) {
			String name = oldCommands[i].getBuilderName();
			if (name.equals(BUILDER_NAME)) {
				found_builder = true;
			}
			else if (name.equals(BUNDLER_NAME)) {
				found_bundler = true;
			}
		}
		
		if (found_builder && found_bundler) {
			// The nature is already there. (Not necessarily in the right
			// order, though!)
			return;			
		}
		
		// Add our builders - BUILDER at the start of the array,
		// BUNDLER at the end. Only if they aren't already there.
		// This code handles broken situations like "one is there and
		// one is not."
		int dest_for_copying = 0;
		int new_length = oldCommands.length;
		
		if (!found_builder) {
			dest_for_copying = 1;
			new_length++;
		}
		if (!found_bundler) {
			new_length++;
		}
		ICommand[] newCommands = new ICommand[new_length];
		if (!found_builder) {
			ICommand builder_cmd = desc.newCommand();
			builder_cmd.setBuilderName(BUILDER_NAME);
			newCommands[0] = builder_cmd;
		}
		System.arraycopy(oldCommands, 0, newCommands, dest_for_copying, oldCommands.length);
		if (!found_bundler) {
			ICommand bundler_cmd = desc.newCommand();
			bundler_cmd.setBuilderName(BUNDLER_NAME);
			newCommands[new_length-1] = bundler_cmd;
		}
		desc.setBuildSpec(newCommands);
		getProject().setDescription(desc, null);
	}

	/**
	 * Return an array which consists of the input array 
	 * minus one element.
	 * @param arr the input array
	 * @param index the index of the element to remove
	 * @return a copy of the array minus the indicated element
	 */
	private ICommand[] removeFromArray(ICommand[] arr, int index) {
		int new_size = arr.length - 1;
		ICommand[] ret = new ICommand[new_size];
		System.arraycopy(arr, 0, ret, 0, index);
		System.arraycopy(arr, index+1, ret, index, arr.length - index);
		return ret;
	}
	/**
	 * Remove all instances of BUILDER and BUNDLER from the build
	 * command array for this project.
	 * 
	 * @see org.eclipse.core.resources.IProjectNature#deconfigure()
	 */
	public void deconfigure() throws CoreException {
		IProjectDescription desc = theProject.getDescription();
		ICommand[] cmds = desc.getBuildSpec();
		boolean found;
		do {
			found = false;
			for (int i = 0; i < cmds.length; i++) {
				String name = cmds[i].getBuilderName();
				if (name.equals(BUILDER_NAME) || name.equals(BUNDLER_NAME)) {
					found = true;
					cmds = removeFromArray(cmds, i);
					break;
				}
			}
		} while (found);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.core.resources.IProjectNature#getProject()
	 */
	public IProject getProject() {
		return theProject;
	}
	/* (non-Javadoc)
	 * @see org.eclipse.core.resources.IProjectNature#setProject(org.eclipse.core.resources.IProject)
	 */
	public void setProject(IProject project) {
		theProject = project;
	}
	
	public static void addProbekitBuildNature(IProject project) 
		throws CoreException
	{
		addProbekitBuildNature(project, null);
	}
	
	public static void addProbekitBuildNature(IProject project, IProgressMonitor monitor) 
		throws CoreException
	{
		if (project.hasNature(NATURE_NAME)) return;
		IProjectDescription desc = project.getDescription();
		String[] ids = desc.getNatureIds();
		String[] newIds = new String[ids.length + 1];
		System.arraycopy(ids, 0, newIds, 0, ids.length);
		newIds[ids.length] = NATURE_NAME;
		desc.setNatureIds(newIds);
		project.setDescription(desc, monitor);
	}
	
	public static void removeProbekitBuildNature(IProject project) 
		throws CoreException
	{
		removeProbekitBuildNature(project, null);
	}
	
	public static void removeProbekitBuildNature(IProject project, IProgressMonitor monitor) 
		throws CoreException
	{
		if (!project.hasNature(NATURE_NAME)) return;
		IProjectDescription desc = project.getDescription();
		String[] ids = desc.getNatureIds();
		String[] newIds = new String[ids.length - 1];
		// Copy all but the matching one
		// BUG: only removes one, even if it go there multiple times.
		int j = 0;
		for (int i = 0; i < ids.length; i++) {
			if (!ids[i].equals(NATURE_NAME)) {
				newIds[j] = ids[i];
				j++;
			}
		}
		desc.setNatureIds(newIds);
		project.setDescription(desc, monitor);
	}
}
