/********************************************************************** 
 * Copyright (c) 2004, 2006 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: CandidateEntry.java,v 1.2 2006/02/06 20:18:19 nmehrega Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 

package org.eclipse.tptp.platform.probekit.registry;

/**
 * This class encapsulates a tentative entry to the registry. It is
 * used to preflight registry entries so that callers can resolve
 * conflicting entries. CandidateEntry's are entirely managed by
 * the ProbeRegistry. Other clients can only make queries.
 * 
 * <p>Use CandidateEntry's with the ProbeRegistry's 
 * addIfNoConflict/commit/discard methods, as follows:</p>
 * 
 * <code>
 * <pre>
 * ProbeRegistry registry = ProbeRegistry.getRegistry();
 * CandidateEntry candidate = registry.addIfNoConflict(someProbekitImportFile);
 * 
 * if ( candidate.getConflict() != null ) {
 *     commit_candidate = do_conflict_resolution();
  *    if ( commit_candidate )
 *        registry.commit(candidate);
 *     else
 *        registry.discard(candidate);
 * }
 * </pre>
 * </code>
 * 
 * <p>Note that a lock is held on the conflicting entry until this
 * candidate is committed, discarded, or garbage collected. The lock
 * ensures that the existence (and nature) of the conflict will not
 * change between creation of the CandidateEntry and final conflict
 * resolution.</p>
 * 
 * <p>Explicitly discarding a candidate is strongly encouraged since 
 * this will immediately release the lock as well as cleaning up any 
 * associated temporary files. See ProbeRegistry for additional information.</p>
 * 
 * @see org.eclipse.tptp.platform.probekit.registry.ProbeRegistry
 * 
 * @author kcoleman
 */
public class CandidateEntry {
	
	/**
	 * There is no conflicting entry in the registry. If this is set,
	 * getConflict() always returns null.
	 */
	public static final int NO_CONFLICT = 0;
	/**
	 * The candidate and it's conflicting entry do not have the same version.
	 * If CANDIDATE_VERSION_OLDER is unset, the candidate entry has the same
	 * or more recent version as the existing (conflicting) entry.
	 */
	public static final int VERSION_CONFLICT = 1;
	/**
	 * There is a version conflict and the candidate version is older
	 * than the existing (conflicting) entry. VERSION_CONFLICT will also
	 * be set if this is true.
	 */
	public static final int CANDIDATE_VERSION_OLDER = 2;
	/**
	 * The conflict entry exists in the workspace. That is, it represents
	 * an authored probe set, not a conflicting imported one.
	 */
	public static final int EXISTS_IN_WORKSPACE = 4;
	
	private ProbeRegistryEntry candidate = null;
	private ProbeRegistryEntry conflict = null;
	private int flags = NO_CONFLICT;
	private boolean dirty = false;
		
	/**
	 * Create a candidate registry <i>entry</i>, which conflicts with <i>incumbent</i>.
	 * <i>incumbent</i> may be null, in which case it is assumed there is no conflict.
	 * These objects are only created on behalf of a probe set that has not yet been
	 * added to the registry. The <i>incumbent</i>, if there is one, represents an
	 * existing registy entry with the same probe set id.
	 * 
	 * @param entry The new candidate entry that has not been committed to the registy.
	 * @param incumbent The conflicting, existing entry, if any.
	 */
	CandidateEntry(ProbeRegistryEntry entry, ProbeRegistryEntry incumbent)
	{
		super();
		candidate = entry;
		conflict = incumbent;
		if ( incumbent != null ) {
			setFlags();
		}
	}
	
	/**
	 * Set flags which indicate the kinds of conflicts that exist in
	 * the registry for this candidate. Nothing is done to check for
	 * or indicate an id conflict - that is implicit in a non-null
	 * <i>conflict</i> member. Instead, we check for refinements on
	 * this basic conflict:
	 * 
	 * <p>The versions are different; the versions differ and the
	 * new (candidate) entry has an older version than the existing
	 * one; the new entry is replacing one from an authored probe
	 * set in the user's workspace.
	 * 
	 * <p>Users of this class may be interested in other kinds of conflicts,
	 * but they'll have to ferret them out themselves by comparing 
	 * information off the underlying registry entries.</p>
	 */
	void setFlags()
	{
		if ( candidate == null || conflict == null ) {
			flags = NO_CONFLICT;
		} else {
			String candVersion = candidate.getVersion();
			String conVersion = conflict.getVersion();
			
			if ( candVersion != null && conVersion != null ) {
				// neither version spec is null
				int compareResult = candVersion.compareTo(conVersion);
				
				if ( compareResult != 0 ) {
					// Versions are not the same
					flags |= VERSION_CONFLICT;
					if ( compareResult < 0 ) {
						flags |= CANDIDATE_VERSION_OLDER;
					}
				}
			} else if ( candVersion != conVersion ) {
				// exactly one version spec is null
				flags |= VERSION_CONFLICT;
				if ( candVersion == null ) {
					// incoming is null, existing conflicting entry is not;
					// treat this like importing an older version
					flags |= CANDIDATE_VERSION_OLDER;
				}
			}
			
			if ( candidate.isAuthored() ) {
				flags |= EXISTS_IN_WORKSPACE;
			}
		}
	}
	
	/**
	 * Change the conflicting entry. This should not normally be necessary.
	 * When the conflicting entry is changed, the flags are automatically
	 * updated to reflect the new conflict details.
	 * 
	 * @param conflict The new conflicting registry entry.
	 */
	void setConflict(ProbeRegistryEntry conflict) {
		this.conflict = conflict;
		flags = NO_CONFLICT;
		setFlags();
	}
	
	/**
	 * Retrieve the candidate registry entry
	 * 
	 * @return The candidate entry
	 */
	public ProbeRegistryEntry getCandidate()
	{
		return candidate;
	}
	
	/**
	 * Retrieve the conflicting registry entry, if any.
	 * 
	 * @return The conflicting entry, or null if there is no conflict.
	 */
	public ProbeRegistryEntry getConflict()
	{
		return conflict;
	}
	
	/**
	 * Retrieve the conflict detail flags. Use the bit masks above to
	 * test for detailed information about the conflict. If there is
	 * no conflict, <i>flags</i> will be NO_CONFLICT.
	 * 
	 * <p>Possible conflict details include: The versions do not match,
	 * the versions do not match <b>and</b> the candidate's version is
	 * older than the existing entry's, and the candidate conflicts
	 * with an entry from an authored probe set in the user's workspace.</p>
	 * 
	 * <p>That the id's conflict is taken for granted - that is what
	 * defines a conflict in this context. The flags just provide additional
	 * information to help guide the user in conflict resolution.</p>
	 * 
	 * @return The conflict detail flags.
	 */
	public int getFlags()
	{
		return flags;
	}
	
	/**
	 * A candidate becomes dirty once it has been either committed or discarded.
	 * At that point, it is no longer a registry entry candidate and cannot be used.
	 * 
	 * @return Returns True if this is no longer a candidate, false otherwise.
	 */
	boolean isDirty() {
		return dirty;
	}
	
	/**
	 * Mark a candidate as dirty and therefore unusable. A candidate becomes dirty 
	 * once it has been either committed or discarded.
	 */
	void markDirty() {
		this.dirty = true;
	}
	
}
