/********************************************************************** 
 * Copyright (c) 2004, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: ProbeRegistry.java,v 1.4 2008/12/15 15:34:40 jcayne Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 

package org.eclipse.tptp.platform.probekit.registry;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Enumeration;

import org.eclipse.hyades.probekit.internal.JarReader;
import org.eclipse.hyades.probekit.ProbekitPlugin;
import org.eclipse.tptp.platform.probekit.util.InvalidProbeBundleException;
import org.eclipse.tptp.platform.probekit.util.ProbeFileBundle;
import org.eclipse.tptp.platform.probekit.util.ProbeResourceBundle;
import org.eclipse.tptp.platform.probekit.util.ProbekitDebugConfig;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;


import java.util.HashMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import com.ibm.icu.text.SimpleDateFormat;

/**
 * The Probe Registry manages probe resources for query, deployment,
 * import, and export operations. There is a probe registry for each
 * workspace.
 * 
 * <p>Entries can be made to the registry in two forms: From a zip file
 * previously created by exporting a .probe file, or from the component
 * pieces. The miniumum pieces include the source file (.probe file) 
 * and all files required to deploy a probe at runtime, such as
 * generated class files and the .probescript file.</p>
 * 
 * <p>Probe sets must be unique within the registry. Probe sets are
 * primarily identified by their id (see org.eclipse.hyades.models.probekit). 
 * If there is an entry in the registry for a given id, it is always
 * replaced by any new entry with the same id. Probe sets which do not
 * have an id will be assigned one by the registry.</p>
 * 
 * <p>In addition, if there is a source file associated with the probe
 * set, the source file is also used in determining probe set equivalence.
 * Two probe sets are equivalent if either their id or their source file
 * names match. When making a new entry, any existing equivalent entries
 * are supplanted by the new entry.</p>
 * 
 * <p>To add a probe set to the registry through an import file,
 * use addIfNoConflict(), along with commit() and discard() for
 * conflict resolution. To add a probe set without worrying about
 * conflicts (e.g. blindly overwrite conflicting entries), use
 * one of the add() methods.</p>
 * 
 * <p>Probes sets added to the registry through import files have their
 * files managed by the registry. They're stored in the plugin's state
 * save area. Probe sets added to the registry by the builder have
 * their files managed by the workbench.</p>
 * 
 * <p>Consumers who want to use the import thru zip file interface
 * should also refer to org.eclipse.tptp.platform.probekit.registry.CandidateEntry.
 * 
 * @author kcoleman
 *
 */
public class ProbeRegistry  {
	
	/**
	 * Timestamp format string for use in generating unique dir names
	 * when unzipping import files. See createEntryFromImportFile.
	 */
	private static final SimpleDateFormat timestamper =
		new SimpleDateFormat("yyyyMMddHHmmss");		//$NON-NLS-1$
	
	/**
	 * Version info used for the format of the saved state. If you 
	 * change the layout of the state info (see save() and restore()),
	 * you need to bump the version. 
	 */
	private static final String VERSION_1 = "1.0";	//$NON-NLS-1$
	private static final String CURRENT_VERSION = VERSION_1;
	
	/**
	 * The registry singleton object. It is expected that there will only
	 * ever be one. If that ever changes, you'll need to reevaluate this
	 * whole class for thread safety.
	 */
	protected static ProbeRegistry theRegistry = null;
	/**
	 * Path to the directory where the registry manages on-disk storage
	 * for imported probe sets.
	 */
	private static IPath importAreaPath = null;
	/**
	 * Counter used for generating unique names. It is only used as a last result
	 * and it doesn't really matter if it wraps around. It is part of the saved
	 * state, so it will re-init every time to where we left off.
	 */
	private int idCounter = 1; 
	
	// These factories are all associated with saving/restoring
	// the registry state. They are only used by save() and restore().
	private static final DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
	private static final TransformerFactory transformerFactory = TransformerFactory.newInstance();
	private static final SAXParserFactory saxParserFactory = SAXParserFactory.newInstance();

	/**
	 * This represents all the probe sets the registry knows about. The
	 * probe set id's are the keys, ProbeRegistryEntry objects are the
	 * values. The registry store is based on HashTable, so it is synchronized.
	 */
	protected Hashtable store = new Hashtable();
	/**
	 * Mapping from source name to registry entry. This is used to 
	 * support operations like lookupBySource(), which in turn is
	 * needed so we can detect conflicting entries even when the id's
	 * change. Only authored probes are entered into the source map.
	 */
	protected HashMap srcMap = new HashMap();
	
	private ArrayList freeList = new ArrayList(5);
	
	protected ProbeRegistry()
	{
		super();
	}
	
	/**
	 * Get a reference to the one and only registry.
	 * 
	 * @return The probe set registry.
	 */
	public static synchronized ProbeRegistry getRegistry()
	{
		if (theRegistry == null) {
			theRegistry = new ProbeRegistry();
		}
		return theRegistry;
	}
	
	/**
	 * Perform registry startup tasks. In particular, make sure we can get our
	 * hands on the storage area the registry uses for imported probe sets.
	 * 
	 */
	public static void startup() 
		throws ProbeRegistryException 
	{
		// Create probekit storage area, if it doesn't already exist.
		File importArea = null;
		try 
		{
			IPath stateRoot = ProbekitPlugin.getDefault().getStateLocation();
			importAreaPath = stateRoot.append(RegistryConstants.REGISTRY_ROOT);
			importArea = importAreaPath.toFile();
			if ( !importArea.exists() )
			{
				importArea.mkdir();
			}
		} catch (Exception e) {
			trace("Unable to create registry storage area");	//$NON-NLS-1$
			throw new ProbeRegistryException("Unable to create registry storage area");
		}	
		if ( !importArea.canRead() || !importArea.canWrite() ) {
			trace("No read/write permission on registry storage area");	//$NON-NLS-1$
			throw new ProbeRegistryException();
		}
	}

	/**
	 * Save the registry state to persistent storage. Use retore() to 
	 * recover what is saved this way.
	 * 
	 * @param saveFile The destination file for the registry state.
	 */
	public void save(File saveFile) 
		throws ProbeRegistryException
	{
		OutputStream stream = null;
		
		try {
			DocumentBuilder docBuilder = documentBuilderFactory.newDocumentBuilder();
			Document doc = docBuilder.newDocument();
			Element regElement = toXML(doc);
			
			if ( regElement != null ) {
				doc.appendChild(regElement);
				
				stream = new FileOutputStream(saveFile);
				StreamResult result = new StreamResult(stream);	
				Transformer transformer;
				synchronized(this) {
					// The factory is not thread safe. Sigh. Neither is a single
					// instance of Transformer, but multiple instances can be used
					// safely on multiple threads.
					transformer = transformerFactory.newTransformer();
				}

				transformer.setOutputProperty(OutputKeys.METHOD, "xml");	//$NON-NLS-1$
				transformer.setOutputProperty(OutputKeys.ENCODING, "UTF-8");	//$NON-NLS-1$
				transformer.setOutputProperty(OutputKeys.INDENT, "yes");	//$NON-NLS-1$
				DOMSource source = new DOMSource(doc);
				transformer.transform(source, result);
			}
		} catch (Exception e) {
			trace("Unable to create registry save file");	//$NON-NLS-1$
			throw new ProbeRegistryException();
		} finally {
			if ( stream != null ) {
				try {
					stream.close();
				} catch (IOException ex) {
					// oh well, we tried
				}
			}
		}
	}
	
	/**
	 * Restore previously saved registry state. This should only be used
	 * to restore state information from a save() operation. The existing
	 * registry contents are completely replaced by the new state.
	 * 
	 * @param saveFile A file created by the registry save() operation.
	 */
	public void restore(File saveFile) 
		throws ProbeRegistryException
	{
		trace("Restoring saved state");
		
		if ( !saveFile.exists() ) {
			throw new ProbeRegistryException("Cannot restore from non-existent file"); //$NON-NLS-1$
		}
		BufferedInputStream instream = null;
		try {
			instream = new BufferedInputStream(new FileInputStream(saveFile));
			InputSource insrc = new InputSource(instream); 
			saxParserFactory.setNamespaceAware(true);
			SAXParser parser = saxParserFactory.newSAXParser();;
			parser.parse(insrc, new RegistryReader());
		} catch (Exception e) {
			trace("Unable to restore registry state");	//$NON-NLS-1$
			throw new ProbeRegistryException(e);
		} finally {
			if ( instream != null ) {
				try {
					instream.close();
				} catch(IOException ex) {
					// oh well, we tried
				}
			}
		}
		
		trace("Restore complete");
	}
	
	static void trace(String message)
	{
		if ( ProbekitDebugConfig.TRACE_REGISTRY && message != null &&
				message.length() > 0) {
			System.out.println("ProbeRegistry: " + message);
		}
	}
	
	/**
	 * Create a registry entry from an import file. Nothing is added to
	 * the registry as a result of calling this method.
	 * 
	 * @param importFileName The name of a file created by exporting a probe set
	 * @return A registry entry corresponding to the import file.
	 * @throws InvalidProbeBundleException Unable to create the entry because
	 *         the import file contents are invalid (missing or inaccessible
	 *         files, corrupt contents)
	 * @throws IOException Unable to create import storage area or unable to
	 *         read one or more of the imported files.
	 */
	private ProbeRegistryEntry createEntryFromImportFile(String importFileName)
		throws IOException, InvalidProbeBundleException
	{
		trace("Adding from import file " + importFileName);

		Path sourcePath = new Path(importFileName);
		String basename = sourcePath.removeFileExtension().lastSegment();
		
		// use simple filename if we can
		File destDir = importAreaPath.append(basename).toFile();	
		synchronized(theRegistry) {
			if ( !destDir.exists() ) {
				destDir.mkdirs();
			} else {			
				// It's not that easy. Add a timestamp.
				basename += "-" + timestamper.format(new Date());
				destDir = importAreaPath.append(basename).toFile();
				while (destDir.exists()) {
					// Keep adding incremental counter until we hit an unused name.
					// In practice, shouldn't happen at all, let alone more than once 
					// through the loop since it means a timestamp collision.
					destDir = importAreaPath.append(basename + idCounter++).toFile();
				}
				destDir.mkdirs();
			}
		}
		basename = null;
		
		// Extract jar file contents
		JarReader reader = new JarReader(sourcePath.toFile());
		reader.extractAll(destDir);
		reader.close();
		reader = null;
		
		ProbeRegistryEntry result = null;
		try {
			result = createEntryFromDisk(destDir);
		} catch (InvalidProbeBundleException e) {
			// remove the temporary storage we created
			deleteEntryStorage(destDir);
			throw e;
		}
		return result;
	}
	
	/**
	 * Synthesize a registry entry from a directory which contains only
	 * the component files. This directory is usually the result of unzipping
	 * an imported probe set or a directory in the registry managed probestore.
	 * The minimum components for a probe set are: a model file (.probe),
	 * probescript, and at least one other (class) file.
	 * 
	 * @param srcDir The directory containing probe set components
	 * @return The newly created registry entry
	 * @throws InvalidProbeBundleException Import file contents are not well-formed.
	 * 			One or more required files are missing or inaccessible. Or there
	 * 			are too many probe, probeinfo or probescript files.
	 * @throws IOException An error occured while reading or copying the contents
	 * 			of the import file.
	 */
	private static ProbeRegistryEntry createEntryFromDisk(File srcDir) 
		throws InvalidProbeBundleException, IOException
	{
		// Create a bundle and use it to construct an entry. The ctor for
		// ProbeRegistryEntry will validate the completeness of the bundle.
		ProbeFileBundle bundle;
		ProbeRegistryEntry entry;
		bundle = new ProbeFileBundle(srcDir);
		entry = new ProbeRegistryEntry(bundle);
		
		// Synthesize an id if one isn't present in the persisted model.
		String id = entry.getProbekit().getId();
		if ( id == null || id.length() == 0 ) {
			// No id in model file. Synthesize one based on the model filename
			File modelFile = bundle.getModelFile();
			Path path = new Path(modelFile.getName());
			String basename = path.removeFileExtension().toString();
			entry.setId(basename);
		}
		
		if ( ProbekitDebugConfig.TRACE_REGISTRY ) {
			trace("Adding " + srcDir);
			System.out.println("   " + bundle.getModelFile());
			System.out.println("   " + bundle.getScript());
			File[] auxFiles = bundle.getSupporting();
			for (int i = 0; i < auxFiles.length; i++ ) {
				if ( auxFiles[i] != null ) {
					System.out.println("   " + auxFiles[i]);
				} else {
					System.out.println("   !!! NULL !!!");
				}
			}
		}
		
		return entry;
	}

	/**
	 * Add a probe set to the registry from a file previously created
	 * by an export operation. If a registry entry for this probe set
	 * is already present, it will be replaced by the contents of
	 * the new file. 
	 * 
	 * <p>Open the import file, extract the pieces to a probe
	 * registry managed location. Perform as much validation of
	 * the file contents as possible. Successful completion of
	 * this operation makes the probe set available for deployment.</p>
	 * 
	 * @param importFileName Reference to a file created by exporting a probe set.
	 * @return the newly created registry entry
	 * @throws IOException Unable to read or extract the import file contents
	 * @throws InvalidProbeBundleException The import file is mal-formed. That is,
	 *     it does not contain the expected files.
	 */
	public ProbeRegistryEntry add(String importFileName)
		throws IOException, InvalidProbeBundleException
	{
		ProbeRegistryEntry newEntry = createEntryFromImportFile(importFileName);
		try {
			add(newEntry);
		} catch (InvalidProbeBundleException e) {
			// whoops, better clean up the storage area
			deleteEntryStorage(newEntry.getLocation());
			throw e;
		}

		return newEntry;
	}
	
	/**
	 * Add a probe set to the registry by supplying all the required pieces in
	 * a bundle. This is usually called by the ProbeBundler after a build. 
	 * If a registry entry for this probe set is already present, it will be 
	 * replaced by the contents of the bundle.
	 * 
	 * @param bundle A complete probe resource bundle, including persisted model
	 * 		file, probescript, and one or more supporting class files.
	 * @return The newly created registry entry.
	 * @throws ProbeRegistryException The probe bundle is incomplete or otherwise
	 * 		badly formed.
	 */	
	public ProbeRegistryEntry add(ProbeResourceBundle bundle)
		throws ProbeRegistryException, InvalidProbeBundleException
	{
		ProbeRegistryEntry newEntry = new ProbeRegistryEntry(bundle);
		String id = newEntry.getProbekit().getId();
		if ( id == null || id.length() == 0 ) {
			// No id in model file. Synthesize one based on the model filename
			String basename = bundle.getSource().getFullPath().removeFileExtension().lastSegment();
			newEntry.setId(basename);
		}
		add(newEntry);
		
		return newEntry;		
	}
	
	/**
	 * Add a previously created entry to the registry, after first removing
	 * any conflicting entries.
	 * 
	 * @param newEntry The entry to add to the registry's database
	 * @return a reference to the newly added entry
	 */
	private synchronized ProbeRegistryEntry add(ProbeRegistryEntry newEntry)
		throws InvalidProbeBundleException
	{
		IResource source = null;
		String id = newEntry.getId();
		
		trace("Adding entry for " + id);		
		if ( newEntry.isAuthored() ) {
			source = newEntry.getSource();
		}

		removeConflictingEntries(newEntry);
		store.put(id, newEntry);
		if ( newEntry.isAuthored() && source != null ) {
			srcMap.put(source, newEntry);
		}
		
		return newEntry;
	}
		
	/**
	 * Restore a probe set from saved registry state. This is usually
	 * invoked while restoring the registry at startup, so the input
	 * parameters come from the registry state file - see restore().
	 * 
	 * <p>This operation fails silently. That is, if the entry cannot be
	 * restored for any reason, we simply add nothing to the registry. If
	 * the failed probe set is not authored, we also remove it from the
	 * persistent store on failure.</p>
	 * 
	 * @param probeFilename Either the probe source file name (authored probe set) or
	 * 		the persistent store simple subdir name where the targetted
	 * 		probe set resides. For an authored probe, this is a full path.
	 * 		For an imported probe set, this is a simple name ("foo"), not an
	 * 		absolute or true relative pathname (".\foo" or "/wherever/foo").
	 * @param id The probe set id
	 * @param authored True if this an authored probe set, false otherwise.
	 * @return The newly created entry, or null if restoration failed.
	 */
	private ProbeRegistryEntry restoreElement(String probeFilename, String id, boolean authored) 
	{
		ProbeRegistryEntry entry = null;
		if ( !authored ) {		
			// not authored, restore from registry managed storage
			File inputDir = importAreaPath.append(probeFilename).toFile();		
			if ( inputDir.exists() ) {
				try {
					entry = createEntryFromDisk(inputDir);
					entry.setId(id);
					add(entry);
				} catch (Exception e) {
					// This entry has become invalid. Remove it from persistent store.
					trace("Unable to restore registry entry for " + id);	//$NON-NLS-1$
					deleteEntryStorage(inputDir);
					entry = null;
				} 
			}
		} else {
			// Authored, try to access related resources. We previously saved
			// the Path to the source (.probe) resource and previously attached
			// info about the other, related files as persistent properties on
			// this resource.
			try {
				ProbeResourceBundle bundle = new ProbeResourceBundle(probeFilename);
				entry = add(bundle);
			} catch (Exception e) {
				trace("Unable to restore registry entry for " + id);	//$NON-NLS-1$
				entry = null;
			}
		}
		return entry;
	}
	
	/**
	 * Scrub the registry of any entries which conflict with <i>newEntry</i>.
	 * Note that <i>newEntry</i> should not already be in the registry. After
	 * this operation, the registry will contain no entries that match
	 * <i>newEntry</i> in either the source or id maps.
	 * 
	 * @param newEntry The entry whose conflicts are to be removed.
	 */
	private void removeConflictingEntries(ProbeRegistryEntry newEntry)
	{	
		String id = newEntry.getId();
		ProbeRegistryEntry matchedById = null;
		if ( id != null ) {
			matchedById = (ProbeRegistryEntry)store.get(id);
		}
		
		IResource source = null;
		ProbeRegistryEntry matchedBySrc = null;
		if ( newEntry.isAuthored() ) {
			try {
				source = newEntry.getSource();
				matchedBySrc = (ProbeRegistryEntry) srcMap.get(source);
			} catch (InvalidProbeBundleException e) {
				matchedBySrc = null;
				return;	// newEntry is invalid, so don't remove anything
			}
		}
		
		if ( matchedById != null ) {
			// Found an entry with the same id in id map. Remove it.
			remove(matchedById);
		} else if ( matchedBySrc != null && matchedById != matchedBySrc ) {
			// Found entry w same source in source map, but either it is not
			// the same as the matching id map entry or there is no id
			// map match. (The latter probably means the author changed
			// the id).
			remove(matchedBySrc);
		}
	}

	/**
	 * Scratch an entry from the registry managed storage area. This just
	 * cleans out the on-disk storage; it does not clean up the source map
	 * or other hash tables containing an entry.
	 * 
	 * <p>It is not guaranteed that the associated files and directory will
	 * be removed - permissions, non-empty directories, etc. may prevent
	 * complete cleanup.
	 * @param location the directory to remove from the registry disk cache
	 */
	private static void deleteEntryStorage(File location) {
		if ( location != null && location.isDirectory() && location.exists() ) {
			File[] contents = location.listFiles();
			for (int i = 0; i < contents.length; i++) {
				if ( contents[i] != null && contents[i].exists() ) {
					contents[i].delete();
				}
			}
			location.delete();
		}
	}

	/**
	 * Remove an entry from the probe registry. If <i>entry</i>
	 * is not in the registry, this call does nothing. If <i>entry</i>
	 * is an imported probe set, its probescript & class files will
	 * also be removed from the registry managed on-disk storage.
	 * 
	 * @param entry registry entry to remove
	 */
	public synchronized void remove(ProbeRegistryEntry entry)
	{
		if ( entry == null ) 
			return;
		
		if ( entry.isAuthored() ) {
			// Authored, so clean up source map
			IResource srcTmp = entry.getSourceUnchecked();
			if ( srcTmp != null ) {
				srcMap.remove(srcTmp);
			}
		} else {
			// Imported, not authored. Clean up registry managed storage.
			deleteEntryStorage(entry.getLocation());
		}
		String id = entry.getId();
		trace("Removing entry for " + id);		//$NON-NLS-1$
		store.remove(id);
	}
	
	/**
	 * Remove the entry corresponding to a particular id from the registry.
	 * 
	 * @param id Probe set identifier of the entry to remove
	 */
	public synchronized void remove(String id)
	{
		if ( id == null ) {
			return;
		}
		
		ProbeRegistryEntry entry = (ProbeRegistryEntry)store.remove(id);
		if ( entry != null ) {
			if ( entry.isAuthored() ) {
				// Authored, so clean up source map
				IResource src = entry.getSourceUnchecked();
				if ( src != null ) {
					srcMap.remove(src);
				}
			} else {
				// Imported, not authored. Clean up registry managed storage.
				deleteEntryStorage(entry.getLocation());
			}
		}
	}
	
	/**
	 * Remove the entry corresponding to a particular .probe source file. 
     * If there there is no entry for this Resource, the request is ignored.
	 * 
	 * @param source The probe source Resource of registry entry to be removed.
	 */
	public synchronized void remove(IResource source)
	{
		if ( source == null ) {
			return;
		}
				
		ProbeRegistryEntry entry = (ProbeRegistryEntry)srcMap.remove(source);
		if ( entry != null ) {
			store.remove(entry.getId());
		}
		// No need to cleanup on-disk registry storage because we know this
		// is an authored probe set and we don't manage the storage.
	}
	
	/**
	 * Create an iterator over the registry contents. The order is not defined.
	 * The resulting enumerator is read-only and cannot be used to modify the
	 * registry contents. If the registry contents change during the lifetime
	 * of the enumerator, the changes will not be reflected in the enumerated set.
	 * This method is equivalent to the iterator() method, except that it
	 * returns an Enumeration instead of an Iterator.
	 * 
	 * @return an enumeration that can be used to walk the registry contents.
	 */
	public synchronized Enumeration contents()
	{
		return new Enumerator();
	}
	
	/**
	 * Create an iterator over the registry contents. The order is not defined.
	 * The resulting iterator is read-only and cannot be used to modify the
	 * registry contents. If the registry contents change during the lifetime
	 * of the enumerator, the changes will not be reflected in the enumerated set.
	 * This method is equivalent to the contents() method, except that it 
	 * returns an Iterator instead of an Enumeration.
	 * 
	 * @return an iterator that can be used to walk the registry contents.
	 */
	public synchronized Iterator iterator()
	{
		return new Enumerator();
	}
	
	/**
	 * Check a registry entry for validity. If it is invalid, remove it from
	 * the registry. This operation performs the cheap quickValidate check,
	 * rather than the more costly fullValidate().
	 * 
	 * @param entry The entry to validate
	 * @return <i>entry</i> or null, if <i>entry</i> is invalid
	 */
	private synchronized ProbeRegistryEntry validateEntry(ProbeRegistryEntry entry)
	{
		if ( entry != null ) {
			try {
				entry.quickValidate();
				return entry;
			} catch (InvalidProbeBundleException e) {
				remove(entry);
			}
		}
		return null;
	}
	
	/**
	 * Locate the entry corresponding to a particular id.
	 * 
	 * @param id Probe set identifier of the entry to locate
	 * @return the matching entry, if found; null if not found.
	 */
	public ProbeRegistryEntry lookupById(String id)
	{
		if ( id != null ) {
			ProbeRegistryEntry entry = (ProbeRegistryEntry)store.get(id);
			return validateEntry(entry);
		} else {
			return null;
		}
	}
		
	/**
	 * Retrieve a probe set based on its source (.probe) file reference. This
	 * query is really only meaningful on authored probes in the workspace.
	 * Imported probes in the registry save area are not resources (and do not
	 * necessarily have source).
	 * 
	 * @param source The probe set source file, as an IPath.
	 * @return The matching entry, or null if not found.
	 */
	public ProbeRegistryEntry lookupBySource(IResource source)
	{
		if ( source != null ) {
			ProbeRegistryEntry entry = (ProbeRegistryEntry)srcMap.get(source);
			return validateEntry(entry);
		} else {
			return null;
		}	
	}
	
	/**
	 * Attempt to import a probe set into the registry. A tentative or
	 * "candidate" registry entry is created and decorated with enough 
	 * information to tell the caller whether or not there is a conflicting
	 * entry in the registry. If there is no conflict, the entry is
	 * automatically added to the registry. Otherwise, the caller should
	 * either commit this candidate entry to the registry or explicitly 
	 * discard it.
	 * 
	 * <p>A conflicting entry is any one with the same probe set id. It is up
	 * to the caller to decide whether or not to proceed with the import
	 * (and thereby overwrite the existing entry). See the commit() and
	 * discard() operations.</p>
	 * 
	 * <p>When there is a conflict, a lock is placed on the conflicting
	 * entry. The lock is released when the candidate is either
	 * committed or discarded. While the conflicting entry is locked, no
	 * other thread can successfully pre-flight a probe set with the same
	 * id or explicitly add() a conflicting entry. This ensures the state
	 * of the contentious probe set does not change between preflight()
	 * and commit()/discard().</p>
	 * 
	 * @param importFileName A previously exported probe set package
	 * @return A candidate registry entry that can be queried for conflict resolution.
	 * @throws IOException Unable to read/extract the import file
	 * @throws InvalidProbeBundleException import file is badly formed. That is, it does
	 *     not contain the expected files.
	 */
	public CandidateEntry addIfNoConflict(String importFileName) 
		throws IOException, InvalidProbeBundleException
	{
		ProbeRegistryEntry newEntry = createEntryFromImportFile(importFileName);
		ProbeRegistryEntry incumbent;
		
		synchronized(this) {
			incumbent = lookupById(newEntry.getId());
			
			if ( incumbent == null ) {
				add(newEntry);
			} else {
				trace("probe in " + importFileName + " conflicts with existing entry");
			}
		}
		return new CandidateEntry(newEntry, incumbent);
	}
	
	/**
	 * Commit a candidate entry to the registry. No conflict resolution is
	 * performed. That is, the conflicting entry (if any) is always overwritten.
	 * This operation is a no-op if there was no conflict to begin with
	 * because addIfNoConflict will already have committed the candidate.
	 * 
	 * <p>Once this operation succeeded, <i>candidate</i> is invalid
	 * and should not be committed or discarded again.</p>
	 * 
	 * @param candidate A candidate registry entry, previously created by preflight()
	 * @throws ProbeRegistryException <i>candidate</i> was previously committed or discarded.
	 */
	public synchronized void commit(CandidateEntry candidate) 
		throws ProbeRegistryException, InvalidProbeBundleException
	{
		if ( candidate.isDirty() ) {
			// this entry has already been committed or discarded
			throw new ProbeRegistryException("Entry already committed or discarded");	//$NON-NLS-1$
		}
		
		if ( candidate.getConflict() != null ) {
			ProbeRegistryEntry newEntry = candidate.getCandidate();
			add(newEntry);
		}
		// else, already added by addIfNoConflict()

		candidate.markDirty();
	}
	
	/**
	 * Throw away a candidate entry. The registry will clean up any temporary
	 * files created by the preflight() operation and release all associated locks.
	 * If there is no conflict, addIfNoConflict() already added it to the registry
	 * and you cannot now meaningfully discard it, so an exception is thrown.
	 * 
	 * <p>Once this operation has succeeded, the candidate entry is invalid and
	 * cannot committed or discarded again.</p>
	 *  
	 * @param candidate The candidate registry entry to discard.
	 * @throws ProbeRegistryException <i>candidate</i>was previously committed or discarded.
	 */
	public synchronized void discard(CandidateEntry candidate) 
		throws ProbeRegistryException
	{
		if ( candidate.isDirty() || candidate.getConflict() == null ) {
			throw new ProbeRegistryException("Candidate registry entry has already been committed or discarded.");
		}
		trace("Discarding candidate entry for " + candidate.getCandidate().getId());
		candidate.markDirty();
		deleteEntryStorage(candidate.getCandidate().getLocation());
	}
	
	/**
	 * Delete entries marked for deferred deletion.
	 * 
	 */
	protected synchronized void purgeFreeList()
	{
		for (int i = 0; i < freeList.size(); i++) {
			ProbeRegistryEntry entry = (ProbeRegistryEntry)freeList.get(i);
			remove(entry);
		}
	}

	/**
	 * Convert the registry contents into XML. See RegistryReader for a
	 * description of the format.
	 * 
	 * @param doc The document which receives the generated XML
	 * @return The XML element representing the root of the registry.
	 */
	private Element toXML(Document doc)
	{
		Element registry = doc.createElement(RegistryConstants.REGISTRY_TAG);
		registry.setAttribute(RegistryConstants.COUNTER_TAG, Integer.toString(idCounter));
		registry.setAttribute(RegistryConstants.VERSION_TAG, CURRENT_VERSION);

		Enumeration entries = contents();
		// if there are no contents
		if(entries.hasMoreElements() == false) {
			return null;
		}
		while (entries.hasMoreElements()) {
			ProbeRegistryEntry entry = (ProbeRegistryEntry)entries.nextElement();
			Element entryElem = entry.toXML(doc);
			if ( entryElem != null ) {
				registry.appendChild(entryElem);
			}
		}
		return registry;
	}

	/**
	 * This is a SAX-based XML parser handler for restoring saved registry
	 * state. It is invoked by the restore() method and contains SAX parser
	 * event callbacks for handling registry and entry tags. The format of
	 * the state save file is:
	 * 
	 * <registry counter=idCounter version=string>
	 *    <entry id=string authored=true|false>
	 *    source path
	 *    </entry>
	 *    <entry>...</entry>
	 *    ...
	 * </registry>
	 * 
	 * (In practice, all attribute values are strings, regardless what the
	 * represent).
	 * 
	 * The idCounter is stored so that the registry maintains an ongoing
	 * unique id counter.
	 * 
	 * Depending on whether or not an entry represents an authored probe set,
	 * the text inside an entry represents either a simple folder name inside
	 * the registry's persistent store area (non-authored), or the probe set
	 * source file absolute path (authored).
	 * 
	 * @author kcoleman
	 */
	private class RegistryReader extends DefaultHandler {
		protected final StringBuffer charBuffer = new StringBuffer();
		protected String entryId = null;
		protected boolean entryAuthored = false;
		
		public RegistryReader() {}
		
		/* (non-Javadoc)
		 * @see org.xml.sax.ContentHandler#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
		 * 
		 * We only care about 2 elements: <registry> and <entry>. Anything else 
		 * is garbage and we should probably bellow about it, but currently, 
		 * this is very lazy.
		 * 
		 * The registry element contains only 1 interesting piece of information: 
		 * The idCounter initial value. The version will get to be interesting 
		 * eventually, but currently is ignored.
		 * 
		 * For entry elements, we save off the id and authored attributes and 
		 * retrieve them later, after we've seen the entire element. 
		 * See endElement().
		 */
		public void startElement(String uri, String elementName, String qname, Attributes attributes) 
			throws SAXException 
		{
			charBuffer.setLength(0);
			String tag = elementName.trim();
			
			if ( tag.equalsIgnoreCase(RegistryConstants.REGISTRY_TAG) ) {
				String counterStr = attributes.getValue(RegistryConstants.COUNTER_TAG);
				if ( counterStr != null && counterStr.trim().length() != 0 ) {
					try {
						idCounter = Integer.parseInt(counterStr);
					} catch (NumberFormatException e) {
						idCounter = 0;
					}
				}
				entryId = null;
				entryAuthored = false;
			} else if ( tag.equalsIgnoreCase(RegistryConstants.ENTRY_TAG) ) {
				charBuffer.setLength(0);
				entryAuthored = false;
				String authStr = attributes.getValue(RegistryConstants.AUTHORED_TAG);
				if ( authStr != null && authStr.trim().length() != 0 ) {
					entryAuthored = Boolean.valueOf(authStr.trim()).booleanValue();
				}
				
				entryId = attributes.getValue(RegistryConstants.ID_TAG);
				if ( entryId != null ) {
					entryId = entryId.trim();
				}
			}
		}
		
		/* (non-Javadoc)
		 * @see org.xml.sax.ContentHandler#characters(char[], int, int)
		 * 
		 * For a well-formed registry save file, this should only be invoked for
		 * the text that represents the source/location of an entry. Just append
		 * it to a buffer, for use later by endElement().
		 */
		public void characters(char[] chars, int offset, int length)
				throws SAXException 
		{
			charBuffer.append(chars, offset, length);
		}
		
		/* (non-Javadoc)
		 * @see org.xml.sax.ContentHandler#endElement(java.lang.String, java.lang.String, java.lang.String)
		 * 
		 * When we get to the end of an <i>entry</i> element, attempt to add the
		 * associated entry to the registry, using the previously accumulated
		 * body parts. Badly formed probe sets are silently discarded.
		 */
		public void endElement(String uri, String elementName, String qname)
				throws SAXException 
		{
			String tag = elementName.trim();
			if ( tag.equalsIgnoreCase(RegistryConstants.ENTRY_TAG) && 
					charBuffer != null ) {
				String src = charBuffer.toString().trim();
				if ( src.length() != 0 ) {
					theRegistry.restoreElement(src, entryId, entryAuthored);
				}
			}
		}
	}	// class RegistryReader
	
	/**
	 * This class encapsulates a read-only Enumeration/Iterator over the 
	 * contents of the registry. To maintain consistent state for the consumer,
	 * the set of entries over which the enumerator operates is fixed at the
	 * time of creation. That is, once the Enumerator is created, it will not
	 * reflect bubsequent additions to or deletions from the registry.
	 * 
	 * All entries in the enumerated set are guaranteed to be valid at the
	 * time of iterator creation.
	 * 
	 * Removal of entries from the registry through an iterator is not
	 * supported. That is, the remove operation will throw an
	 * UnsupportedOperation Exception.
	 * 
	 * @author kcoleman
	 */
	protected class Enumerator implements Enumeration, Iterator 
	{
		Iterator iter;
		ProbeRegistryEntry next_elem;
		ArrayList list;
		
		public Enumerator()
		{
			ProbeRegistryEntry nextElem;
			
			list = new ArrayList(store.size());
			
			Enumeration elems = store.elements();
			while ( elems.hasMoreElements() ) {
				nextElem = (ProbeRegistryEntry)elems.nextElement();
				try {
					nextElem.quickValidate();
					list.add(nextElem);
				} catch (InvalidProbeBundleException e) {
					ProbeRegistry.this.remove(nextElem);
				}
			}
			iter = list.iterator();
		}
		
		/* (non-Javadoc)
		 * @see java.util.Iterator#hasNext()
		 * hasNext() always leaves the logical next element in next_elem.
		 */
		public boolean hasNext() 
		{
			return iter.hasNext();
		}

		/* (non-Javadoc)
		 * @see java.util.Iterator#next()
		 */
		public Object next() 
		{
			return iter.next();
		}

		/* (non-Javadoc)
		 * @see java.util.Iterator#remove()
		 */
		public void remove() {
			throw new UnsupportedOperationException();
		}

		/* (non-Javadoc)
		 * @see java.util.Enumeration#hasMoreElements()
		 */
		public boolean hasMoreElements() {
			return iter.hasNext();
		}

		/* (non-Javadoc)
		 * @see java.util.Enumeration#nextElement()
		 */
		public Object nextElement() {
			return iter.next();
		}
	} // Enumerator
}

