/********************************************************************** 
 * Copyright (c) 2004, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: ProbeRegistryEntry.java,v 1.5 2008/12/12 22:21:32 jcayne Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 

package org.eclipse.tptp.platform.probekit.registry;

/**
 * A ProbeRegistryEntry is the unit of storage in the ProbeRegistry.
 * An entry encapsulates everything you need to deploy a probe set
 * except end-user settable configurations, such as filters.
 * 
 * <p>There are two flavors of entry: Authored and imported. An authored
 * probe is one whose source file is part of a project in the workspace.
 * It is under active development and all the files are managed by the
 * normal workbench project framework.</p>
 * 
 * <p>An imported probe is the result of importing a previously exported
 * probe set. This usually takes the form of a zip containing all the
 * required pieces. The registry manages the storage for imported probe
 * sets. See ProbeRegistry for more details</p>
 * 
 * <p>There is a ProbeRegistryEntry constructor for each of these two
 * flavors. The key difference is in how the related files are managed.
 * Imported probe sets have a source File; authored probes have a source
 * Resource. This affects how we persist an entry and how we detect
 * invalid entries.</p>
 * 
 * <p>An entry becomes invalid if any of its constituent files become
 * unavailable. For an imported probe set, this just means the files in the
 * registry on-disk cache are removed or become unreadable. But an authored
 * probe set can also become invalid if the enclosing project is closed or
 * deleted. The files may still be on disk, but the probe is no longer usable
 * in the semantic model.</p>
 * 
 * <p>We cannot reliably tell when entries become invalidated. We cannot
 * prevent users from removing key files from the file system, outside the
 * workbench. Currently we also do not receive registration of project
 * closure or deletion. So, we re-check validity whenever someone pulls on 
 * an entry. Not terribly efficient, but it should have the desired effect.</p> 
 * 
 * @author kcoleman
 */

import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.internal.probekit.Label;
import org.eclipse.hyades.models.internal.probekit.Probe;
import org.eclipse.hyades.models.internal.probekit.Probekit;
import org.eclipse.tptp.platform.probekit.util.InvalidProbeBundleException;
import org.eclipse.tptp.platform.probekit.util.ProbeFileBundle;
import org.eclipse.tptp.platform.probekit.util.ProbeResourceBundle;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.ibm.icu.util.ULocale;


public class ProbeRegistryEntry {	
	/**
	 * A handle on this probe set as an EMF model. This is where we get
	 * information like the name, description, and version.
	 */
	private Probekit probekit = null;	
	/**
	 * TRUE if this entry is valid, FALSE otherwise. Entries become invalid
	 * if one or more of their required files becomes unavailable, such as
	 * the probe model file or the related class files.
	 */
	private boolean valid = true;
	/**
	 * List of files required for probe set deployment, in addition to the
	 * probescript. This always includes one or more class files, but could
	 * include other files - the nature of these files is opaque to the
	 * registry. The only constraint is that there must be atleast one of
	 * these extra files.
	 */
	private File componentFiles[];
	/**
	 * The probescript file generated by the probe compiler. A separate handle
	 * is kept to this file just to make life easier for the deployment system.
	 * It saves the launcher from rummaging through the component files list.
	 */
	private File probescript = null;
	/**
	 * The unique identifier by which a probe set is known to the registry.
	 * The registry never contains more than one probe set with the same id.
	 * This field is only filled in if the model (<i>probekit</i>) does not
	 * include an explicit id. If the model includes an id, it is always
	 * used and cannot be overridden by calling setId().
	 */
	private String id = null;
	/**
	 * The location of registry managed storage for an imported probe set.
	 * This field is always null for an authored probe set. For an imported
	 * probe, it references the sub-dir of the registry disk cache which
	 * contains all files associated with this entry.
	 */
	private File location = null;
	/**
	 * The Resource which represents a probe source file being authored in
	 * the workspace. This field is always null for an imported probe set.
	 * For an authored probe, this resource represents the .probe file.
	 */
	private IResource source = null;
	/**
	 * TRUE if this is an authored probe in the workspace, FALSE if this is
	 * an imported probe set. The setting of this field is based solely on
	 * how this entry was created.
	 */
	private boolean authored = false;
	/**
	 * Cache for the locale-specific label (name/desription) for this
	 * probe set. The value is cached on demand when getName() or
	 * getDescription() is called. This is purely a performance optimization
	 * since scrouging around for the best locale match is expensive. It
	 * is based on the assumption that the locale does not subsequently change.
	 */
	private Label label = null;
	
	private InvalidProbeBundleException invalidationEx = null;
	
	/**
	 * Create a registry entry from a probe bundle. The resulting entry includes
	 * everything needed to deploy a probe set, so the bundle must be complete.
	 * Authored entries have a source resource (e.g. .probe file), while
	 * imported entries have a registry storage location. This information is
	 * carried on the probe entry to assist with registry save/restore operations.
	 * 
	 * @param bundle A complete, well-formed probe bundle
	 * @throws InvalidProbeBundleException The bundle is not complete or is
	 * 		otherwise corrupted (e.g. cannot instantiate a the model or 
	 * 		cannot access some of the files/resources).
	 * @throws InvalidProbeBundleException
	 */
	ProbeRegistryEntry(ProbeFileBundle bundle)
		throws InvalidProbeBundleException
	{
		super();
		bundle.validate();		// throws ex if bad bundle
		
		// Try to instantiate the model
		probekit = null;
		try {
			probekit = bundle.instantiateModel();
		} finally {
			if (probekit == null) {
				throw new InvalidProbeBundleException(
						InvalidProbeBundleException.INVALID_MODEL, 
						bundle.getModelFile());
			}
		}
		
		// Get the probescript
		probescript = bundle.getScript();
		
		// deal with the class and other auxiliary files
		componentFiles = bundle.getSupporting();
		
		// set up source/location, dependent on authoring attribute.
		this.authored = false;
		// Strip the enclosing directory off the source info.
		this.location = bundle.getModelFile().getParentFile();
	}
		
	/**
	 * Create a registry entry from a probe bundle. The resulting entry includes
	 * everything needed to deploy a probe set, so the bundle must be complete.
	 * Authored entries have a source resource (e.g. .probe file), while
	 * imported entries have a registry storage location. This information is
	 * carried on the probe entry to assist with registry save/restore operations.
	 * 
	 * @param bundle A complete, well-formed probe bundle
	 * @throws InvalidProbeBundleException The bundle is not complete or is
	 * 		otherwise corrupted (e.g. cannot instantiate a the model or 
	 * 		cannot access some of the files/resources).
	 */
	ProbeRegistryEntry(ProbeResourceBundle bundle)
		throws InvalidProbeBundleException
	{
		super();
		bundle.validate();	// throws ex if bad
		
		// Try to instantiate the model
		probekit = null;
		try {
			probekit = bundle.instantiateModel();
		} finally {
			if (probekit == null) {
				throw new InvalidProbeBundleException(
						InvalidProbeBundleException.INVALID_MODEL, 
						bundle.getModelResource());
			}
		}
		
		// Get the probescript
		probescript = bundle.getScriptFile();
		
		// deal with the class and other auxiliary files
		componentFiles = bundle.getSupportingFiles();
		
		// set up source/location, dependent on authoring attribute.
		this.authored = true;
		source = bundle.getSource();
	}
		
	/**
	 * Retrieve an instance of the probekit model for this entry. Note that 
	 * this may not be a fully populated instance of the model. For example,  
	 * the probe fragments may be unavailable.
	 * @return An instance of the probekit model for this probe set.
	 */
	public Probekit getProbekit() 
	{
		return probekit;
	}
		
	/**
	 * Retrieve the list of labels for this probe set. The list may be
	 * empty. It is up to the caller to select the description corresponding
	 * to the current locale.
	 *
	 * @return The (possibly empty) set of locale-specific labels of this probe set.
	 *   The list element type is determined by the model, but should be 
	 *   org.eclipse.hyades.models.probekit.Label.
	 * @see org.eclipse.hyades.models.internal.probekit
	 */
	public EList getLabels() 
	{
		return probekit.getLabel();
	}
	
	/**
	 * Retrieve the locale-specific name of this probe set. If there is no match
	 * for the current locale, the default specified by the author will be
	 * returned. Note that this may not match the current locale and so may
	 * not be meaningfully renderable. If the probe set is unnamed, the
	 * id will be returned.
	 * 
	 * @return The name of this probe set, or the id if no name is available.
	 */
	public String getName()
	{
		if ( label == null ) {
			label = getLocalizedLabel(ULocale.getDefault());
		}
		return (label == null ? getId() : label.getName());
	}
	
	/**
	 * Retrieve the locale-specific description of this probe set, if available.
	 * If there is no description matching the current locale, the author
	 * specified default description will be returned. Note that since this
	 * may not match the current locale, it may not be meaningfully renderable.
	 * If the probe set has no description, null is returned. 
	 * 
	 * @return The description of this probe set, or null if there is no description.
	 */
	public String getDescription()
	{
		if ( label == null ) {
			label = getLocalizedLabel(ULocale.getDefault());
		}
		return (label == null ? null : label.getDescription());
	}
	
	/**
	 * Pick off the country prefix at the beginning of an xml:lang string value.
	 * These strings have the form:
	 * 		country{-subset}*
	 * That is, a country specifier followed by 0 or more subclassifications. The
	 * country and subset tags are always alphanumeric.
	 * 
	 * @param lang A legal xml:lang string value
	 * @return The country (or first) tag in <i>lang</i>
	 */
	private static String getCountry(String lang)
	{
		int end = lang.indexOf('-');
		
		return (end > 0) ? lang.substring(0,end) : lang;
	}
	
	private static String[] buildNLVariants(String nl) {
		ArrayList result = new ArrayList();
		int lastSeparator;
		while ((lastSeparator = nl.lastIndexOf('_')) != -1) {
			result.add(nl);
			if (lastSeparator != -1) {
				nl = nl.substring(0, lastSeparator);
			}
		}
		result.add(nl);
		// always add the default locale string
		result.add(""); //$NON-NLS-1$
		return (String[]) result.toArray(new String[result.size()]);
	}

	/**
	 * Create a Java Locale object from a language string that came out of the
	 * probekit model (xml:lang format).
	 * 
	 * <p>We want to work through a Locale object rather than the language string
	 * so that we can control impedence mismatch, such as the use of dashes
	 * vs. underscores as separators and old vs. new country codes.</p>
	 * 
	 * <p>Variant (as in language-country-{variant*} are not parsed out into
	 * individual variants. This is consistent with the Locale ctors.</p>
	 * 
	 * @param input A language string in xml:lang format
	 * @return A Locale object representing labelLang.
	 */
	private static ULocale createLocaleFromXML(String input)
	{
	    final String PATTERN = "[-_]";	//$NON-NLS-1$
	    final int PIECES = 3;
	    
        String lang = ""; 				//$NON-NLS-1$
        String country = ""; 			//$NON-NLS-1$
        String variant = ""; 			//$NON-NLS-1$
        String[] result = input.split(PATTERN, PIECES);
        
        if ( result != null ) {
            lang = result[0];
            if ( result.length > 1 ) {
                country = result[1];
                if ( result.length > 2 ) {
                    // Normalize dashes to underscores so there's no conflict of
                    // syntax between XML vs. JVM.
                    variant = result[2].replace('-', '_');
                }
            }
        }
        return new ULocale(lang, country, variant);
	}
	/**
	 * Return the probe set's Label element that matches the locale specified 
	 * in the argument string. You really shouldn't call this from outside this
	 * package, use getName() and getDescription() instead; this method is
	 * public only to facilitate testing.
	 * 
	 * <p>In order of preference, this function returns:</p>
	 * <ol>
	 * <li>An exact match
	 * <li>The label which matches language and country, but not variant
	 * <li>A language-only label which exactly matches the target language. 
	 *     Example: If lang is "en-US", list contains no exact match, but does 
	 *     contain "en" entry.</li>
	 * <li>A language-country label with a matching language, but mismatched
	 *     country. Example: lang is "en-US", there is no "en-US" or "en" match, 
	 *     but there is a "en-GB" match.
	 * <li>The author-specified default label. Default is indicated by having an
	 *     empty lang on a Label.</li>
	 * <li>The first Label in the list.</li>
	 * <li>Null if there are no labels at all.</li>
	 * </ol>
	 * 
	 * @param targetLocale The locale for which to get a label.
	 * @return the Label for this locale, or a default as described above. 
	 * 		   Or null if there are none.
	 */
	public Label getLocalizedLabel(ULocale targetLocale)
	{
		EList labels = probekit.getLabel();
		Iterator iter = labels.iterator();
		Label defaultLabel = null;
		Label exactCountryMatch = null;
		Label langMatchCountryMismatch = null;
		Label langOnlyMatch = null;
		String lang = targetLocale.getLanguage();
		String country = targetLocale.getCountry();
		
		while (iter.hasNext()) {
			Object o = iter.next();
			if (o instanceof Label) {
				Label thisLabel = (Label) o;
				if (defaultLabel == null) {
					// Set default first time thru. This guarantees that if
					// there is only 1 name, we'll pick it up as default. The
					// probe compiler guarantees that if there is more than 1,
					// there must be an explicit default, which we'll pick up
					// below.
					defaultLabel = thisLabel;
				}
				
				String thisLabelLang = thisLabel.getLang();
				if (thisLabelLang == null || thisLabelLang.length() == 0) {
				    // Found explicit default. Save it in case we find no lang
				    // match.
				    defaultLabel = thisLabel;
				} else {
				    ULocale thisLabelLocale = createLocaleFromXML(thisLabelLang);
				    if ( thisLabelLocale.equals(targetLocale) ) {
				        // Exact match
				        return thisLabel;
				    } else if ( thisLabelLocale.getLanguage().compareToIgnoreCase(lang) == 0 ) {
				        // Language matches
				        String thisCountry = thisLabelLocale.getCountry();
				        if ( thisCountry.length() == 0 ) {
				            // This is the lang-only label. Will use it if we
				            // don't find a better match.
				            langOnlyMatch = thisLabel;
				        } else if ( thisCountry.compareToIgnoreCase(country) == 0 ) {
				            // Language and country match, but variant doesn't.
				            // (If variant had matched, we'd have found exact match)
				            exactCountryMatch = thisLabel;
				        } else {
				            // Language matches, but country doesn't.
				            langMatchCountryMismatch = thisLabel;
				        }
				    } // else, language doesn't match, so not a candidate.
				}
			}
		} // loop
		
		// No exact match was found. Pick the best fallback.
		if ( exactCountryMatch != null ) {
		    return exactCountryMatch;
		} else if ( langOnlyMatch != null ) {
		    return langOnlyMatch;
		} else if ( langMatchCountryMismatch != null ) {
		    return langMatchCountryMismatch;
		} else {
		    return defaultLabel;
		}
	}

	/**
	 * Retrieve the probe set's unique id. This id may have been
	 * specified by the author or assigned by the registry. It is
	 * guaranteed to be unique within this workspace. Any user of the
	 * probe registry should always use this method rather than calling
	 * getId directly on the model instance.
	 * @return the probe set identifier 
	 * @see org.eclipse.hyades.models.internal.probekit
	 */
	public String getId() 
	{
		return (id == null) ? probekit.getId() : id;
	}
		
	/**
	 * Explicitly set the probe set identifier. Normally, the id is taken
	 * from the model, but older model definitions did not include this
	 * attribute, so it may not be present. 
	 * 
	 * <p>If there is an id on the model, this operation does nothing.
	 * You cannot override the author-assigned id embedded in the model.</p>
	 * 
	 * @param newId The new probe set id.
	 */
	synchronized void setId(String newId)
	{
		if (newId != null && newId.length() > 0) {
			String builtinId = probekit.getId();
			if ( builtinId == null || builtinId.length() == 0) {
				// Don't let someone accidentally override what's in the model.
				id = newId;
			}
		}
	}
	
	/**
	 * Retrieve the (optional) version string associated with this probe set.
	 * @return the probe set version string, or null if there is no version info.
	 * @see org.eclipse.hyades.models.internal.probekit
	 */
	public String getVersion() 
	{
		return probekit.getVersion();
	}
	
	/**
	 * Retrieve the original source file for the probe set, if any. Source
	 * is only available if the entry came into the registry as part of 
	 * an authoring operation.
	 * @return the source (.probe) file for the probe set, or null. 
	 * @throws InvalidProbeBundleException The entry is unusable
	 */
	public IResource getSource() 
		throws InvalidProbeBundleException
	{
		quickValidate();
		return (authored == true ? source : null);
	}
	
	/**
	 * Retrieve the probe source file in the workspace, if any. Source is only
	 * available if the entry came into the registry as part of an authoring
	 * operation.
	 * 
	 * This unchecked call will not validate the accessibility of the source
	 * file and therefore will not notice an invalid entry. It is used by
	 * the registry when validity isn't interesting, such as when removing
	 * a (previously invalidated) entry.
	 * 
	 * @return The source (.probe) file, or null if there is no source.
	 */
	IResource getSourceUnchecked()
	{
		return (authored == true ? source : null);
	}
		
	/**
	 * Retrieve the files which are required for probe deployment, such as the
	 * probescript and class files generated by the build operation.
	 * @return list of files required for probe deployment
	 * @throws InvalidProbeBundleException This entry is not usable.
	 */
	public File[] getFiles() 
		throws InvalidProbeBundleException
	{
		fullValidate();
		return componentFiles;
	}
	
	/**
	 * Retrieve the probe compiler-generated probescript for this
	 * probe set. A valid entry always has a non-null probescript file.
	 * 
	 * @return This probe set's probescript file.
	 * @throws InvalidProbeBundleException This entry is not usable.
	 */
	public File getProbescript()
		throws InvalidProbeBundleException
	{
		fullValidate();
		return probescript;
	}
	
	/**
	 * Determine whether or not the probe set includes probes with targets.
	 * If it does, the consumer cannot re-target the probe set. 
	 * 
	 * @return True if at least one probe in the probeset includes a target spec.
	 * @see org.eclipse.hyades.models.internal.probekit.Probe
	 */
	public boolean hasTargets()
	{
		Iterator iter = probekit.getProbe().iterator();
		while (iter.hasNext()) {
			Object o = iter.next();
			if (o instanceof Probe) {
				Probe probes = (Probe) o;
				if (probes.getTarget().size() > 0) {
					return true;
				}
			}
		}
		return false;
	}
	
	/**
	 * Cheapest possible check for entry validity. This is not a thorough check
	 * and lets a number of possible problems through. If you want to be
	 * absolutely certain, use fullValidate().
	 * 
	 * In particular, this operation will only return false if the entry was
	 * previously invalidated or if the .probe source of an authored probe
	 * is inaccessible. 
	 * 
	 * Imported probes are assumed to be always OK, which might not be true if
	 * someone tinkers with the plugin storage area, but that's a pretty
	 * pathological condition. It is only checked for during startup.
	 * 
	 * Authored probes may be invalidated by closing the contain project, by
	 * deleting the probe source, or by deleting one or more of the derived
	 * resources (class files, probescript). The latter situation is not
	 * checked for by this operation.
	 * 
	 * @throws InvalidProbeBundleException The entry is invalid due to a missing 
	 * 		or invalid source resource or other problem.
	 */
	public void quickValidate() throws InvalidProbeBundleException
	{
		if ( !valid ) {
			throw invalidationEx;
		} else if ( !isAuthored() ) {
			// we don't check up on imports except when reloading from disk
		} else if ( source == null || source.isAccessible() ) {
			// There is no source or the source exists and is accessible in
			// the workspace. Source can only be null during initial creation.
			// It may be inaccessible if the contaiing project is closed.
		} else {
			// source exists, but is not accessible
			InvalidProbeBundleException ex =
				new InvalidProbeBundleException(
					InvalidProbeBundleException.INACCESSIBLE_MODEL_FILE, source);
			invalidate(ex);
		}
	}
	
	/**
	 * Determine whether or not a probe set can be deployed. Probe sets are
	 * invalidated if any of their required files are missing or invalid.
	 * This operation does everything quickValidate() does, plus it checks
	 * the existence and readability of all deployment related files.
	 * 
	 * @throws InvalidProbeBundleException One or more of the files or resources
	 * 		that comprise a well-formed entry are either missing or inaccessible.
	 */
	public void fullValidate() throws InvalidProbeBundleException
	{
		quickValidate();
		int reason = 0;
		File theFile = null;
		
		// check all the things the quickie skips over
		if ( !probescript.exists() ) {
			reason = InvalidProbeBundleException.MISSING_SCRIPT_FILE;
			theFile = probescript;
		}
		else if ( !probescript.canRead() ) {
			reason = InvalidProbeBundleException.INACCESSIBLE_SCRIPT_FILE;
			theFile = probescript;
		} else {
			// All of the other deployment related files must exist and be readable
			for (int i = 0; i < componentFiles.length; i++) {
				if ( !componentFiles[i].exists() ) {					
					reason = InvalidProbeBundleException.MISSING_SUPPORT_FILE;
				} else if (	!componentFiles[i].canRead() ) {
					reason = InvalidProbeBundleException.INACCESSIBLE_SUPPORT_FILE;
					theFile = componentFiles[i];
					break;
				}
			}
		}
		if ( reason != 0 ) {
			InvalidProbeBundleException ex = 
				new InvalidProbeBundleException(reason, theFile);
			invalidate(ex);
		}
	}
	
	/**
	 * Mark a registry entry as invalid. When invalidated, the entry is eligible  
	 * for garbage collection, it will not be returned by an enumerated walk of 
	 * the registry, and you cannot deploy it. 
	 */
	synchronized void invalidate(InvalidProbeBundleException ex)
		throws InvalidProbeBundleException
	{
		ProbeRegistry.trace("Invalidate ProbeRegistryEntry for " + 	//$NON-NLS-1$
				this.getId() + ", reason: " + ex.getReason()); 		//$NON-NLS-1$
		invalidationEx = ex;
		valid = false;
		throw ex;
	}
	
	/**
	 * @return True if this is a probe being authored in the workspace
	 */
	public boolean isAuthored()
	{
		return authored;
	}
		
	/**
	 * Retrieve the storage location of this entry. That is, where the .probescript
	 * and associated class files are stored. Only imported probe sets have a
	 * location. The storage for an authored probe set is not managed by the
	 * registry, so they have no location information.
	 * @return The storage location for this entry, or null if it is authored.
	 */
	File getLocation()
	{
		return ( !authored ? location : null );
	}
		
	/**
	 * Convert this entry into XML. A tag is generated of the form:
	 * <pre>
	 *    <entry authored={"true" | "false") id=idstr>
	 *    location_or_source_text_representation
	 *    </entry>
	 * </pre>
	 * <p>If this is an authored entry, the text node contains the stringified
	 * resource references. Otherwise, it is the storage location, relative to
	 * the registry store.</p>
	 * 
	 * @param doc
	 * @return An Element representing a ProbeRegistryEntry.
	 */
	Element toXML(Document doc)
	{
		Element entry = doc.createElement(RegistryConstants.ENTRY_TAG);
		entry.setAttribute(RegistryConstants.AUTHORED_TAG, isAuthored() ? "true" : "false"); //$NON-NLS-1$ //$NON-NLS-2$
		entry.setAttribute(RegistryConstants.ID_TAG, getId());
		
		try {
			String modelLocation = null;
			if ( !isAuthored() ) {
				// Just pick off the last folder in the registry save area
				modelLocation = getLocation().getCanonicalPath();
				Path srcPath = new Path(modelLocation);
				modelLocation = srcPath.lastSegment();
			} else {
				modelLocation = source.getFullPath().toString();
			}
			entry.appendChild(doc.createTextNode(modelLocation));	
		} catch (Exception e) {
			// if we have a problem, just don't save this entry.
			entry = null;
		}
		return entry;
	}
}
