/********************************************************************** 
 * Copyright (c) 2004, 2006 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: InvalidProbeBundleException.java,v 1.2 2006/02/06 20:18:15 nmehrega Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 

package org.eclipse.tptp.platform.probekit.util;

import java.io.File;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;

/**
 * This exception encapsulates a number of things that might be wrong with a
 * a collection of files/resources that represent a probe. Though this class
 * was created primarily for use with ProbeFileBundle and ProbeResourceBundle,
 * it may be useful elsewhere, such as by the registry. 
 * 
 * <p>The exception can hold one or many reasons for a bundle to be considered
 * invalid - <i>reason</i> is a bit mask, not a single code. Additional bits
 * of information may also be added, such as which file/resource is inaccessible. 
 * Callers should be prepared, however, for this extra info to be missing.</p>
 * 
 * <p>Unfortunately, since this exception does double duty for both File and
 * IResource problems, pulling off the extra details can be tricky. You can
 * always get it as an Object or a File, but not always as a resource (all
 * Resources are Files, but not all Files are Resources).</p>
 * 
 * <p>To mitigate this problem somewhat, getFileOrResourceName() can be used
 * to retrieve the most commonly interesting detail.</p>
 * 
 * @author kcoleman
 *
 * TODO To change the template for this generated type comment go to
 * Window - Preferences - Java - Code Style - Code Templates
 */
public class InvalidProbeBundleException extends Exception {
    private static final long serialVersionUID = 3690759483865444403L;
    private transient int reason = 0;
	private transient String extension = null;
	private transient Object problemFileOrRsrc = null;
	
	// Possible sources of trouble
	public static final int MISSING_FILE = 1;
	public static final int INACCESSIBLE_FILE = 2;
	public static final int TOO_MANY_FILES_WITH_EXT = 4;
	public static final int INVALID_MODEL = 8;
	
	// Supplemental details for the above, where it makes sense.
	// Most of the sensible cases have been combined in the compound
	// error codes below.
	public static final int MODEL_FILE_ERR = 16;
	public static final int SCRIPT_FILE_ERR = 32;
	public static final int SUPPORT_FILE_ERR = 64;
	
	// Compound error codes, for those who wish to supply more detail
	public static final int MISSING_MODEL_FILE =
		MISSING_FILE | MODEL_FILE_ERR;
	public static final int MISSING_SCRIPT_FILE =
		MISSING_FILE | SCRIPT_FILE_ERR;
	public static final int MISSING_SUPPORT_FILE = 
		MISSING_FILE | SUPPORT_FILE_ERR;
	public static final int INACCESSIBLE_MODEL_FILE =
		INACCESSIBLE_FILE | MODEL_FILE_ERR;
	public static final int INACCESSIBLE_SCRIPT_FILE =
		INACCESSIBLE_FILE | SCRIPT_FILE_ERR;
	public static final int INACCESSIBLE_SUPPORT_FILE =
		INACCESSIBLE_FILE | SUPPORT_FILE_ERR;
	
	private InvalidProbeBundleException()
	{
	}

	public InvalidProbeBundleException(int reason)
	{
		super();
		this.reason = reason;
	}
	
	public InvalidProbeBundleException(int reason, Throwable e)
	{
		super(e);
		this.reason = reason;
	}
	
	public InvalidProbeBundleException(int reason, String extension)
	{
		this.extension = extension;
		this.reason = reason;
	}
	
	public InvalidProbeBundleException(int reason, Object o)
	{
		problemFileOrRsrc = o;
		this.reason = reason;
	}
	
	public InvalidProbeBundleException(int reason, Object o, String extension)
	{
		this.reason = reason;
		problemFileOrRsrc = o;
		this.extension = extension;
	}
	
	/**
	 * The reasons behind this exception. This is a bit field, composed of the
	 * bit flag constants declared above (MISSING_FILE, etc.). It should always
	 * be non-zero.
	 * 
	 * @return Causes of this exception. May be one or many values.
	 */
	public int getReason()
	{
		return reason;
	}
	
	/**
	 * If <i>reason</i> includes TOO_MANY_FILES_WITH_EXT, calling this method
	 * should reveal which type of file we found too many of.
	 * 
	 * @return The file extension causing TOO_MANY_FILES_WITH_EXT, or null.
	 */
	public String getExtension()
	{
		return extension;
	}
	
	/**
	 * Retrieve exception detail as an Object. If non-null, this should in theory
	 * be either a File or an IResource. It is expected that users will use
	 * getFileDetail() or getResourceDetail in preference to this method.
	 * 
	 * @return The exception detail as an Object, or null if none is available.
	 */
	public Object getObjectDetail()
	{
		return problemFileOrRsrc;
	}
	
	/**
	 * Retrieve exception detail as a File. If the exception was not decorated
	 * with either file or resource detail, null is returned. If it was decorated
	 * with an IResource, the resource is converted into a File and returned.
	 * 
	 * @return The exception detail as a File, or null.
	 */
	public File getFileDetail()
	{
		if ( problemFileOrRsrc != null ) {
			if ( problemFileOrRsrc instanceof File ) {
				return (File)problemFileOrRsrc;
			} else if ( problemFileOrRsrc instanceof IResource ) {
				IResource r = (IResource) problemFileOrRsrc;
				return r.getLocation().toFile();
			}
		}
		return null;
	}
	
	/**
	 * Retrieve exception detail as an IResource. If the exception was not
	 * decorated with an IResource, null will be returned. This could mean
	 * either that no extra information was supplied, or that only a File
	 * was supplied.
	 * 
	 * @return The exception detail as a IResource, or null.
	 */
	public IResource getResourceDetail()
	{
		if ( hasResourceDetail() ) {
			return (IResource)problemFileOrRsrc;
		}
		return null;
	}
	
	/**
	 * Determine whether or not there is File exception detail available. This
	 * will be the case whether the exception object was decorated with a File
	 * or a IResource.
	 * 
	 * @return true if calling getFileDetail would return a non-null value.
	 */
	public boolean hasFileDetail()
	{
		return problemFileOrRsrc != null; 
	}
	
	/**
	 * Determine whether or not there is IResource exception detail available.
	 * 
	 * @return true if calling getResourceDetail would return a non-null value.
	 */
	public boolean hasResourceDetail()
	{
		return problemFileOrRsrc != null && problemFileOrRsrc instanceof IResource;
	}
	
	/**
	 * Get the simple name of the file/resource detail, if available. The simple
	 * name is just the filename, such as mumble.probe. 
	 * 
	 * @return The simple name of the file/resource detail, or none if no detail
	 * 		is available.
	 */
	public String getFileOrResourceName()
	{
		File f = getFileDetail();
		if ( f != null ) {
			return f.getName();
		} else {
			return null;
		}
	}
	
	/**
	 * Test to see whether or not <i>mask</i> is one of the causes of this
	 * exception. <i>Mask</i> should be one of the constants declared above.
	 * 
	 * @param mask One of the constants declared by this class
	 * @return true if <i>mask</i> is a cause of this exception.
	 */
	public boolean testReason(int mask)
	{
		return (reason & mask) == mask;
	}
	
	private synchronized void writeObject(ObjectOutputStream s) throws IOException
	{
		s.defaultWriteObject();
		s.writeInt(reason);
		s.writeObject(extension);
		
		if ( hasResourceDetail() ) {
			IResource r = (IResource)problemFileOrRsrc;
			s.writeBoolean(true);
			s.writeObject(r.getFullPath().toString());
		} else {
			s.writeBoolean(false);
			if ( problemFileOrRsrc != null ) {
				File f = (File)problemFileOrRsrc;
				s.writeObject(f.getCanonicalPath());
			} else {
				s.writeObject("null");
			}
		}
	}
	
	private synchronized void readObject(ObjectInputStream s) throws IOException, ClassNotFoundException
	{
		s.defaultReadObject();
		reason = s.readInt();
		extension = (String)s.readObject();
		
		boolean isResource = s.readBoolean();
		String str = (String) s.readObject();
		if ( isResource ) {
			// Attempt to restore from workspace relative resource path.
			// We assume, by nature of this exception that it is always a
			// file resource (IFile).
			IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();		
			problemFileOrRsrc = root.getFile(new Path(str));
		} else {
			if ( str == null || str.compareTo("null") == 0 ) {
				problemFileOrRsrc = null;
			} else {
				problemFileOrRsrc = new File(str);
			}
		}
	}
}
