/********************************************************************** 
 * Copyright (c) 2004, 2006 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: ProbeFileBundle.java,v 1.2 2006/02/06 20:18:15 nmehrega Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 

package org.eclipse.tptp.platform.probekit.util;
import java.io.File;
import java.io.IOException;
import java.util.Iterator;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.models.internal.probekit.DocumentRoot;
import org.eclipse.hyades.models.internal.probekit.Probekit;


/**
 * A probe bundle is all the files necessary to export, import, or deploy
 * a probe set. This consists of a description in the form of a persisted
 * model (either source or probeinfo) and all the items generated by the
 * probekit builder. This class represents a probe bundle based on the java 
 * File class. This class is primarily for use by the registry, which must 
 * manipulate low level File objects for imported probe sets. See ProbeRegistry 
 * for details.
 * 
 * <p>The methods in this class intentionally mirrors those of ProbeResourceBundle.
 * They conceptually serve the same purpose, but the two classes manipulate
 * probe bundles as different data types. It was not possible to usefully
 * abstract these two classes into a Java Interface, primarily because we
 * cannot legally use a concrete type for resources (IResource vs. the internal
 * Eclipse class Resource, for example). Rather than go through a lot of hoops
 * for OO purity, I just created two similar, but different classes. The
 * probe builder/bundler primarily uses ProbeResourceBundle; the registry
 * accepts both, but ultimately reduces everything down to java File's.</p>
 * 
 * @author kcoleman
 */
public class ProbeFileBundle {

	private File source = null;
	private File script = null;
	private File probeinfo = null;
	private File[] supportFiles = null;
	private Probekit probekit = null;
	
	/**
	 * Create a probe bundle from the files contained in a sub-directory.
	 * This constructor supports the registry in handling imported probe sets.
	 * It is suitable for use either during initial import or for recreating
	 * saved registry state. It expects srcDir to contain only the files for a
	 * single probe bundle. An exception is thrown if it is not possible to
	 * create a well-formed bundle from the directory contents.
	 * 
	 * @param srcDir A sub-directory containing the pieces of a probe bundle.
	 * @throws InvalidProbeBundleException srcDir does not contain the components of
	 * 		a well-formed bundle. For example, required files are missing or
	 * 		there are too many instances of singleton files like the model.
	 * @throws IOException srcDir is not a readable directory.
	 */
	public ProbeFileBundle(File srcDir) 
		throws InvalidProbeBundleException, IOException 
	{
		if ( srcDir == null || !srcDir.isDirectory() || !srcDir.canRead() ) {
			throw new IOException("Parameter is not a readable directory");	//$NON-NLS-1$
		}
		
		File[] files = srcDir.listFiles();
		File modelFile = null;
		int nfiles = 0;
		int probeinfoIdx = 0;
		int probescriptIdx = 0;
		int sourceIdx = 0;
		
		// Walk the dir contents and identify the required components
		for (int i = 0; i < files.length; i++) {
			if ( files[i].isFile() ) {
				String filename = files[i].getName();
				if ( filename.endsWith(ProbekitConstants.PROBE_INFO_EXT) ) {
					if ( probeinfo != null ) {
						throw new InvalidProbeBundleException(
								InvalidProbeBundleException.TOO_MANY_FILES_WITH_EXT,
								ProbekitConstants.PROBE_INFO_EXT);
					}
					probeinfo = files[i];
					probeinfoIdx = i;
				} else if (	filename.endsWith(ProbekitConstants.PROBE_SRC_EXT) ) {
					if ( source != null ) {
						throw new InvalidProbeBundleException(
								InvalidProbeBundleException.TOO_MANY_FILES_WITH_EXT,
								ProbekitConstants.PROBE_SRC_EXT);
					}
					source = files[i];
					sourceIdx = i;
				} else if (filename.endsWith(ProbekitConstants.PROBE_SCRIPT_EXT)) {
					if ( script != null ) {
						throw new InvalidProbeBundleException(
								InvalidProbeBundleException.TOO_MANY_FILES_WITH_EXT,
								ProbekitConstants.PROBE_SCRIPT_EXT);
					}
					script = files[i];
					probescriptIdx = i;
				} else {
					// Count up the number of other, supporting files so we know
					// how large the containing array needs to be.
					nfiles++;
				}
			}
		}
		
		// Now that we know how many support files there are, go round 'em up.
		supportFiles = new File[nfiles];
		for (int i = 0, j = 0; i < files.length; i++) {
			if ( i != probeinfoIdx && i != sourceIdx && 
					i != probescriptIdx ) {
				supportFiles[j++] = files[i];
			}
		}
		
		validate();
	}

	/**
	 * Attach an optional probe source file. This is typically a full model,
	 * created by the probe author. Inclusion of source is optional. See 
	 * isComplete().
	 * 
	 * @param file The probe source file to add to the bundle.
	 */
	public void setSource(File file) 
	{
		source = file;
	}
	
	/**
	 * Retrieve the probe source file that is part of this bundle.
	 * @return The probe source or null if no source is available.
	 */
	public File getSource() 
	{
		return source;
	}

	/**
	 * Attach a probescript file to the bundle. This is usually a file generated
	 * by the builder. A well-formed bundle always includes a probescript.
	 * @param file The probescript to add to this bundle.
	 */
	public void setScript(File file) 
	{
		script = file;
	}

	/**
	 * Retrieve the probescript that is part of this bundle. 
	 * @return The probescript associated with this bundle, or null.
	 */
	public File getScript() 
	{
		return script;
	}

	/**
	 * Attach a probeinfo file to the bundle. This is usually a reduced version
	 * of the full probe source model, generated by the builder. A well-formed
	 * bundle always includes either a source or probeinfo file, or both.
	 * @param file The probeinfo file to add to this bundle.
	 */
	public void setProbeInfo(File file) 
	{
		probeinfo = file;
		// clear probekit because we probably need to regen the model now
		probekit = null;
	}

	/**
	 * Retrieve the probeinfo that is part of this bundle.
	 * @return The probeinfo associated with thsi bundle, or null if there is
	 * 		none.
	 */
	public File getProbeInfo() 
	{
		return probeinfo;
	}

	/**
	 * Attach a list of supporting files (such as class files) needed to deploy
	 * the probe set. This set should not include the source, script or 
	 * probeinfo files. It is usually the set of class files generated by the
	 * builder. A well-formed bundle always includes at least one such file.
	 * @param files The set of supporting files to add to the bundle.
	 */
	public void setSupporting(File[] files) 
	{
		supportFiles = files;
	}

	/**
	 * Retrieve the set of supporting files associated with this bundle.
	 * @return The supporting files, or null if there are none.
	 */
	public File[] getSupporting() 
	{
		return supportFiles;
	}

	/**
	 * Determine whether or not a Java File object is truly a file (not a
	 * directory, for example) and we can read it. Convenience function for
	 * use by isComplete().
	 * 
	 * @param f The file to check
	 * @return True if this is a file in the file system and it is readable.
	 */
	protected static boolean isReadableFile(File f) {
		return (f != null && f.isFile() && f.canRead());
	}
	
	/**
	 * Determine whether or not the bundle is complete and well-formed. Such a
	 * bundle includes:
	 * <ul>
	 *   <li>At least one of source or probeinfo files.</li>
	 *   <li>A probescript</li>
	 *   <li>One or more supporting files</li>
	 * </ul>
	 * <p>If any of the above conditions does not hold, the bundle is incomplete.
	 * Implementations of this interface may impose additional, stricter
	 * constraints, but should not relax this set.</p>
	 * 
	 * <p>This method is just a yes/no answer. If you need more details about
	 * what is wrong, use validate() and collect details off the thrown
	 * exception.</p>
	 * 
	 * @return true if the bundle is well-formed and complete, false otherwise.
	 */
	public boolean isComplete() 
	{
		try {
			validate();
		} catch (InvalidProbeBundleException ex) {
			return false;
		}
		return true;
	}

	public void validate() throws InvalidProbeBundleException
	{
		int reason = 0;
		File theFile = null;
		
		// one or the other of source and probeinfo must be set
		if ( probeinfo == null && source == null ) {
			reason |= InvalidProbeBundleException.MISSING_MODEL_FILE;
		}
		
		// if present, probeinfo file must be a readable file
		if ( probeinfo != null && !isReadableFile(probeinfo) ) {
			reason |= InvalidProbeBundleException.INACCESSIBLE_FILE;
			theFile = probeinfo;
		}
		
		// if present, source must be a readable file
		if ( source != null && !isReadableFile(source) ) {
			reason |= InvalidProbeBundleException.INACCESSIBLE_FILE;
			theFile = source;
		}
		
		// script must be present and must be a readable file
		if ( script == null ) {
			reason |= InvalidProbeBundleException.MISSING_SCRIPT_FILE;
		} else if ( !isReadableFile(script) ) {
			reason |= InvalidProbeBundleException.INACCESSIBLE_FILE;
			theFile = script;
		}
		
		// There must be at least 1 support file and all of them must be
		// readable files.
		if ( supportFiles == null || supportFiles.length == 0 ) {			
			reason |= InvalidProbeBundleException.MISSING_SUPPORT_FILE;
		} else {
			for (int i = 0; i < supportFiles.length; i++ ) {
				if ( supportFiles[i] == null ) {					
					reason |= InvalidProbeBundleException.MISSING_SUPPORT_FILE;
				} else if ( !isReadableFile(supportFiles[i]) ) {					
					reason |= InvalidProbeBundleException.INACCESSIBLE_FILE;
					theFile = supportFiles[i];
				}
			}
		}
		
		if ( reason != 0 ) {
			// There is at least one problem.
			if ( theFile != null ) {
				throw new InvalidProbeBundleException(reason, theFile);
			} else {
				throw new InvalidProbeBundleException(reason);
			}
		}		
	}
	
	/**
	 * Instantiate an EMF model for the probeset described by this bundle.
	 * Since the model may be instantiated from either the source or the
	 * probeinfo file, consumers should not assume more information than is
	 * available from the probeinfo model.
	 * 
	 * @return The instantiated model, or null if it was not possible to
	 * 		instantiate it.
	 */
	public Probekit instantiateModel() 
	{
		if ( probekit != null ) {
			return probekit;
		}
		
		File f = getModelFile();
		if ( f != null ) {
			URI fileURI = URI.createFileURI(f.getAbsolutePath());
			ResourceSet resourceSet = new ResourceSetImpl();
			Iterator iter = null;
			
			try {
				iter = resourceSet.getResource(fileURI, true).getContents().iterator();
			} catch(RuntimeException ex) {
				// Unable to create the model. Something is probably wrong with
				// the source input file.
				probekit = null;
			}
			
			while ( iter.hasNext() )
			{
				Object o = iter.next();
				if ( o instanceof DocumentRoot )
				{
					// EMF 2.0 and later: Root of the document should be a Probekit.
					DocumentRoot root = (DocumentRoot)o;
					probekit = root.getProbekit();
				} else if ( o instanceof Probekit ) {
					// EMF 1.x: Probekit was the top level object.
					probekit = (Probekit)o;
				} 
			}
		}
		return probekit;
	}

	/**
	 * Retrieve a file which can be used to instantiate an EMF model of the
	 * probe. The model may not be complete, since full source is not always
	 * available.
	 * 
	 * @return Either a full or partial persisted model of the probe.
	 */
	public File getModelFile()
	{
		if ( probeinfo != null ) {
			return probeinfo;
		} else {
			return source;
		}
	}
}
