/********************************************************************** 
 * Copyright (c) 2004, 2006 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: ProbeLaunchConfigString.java,v 1.2 2006/02/06 20:18:15 nmehrega Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 

package org.eclipse.tptp.platform.probekit.util;

public class ProbeLaunchConfigString {
	/**
	 * The name prefix to use for name/value pairs in the launch configuraiton
	 * which are meant for Probekit's attention. Add your own suffix to this
	 * string when forming your entry, so you don't collide with anybody else.
	 */
	public static final String LAUNCH_CONFIG_NAME_PREFIX = "SETOPTION_org.eclipse.tptp.platform.probekit.Probespec.";
	public static final String AGENT_CONFIG_NAME_PREFIX = "org.eclipse.tptp.platform.probekit.Probespec.";
	
	/**
	 * Constants for the values returned by getType method
	 */
	public static final int TYPE_REGISTRY_ID = 1;
	public static final int TYPE_PROBE_SCRIPT = 2;

	/**
	 * When true, the probeSpec string is a Registry ID.
	 * When false, it's the full engine script.
	 */
	private boolean hasRegistryId;

	/**
	 * See hasRegistryID: either a registry ID or an engine script.
	 */
	private String probeSpec;

	/**
	 * When true, use the Run With Profiling dialog's main filter set.
	 * When false, use the filters string, or none if it's not set.
	 */
	boolean useDefaultFilters;

	/**
	 * The filters specification, or null if none supplied.
	 */
	private String filters;

	public void addProbeID(String id_string) {
		hasRegistryId = true;
		probeSpec = id_string;
	}

	public void addProbescript(String probe_script) {
		hasRegistryId = false;
		probeSpec = probe_script;
	}

	public void useDefaultFilters() {
		useDefaultFilters = true;
		filters = null;
	}

	public void addFilters(String filter_string) {
		if (filter_string != null && filter_string.length() > 0) {
			useDefaultFilters = false;
			filters = filter_string;			
		}
		else {
			useDefaultFilters();
		}
	}

	public String toString() {
		StringBuffer outstr = new StringBuffer();
		if (hasRegistryId) {
			outstr.append("A");
		}
		else {
			outstr.append("B");
		}
		outstr.append(Integer.toString(probeSpec.length()));
		outstr.append(",");
		outstr.append(probeSpec);
		if (useDefaultFilters) {
			outstr.append("D");
		}
		if (filters != null) {
			outstr.append(Integer.toString(filters.length()));
			outstr.append(",");
			outstr.append(filters);
		}
		
		return outstr.toString();
	}

	/**
	 * Given an encoded string, decode it into a new 
	 * ProbeLaunchConfigString object.
	 * @param str The string to decode.
	 * @returns a new ProbeLaunchCOnfigString decoded from the string
	 * @throws IllegalArgumentException if there are decoding errors
	 */
	static public ProbeLaunchConfigString fromString(String str) {
		ProbeLaunchConfigString plcs = new ProbeLaunchConfigString();
		if (str.startsWith("A")) {
			plcs.hasRegistryId = true;
		}
		else if (str.startsWith("B")) {
			plcs.hasRegistryId = false;
		}
		else throw new IllegalArgumentException();

		int[] next_a = new int[1];
		plcs.probeSpec = extractString(str, 1, next_a);
		int next = next_a[0];

		// Get the filter spec
		if (next != str.length()) {
			if (str.charAt(next) == 'D') {
				plcs.useDefaultFilters = true;
			}
			else {
				plcs.filters = extractString(str, next, next_a);
				if (next_a[0] != str.length())
					throw new IllegalArgumentException();
			}
		}
		return plcs;
	}

	static private String extractString(String str, int num_start, int[] next_index) {
		int num_end;
		for (num_end = num_start; num_end < str.length(); num_end++) {
			if (!Character.isDigit(str.charAt(num_end)))
				break;
		}
		if (num_end == str.length() || 
		    num_end == num_start ||
		    str.charAt(num_end) != ',')
		{
			throw new IllegalArgumentException();
		}

		String length_str = str.substring(num_start, num_end);
		Integer length_int = Integer.decode(length_str);
		int length = length_int.intValue();

		if (num_end + 1 + length > str.length())
			throw new IllegalArgumentException();

		next_index[0] = num_end + 1 + length;
		return str.substring(num_end + 1, next_index[0]);
	}

	/* Getters ------------------------------------------------------- */

	/**
	 * Get the type of this entry.
	 * @returns 1 for Probe Registry ID, 2 for probescript.
	 */
	public int getType() {
		if (hasRegistryId) return TYPE_REGISTRY_ID;
		else return TYPE_PROBE_SCRIPT;
	}

	/**
	 * Get the Registry ID for this entry.
	 * @returns the Registry ID, or null if it's the wrong type.
	 */
	public String getRegistryId() {
		if (hasRegistryId) return probeSpec;
		else return null;
	}

	/**
	 * Get the value of useDefaultFilters flag
	 * 
	 * @returns true if default filter set should be used.
	 */
	public boolean getUseDefaultFilters() {
		return useDefaultFilters;
	}

	/**
	 * Get the probescript for this entry.
	 * @returns the probescript, or null if it's the wrong type.
	 */
	public String getProbeScript() {
		if (!hasRegistryId) return probeSpec;
		else return null;
	}

	/**
	 * Get the filter spec for this entry.
	 * @returns The letter "D" for the default filter set,
	 * or the filter spec string (can be null if never set).
	 */
	public String getFilters() {
		if (useDefaultFilters) return "D";
		else return filters;
	}

	public static void main(String args[]) {
		if (Test()) {
			System.out.println("Test passed");
		}
		else {
			System.out.println("Test failed");
		}
	}

	public static boolean Test() {
		// Test one example of each type
		// with a filter, with default filters, and without filters.

		ProbeLaunchConfigString s;
		String result;

		try {
			s = new ProbeLaunchConfigString();
			s.addProbeID("someProbeId");
			s.addFilters("java*:*:EXCLUDE&sun*:*:EXCLUDE");
			result = s.toString();
			if (!result.equals("A11,someProbeId30,java*:*:EXCLUDE&sun*:*:EXCLUDE")) {
				throw new Exception("1");
			}
			
			s = new ProbeLaunchConfigString();
			s.addProbeID("someProbeId");
			s.useDefaultFilters();
			result = s.toString();
			if (!result.equals("A11,someProbeIdD")) {
				throw new Exception("2");
			}
	
			s = new ProbeLaunchConfigString();
			s.addProbeID("someProbeId");
			result = s.toString();
			if (!result.equals("A11,someProbeId")) {
				throw new Exception("3");
			}
	
			s = new ProbeLaunchConfigString();
			s.addProbescript("this is my probescript");
			s.addFilters("some other filters");
			result = s.toString();
			if (!result.equals("B22,this is my probescript18,some other filters")) {
				throw new Exception("4");
			}
	
			s = new ProbeLaunchConfigString();
			s.addProbescript("this is my probescript");
			s.useDefaultFilters();
			result = s.toString();
			if (!result.equals("B22,this is my probescriptD")) {
				throw new Exception("5");
			}
	
			s = new ProbeLaunchConfigString();
			s.addProbescript("this is my probescript");
			result = s.toString();
			if (!result.equals("B22,this is my probescript")) {
				throw new Exception("6");
			}
		}
		catch (IllegalArgumentException iae) {
			System.err.println("IllegalArgumentException!");
			return false;
		}
		catch (Exception e) {
			System.err.println("Error in test number " + e.getMessage());
			return false;
		}
		return true;
	}
}
