/********************************************************************** 
 * Copyright (c) 2004, 2006 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: ProbeResourceBundle.java,v 1.2 2006/02/06 20:18:15 nmehrega Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 


package org.eclipse.tptp.platform.probekit.util;

import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.models.internal.probekit.DocumentRoot;
import org.eclipse.hyades.models.internal.probekit.Probekit;


/**
 * A probe bundle is all the files necessary to export, import, or deploy
 * a probe set. This consists of a description in the form of a persisted
 * model (either source or probeinfo) and all the items generated by the
 * probekit builder. This class represents a probe bundle based on Eclipse 
 * IResource objects. The builder and bundler generally work with IResource's 
 * since they're intimately involved with the workbench and the UI. All 
 * IResources in the bundle should actually be file resources (IFile).
 * 
 * <p>Since this class exists primarily to support the use case of probe
 * authoring, it includes an operation for creating connections among the
 * resources using persistent properties (see createResourceAssociations()).</p>
 * 
 * <p>Also, because this class exists primarily to support authoring, a probe 
 * source file must be part of the bundle for it to be considered complete. See 
 * isComplete().</p>
 * 
 * <p>Many methods of this class intentionally mirror those of ProbeFileBundle.
 * They conceptually serve the same purpose, but the two classes manipulate
 * bundles different component types. It was not possible to usefully
 * abstract these two classes into a Java Interface, primarily because we
 * cannot legally use a concrete type for resources (IResource vs. the internal
 * Eclipse class Resource, for example). Rather than go through a lot of hoops
 * for OO purity, I just created two similar, but different classes. The
 * probe builder/bundler primarily uses ProbeResourceBundle; the registry
 * accepts both, but ultimately reduces everything down to java File's.</p>
 * 
 * <p>One unique aspect of resource bundles is the use of Eclipse persistent
 * resource properties for persisting the connection between a source resource
 * and it's related bundle resources:</p>
 * 
 * <p>When you call createResourceAssociations() on a (properly formed) bundle,
 * persistent property are attached to the source resource to tag the source
 * with references to the probescript, probeinfo, and supporting resources.
 * Each property's value is a string that can subsequently be used to recreate
 * an IResource using something like IWorkspaceRoot.findMember().</p>
 * 
 * <p>The properties for the supporting files (like class files) are problematic
 * because there is a variable number of them. I'm sure there are better ways
 * to do it, but my wimpy solution was one property to hold the file count (N)
 * and then N additional properties to hold the paths, with each property name
 * have an index component (file0, file1, etc.).</p>
 * 
 * <p>So, the property that are created by createResource Associations are:</p>
 * <ul>
 *   <li>org.eclipse.tptp.platform.probekit.bundle.probescript</li>
 *   <li>org.eclipse.tptp.platform.probekit.bundle.probeinfo</li>
 *   <li>org.eclipse.tptp.platform.probekit.bundle.nfiles</li>
 *   <li>org.eclipse.tptp.platform.probekit.bundle.file0</li>
 *   <li> ... </li>
 *   <li>org.eclipse.tptp.platform.probekit.bundle.fileN</li>
 * </ul>
 * 
 * <p>A bundle can be restored from these properties using the constructor
 * which accepts a String path to the source resource. The property names can
 * be retrieved for other uses using the get*Property() methods. There are also
 * operations for removing the property settings.</p>
 * 
 * <p>These properties are used by the Probe Registry in restoring registry
 * state. They're also used during export, as an easy means of gathering up
 * all the pieces that need to be exported.</p>
 * 
 * @author kcoleman
 */
public class ProbeResourceBundle {

	// The following constants are used in generating peristent properties on
	// the probe source file. See createResourceAssociations().
	private static final String PROP_NAMESPACE = "org.eclipse.tptp.platform.probekit.bundle";	//$NON-NLS-1$
	private static final String FILE_PROP_PREFIX = "file";	//$NON-NLS-1$
	private static final String PROBESCRIPT_PROP = "probescript";	//$NON-NLS-1$
	private static final String PROBEINFO_PROP = "probeinfo";	//$NON-NLS-1$
	private static final String NFILES_PROP = "nfiles";	//$NON-NLS-1$
	// The above property elements as the QualifiedNames required by
	// IResource.setPersistentProperty. The FILE_PROP property has to be
	// built on the fly.
	private static final QualifiedName probescriptProperty = 
		new QualifiedName(PROP_NAMESPACE, PROBESCRIPT_PROP);
	private static final QualifiedName probeinfoProperty =
		new QualifiedName(PROP_NAMESPACE, PROBEINFO_PROP);
	private static final QualifiedName nfilesProperty = 
		new QualifiedName(PROP_NAMESPACE, NFILES_PROP);	
	// The minimum number of persistent properties that should be present.
	// (probescript, probeinfo, file count). See initBundleFromSourceResource.
	private static final int MIN_PROPERTIES = 3;
	
	IResource source = null;
	IResource script = null;
	IResource probeinfo = null;
	Probekit probekit = null;
	
	/**
	 * Set of supporting ProbeResource's, such as class files.
	 */
	//IResource[] supportFiles = null;
	ArrayList supportFilesList = null;
	
	/**
	 * Use this constructor to manually build up a bundle from its parts.
	 * Use isComplete() to validate the bundle when you're done.
	 */
	public ProbeResourceBundle()
	{
		super();
		supportFilesList = new ArrayList();
	}
	
	/**
	 * Create an empty bundle that is eventually expected to include the
	 * given number of support files.
	 * 
	 * @param nSupportFiles Max number of support files this bundle can hold.
	 */
	public ProbeResourceBundle(int nSupportFiles)
	{
		if (nSupportFiles > 0 ) {
			supportFilesList = new ArrayList(nSupportFiles);
		} else {
			supportFilesList = new ArrayList();
		}
	}
	
	/**
	 * Create a probe bundle, given the workspace relative path to its source
	 * resource. The file resources (probescript, class files, etc.) associated
	 * with the source should be restorable from persistent properties attached
	 * to the source resource by the ProbeBuilder. Callers who care about the validity
	 * of the resulting bundle should call isComplete() to verify the result.
	 * 
	 * @param resourcePath The workspace relative path to the source resource.
	 * 		For example project/folder/whatever.probe.
	 * @throws InvalidProbeBundleException A bundle cannot be constructed from
	 * 		resourcePath. The source may be inaccessible, the properties may
	 * 		be missing, or the referenced resources may not be accessible.
	 * @throws ProbeBundleException Some other problem besides accessibility
	 * 		occurred, such as a missing or corrupted property value, or
	 * 		rsrc does not represent a viable Resource. In effect, an internal
	 * 		error (possibly caused by the caller), rather than an expected
	 * 		problem.
	 * @throws ProbeBundleBuildRequiredException None of the require properties
	 * 		are set, suggesting strongly the <i>rsrc</i> needs to be rebuilt.
	 */
	public ProbeResourceBundle(String resourcePath) 
			throws ProbeBundleException, InvalidProbeBundleException, ProbeBundleBuildRequiredException
	{
		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		
		// Find the source resource
		source = root.getFile(new Path(resourcePath));
		initBundleFromSourceResource();
	}
	
	/**
	 * Create a probe bundle, given the workspace relative path to its source
	 * resource. The file resources (probescript, class files, etc.) associated
	 * with the source should be restorable from persistent properties attached
	 * to the source resource by the ProbeBuilder. Callers who care about the validity
	 * of the resulting bundle should call isComplete() to verify the result.
	 * 
	 * @param rsrc An IFile resource which represents a probe source file in
	 * 		the workspace. For example, mumble.probe.
	 * @throws InvalidProbeBundleException A bundle cannot be constructed from
	 * 		rsrc. The source may be inaccessible, the properties may
	 * 		be missing, or the referenced resources may not be accessible.
	 * @throws ProbeBundleException Some other problem besides accessibility
	 * 		occurred, such as a missing or corrupted property value, or
	 * 		rsrc does not represent a viable Resource. In effect, an internal
	 * 		error (possibly caused by the caller), rather than an expected
	 * 		problem.
	 * @throws ProbeBundleBuildRequiredException None of the require properties
	 * 		are set, suggesting strongly the <i>rsrc</i> needs to be rebuilt.
	 */
	public ProbeResourceBundle(IResource rsrc)
		throws ProbeBundleException, InvalidProbeBundleException, ProbeBundleBuildRequiredException
	{
		source = rsrc;
		initBundleFromSourceResource();
	}
	
	/**
	 * This is the function that does all the work to support the String and
	 * resource constructors: Pulling the property values off a source resource
	 * and using them to populate the rest of the bundle. Callers should set
	 * <i>source</i> to the resource from which the rest of the bundle will
	 * be initialized.
	 * 
	 * @throws InvalidProbeBundleException A bundle cannot be constructed from
	 * 		rsrc. The source may be inaccessible, the properties may
	 * 		be missing, or the referenced resources may not be accessible.
	 * @throws ProbeBundleException Some other problem besides accessibility
	 * 		occurred, such as a missing or corrupted property value, or
	 * 		rsrc does not represent a viable Resource. In effect, an internal
	 * 		error (possibly caused by the caller), rather than an expected
	 * 		problem.
	 * @throws ProbeBundleBuildRequiredException None of the require properties
	 * 		are set, suggesting strongly the <i>rsrc</i> needs to be rebuilt.
	 */
	protected void initBundleFromSourceResource()
		throws ProbeBundleException, InvalidProbeBundleException, ProbeBundleBuildRequiredException
	{
		// TODO isAccessible might be too constrained a check. Is it possible
		// that the workbench isn't fully up and so resources aren't available?
		if ( source == null ) {
			throw new ProbeBundleException("Null source resource in bundle");	//$NON-NLS-1$
		} else if ( !source.isAccessible() ) {
			throw new InvalidProbeBundleException(
					InvalidProbeBundleException.INACCESSIBLE_MODEL_FILE, source);
		}
		
		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		int missing = 0;
		
		// restore the probeinfo file, if there is one
		try {
			String s = source.getPersistentProperty(probeinfoProperty);					
			if ( s != null && s.length() > 0 ) {
				probeinfo = root.getFile(new Path(s));
				if ( probeinfo == null ) {				
					throw new InvalidProbeBundleException(
							InvalidProbeBundleException.MISSING_MODEL_FILE);
				} else if ( !probeinfo.isAccessible() ) {	
					throw new InvalidProbeBundleException(
							InvalidProbeBundleException.INACCESSIBLE_MODEL_FILE, probeinfo);
				}
			} else if (s == null) {
				missing++;
			}
		} catch (CoreException e) {
			// getPersistentProperty throws CoreExceptions for reasons
			// that translate into an inaccessible resource in our parlance.
			throw new InvalidProbeBundleException(
					InvalidProbeBundleException.INACCESSIBLE_MODEL_FILE, source);
		}
		
		// restore the probescript
		try {
			String s = source.getPersistentProperty(probescriptProperty);					
			if ( s != null && s.length() > 0 ) {
				script = root.getFile(new Path(s));
				if ( script == null ) {
					throw new InvalidProbeBundleException(
							InvalidProbeBundleException.MISSING_SCRIPT_FILE);
				} else if ( !script.isAccessible() ) {
					throw new InvalidProbeBundleException(
							InvalidProbeBundleException.INACCESSIBLE_SCRIPT_FILE, script);
				}
			} else if (s == null) {
				missing++;
			}
		} catch (CoreException e) {
			// getPersistentProperty throws CoreExceptions for reasons
			// that translate into an inaccessible resource in our parlance.
			throw new InvalidProbeBundleException(
					InvalidProbeBundleException.INACCESSIBLE_MODEL_FILE, source);
		}
		
		// restore the supporting class files, etc.
		try {
			// Determine how many supporting files there are
			String nFilesStr = source.getPersistentProperty(nfilesProperty);
			if ( nFilesStr == null ) {
				missing++;
				if ( missing >= MIN_PROPERTIES ) {
					// Give it up: NONE of the properties are set. This file
					// has never been compiled or has not been re-built since
					// a Clean. Call this out as a special case since it is
					// clear what's wrong and what to do about it. If only some are
					// missing, that's a different problem, caught by isComplete().
					// Why catch this case here? Mostly because it is convenient.
					throw new ProbeBundleBuildRequiredException(source);
				} else {
					throw new ProbeBundleException("Supporting files property missing");	//$NON-NLS-1$					
				}
			} else if ( nFilesStr.length() == 0 ) {
				throw new ProbeBundleException("Supporting files property missing");	//$NON-NLS-1$
			}
			
			int nFiles;
			try {
				nFiles = Integer.parseInt(nFilesStr);
			} catch (NumberFormatException e) {
				throw new ProbeBundleException("Invalid supporting file count");	//$NON-NLS-1$
			}
			if ( nFiles <= 0 ) {
				throw new ProbeBundleException("Invalid supporting file count");	//$NON-NLS-1$
			}
			// Create resources for each supporting file
			supportFilesList = new ArrayList(nFiles);
			//supportFiles = new IResource[nFiles];
			for (int i = 0; i < nFiles; i++) {
				String s = source.getPersistentProperty(getFileProperty(i));
				if ( s != null && s.length() > 0 ) {
					IResource r = root.findMember(s);
					if ( r.isAccessible() ) {
						supportFilesList.add(i, r);
					} else {
						throw new InvalidProbeBundleException(
								InvalidProbeBundleException.INACCESSIBLE_SUPPORT_FILE, r);
					}
				} 
			}
		} catch (CoreException e) {
			// getPersistentProperty throws CoreExceptions for reasons
			// that translate into an inaccessible resource in our parlance.
			throw new InvalidProbeBundleException(
					InvalidProbeBundleException.INACCESSIBLE_MODEL_FILE, source);
		}			
	}
	
	/**
	 * Attach a probe source resource. This is typically a full model,
	 * created by the probe author. See isComplete().
	 * @param r The probe source resource to add to the bundle. It should be
	 * 		a file resource.
	 */
	public void setSource(IResource r) 
	{
		source = r;
	}
	
	/**
	 * Retrieve the probe source resource that is part of this bundle.
	 * @return The probe source as a Resource ,or null if no source is available.
	 */
	public IResource getSource() 
	{
		return source;
	}
	
	public File getSourceFile()
	/**
	 * Retrieve the probe source file that is part of this bundle.
	 * @return The probe source as a File, or null if no source is available.
	 */
	{
		if ( source != null ) {
			return source.getLocation().toFile();
		} else {
			return null;
		}
	}

	/**
	 * Attach a probescript (file) resource to the bundle. This is usually generated
	 * by the builder. A well-formed bundle always includes a probescript.
	 * @param r The probescript to add to this bundle.
	 */
	public void setScript(IResource r) 
	{
		script = r;
	}
	
	/**
	 * Retrieve the probescript that is part of this bundle. 
	 * @return The probescript associated with this bundle, or null.
	 */
	public IResource getScript() 
	{
		return script;
	}

	/**
	 * Attach a probeinfo (file) resource to the bundle. This is usually a 
	 * reduced version of the full probe source model, generated by the builder. 
	 * A well-formed bundle always includes either a source or probeinfo file, 
	 * or both.
	 * 
	 * @param rsrc The probeinfo file to add to this bundle.
	 */
	public void setProbeInfo(IResource rsrc)
	{
		probeinfo = rsrc;
	}

	/**
	 * Retrieve the probeinfo resource that is part of this bundle.
	 * 
	 * @return The probeinfo associated with thsi bundle, or null if there is
	 * 		none.
	 */
	public IResource getProbeInfo()
	{
		return probeinfo;
	}
	
	/**
	 * Retrieve the probeinfo java.io.File that is part of this bundle.
	 * 
	 * @return The probeinfo file associated with is bundle, or null if there
	 * 		is none.
	 */
	public File getProbeInfoFile()
	{
		if ( probeinfo != null ) {
			return probeinfo.getLocation().toFile();
		} else {
			return null;
		}
	}

	/**
	 * Attach a list of supporting (file) resources, such as class files, needed
	 * to deploy this bundle. This set should not include the source, script or 
	 * probeinfo resources. It is usually the set of class resource generated 
	 * by the builder. A well-formed bundle always includes at least one such 
	 * resource.
	 * 
	 * @param files The set of supporting files to add to the bundle.
	 */
	public void setSupporting(IResource[] files) 
	{
		//supportFiles = files;
		supportFilesList = new ArrayList(files.length);
		for (int i = 0; i < files.length; i++ ) {
			supportFilesList.add(i, files[i]);
		}
	}
	
	/**
	 * Retrieve the set of supporting resources associated with this bundle.
	 * @return The supporting resources, or null if there are none.
	 */
	public IResource[] getSupporting() 
	{
		if ( supportFilesList != null ) {
			IResource[] result = new IResource[supportFilesList.size()];
			result = (IResource[]) supportFilesList.toArray(result);
			return result;
		} else {
			return null;
		}
	}
	
	public void addResource(IResource r) throws InaccessibleProbeRsrcException
	{
		if ( !isAccessibleFileResource(r) ) {
			throw new InaccessibleProbeRsrcException(r);
		}
		
		String basename = r.getName();
		if (basename.endsWith(ProbekitConstants.PROBE_SRC_EXT)) {
			source = r;
		}
		else if (basename.endsWith(ProbekitConstants.PROBE_SCRIPT_EXT)) {
			script = r;
		}
		else if (basename.endsWith(ProbekitConstants.PROBE_INFO_EXT)) {
			probeinfo = r;
		}
		else {
			// This is a supporting file, like a class file.
			if ( supportFilesList == null ) {
				supportFilesList = new ArrayList();
			}
			supportFilesList.add(r);
		}	
	}
	
	/**
	 * Determine whether or not this is a resource we can use in the bundle.
	 * Usable means it is an IFile (not a directory, etc.) and it is
	 * accessible in the workspace.
	 * 
	 * @param r The resource to validate.
	 * @return True if the resource is of the expected type and is accessible,
	 * 		false otherwise.
	 */
	protected static boolean isAccessibleFileResource(IResource r)
	{
		if ( r == null || (r.getType() != IResource.FILE) ||
				!r.isAccessible() ) {
			return false;
		} else {
			// It is an accessible file resource, but is it readable?
			File f = r.getLocation().toFile();
			return f.canRead();
		}
	}
	
	/**
	 * Determine whether or not the bundle is complete and well-formed. Such a
	 * bundle includes:
	 * <ul>
	 *   <li>A probe source</li>
	 *   <li>A probeinfo</li>
	 *   <li>A probescript</li>
	 *   <li>One or more supporting files</li>
	 * </ul>
	 * <p>All of the above resources must be accessible (as in 
	 * IResource.isAccessible()). You must check for this because it is possible
	 * to create IResource's for things which do not exist. Resources also
	 * become inaccessible if their containning project is closed or deleted.</p>
	 * 
	 * <p>In addition to being accessible, all the above resources should have
	 * an underlying type of FILE. They should not represent things like 
	 * directories or projects.</p>
	 * 
	 * <p>If any of the above conditions does not hold, the bundle is incomplete.</p>
	 * 
	 * <p>If details are needed about the nature of the problems, use validate()
	 * instead of isComplete().</p>
	 * 
	 * @return true if the bundle is well-formed and complete, false otherwise.
	 */
	public boolean isComplete() 
	{
		try {
			validate();
		} catch (InvalidProbeBundleException ex){
			return false;
		}
		return true;
	}
	
	/**
	 * Determine whether or not the bundle is complete and well-formed. Such a
	 * bundle includes:
	 * <ul>
	 *   <li>A probe source</li>
	 *   <li>A probeinfo</li>
	 *   <li>A probescript</li>
	 *   <li>One or more supporting files</li>
	 * </ul>
	 * <p>All of the above resources must be accessible (as in 
	 * IResource.isAccessible()). You must check for this because it is possible
	 * to create IResource's for things which do not exist. Resources also
	 * become inaccessible if their containning project is closed or deleted.</p>
	 * 
	 * <p>In addition to being accessible, all the above resources should have
	 * an underlying type of FILE. They should not represent things like 
	 * directories or projects.</p>
	 * 
	 * <p>If any of the above conditions does not hold, the bundle is incomplete.</p>
	 * 
	 * <p>This method differs from isComplete in that it will throw an exception
	 * contain details of any identified problems. Note that for missing files,
	 * only the last discovered missing file is reported.</p>
	 * 
	 * @throws InvalidProbeBundleException The bundle is invalid/incomplete.
	 * 		Details are available through the exception.
	 */
	public void validate() throws InvalidProbeBundleException
	{
		int reason = 0;
		IResource rsrc = null;
		
		// source, probeinfo, and script must all be present and accessible
		if ( source == null ) {
			reason = InvalidProbeBundleException.MISSING_MODEL_FILE;
		} else if ( !isAccessibleFileResource(source) ) {
			reason = InvalidProbeBundleException.INACCESSIBLE_FILE;
			rsrc = source;
		}
		if ( probeinfo == null ) {
			reason = InvalidProbeBundleException.MISSING_MODEL_FILE;
		} else if ( !isAccessibleFileResource(probeinfo) ) {
			reason = InvalidProbeBundleException.INACCESSIBLE_FILE;
			rsrc = probeinfo;
		}
		if ( script == null ) {
			reason = InvalidProbeBundleException.MISSING_SCRIPT_FILE;
		} else if ( !isAccessibleFileResource(script) ) {
			reason = InvalidProbeBundleException.INACCESSIBLE_FILE;
			rsrc = script;
		}

		// There must be at least one support file and all support files must
		// be accessible.
		if ( supportFilesList == null || supportFilesList.size() == 0 ) {
			reason = InvalidProbeBundleException.MISSING_SUPPORT_FILE;
		} else {
			for ( int i = 0; i < supportFilesList.size(); i++ ) {
				IResource r = (IResource) supportFilesList.get(i);
				if ( r == null ) {
					reason = InvalidProbeBundleException.MISSING_SUPPORT_FILE;
				} else if ( !isAccessibleFileResource(r) ) {
					reason = InvalidProbeBundleException.INACCESSIBLE_FILE;
					rsrc = r;
				}
			}
		}
		
		if ( reason != 0 ) {
			// At least one problem was identified.
			if ( rsrc != null ) {
				throw new InvalidProbeBundleException(reason, rsrc);
			} else {
				throw new InvalidProbeBundleException(reason, rsrc);				
			}
		}
	}

	/**
	 * Instantiate an EMF model for the probeset described by this bundle.
	 * Since the model may be instantiated from either the source or the
	 * probeinfo file, consumers should not assume more information than is
	 * available from the probeinfo model.
	 * 
	 * @return The instantiated model, or null if it was not possible to
	 * 		instantiate it.
	 */
	public Probekit instantiateModel() 
	{
		if ( probekit != null ) {
			return probekit;
		}
		
		// Instantiate the model off the source, not the probeinfo so that
		// the max amount of information is available.
		IResource modelSrc = (source != null) ? source : probeinfo;
		if ( modelSrc != null ) {
			URI fileURI = URI.createFileURI(modelSrc.getLocation().toOSString());
			ResourceSet resourceSet = new ResourceSetImpl();
			Iterator iter = null;
			
			try {
				iter = resourceSet.getResource(fileURI, true).getContents().iterator();
			} catch(RuntimeException ex) {
				// Unable to create the model. Something is probably wrong with
				// the source input file.
				return null;
			}
			
			while ( iter.hasNext() )
			{
				Object o = iter.next();
				if ( o instanceof DocumentRoot )
				{
					// EMF 2.0 and later: Root of the document should be a Probekit.
					DocumentRoot root = (DocumentRoot)o;
					probekit = root.getProbekit();
					break;
				} else if ( o instanceof Probekit ) {
					// EMF 1.x: Probekit was the top level object.
					probekit = (Probekit)o;
					break;
				} 
			}
		}
		return probekit;
	}
	
	/**
	 * Retrieve a Resource which can be used to instantiate an EMF model of the
	 * probe. The model may not be complete, since full source is not always
	 * available. If a source Resource is available, it will always be returned
	 * in preference to the probeinfo, in order to maximize the available
	 * information.
	 * 
	 * @return Resource representing either a full or partial persisted model 
	 * 			of the probe.
	 */
	public IResource getModelResource() 
	{
		IResource r = (source == null) ? probeinfo : source;
		return r;
	}

	/**
	 * Retrieve the generated probescript resource as a Java File
	 * 
	 * @return a java.io.File reference to the probescript resource
	 */
	public File getScriptFile() 
	{
		if ( script != null ) {
			return script.getLocation().toFile();
		} else {
			return null;
		}
	}

	/**
	 * Retrieve the set of supporting files associated with this bundle.
	 * This returns the same information as getSupporting(), but the
	 * file Resources are first converted into Java File references.
	 * 
	 * @return The supporting files, or null if there are none.
	 */
	public File[] getSupportingFiles() 
	{
		if ( supportFilesList != null && supportFilesList.size() > 0 ) {
			int nfiles = supportFilesList.size();
			File[] result = new File[nfiles];
			for ( int i = 0; i < nfiles; i++ ) {
				IResource r = (IResource)supportFilesList.get(i);
				if ( r != null ) {
					result[i] = r.getLocation().toFile();
				} else {
					result[i] = null;
				}
			}
			return result;
		} else {
			// there are no supporting file resources
			return null;
		}
	}
	
	/**
	 * Attach persistent properties to the probe source which connect it to
	 * all the generated, derived resources like the probeinfo, probescript
	 * and class files. You can recover a bundle from this set of properties.
	 * The builder, bundler, and registry all rely on this mechansim. The 
	 * following properties are set (<i>namespace</i> is a qualified name such as
	 * org.eclipse.tptp.platform.probekit, defined by PROP_NAMESPACE):
	 * <ul>
	 *   <li>namespace.probescript</li>
	 *   <li>namespace.probeinfo</li>
	 *   <li>namespace.nfiles</li>
	 *   <li>namespace.file0 ... namepsace.fileN</li>
	 * </ul>
	 * <p>The nfiles and fileN properties are used to handle the list of support
	 * files. nfiles is set to the number of supportfile files, then each supporting
	 * file gets a property of "file" + an index suffix (file0, file1,...,fileN).</p>
	 * 
	 * <p>What's in the property value? The "full path" to the resource. This is
	 * a workspace relative path which can be used to re-constitute a reference
	 * to the associated resource. For example, for some myprobe.probescript
	 * file in project foo, the value is "/foo/myprobe.probescript". For
	 * information on how to get from the property to the resource, see the
	 * constructor ProbeResourceBundle(String).</p>
	 * 
	 * @throws CoreException Unable to set the properties. Possibly one or more of
	 * 		the associated resources in not available.
	 */
	public void createResourceAssociations()
		throws CoreException
	{
		if ( source != null ) {
			// Set properties relating to the support (class) files.
			int nfiles = 0;
			if ( supportFilesList != null && supportFilesList.size() > 0 ) {
				for (int i = 0; i < supportFilesList.size(); i++) {
					IResource r = (IResource) supportFilesList.get(i);
					if ( r != null ) {
						source.setPersistentProperty(getFileProperty(i), 
								r.getFullPath().toString());
						nfiles++;
					}
				}
			}
			source.setPersistentProperty(nfilesProperty, Integer.toString(nfiles));
			
			// set probescript property
			if ( script != null ) {
				source.setPersistentProperty(probescriptProperty, 
						script.getFullPath().toString());
			}
			// set probeinfo property
			if ( probeinfo != null ) {
				source.setPersistentProperty(probeinfoProperty, 
						probeinfo.getFullPath().toString());
			}
		}
	}
	
	/**
	 * Remove/reset the persistent property decorations on a probe source
	 * resource. Properties cannot be removed, per se, so we just set the
	 * values to null, if the properties already exist. 
	 */
	public void removeResourceAssociations()
	{
		removeResourceAssociations(source);
	}
	
	/**
	 * Remove/reset the persistent property decorations on a probe source
	 * resource. Properties cannot be removed, per se, so we just set the
	 * values to null, if the properties already exist. 
	 */
	public static void removeResourceAssociations(IResource source) 
	{	
		String s;
		
		if ( source != null && source.isAccessible() ) {
			// probeinfo
			try {
				s = source.getPersistentProperty(probeinfoProperty);					
				if ( s != null && s.length() > 0 ) {
					source.setPersistentProperty(probeinfoProperty, null);
				}
			} catch (CoreException e) {
				// shrug
			}
			
			// probescript
			try {
				s = source.getPersistentProperty(probescriptProperty);					
				if ( s != null && s.length() > 0 ) {
					source.setPersistentProperty(probescriptProperty, null);
				}
			} catch (CoreException e) {
				// shrug
			}
			
			// the supporting class files, etc.
			try {
				// Determine how many supporting file properties there are
				String nFilesStr = source.getPersistentProperty(nfilesProperty);
				if ( nFilesStr != null && nFilesStr.length() > 0 ) {
					try {
						source.setPersistentProperty(nfilesProperty, null);
					} catch (CoreException e) {
						// shrug
					}
					
					int nFiles;
					try {
						nFiles = Integer.parseInt(nFilesStr);
					} catch (NumberFormatException e) {
						return;	// can't do anything about the file props now
					}
					if ( nFiles > 0 ) {
						for (int i = 0; i < nFiles; i++) {
							QualifiedName propName = getFileProperty(i);
							s = source.getPersistentProperty(propName);
							if ( s != null && s.length() > 0 ) {
								source.setPersistentProperty(propName, null);
							}
						}
					}
				}
			} catch (CoreException e) {
				// shrug
			}
		} 
	}

	/**
	 * @return The name of the persistent resource property that represents a
	 * 		probescript file resource.
	 */
	public static QualifiedName getProbescriptProperty()
	{
		return probescriptProperty;
	}
	
	/**
	 * @return The name of the persistent resource property that represents a
	 * 		probeinfo file resource.
	 */
	public static QualifiedName getProbeInfoProperty()
	{
		return probeinfoProperty;
	}
	
	/**
	 * @return The name of the persistent resource property that represents the
	 * 		number of supporting files for a bundle.
	 */
	public static QualifiedName getNFilesProperty()
	{
		return nfilesProperty;
	}
	
	/**
	 * Retrieve the name of the property representing the Ith supporting file
	 * resource. The file number (0..N) is part of the property string.
	 *
	 * @param i The number (index) of the file property you want to retrieve.
	 * @return The name of the persistent resource property that represents the
	 * 			Ith supporting file resource.
	 */
	public static QualifiedName getFileProperty(int i)
	{
		return new QualifiedName(PROP_NAMESPACE, FILE_PROP_PREFIX + i);
	}
}
