/********************************************************************** 
 * Copyright (c) 2004, 2006 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: ProbeSrcListener.java,v 1.2 2006/02/06 20:18:15 nmehrega Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 

package org.eclipse.tptp.platform.probekit.util;

import java.util.ArrayList;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.tptp.platform.probekit.builder.ProbeNature;
import org.eclipse.tptp.platform.probekit.registry.ProbeRegistry;
import org.eclipse.tptp.platform.probekit.registry.ProbeRegistryException;


/**
 * This class listens for modifications to probe source files in the workspace.
 * In particular, we are trying to catch renaming, moving, and copying of
 * source files, plus re-opening of projects containing probe source.
 * 
 * <p>Why are these events interesting? For registry and property maintenance. We
 * use persistent properties on the probe source to keep track of the files
 * generated by a probe build. The registry uses this information for restoring
 * state and the rest of probekit uses it for resource cleanup and for exporting
 * probe sets.</p>
 * 
 * <p>Unfortunately, persistent properties are...persistent. They get copied
 * and moved around with the associated resource. So, if you copy a probe,
 * the new file ends up with properties referencing derived resources from the
 * original. Similarly, if you move a probe to a new project/folder, it will
 * still carry properties pointing back to the old (now invalid) derived
 * resources. Doh!</p>
 * 
 * <p>Also, when a project containing probes is closed and re-opened, the probes
 * should magically come back into existence in the workspace. That is, be
 * deployable, show up in the registry, etc. The properties makes this possible,
 * but only if we can tell when a probe source file has reappeared.</p>
 * 
 * <p>It turns out that all these situations can be handled by watching for
 * POST_CHANGE events in the workspace and catching ADDED deltas on probe
 * source files. It must be POST_CHANGE because these are the events which
 * carry the extra flags that allow us to distinguish rename/move from
 * copy/re-open.</p>
 * 
 * <p>POST_CHANGE events occur very frequently, so we try to minimize
 * our involvement by only delving into projects with the probe build nature.
 * And within such projects, only the top level of source directories.</p>
 * 
 * <p>It is critical to keep our processing as lightweight as possible since
 * change listeners put a burden on the workspace. If you grow a need to do
 * more work here, you'll need to make this more sophisticated so that the
 * changes can be done on a separate runnable thread.</p>
 * 
 * <p>This listener is used in 2 places: It is installed as a standard change
 * listener while our plugin is active, and it is used for a one-time walk
 * when our plugin first starts up. The latter is required because there may
 * be relevant resource changes while our plugin is inactive. For example, just
 * launching the workbench and moving a probe source file won't activate our
 * plugin (until a build occurs). To be sure we see everything, we need to let
 * our save participant walk the changes since the last activation. All this
 * handled in the main plugin class (ProbekitPlugin).</p>
 * 
 * <p>The work we do in both cases is exactly the same. However, during the
 * startup traversal, we only receive POST_AUTO_BUILD events (see the help on
 * ISaveParticipant). During normal execution (while our plugin is active) we
 * only listen to POST_CHANGE events. So, though you'll see code in here to
 * handle both event types, only one of them at a time is ever active.</p>
 * 
 * @author kcoleman
 * @see org.eclipse.tptp.platform.probekit.ProbekitPlugin
 * @see org.eclipse.tptp.platform.probekit.registry.ProbeRegistry
 * @see org.eclipse.tptp.platform.probekit.util.ProbeResourceBundle
 */
public class ProbeSrcListener implements IResourceChangeListener 
{	
	private final ProbeRegistry registry = ProbeRegistry.getRegistry();
	
	/**
	 * Defines what events this listener expects to handle during normal
	 * (active) execution of the plugin. You'll find that the code below also 
	 * handles POST_AUTO_BUILD events. This is done ONLY to support plugin 
	 * activation. The only time we should ever receive POST_AUTO_BUILD events 
	 * is when invoked via processResourceChangeEvents at startup. I've defined 
	 * this public mask to prevent future errors of the form "Oh, I wonder why 
	 * they left POST_AUTO_BUILD out of the mask but still process it".</p>
	 */
	public static int ACTIVE_EVENT_MASK = IResourceChangeEvent.POST_CHANGE;
	
	/**
	 * This is the top level entry into the listener. A given instance will be
	 * called either for POST_CHANGE events or for POST_AUTO_BUILD, but never
	 * both, as explained above. All the real work is done by the visitor class.
	 * 
	 * @see org.eclipse.core.resources.IResourceChangeListener#resourceChanged(org.eclipse.core.resources.IResourceChangeEvent)
	 */
	public void resourceChanged(IResourceChangeEvent evt) 
	{
		ArrayList srcFolders = null;
		IResource r = evt.getResource();
		IProject p = null;
		try {
			switch (evt.getType()) {
			case IResourceChangeEvent.POST_CHANGE:
			case IResourceChangeEvent.POST_AUTO_BUILD:	// plugin activation only
			{
//				trace("Change event received");
				evt.getDelta().accept(new ProbeSrcChangeVisitor());
			}
			break;
			}
		} catch (CoreException e) {
			// TODO do something about this
			trace("Got CoreException in resourceChanged");
		} catch (Exception e) {
			trace("Caught " + e + "resourceChanged");
		}
	}
	
	/**
	 * Grovel over a Java project and find all the source folders. Probe source
	 * files will only occur in the top level of project source folders. Source
	 * folders (containers) are identified in a Java project as classpath
	 * entries of type CPE_SOURCE.
	 * 
	 * @param p A Probe project (and therefore, known to also be a Java project).
	 * @return A list of all the source folders, as IPath's.
	 */
	static ArrayList buildJavaSrcFolderList(IProject p)
	{
		ArrayList result = null;
		IJavaProject javaProject = JavaCore.create(p);
		IClasspathEntry[] classpath;
		try {
			classpath = javaProject.getRawClasspath();
			result = new ArrayList(classpath.length);
			for (int i = 0, j = 0; i < classpath.length; i++) {
				IClasspathEntry entry = classpath[i];
				if (entry.getEntryKind() == IClasspathEntry.CPE_SOURCE) {
					result.add(entry.getPath());
				}
			}
		} catch (JavaModelException e) {
			// TODO Do something about this
			trace("Got JavaModelException while building src folder list");
		}
		return result;
	}
	
	/**
	 * Determine whether or not a folder in project under traversal might
	 * contain interesting probe source files. Probe source files can only
	 * occur in top level java project source containers.
	 * 
	 * <p>Note that it is not sufficient to check for an exact match because 
	 * we might be traversing our way through something like this: If a
	 * java project's build path specfies proj/fldr/fldr2 as a source
	 * container and we're at "proj/fldr", we need to keep going.</p>
	 * 
	 * @param srcFolders The list of source containers under consideration. The
	 * 		list elements should be IPath's.
	 * @param f The folder under consideration
	 * @return True if <i>f</i> is a source container or the parent of a
	 * 		source container.
	 */
	public boolean mightContainProbeSrc(ArrayList srcFolders, IFolder f) 
	{
		int nFolders = (srcFolders != null) ? srcFolders.size() : 0;
		IPath candPath = f.getFullPath();
		
		for (int i = 0; i < nFolders; i++ ) {
			IPath srcPath = (IPath)srcFolders.get(i);
			if ( srcPath.equals(candPath) || candPath.isPrefixOf(srcPath) ) {
				return true;
			}
		}
		return false;
	}
	
	/**
	 * Try to reconstruct a probe registry entry from a probe source resource.
	 * 
	 * <p>Reconstruction is done by ProbeResourceBundle, using the persistent
	 * properties on the source. If it fails because, for example, one or more
	 * of the derived files isn't available, the bundle properties are no longer
	 * valid and we erase them.</p>
	 * <p>If it fails because the props are missing, we just shrug it off - it
	 * is normal for this method to be called for probe source files which have 
	 * never been built and so have no bundle properties.</p>
	 * 
	 * @param r A probe source IFile resource to restore
	 */
	private void reconstructRegistryEntry(IResource r)
	{
		try {
			ProbeResourceBundle bundle = new ProbeResourceBundle(r);
			if ( bundle.isComplete() ) {
				registry.add(bundle);
			} else {
				bundle.removeResourceAssociations();
			}
		} catch (ProbeBundleBuildRequiredException e) {
			// this is a no-op...it means we've probably never
			// seen this file before, so there's no work to do
		} catch (ProbeBundleException e) {
			// erase the property settings to prevent future
			// difficulties.
			ProbeResourceBundle.removeResourceAssociations(r);
		} catch (InvalidProbeBundleException e) {
			// erase the property settings to prevent future
			// difficulties.
			ProbeResourceBundle.removeResourceAssociations(r);
		} catch (ProbeRegistryException e) {
			// erase the property settings to prevent future
			// difficulties.
			ProbeResourceBundle.removeResourceAssociations(r);
		}
	}
	
	/**
	 * This is where the rubber meets the road: Catch interesting probe source
	 * file change events and maintain the registry state and bundle
	 * properties accordingly.
	 * 
	 * <p>We try to trim the traversal as much as possible by only considering
	 * project with the probe build nature, and within such projects, only
	 * considering the top level of a source container.</p>
	 * 
	 * <p>When we finally hit a probe source file with an ADDED change, we
	 * try to determine whether it is a rename/move, a copy, or an add from
	 * re-opening a project. The actions associated with these events are:</p>
	 * <ul>
	 *  <li>Rename/move: Remove the registry entry and erase the bundle
	 * 		properties. The probe cannot be deployed or exported until it is
	 * 		rebuilt.</li>
	 *  <li>Copy: Erase the bundle properties that have been copied from the
	 * 		original file.</li>
	 *  <li>Re-open: Use the bundle properties to restore the probe to the
	 * 		registry, if possible.</li>
	 * </ul>
	 * 
	 * <p>We make an ordering assumption about the traversal: It will be top
	 * down. That is, we will always see a project before its folders and
	 * always see a folder before its contents. This is critical because it
	 * allows us to maintain the notion of "current project" across multiple
	 * invocations of visit().</p>
	 * 
	 * <p>Why do we care? Because we only want to delve into the source folders
	 * for the current project and we don't want to rebuild the folder list
	 * every time we traverse a folder in the project.</p>
	 * 
	 * <p>Also, to tell the difference between a copy and a the addition of a
	 * probe source during project re-open, we have to maintain current project
	 * state indicating whether or not this is fallout from an OPEN event on
	 * the project. Both opening and closing a project set the OPEN flag, but
	 * we don't have to worry about close because IProject.isOpen() will save
	 * us from the situation up front.</p.
	 * 
	 * @author kcoleman
	 */
	class ProbeSrcChangeVisitor implements IResourceDeltaVisitor
	{
		ArrayList srcFolders = null;
		boolean projOpening = false;
		
		public boolean visit(IResourceDelta delta) throws CoreException 
		{
	        IResource res = delta.getResource();
	        IProject p = null;
	        
	        // Examine the resource type to look for traversal trimming
	        // opportunities. You can't start with the delta type because
	        // that doesn't allow you to trim the walk. Trim out:
	        // Trim the tree walk by excluding;
	        // - projects w/o the probe build nature
	        // - non Java source containers
	        // - anything below the top level of a Java source container	        
	        switch (res.getType()) {
	        	case IResource.PROJECT:
	        		// Continue only if this is a project which might contain
	        		// probe source. (You must check for openness because this
	        		// visitor will run after PRE_CLOSE).
	        		p = (IProject) res;
	        		if ( p.isOpen() && p.hasNature(ProbeNature.NATURE_NAME) ) {
	        			srcFolders = buildJavaSrcFolderList(p);
	        			projOpening = (delta.getFlags() & IResourceDelta.OPEN) != 0;
	        			return true;
	        		} else {
	        			srcFolders = null;
	        			projOpening = false;
	        			return false;
	        		}
	        	case IResource.FOLDER:
	        		// Continue only if this is a folder which might contain
	        		// probe source. Note that we've already cut out folders in
	        		// non-probe nature projects, above.
	        		return mightContainProbeSrc(srcFolders, (IFolder)res);
	        	case IResource.FILE:
	        		// We're in a source folder of a probe-natured project. Do
	        		// we have a probe source file?
	        		if ( !res.getName().endsWith(ProbekitConstants.PROBE_SRC_EXT) ) {
	        			// not a probe source file
	        			return true;
	        		}
	        		break;
	        	default:
	        		// This traps things like IResource.ROOT (top of workspace)
	        		return true;
	        }
	        
        	// If we get here, we know we have are examining an a probe 
        	// source file located in a suitable project/folder.
	        
	        int changeType = delta.getKind();
	        int flags = delta.getFlags();
	        boolean moving = (flags & IResourceDelta.MOVED_TO) != 0 || 
							 (flags & IResourceDelta.MOVED_FROM) != 0;
	        if ( changeType == IResourceDelta.REMOVED && moving ) {
	        	// We're examining a probe source file that was moved to
	        	// another file/location. The properties come with it, but are
	        	// not valid in the new location, so we scrub them.
	        	//
	        	// We handle this on REMOVED not ADDED because if the destination 
	        	// is not in a probe project, we will never visit it and therefore 
	        	// never remove the persistent properties, so do it now. We need
	        	// to make sure the properties are scrubbed, in case the target
	        	// project subsequently acquires the probe build nature.
	        	IPath movedToPath = delta.getMovedToPath();
	        	IFile dest = 
	        		ResourcesPlugin.getWorkspace().getRoot().getFile(movedToPath);
	        	IResource r = delta.getResource();
	        	
	        	trace("Removing registry entry and bundle for " + dest);
	        	registry.remove(r);
	        	ProbeResourceBundle.removeResourceAssociations(dest);
	        } else if ( changeType == IResourceDelta.ADDED ) {
	        	// This is the target of a copy/move/rename, or a newly
	        	// added file (from creating a new file or re-opening a project).
	        	IResource r = delta.getResource();
	        	
	        	if ( moving ) {
	        		// This is the destination of a move/rename. We already
	        		// handled this above, when we saw the REMOVED change for
	        		// the original file. Do nothing now.
	        	} else if ( projOpening ) {
	        		// We are  re-opening a project which contains a probe source. 
	        		// Try to recreate the resource bundle and add it back to the
	        		// registry.
	        		trace("Reconstructing bundle for " + res.getFullPath());
	        		reconstructRegistryEntry(r);
	        	}  else {
	        		// This is a new probe source file. It may be the result
	        		// of copying an existing file, in which case, we need to
	        		// zorch the persistent properties as they belong to the
	        		// originator.
	        		trace("Removing bundle for " + res.getFullPath());
	        		ProbeResourceBundle.removeResourceAssociations(r);
	        	}
	        }
	        // visit the children (of which there are none anyway since if
	        // we end up here, we were visiting a file).
	        return true;
		}	
	} // class ProbeSrcChangeVisitor
	
	private static void trace(String message)
	{
		if ( ProbekitDebugConfig.TRACE_SRC_LISTENER && 
				message != null && message.length() > 0 ) {
			System.out.println("ProbeSrcListener: " + message);
		}
	}
		
/* I took these classes out because I don't think we need them. I think we can
 * do everything we need to just by listening to additions during POST_CHANGE.
 * But just in case I'm wrong... 
 */
//	class ProjCloseVisitor implements IResourceVisitor
//	{
//		ArrayList srcFolders = null;
//		
//		public ProjCloseVisitor(IProject p)
//		{
//			srcFolders = buildJavaSrcFolderList(p);
//		}
//		
//		public boolean visit(IResource r) throws CoreException 
//		{
//			switch (r.getType()) {
//				case IResource.FOLDER:
//					return isJavaSrcFolder(srcFolders, (IFolder)r);
//				case IResource.FILE:
//				{
//					IFile f = (IFile)r;
//					if ( r.getName().endsWith(ProbekitConstants.PROBE_SRC_EXT) ) {
//						// Remove the registry entry for this probe, but do not
//						// trash the properties...we want to use them to restore
//						// the registry entry when the project is re-opened.
//						trace("    Visiting " + f.getFullPath());
//						registry.remove(f);
//						return true;
//					} 
//					break;
//				}
//				default:
//					break;
//			}
//			return true;
//		}	
//	}	// class ProjCloseVisitor
//
//	class ProjDeleteVisitor implements IResourceVisitor
//	{
//		ArrayList srcFolders = null;
//		
//		public ProjDeleteVisitor(IProject p)
//		{
//			srcFolders = buildJavaSrcFolderList(p);
//		}
//		
//		public boolean visit(IResource r) throws CoreException 
//		{
//			switch (r.getType()) {
//				case IResource.FOLDER:
//					return isJavaSrcFolder(srcFolders, (IFolder)r);
//				case IResource.FILE:
//				{
//					IFile f = (IFile)r;
//					if ( r.getName().endsWith(ProbekitConstants.PROBE_SRC_EXT) ) {
//						trace("    Visiting " + f.getFullPath());
//						registry.remove(f);
//						ProbeResourceBundle.removeResourceAssociations(f);
//						return true;
//					} 
//					break;
//				}
//				default:
//					break;
//			}
//			return true;
//		}	
//	}	// class ProjCloseVisitor
}
