/********************************************************************** 
 * Copyright (c) 2008, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: class_interface.cpp,v 1.6 2009/11/21 22:19:35 jwest Exp $ 
 * 
 * Contributors: 
 * Intel - Initial API and implementation 
 **********************************************************************/ 
#ifdef WIN32
#pragma warning(disable:4786)
#endif

#include <stdlib.h>
#include <string.h>

#include <map>

#include <assert.h>

#include "class_inerface_int.h"
#include "BCIEng.h"
#include "JClassBuilder.h"
#include "ModuleJ.h"

#include "JMemStream.h"

#include <jni.h>

typedef unsigned long intptr;

static JNIEnv *(*m_get_vm_pointer)(void **);
static jclass m_j_l_class;
static jclass m_j_l_classloader;
static jmethodID m_j_l_class_getName; /*()Ljava/lang/String;*/
static jmethodID m_j_l_class_isInterface;/* ()Z */
static jmethodID m_j_l_class_getClassLoader; /*()Ljava/lang/ClassLoader;*/
static jmethodID m_j_l_classloader_findLoadedClass; /*(Ljava/lang/String;)Ljava/lang/Class;*/
static jmethodID m_j_l_classloader_getSystemClassLoader; /*()Ljava/lang/ClassLoader;*/
static jobject m_system_classloader;
/**
 * Define class handler
 */
/*[0xdead0001 - simplefied, 0xdead0002 - instrumented, 0xdead0000 - none]*/
#define CLASS_HANDLER_MASK 0xdeadffff
#define CLASS_HANDLER_SIMPLEFIED 0xdead0001
#define CLASS_HANDLER_INSTRUMENTED 0xdead0002
#define CLASS_HANDLER_NONE 0xdead0000
#define IS_CLASS_INSTRUMENTED(x) (((x)->magic & CLASS_HANDLER_MASK) == CLASS_HANDLER_INSTRUMENTED)
struct Class{
	int magic; 
	struct {
		class CModuleJ *instrumented_class;
		void *simplified_class;	
	} X; /*XXX: possible union here will be more applieble */
	int loaded;
	const char *class_name; /*used if loaded = 0*/
	struct Class *super_class;
};

typedef std::map<std::string,class_handler> name2ch_t;
static name2ch_t m_name2ch;

#define MEHTOD_HANDLER_MASK 0xbabeffff
#define METHOD_HANDLER_SELF 0xbabe0001
#define METHOD_HANDLER_EXT 0xbabe0002
#define METHOD_HANDLER_NONE 0xbabe0000
struct Method {
	int magic; 
	struct {
		class CMethodJ *instrumented_method;
		void *simplefied_method;
	} X; /*XXX: possible union here will be more applieble */
	class_handler class_owner;
	const char *method_name; /*used for METHOD_HANDLER_EXT only*/
	const char *method_descriptor; /*used for METHOD_HANDLER_EXT only*/
};

static struct Class *m_j_l_object_ch; 
/**
 * Define field handler
 */
typedef struct Field_ * field_handler;

/**
 * Define method handler
 */
//typedef struct Method_ * method_handler;

/**
 * Define class loader handler
 */
typedef struct ClassLoader_ * classloader_handler;

static int 
create_class_handler(class_handler *pch, int flag, const char *class_name, int loaded, void *dsk)
{
	class_handler ch0 = m_name2ch[class_name]; /*for debug*/
	if (m_name2ch[class_name] != NULL 
		&& m_name2ch[class_name]->magic == flag) {
			*pch = m_name2ch[class_name];
			return (0);
	} 
	class_handler ch = (class_handler)malloc(sizeof(struct Class));
	memset(ch, 0, sizeof(struct Class));
	assert(ch);
	ch->loaded = loaded;
	if (loaded == 0) {
		ch->class_name = class_name;
	}
	ch->magic = flag;
	switch(flag) {
	case (CLASS_HANDLER_SIMPLEFIED):
		ch->X.simplified_class = dsk;
		if (loaded == 0) {
			ch->class_name = class_name;
		}
		ch->loaded = loaded;
		break;
	case (CLASS_HANDLER_INSTRUMENTED):
		ch->X.instrumented_class = (CModuleJ *)dsk;
		break;
	default:
		assert(!"shouldn't be here!!!");
	}
	*pch = ch;
	m_name2ch[class_name] = ch;
	return (0);
}

extern "C" int
create_method_handler(method_handler *pmh, class_handler ch, int flag, void *dsk) 
{
	method_handler mh = (method_handler)malloc(sizeof(struct Method));
	assert(mh);
	memset(mh, 0, sizeof(struct Method));
	switch (flag) {
	case METHOD_HANDLER_SELF:
		mh->X.instrumented_method = (class CMethodJ *)dsk;
		break;
	case METHOD_HANDLER_EXT:
		mh->X.simplefied_method = dsk;
		break;
	default:
		assert(!"shouldn't enter here");
	}
	mh->magic = (IS_CLASS_INSTRUMENTED(ch))?METHOD_HANDLER_SELF:flag;
	mh->class_owner = ch;
	*pmh = mh;
	return (0);
}
/**
 * Class interface
 */

extern "C" unsigned short 
class_get_version( class_handler klass ) {
	if (IS_CLASS_INSTRUMENTED(klass)) {
		return klass->X.instrumented_class->GetClassBuilder().GetMajorVersion();
	} else {
		assert (!"not implemented yet!");
	}
	return (0);
}

extern "C" const char *
class_get_name( class_handler klass ) 
{
	JNIEnv *env;
	jstring class_name;
	if (IS_CLASS_INSTRUMENTED(klass)) {
		return klass->X.instrumented_class->GetName();
	} 
	else {
		if (klass->loaded == 0) {
			return(klass->class_name);
		}
		env = m_get_vm_pointer((void **)&env);
		class_name = (jstring)env->CallObjectMethod((jobject)klass->X.simplified_class, m_j_l_class_getName);
		if (class_name == NULL)
			return (NULL);
		char *class_name_simple = (char *)env->GetStringUTFChars(class_name, NULL);
		char *ch = strchr(class_name_simple, '.');
		while (ch != NULL) {
			*ch = '/';
			ch = strchr(ch, '.');
		}
		return(class_name_simple);
	
	}
	return (NULL);
}

extern "C" classloader_handler
class_get_class_loader( class_handler klass )
{
	if (IS_CLASS_INSTRUMENTED(klass)) {
		return (0);
	} else {
		assert(!"simplefied");
	}
	return (NULL);
}

extern "C" class_handler
class_get_super_class( class_handler klass )
{

	const char *class_name;
	const char *super_class_name = NULL;
	jobject this_class = NULL;
	jobject super_class;
	JNIEnv *env = NULL;
	env = m_get_vm_pointer((void **)&env);
	assert(env);
	int loaded = 1;
	if (klass->loaded == 0 && strcmp(klass->class_name,"java/lang/Object") == 0) return (NULL);
	if (klass->super_class != NULL) return (klass->super_class);
	
	if (IS_CLASS_INSTRUMENTED(klass)) {
		class_name = klass->X.instrumented_class->GetName();
		CConstPool* cpool = klass->X.instrumented_class->GetClass().GetConstPool();
		CCPUtf8Info* utfinfo = cpool->GetClass(klass->X.instrumented_class->GetClass().GetSuperClass());
		super_class_name = (const char *)utfinfo->GetBytes();
		assert(super_class_name);	
		super_class = (jclass)env->FindClass(super_class_name);		
	} 
	else {
		if (klass->loaded == 0) {
			assert(klass->class_name);
			this_class = (jclass)env->FindClass(klass->class_name);
			if (this_class == NULL && env->ExceptionCheck()) {
				//env->ExceptionDescribe();
				env->ExceptionClear();
				//XXX: class wasn't found or loaded
				return (m_j_l_object_ch);
				//return (NULL);
			}
		}
		this_class = this_class?this_class:(jobject)klass->X.simplified_class;
		super_class = (jobject)env->GetSuperclass((jclass)this_class);
	}

	if (super_class == NULL) {
		if (env->ExceptionCheck()) {
			//env->ExceptionDescribe();
			env->ExceptionClear();
			//XXX: class wasn't found or loaded
			loaded = 0;
			//return (NULL);
		}
		else {
			klass->super_class = m_j_l_object_ch;
			return (m_j_l_object_ch);
			//return (NULL); /*it happends when klass is java.lang.Object*/
		}
	}
	else {
		jstring jclass_name = (jstring)env->CallObjectMethod(super_class, m_j_l_class_getName);
		if (jclass_name == NULL) {
			return (m_j_l_object_ch);
		}
		char *class_name_simple = (char *)env->GetStringUTFChars(jclass_name, NULL);
		assert (class_name_simple);
		char *ch = strchr(class_name_simple, '.');
		while (ch != NULL) {
			*ch = '/';
			ch = strchr(ch, '.');
		}
		super_class_name = class_name_simple;
	}
	
	//assert(super_class);
	class_handler ch;
	jobject super_class_ref = env->NewGlobalRef(super_class);
	env->DeleteLocalRef(super_class);
	int status = create_class_handler(&ch, CLASS_HANDLER_SIMPLEFIED, super_class_name, loaded, super_class_ref);
	assert(status == 0);
	return (ch);
}

extern "C" unsigned
class_is_same_class( class_handler klass1, class_handler klass2 )
{
	assert(!"verifier entered!!");
	return 0;
}

#ifdef _SOLARIS	
	static char*  package_name(char* kname)
#else
	static char*  package_name(const char* kname)
#endif
{
	char *tokp;
	char *tok0;
	tokp = tok0 = (char *)&kname[0];
	tok0 = strchr((char *)kname, '/');
	while (tok0 != NULL) {
		tokp = tok0;
		tok0 = strchr(tok0 + 1, '/');
	}
	return (tokp);
}

extern "C" unsigned
class_is_same_package( class_handler klass1, class_handler klass2 ) 
{
	if (klass1 == klass2) {
		return (1);
	}
	else {
	
	#if defined(_SOLARIS) || defined(_SOLARISX86)
		const char *klass1_name = class_get_name(klass1);
		const char *klass2_name = class_get_name(klass2);

		// My solaris compiler barfs when converting chars* to const chars* implicitly - pk
		char* package1 = package_name((char*)klass1_name);
		char* package2 = package_name((char*)klass2_name);
	#else		
		const char *klass1_name = class_get_name(klass1);
		const char *klass2_name = class_get_name(klass2);

		char* package1 = package_name(klass1_name);
		char* package2 = package_name(klass2_name);
	#endif
	
		int package1_len = (package1 - klass1_name);
		int package2_len = (package2 - klass2_name);
		if (package1_len != package2_len) {
			return (0);
		}
	
		return(strncmp(klass1_name, klass2_name, package2_len) == 0);
		assert(!"verifier entered!!");
	}
	return (0);
}

extern "C" unsigned
class_is_interface( class_handler klass ) 
{
	JNIEnv *env;
	env = m_get_vm_pointer((void **)&env);
	assert(env);
	if (IS_CLASS_INSTRUMENTED(klass)) {
		return (0);
	} 
	else {
		return (unsigned)(env->CallBooleanMethod((jobject)klass->X.simplified_class, m_j_l_class_isInterface));
	}
	
	return (0);
}

extern "C" unsigned
class_is_array( class_handler klass ) 
{
	assert(!"verifier entered!!");
	return (0);
}

extern "C" unsigned
class_is_final_( class_handler klass ) 
{
	assert(!"verifier entered!!");
	return (0);
}

extern "C" unsigned short
class_get_superinterface_number( class_handler klass )
{
	assert(!"verifier entered!!");
	return (0);
}

extern "C" class_handler
class_get_superinterface( class_handler klass, unsigned short index )
{
	assert(!"verifier entered!!");
	return (NULL);
}

extern "C" class_handler
class_get_array_element_class( class_handler klass )
{
	assert(!"verifier entered!!");
	return (NULL);
}


extern "C" class_handler
class_get_extended_class( class_handler klass, const char *super_name )
{
	return (NULL);
}

extern "C" unsigned short
class_get_method_number( class_handler klass )
{
	assert(!"verifier entered!!");
	return (0);
}

extern "C" method_handler
class_get_method( class_handler klass, unsigned short index )
{
	assert(!"verifier entered!!");
	return (NULL);
}


/**
 * Constant pool inteface
 */

extern "C" unsigned short
class_cp_get_size( class_handler klass )
{
	assert(klass);
	if (IS_CLASS_INSTRUMENTED(klass)) {
		assert(klass->X.instrumented_class);
		assert(klass->X.instrumented_class->GetClassBuilder().GetConstPool());
		return klass->X.instrumented_class->GetClassBuilder().GetConstPool()->GetSize();
	} else {
		assert (!"not implemented yet!");
	}
	assert(!"verifier entered!!");
	return (0);
}


extern "C" unsigned char
class_cp_get_tag( class_handler klass, unsigned short index )
{
	if (IS_CLASS_INSTRUMENTED(klass)) {
		return (*klass->X.instrumented_class->GetClassBuilder().GetConstPool())[index]->GetTag();
	} else {
		assert (!"not implemented yet!");
	}
	return (0);
}

extern "C" unsigned short
class_cp_get_class_name_index( class_handler klass, unsigned short index )
{
	if (IS_CLASS_INSTRUMENTED(klass)) {
		CCPInfo * ccinfo = (*klass->X.instrumented_class->GetClassBuilder().GetConstPool())[index];
		assert(ccinfo->GetTag() == CONSTANT_Class);
		return ((CCPClassInfo *)ccinfo)->GetClassInd();
	} else {
		assert (!"not implemented yet!");
	}
	return (0);
}

extern "C" unsigned short
class_cp_get_ref_class_index( class_handler klass, unsigned short index )
{
	if (IS_CLASS_INSTRUMENTED(klass)) {
		CCPInfo * ccinfo = (*klass->X.instrumented_class->GetClassBuilder().GetConstPool())[index];
		unsigned char tag = ccinfo->GetTag();
		switch(tag) {
		case CONSTANT_Fieldref:
			return ((CCPFieldrefInfo *)ccinfo)->GetClassInd();
		break;
		case CONSTANT_Methodref:
			return ((CCPMethodrefInfo *)ccinfo)->GetClassInd();
		break;
		case CONSTANT_InterfaceMethodref:
			return ((CCPInterfaceMethodrefInfo *)ccinfo)->GetClassInd();
		break;
		default:
			assert(!"unsupported tag");
		}
	} else {
		assert (!"not implemented yet!");
	}
	return (0);
}

extern "C" unsigned short
class_cp_get_ref_name_and_type_index( class_handler klass, unsigned short index )
{
	if (IS_CLASS_INSTRUMENTED(klass)) {
		CCPInfo * ccinfo = (*klass->X.instrumented_class->GetClassBuilder().GetConstPool())[index];
		unsigned char tag = ccinfo->GetTag();
		switch(tag) {
		case CONSTANT_Fieldref:
			return ((CCPFieldrefInfo *)ccinfo)->GetNameAndTypeInd();
		break;
		case CONSTANT_Methodref:
			return ((CCPMethodrefInfo *)ccinfo)->GetNameAndTypeInd();
		break;
		case CONSTANT_InterfaceMethodref:
			return ((CCPInterfaceMethodrefInfo *)ccinfo)->GetNameAndTypeInd();
		break;
		default:
			assert(!"unsupported tag");
		}
	} else {
		assert (!"not implemented yet!");
	}
	return (0);
}

extern "C" unsigned short
class_cp_get_string_index( class_handler klass, unsigned short index )
{
	assert(!"verifier entered!!");
	return (0);
}

extern "C" unsigned short
class_cp_get_name_index( class_handler klass, unsigned short index )
{
	if (IS_CLASS_INSTRUMENTED(klass)) {
		CCPInfo * ccinfo = (*klass->X.instrumented_class->GetClassBuilder().GetConstPool())[index];
		unsigned char tag = ccinfo->GetTag();
		assert(tag == CONSTANT_NameAndType);
		return ((CCPNameAndTypeInfo *)ccinfo)->GetNameInd();
	} else {
		assert (!"not implemented yet!");
	}
	return (0);
}

extern "C" unsigned short
class_cp_get_descriptor_index( class_handler klass, unsigned short index )
{
	if (IS_CLASS_INSTRUMENTED(klass)) {
		CCPInfo * ccinfo = (*klass->X.instrumented_class->GetClassBuilder().GetConstPool())[index];
		unsigned char tag = ccinfo->GetTag();
		assert(tag == CONSTANT_NameAndType);
		return ((CCPNameAndTypeInfo *)ccinfo)->GetDescriptorInd();
	} else {
		assert (!"not implemented yet!");
	}
	return (0);
}

extern "C" const char *
class_cp_get_utf8_bytes( class_handler klass, unsigned short index )
{
	if (IS_CLASS_INSTRUMENTED(klass)) {
		return (const char *)((CCPUtf8Info *)(*klass->X.instrumented_class->GetClassBuilder().GetConstPool())[index])->GetBytes();
	} else {
		assert (!"not implemented yet!");
	}
	return (0);
}

extern "C" void
class_set_verify_data_ptr( class_handler klass, void *data )
{
	assert(!"verifier entered!!");
	return;
}

extern "C" void *
class_get_verify_data_ptr( class_handler klass )
{
	assert(!"verifier entered!!");
	return (NULL);
}

extern "C" method_handler
class_resolve_method( class_handler klass, unsigned short index )
{
	assert(!"verifier entered!!");
	return (NULL);
}

extern "C" field_handler
class_resolve_nonstatic_field( class_handler klass, unsigned short index )
{
	assert(!"verifier entered!!");
	return (NULL);
}

/**
 * Method interface
 */

extern "C" class_handler
method_get_class( method_handler hmethod )
{
	assert(hmethod);
	return hmethod->class_owner;
}

extern "C" const char *
method_get_name( method_handler method )
{
	if (method->magic == METHOD_HANDLER_SELF) {
		return (method->X.instrumented_method->GetName());
	}
	else {
		assert(!"verifier entered!!");
	}
	return (NULL);
}

extern "C" const char *
method_get_descriptor( method_handler method )
{
	if (method->magic == METHOD_HANDLER_SELF) {
		return (method->X.instrumented_method->GetSignature());
	}
	else {
		assert(!"verifier entered!!");
	}
	return (NULL);
}

extern "C" unsigned
method_get_bytecode_length( method_handler method )
{
	/*insert check about instrumentation*/
	if (method->magic == METHOD_HANDLER_SELF) {
		unsigned size = 0;
		CInstructions::const_iterator insIterator;
		if (method->X.instrumented_method->GetCodeAttribute() == NULL) return (0);
		return (method->X.instrumented_method->GetCodeAttribute()->GetCodeLength());
	}
	else {
		assert(!"verifier entered!!");
	}
	return (0);	
}

extern "C" unsigned char *
method_get_bytecode( method_handler method )
{
	if (method->magic == METHOD_HANDLER_SELF) {
		return (method->X.instrumented_method->GetCodeAttribute()->GetCode());
	}
	else {
		assert(!"verifier entered!!");
	}
	return (NULL);	
}

extern "C" unsigned
method_get_max_locals( method_handler method )
{
	if (method->magic == METHOD_HANDLER_SELF) {
		/*XXX: add check for instrumented method*/
		CMethodJ * jmethod = (CMethodJ *)(method->X.instrumented_method);
		return (jmethod->GetCodeAttribute()->GetMaxLocals());
	}
	else {
		assert(!"verifier entered!!");
	}
	return (0);	
	
}

extern "C" unsigned 
method_get_max_stack( method_handler method )
{
	if (method->magic == METHOD_HANDLER_SELF) {
		CMethodJ * jmethod = (CMethodJ *)(method->X.instrumented_method);
		return (jmethod->GetCodeAttribute()->GetMaxStack());
	}
	else {
		assert(!"verifier entered!!");
	}
	return (0);	
}

extern "C" unsigned
method_is_static( method_handler method )
{
	if (method->magic == METHOD_HANDLER_SELF) {
		return (method->X.instrumented_method->GetAccessFlags() & ACC_STATIC);
	}
	else {
		//assert(!"verifier entered!!");
		return (0); /*we're assuming that this function called after class_resolve_method and couldn't return static methods*/
	}
	return (0);	
	
}

extern "C" unsigned
method_is_protected( method_handler method )
{
	assert(!"verifier entered!!");
	return (0);
}

/**
 * Method exception handler
 */

extern "C" unsigned short
method_get_exc_handler_number( method_handler method )
{
	return (method->X.instrumented_method->GetExTable()->size());
}

extern "C" void
method_get_exc_handler_info( method_handler method, unsigned short index,
                             unsigned short *start_pc, unsigned short *end_pc,
                             unsigned short *handler_pc, unsigned short *catch_type )
{
	class CExTable& exc = method->X.instrumented_method->GetCodeAttribute()->GetExTable();
	*start_pc = exc[index].GetStartPC();
	*end_pc = exc[index].GetEndPC();
	*handler_pc = exc[index].GetHandlerPC();
	*catch_type = exc[index].GetCatchtype();

	return;
}

extern "C" unsigned short
method_get_number_exc_method_can_throw( method_handler hmethod )
{
	assert(!"verifier entered!!");
	return (0);
}

extern "C" const char *
method_get_exc_method_can_throw( method_handler hmethod, unsigned short index )
{
	assert(!"verifier entered!!");
	return (NULL);
}


extern "C" unsigned char *
method_get_stackmaptable( method_handler hmethod )
{
	
	CStackMapTableAttribute *attr = hmethod->X.instrumented_method->GetCodeAttribute()->GetStackMaps();
	if (attr == NULL)
		return (NULL);
	u4 length = attr->GetLength();
	u1 *stackmap = (u1 *)malloc(length);
	CJMemStream mem_stream;
	mem_stream.Open(stackmap, length);
	CJStream stream(&mem_stream);
	attr->Write(stream);
	//return (attr->GetInfo());
	return (stackmap);
}

extern "C" void 
method_modify_exc_handler_info(method_handler method, 
                                    unsigned short idx, 
                                    unsigned short start_pc,
                                    unsigned short end_pc, 
                                    unsigned short handler_pc, 
                                    unsigned short handler_cp_index )
{

	class CExTable& exc = method->X.instrumented_method->GetCodeAttribute()->GetExTable();
	exc[idx].SetStartPC(start_pc);
	exc[idx].SetEndPC(end_pc);
	exc[idx].SetHandlerPC(handler_pc);
	return;
}

extern "C" void 
method_remove_exc_handler( method_handler method, unsigned short idx )
{
	assert(!"verifier entered!!");
	return;
}

/**
 * Class loader interface
 */

extern "C" void
class_loader_set_verifier_data_ptr( classloader_handler classloader, void *data )
{
	assert(!"verifier entered!!");
	return;
}

extern "C" void *
class_loader_get_verifier_data_ptr( classloader_handler classloader )
{
	assert(!"verifier entered!!");
	return (NULL);
}

extern "C" void
class_loader_lock( classloader_handler classloader )
{
	assert(!"verifier entered!!");
	return;
}

extern "C" void
class_loader_unlock( classloader_handler classloader )
{
	assert(!"verifier entered!!");
	return;
}

extern "C" class_handler
class_loader_lookup_class( classloader_handler classloader, const char *name )
{
	if (m_name2ch[name] != NULL && IS_CLASS_INSTRUMENTED(m_name2ch[name])) 
		return (m_name2ch[name]);
	else
		return (NULL);
}

extern "C" class_handler
class_loader_load_class( classloader_handler classloader, const char *name )
{
	assert(!"verifier entered!!");
	return (NULL);
}

extern "C" unsigned
field_is_protected( field_handler field )
{
	assert(!"verifier entered!!");
	return (0);
}

class_handler
get_class_handler_from_builder(class CModuleJ *module) 
{
	class_handler ch;

	m_name2ch.clear();
	int status = create_class_handler(&m_j_l_object_ch, CLASS_HANDLER_SIMPLEFIED, "java/lang/Object", 0, NULL);
	assert(!status);
	
	status = create_class_handler(&ch, CLASS_HANDLER_INSTRUMENTED, module->GetName(), 1, module);
	assert(!status);
	return (ch);
}

method_handler 
get_method_handler_for_cmethod(class CMethod *method)
{
	class_handler ch = NULL;
	method_handler mh = NULL;
	int status;
	status = create_class_handler(&ch, CLASS_HANDLER_INSTRUMENTED, (const char *)method->GetModule()->GetName(), 1, method->GetModule());
	assert(!status);
	status = create_method_handler(&mh, ch, METHOD_HANDLER_SELF, method);
	assert(!status);
	return (mh);
}

extern "C" unsigned short
class_cp_get_class_entry(class_handler klass, const char* name)
{

	
	CCPClassInfo *cinfo;
	if (IS_CLASS_INSTRUMENTED(klass)) {
		/*for this class we'll have entry somewhere in some future */
		if (strcmp(klass->X.instrumented_class->GetName(), name) == 0) {
			return (klass->X.instrumented_class->GetClass().GetThisClass());
		}
		assert(name);
		cinfo = (klass->X.instrumented_class->GetClassBuilder().FindClass(name));
		if (cinfo == NULL) {
			/*should be created new class antry */
			cinfo = *klass->X.instrumented_class->GetClassBuilder().CreateClassConstant(name);
			assert(cinfo);
		} 
		return(cinfo->GetCpIndex());
	} else {
		assert (!"not implemented yet!");
	}
	return (0);
}

int
initialize_dynamic(get_vm_pointer_t a_get_vm_pointer)
{
	m_get_vm_pointer = (JNIEnv *(*)(void **))a_get_vm_pointer;
	JNIEnv *env;
	
	env = m_get_vm_pointer((void **)&env);
	assert(env);
	
	if (m_j_l_class) return (0);
	
	m_j_l_class = (jclass)env->NewGlobalRef(env->FindClass("java/lang/Class"));
		assert(m_j_l_class);
	
	m_j_l_class_getName = env->GetMethodID(m_j_l_class, "getName", "()Ljava/lang/String;");
	assert(m_j_l_class_getName);
	
	m_j_l_class_getClassLoader = env->GetMethodID(m_j_l_class, "getClassLoader", "()Ljava/lang/ClassLoader;");
	assert(m_j_l_class_getClassLoader);
	
	m_j_l_class_isInterface = env->GetMethodID(m_j_l_class, "isInterface", "()Z");
	assert(m_j_l_class_isInterface);
	
	m_j_l_classloader = (jclass)env->NewGlobalRef(env->FindClass("java/lang/ClassLoader"));
	assert(m_j_l_classloader);

	m_j_l_classloader_findLoadedClass = env->GetMethodID(m_j_l_classloader, "findLoadedClass", "(Ljava/lang/String;)Ljava/lang/Class;"); 
	assert(m_j_l_classloader_findLoadedClass);

	return (0);
}

void 
class_destroy_handler(class_handler ch) 
{
	if (ch == NULL) return;
	if (!IS_CLASS_INSTRUMENTED(ch) && ch->loaded == 1) {
		JNIEnv *env;
		env = m_get_vm_pointer((void **)&env);
		assert(env);
		env->DeleteGlobalRef((jobject)ch->X.simplified_class);
	
	}
	free(ch);
	ch = NULL;
}
