/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProbeAgentExtension.c,v 1.7 2009/02/26 15:49:57 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

/*******************************************************************************
 * This file contains the Probekit agent extension for the Hyades piAgent.
 * Agent extensions are libraries that are loaded by the main Hyades agent,
 * and they must export a function called agent_extension_init.
 *
 * ABOUT CONFIGURATION OPTIONS
 *
 * Configuration options can be read from the command line options
 * passed by the main agent (the string following -XrunLIBNAME:)
 * or from the RAC using ra_getDefaultConfiguration.
 *
 * When they're on the command line they have to have a special
 * prefix, "ext-pk-" so the piAgent option parser passes them through.
 *
 * The names of the command-line options are the same as the "types" of
 * the RAC configuration options that appear in serviceconfig.xml or
 * the agent configuration section of a pluginconfig.xml.
 *
 *
 * This agent extension respects these configuration options:
 *	logFile			the name or full path to the log file to append to
 *	logLevel		the log level: FINEST through SEVERE or NONE
 *	BCILibraryName	the name or full path to the BCI engine library to load
 *	engineScript	the name or full path to the engine script to load
 *					(usually comes on command line, not part of permanent config)
 *
 * Remember, when they're on the command line they have to have the
 * ext-pk- prefix.
 */

/*
 * The typedefs and structs necessary are defined in piAgentExtension.h,
 * which can be found in the Hyades source folder
 * org.eclipse.hyades.datacollection\collection\collectors\native\java_profiler.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h> /* for malloc */
#ifdef MVS
#include <unistd.h> /* for __etoa */
#endif
#include <jvmpi.h>

#include "piAgentExtension.h"

/*
 * The BCI engine is a DLL that exports a specific C-callable interface.
 * That interface is defined in BCIEngInterface.h.
 */
#include "BCIEngInterface.h"

#include "porting.h"
#include "RAComm.h"	/* for agent controller shared library locking logic */

/*
 * Forward references for delay-handled event registration
 */
static void processClassLoadHookEvent(JVMPI_Event *event, void **thread_local);
static void command_handler(ra_command_t *command);
static void init_done_handler(JVMPI_Event *event, void **thread_local);
static SetEventHandlerFunction set_event_handler;
static void InitializeEventHandlers(BOOL i_bDoAttach);

/*************************************************************************
 * OS400 specific stuff
 *
 * Put this whole file in ASCII mode using convert(819).
 *
 * Later we'll selectively use pragmas to undo this.
 *
 * Define __atoe and __etoa so they rewrite strings in place a la zOS.
 * Define string constants to minimize praga use
 */

#ifdef __OS400__

static void __etoa(char *str)
{
	char* temp = as400_etoa(str);
	strcpy(str, temp);
	ra_free(temp);
}

static void __atoe(char *str)
{
	char* temp = as400_atoe(str);
	strcpy(str, temp);
	ra_free(temp);
}

/* Can't get this to come from string.h properly */
static char* strdup(const char* input)
{
	char* output = (char*)malloc(strlen(input) + 1);
	strcpy(output, input);
	return output;
}

#pragma convert(819)	/* remainder of the file is ASCII unless suspended */
#endif

#if defined(__OS400__)
#	pragma convert(0)
	char fopen_mode_read[2] = "r";
	char fopen_mode_append[2] = "a";
#	pragma convert(819)
#elif defined(MVS)
#	pragma convlit(suspend)
	char fopen_mode_read[2] = "r";
	char fopen_mode_append[2] = "a";
#	pragma convlit(resume)
#else
	char fopen_mode_read[2] = "r";
	char fopen_mode_append[2] = "a";
#endif

/*******************************************************************************
 * Locking-related things
 *
 * We create just one BCI engine in this file, and it's not reentrant,
 * so we  have to serialize access to it. To do so we use a lock as
 * defined in the RAC SDK files hccomm.h and hcclco.lib: ra_mutexCreate,
 * ra_mutexEnter, ra_mutexExit.
 *
 * Note: this is a non-reentrant lock, so be sure no code path that
 * holds the lock can ever reach an ra_mutexEnter call!
 */

ra_critsec_t engineInstanceLock;

/*******************************************************************************
 * typedefs for the BCI engine's interface functions.
 * TODO: Why aren't these defined in the interface header file?
 */

typedef unsigned (*CreateBCIEngine_t)(pbcieng_t* o_eng);
typedef unsigned (*DestroyBCIEngine_t)(pbcieng_t i_eng);
typedef unsigned (*Initialize_t)(pbcieng_t i_pbcieng, char* i_pchOptions, size_t i_cbOptions);
typedef unsigned (*SetAllocator_t)(pbcieng_t i_pbcieng, pfnMalloc_t i_pfnMalloc);
typedef unsigned (*SetCallback_t)(pbcieng_t i_pbcieng, pfnCallback_t i_pfnCallback, unsigned i_uFlags);
typedef unsigned (*Instrument_t)(pbcieng_t i_pbcieng,
								  void* i_pInClass, size_t i_cbInClass,
							      void** o_ppOutClass, size_t* o_pcbOutClass);

/*******************************************************************************
 * Macro defining the option prefix for Probekit configuration options
 * in the piAgent option string.
 */

#define OPT_PREFIX "ext-pk-"
#define OPT_PREFIX_LEN 7

/*******************************************************************************
 * Log-related enums, constants, etc.
 */

enum {
	LOG_LEVEL_FINEST = 1,
	LOG_LEVEL_FINER,
	LOG_LEVEL_FINE,
	LOG_LEVEL_CONFIG,
	LOG_LEVEL_INFO,
	LOG_LEVEL_WARNING,
	LOG_LEVEL_SEVERE,
	LOG_LEVEL_NONE
};

static char* logLevelStrings[] = {
	"(zero)",
	"finest",
	"finer",
	"fine",
	"config",
	"info",
	"warning",
	"severe",
	"none",
	NULL
};

/*******************************************************************************
 * Globals
 */

static CreateBCIEngine_t	CreateBCIEngine_fn;
static DestroyBCIEngine_t	DestroyBCIEngine_fn;
static Initialize_t			Initialize_fn;
static SetAllocator_t		SetAllocator_fn;
static SetCallback_t		SetCallback_fn;
static Instrument_t			Instrument_fn;

static MODULE_REFERENCE		hProbeLib;
static BOOL					BCIEngineActive = FALSE;
static pfnMalloc_t			pfnMalloc = NULL;
static pbcieng_t			pbcieng = NULL;

static RA_AGENT_HANDLE		rac_agent_handle;
static char*				option_string;
static int					is_init_done;		/* initially false */
static int					global_debug_flag;	/* controls debugging output */

/* This flag prevents us from trying to reenter the lock */
static int					inside_init_done_handler;	/* initially false */

static BOOL*				jvm_init_done_ptr;	/* points to a flag which goes true when... */

static int					logLevel = LOG_LEVEL_SEVERE;
static BOOL					events_enabled = FALSE;	/* True if event handlers are initialized */

/*
 * The ATTACH_THREAD macro requires that our static JVM pointer variable have
 * the name _jvmpiAgent_jvm. So it does:
 */
static JavaVM*				_jvmpiAgent_jvm = NULL;



/******************************************************************************
 * Structure for the delay-load system.
 *
 * This agent is going to get commands from the Workbench through the RAC
 * giving it class files to instantiate. Those commands are going to come
 * before JVM_INIT_DONE, and we can't call DefineClass until after.
 * So we stash the classes we get into a list, and process the list
 * at JVM_INIT_DONE time.
 */

typedef struct DelayClassLoadList
{
	unsigned int	Size;
	unsigned char*	Data;
	char*			Name;
	BOOL			Loaded;
	struct DelayClassLoadList* pNext;
} DelayClassLoadList_t;

static DelayClassLoadList_t* pDelayLoad = NULL;

/*******************************************************************************
 * DenyInstrList: a list of class names that should NEVER be instrumented.
 * Every time we load a class dynamically, we record its name here
 * as one that should not be instrumented.
 *
 * TODO: management of this list is not thread safe; should it be?
 */

static struct _DenyInstrList {
	char* className;
	struct _DenyInstrList* next;
} *DenyInstrList = NULL;

/*******************************************************************************
 * get_value_from_option_string
 *
 *     -------------------------------------------------------------
 *     |                                                           |
 *     | THIS FUNCTION RETURNS A STRING THAT THE CALLER MUST FREE! |
 *     |                                                           |
 *     -------------------------------------------------------------
 *
 * Scan option_string for "pattern" preceded by comma or start of string,
 * and followed by an equals sign or the end of the string.
 * If there's a match (using STRICOLL), pull out the value (what follows
 * the equals sign) up to the next comma or the end of the string,
 * in a malloc'ed string buffer that the caller must free.
 *
 * Return null if not found.
 *
 * ASCII/EBCDIC note on MVS (zOS, 390): everything here is ASCII: the
 * pattern argument, the option_string, and the character literals
 * in this source function.
 *
 * On AS400, the option string from the command line is EBCDIC.
 */
char*
get_value_from_option_string(const char* pattern)
{
	int patlen;
	int wordlen;
	int valuelen;
	const char* eq;
	const char* end;
	char* wordbuf;
	char* return_string = NULL;
	const char* start = option_string;
	if (start == NULL)
		return NULL;

#if defined(__OS400__)
	__etoa(option_string);
#endif

	patlen = strlen(pattern);

	/* "while" loop scans by commas */
	while (start != NULL && *start) {
		/* find next equals sign */
		eq = strchr(start, '=');
		if (eq == NULL) {
			/* return_string is null, return it. */
			break;
		}
		wordlen = eq - start;
		if (wordlen == patlen) {
			wordbuf = (char*)malloc(wordlen + 1);
			strncpy(wordbuf, start, wordlen);
			wordbuf[wordlen] = '\0';
			if (STRICOLL(wordbuf, pattern) == 0) {
				/* Match! Value starts after equals sign */
				start += wordlen + 1;
				end = strchr(start, ',');
				if (end == NULL)
					end = start + strlen(start);
				valuelen = end - start;
				free(wordbuf);
				return_string = (char*)malloc(valuelen + 1);
				strncpy(return_string, start, valuelen);
				return_string[valuelen] = '\0';
				break;
			}
			else {
				/* No match; free "word" and fall through */
				free(wordbuf);
			}
		}
		/* else the option in the string isn't even the
		 * same length as the pattern we want, so fall through */

		/* advance past the next comma, if any */
		start = strchr(start, ',');
		if (start != NULL)
			++start;
	}

	/* fall out on match or no match */
#if defined(__OS400__)
	__atoe(option_string);	/* undo the conversion */
#endif
	return return_string;
}

/*******************************************************************************
 * getConfigurationOption
 *
 *     -------------------------------------------------------------
 *     |                                                           |
 *     | THIS FUNCTION RETURNS A STRING THAT THE CALLER MUST FREE! |
 *     |                                                           |
 *     -------------------------------------------------------------
 *
 * Read a configuration option from wherever options come from.
 *
 * One place they come from is the command line options, which should
 * be stored in option_string by now. That is a comma-separated
 * list of name=value expressions, and the "name" will be matched
 * against type_pattern in this function.
 *
 * If there isn't a match from the g_optionString, and rac_agent_handle
 * is not NULL, then this function consults the name/type/value configuration
 * set available from ra_getDefaultConfiguration().
 * This is the agent configuration; it includes things that appear
 * in the master serviceconfig.xml and proper parts of our pluginconfig.xml.
 * "Proper parts" are those within <Agent> that look like this:
 * <Option name="somename" type="sometype" value="somevalue" />
 *
 * In that case, name_pattern and type_pattern are both used.
 *
 * Implementation note: if the string comes from the command-line option
 * string, the caller has to free it. If it comes from getDefaultConfiguration,
 * we make a copy so the caller STILL has to free it - for consistency.
 *
 * ASCII/EBCDIC note on MVS (zOS, 390): the name_pattern and type_pattern
 * arguments are ASCII, but the configuration strings are EBCDIC. We have
 * to do conversions in order to compare, and convert the value string
 * to ASCII while we're returning it.
 */

static char* getConfigurationOption(char* name_pattern, char* type_pattern)
{
	/*     -------------------------------------------------------------
	 *     |                                                           |
	 *     | THIS FUNCTION RETURNS A STRING THAT THE CALLER MUST FREE! |
	 *     |                                                           |
	 *     -------------------------------------------------------------
	 */

	char* value;
	char* prefixed_type_pattern = (char*)malloc(strlen(type_pattern) + OPT_PREFIX_LEN + 1);
	strcpy(prefixed_type_pattern, OPT_PREFIX);
	strcat(prefixed_type_pattern, type_pattern);

	value = get_value_from_option_string(prefixed_type_pattern);
	free(prefixed_type_pattern);
	if (value != NULL) {
		return value;
	}

	/* didn't find it in option_string, look in the RAC agent configuration */
	if (rac_agent_handle != NULL) {
		ra_agentConfigList_t* clist = ra_getDefaultConfiguration(rac_agent_handle);
		ra_agentConfigListEntry_t* listEntry;
		if (clist == NULL)
			return NULL;

#if defined(MVS) || defined(__OS400__)
		name_pattern = strdup(name_pattern);
		type_pattern = strdup(type_pattern);
		__atoe(name_pattern);
		__atoe(type_pattern);
#endif
		for (listEntry = clist->head; listEntry != NULL; listEntry = listEntry->next) {
			ra_agentConfigEntry_t* e = &listEntry->entry;
			const char* name, *type, *value;
			if (e->name.length == 0) name = "";
			else name = e->name.data;
			if (e->type.length == 0) type = "";
			else type= e->type.data;
			if (e->value.length == 0) value = "";
			else value = e->value.data;
			if ((STRICOLL(name, name_pattern) == 0) &&
				(STRICOLL(type, type_pattern) == 0))
			{
				/* Return a copy of the string, so the caller must free it. */
				char *ret = strdup(value);
#if defined(MVS) || defined(__OS400__)
				free(name_pattern);
				free(type_pattern);
				/* convert the matching value to ASCII for returning */
				__etoa(ret);
#endif
				return ret;
			}
		}
	}

	/* all failed */
#if defined(MVS) || defined(__OS400__)
	/* undo conversion of name_pattern and type_pattern */
	__etoa(name_pattern);
	__etoa(type_pattern);
#endif
	return NULL;

}

/*------------------------------------------------------------------------------
 * Function to make a string xml-safe for being in a string.
 * Fix quotation marks, ampersands, greater-than, less-than.
 * Always returns a newly-malloc'ed string: like strdup in that regard.
 */
struct {
	char in;
	const char* out;
} xml_chars[] = {
	{ '"', "&quot;" },
	{ '&', "&amp;" },
	{ '<', "&lt;" },
	{ '>', "&gt;" }
};
#define NUM_XML_CHARS (sizeof(xml_chars) / sizeof(xml_chars[0]))
#define MAX_XML_SUBST_LENGTH 6

static char*
makeXMLSafe(const char* msg)
{
	int i;
	int offender_count = 0;
	for (i = 0; i < NUM_XML_CHARS; i++) {
		if (strchr(msg, xml_chars[i].in) != NULL) {
			offender_count++;
		}
	}
	if (offender_count == 0) {
		/* No need to change anything */
		return strdup(msg);
	}
	else {
		/* Else we have to change something. */
		/* Allocate a new string, long enough as if all offenders will have max substitution length */
		int new_len = strlen(msg) + (MAX_XML_SUBST_LENGTH * offender_count) + 1;
		char* new_msg = (char*)malloc(new_len);

		const char* p = msg;
		char* q = new_msg;
		while (*p) {
			for (i = 0; i < NUM_XML_CHARS; i++) {
				if (*p == xml_chars[i].in) {
					strcpy(q, xml_chars[i].out);
					q += strlen(q);
					break;
				}
			}
			if (i == NUM_XML_CHARS) {
				/* did not change this char */
				*q++ = *p;
			}
			p++;
		}
		*q = '\0';
		return new_msg;
	}
}

/*
 * getLogFile: return the FILE* to use for logging.
 * This function tries a variety of methods the first time it's called,
 * and after that returns the result of the first try, whatever it was.
 */

static FILE* getLogFile(void)
{
	static int already_attempted = FALSE;
	static FILE* logFile = NULL;

	if (!already_attempted) {
		char* logFileName;

		already_attempted = TRUE;
		logFileName = getConfigurationOption("ProbeAgentExtension", "logFile");
		if (logFileName == NULL) {
			return NULL;
		}
		else {
			if (STRICOLL(logFileName,"stderr") == 0) {
				/* Special case string value: log to stderr */
				logFile = stderr;
			}
			else {
#if defined(MVS) || defined(__OS400__)
				logFile = fopen(logFileName, fopen_mode_append);
				if (logFile == NULL) {
					/* maybe it was ASCII */
					__atoe(logFileName);
					logFile = fopen(logFileName, fopen_mode_append);
				}
#else
				logFile = fopen(logFileName, "a");
#endif
			}
			/* must free strings obtained from getConfigurationOption */
			free(logFileName);
		}
	}

	/* Return the result of the first attempt - even if it failed. */
	/* Due to already_attempted, will not keep trying to open the file. */
	return logFile;
}

/*------------------------------------------------------------------------------
 * logMessage: report a message to the log file using standard formatting,
 * if logLevel is low enough.
 *
 * ASCII/EBCDIC note: on MVS (zOS, 390) and AS400 the message and arg
 * are ASCII. We convert to EBCDIC before writing. This whole function
 * is in convlit(suspend) so string literals (e.g. in sprintf) are
 * EBCDIC. Also, we use sprintf and then fputs because of a bug in zOS
 * version 1.5 and 1.6 runtime libraries. Ref z/OS patch PQ89263.
 *
 * Remember, makeXMLSafe always dup's its argument, so we don't need
 * to un-convert the result.
 */

/* Time and timeb macros lifted from Hyades RAC sources. */
#include <time.h>
#ifndef __OS400__
#include <sys/timeb.h>
#endif
#define TM		struct tm
#ifdef _WIN32
#define TIMEB  struct _timeb
#define FTIME(param)  _ftime(param)
#elif __OS400__
#define TIMEB time_t
#define FTIME(param) time(param)
#else											/* else */
#define TIMEB	struct timeb
#define FTIME(param)  ftime(param)
#endif

/* the entire logMessage function needs EBCDIC literals for sprintf etc. */
#if defined(MVS)
#pragma convlit(suspend)
#elif defined(__OS400__)
#pragma convert(0)
#endif

static void logMessage(int level, const char* msg, const char* arg)
{
	if (logLevel <= level) {
		FILE* logFile = getLogFile();
		if (logFile != NULL) {
			char* sev = logLevelStrings[level];
			char* safe_msg = makeXMLSafe(msg);
			TM* timeFields;
			TIMEB timeb;
			char timebuf[20];

			FTIME(&timeb);
#ifdef __OS400__
			timeFields=localtime(&timeb);
#else
			timeFields=localtime(&(timeb.time));
#endif
			sprintf(timebuf, "%d:%d:%d:%d:%d:%d",
				timeFields->tm_year+1900,
				timeFields->tm_mon+1,
				timeFields->tm_mday,
				timeFields->tm_hour,
				timeFields->tm_min,
				timeFields->tm_sec);

#if defined(MVS) || defined(__OS400__)
			sev = strdup(sev);
			__atoe(sev);
			__atoe(safe_msg);
#endif
			if (arg == NULL) {
				{ /* substitute for fprintf, which crashes on MVS zOS */
				  char *msgbuf = (char*)malloc(strlen(safe_msg) + strlen(timebuf) + strlen(sev) + 64);
				  sprintf(msgbuf, "<MESSAGE time=\"%s\" severity=\"%s\" text=\"%s\"/>\n", timebuf, sev, safe_msg);
				  fputs(msgbuf, logFile);
				  free(msgbuf);
				}
			}
			else {
				char* safe_arg = makeXMLSafe(arg);
#if defined(MVS) || defined(__OS400__)
				__atoe(safe_arg);
#endif
				{ /* substitute for fprintf, which crashes on MVS zOS */
				  char *msgbuf = (char*)malloc(strlen(safe_msg) + strlen(safe_arg) + strlen(timebuf) + + strlen(sev) + 64);
				  sprintf(msgbuf, "<MESSAGE time=\"%s\" severity=\"%s\" text=\"%s\" param=\"%s\"/>\n", timebuf, sev, safe_msg, safe_arg);
				  fputs(msgbuf, logFile);
				  free(msgbuf);
				}
				free(safe_arg);
			}
			free(safe_msg);
#if defined(MVS) || defined(__OS400__)
			free(sev);
#endif
			fflush(logFile);
		}
	}
}
#if defined(MVS)
#pragma convlit(resume)
#elif defined(__OS400__)
#pragma convert(819)
#endif

/*------------------------------------------------------------------------------
 * This function is passed to the BCI engine, and the engine calls it
 * in order to ask, "are you *sure* you want me to instrument this?"
 * We should say no if we haven't seen JVM_INIT_DONE yet, and also
 * if the class name matches our (harsh, hard-coded) class list
 * specified above in szFilters.
 */
static int BCICallback(const char* i_pInfo, size_t i_cbInfo, unsigned i_wFlags)
{
	int nRet = 1;
	int i = 0;
	char *szName;

	if(!is_init_done)
	{
		nRet = 0;
	}
	else if(i_wFlags == BCIENGINTERFACE_CALLBACK_MODULE)
	{
		struct _DenyInstrList* deny_item;

		szName = (char*)malloc(i_cbInfo + 1);
		memcpy(szName, i_pInfo, i_cbInfo);
		szName[i_cbInfo] = 0;

		/* look on the list */
		deny_item = DenyInstrList;
		while (deny_item != NULL) {
			if(strcmp(deny_item->className, szName) == 0)
			{
				nRet = 0;
				break;
			}
			deny_item = deny_item->next;
		}
		free(szName);
	}
	else if (i_wFlags == BCIENGINTERFACE_CALLBACK_MODULE_INSTR) {
		/*
		 * This is a report that this class was actually instrumented.
		 * If the logging level is high enough, write a message saying so.
		 * (Because of the way the API is defined, we must make a null-terminated string by hand.)
		 */
		if (logLevel <= LOG_LEVEL_FINEST) {
			char* nameBuf = (char*)malloc(i_cbInfo + 1);
			memcpy(nameBuf, i_pInfo, i_cbInfo);
			nameBuf[i_cbInfo] = '\0';
			logMessage(LOG_LEVEL_FINEST, "IWAC0101I: Class got some instrumentation", nameBuf);
			free(nameBuf);
		}
	}
	return nRet;
}

/*******************************************************************************
 * initializeProbeEngine: called at agent_extension_init time to load and
 * initialize the BCI engine library.
 *
 * This function is only called when we hold the lock.
 */
static jint initializeProbeEngine(void)
{
	unsigned uResult = 0;
	char* bciLibraryName;

	bciLibraryName = getConfigurationOption("ProbeAgentExtension", "BCILibraryName");
	if (bciLibraryName == NULL) {
		logMessage(LOG_LEVEL_SEVERE, "IWAC0102S: can't get BCI library name from agent configuration", NULL);
		return JNI_ERR;
	}
	logMessage(LOG_LEVEL_CONFIG, "IWAC0103I: BCI library name", bciLibraryName);

#if defined(MVS) || defined(__OS400__)
	__atoe(bciLibraryName);
#endif
	hProbeLib = LOAD_LIBRARY(bciLibraryName);
#if defined(MVS) || defined(__OS400__)
	__etoa(bciLibraryName);
#endif

	if(NULL == hProbeLib)
	{
		logMessage(LOG_LEVEL_SEVERE, "IWAC0104S: can't load BCI library", bciLibraryName);
		logMessage(LOG_LEVEL_SEVERE, "IWAC0105S: additional load failure information", DLERROR());
		free(bciLibraryName);
		return JNI_ERR;
	}

#if defined(MVS)
#pragma convlit(suspend)
#elif defined(__OS400__)
#pragma convert(0)
#endif
	CreateBCIEngine_fn = (CreateBCIEngine_t)RESOLVE_ENTRY_POINT(hProbeLib, "CreateBCIEngine");
	DestroyBCIEngine_fn = (DestroyBCIEngine_t)RESOLVE_ENTRY_POINT(hProbeLib, "DestroyBCIEngine");
	Initialize_fn = (Initialize_t)RESOLVE_ENTRY_POINT(hProbeLib, "Initialize");
	SetAllocator_fn = (SetAllocator_t)RESOLVE_ENTRY_POINT(hProbeLib, "SetAllocator");
	SetCallback_fn = (SetCallback_t)RESOLVE_ENTRY_POINT(hProbeLib, "SetCallback");
	Instrument_fn = (Instrument_t)RESOLVE_ENTRY_POINT(hProbeLib, "Instrument");
#if defined(MVS)
#pragma convlit(resume)
#elif defined(__OS400__)
#pragma convert(819)
#endif

	if(NULL == CreateBCIEngine_fn || NULL == DestroyBCIEngine_fn)
	{
		logMessage(LOG_LEVEL_SEVERE, "IWAC0106S: invalid BCI library", bciLibraryName);
		hProbeLib = NULL;
		free(bciLibraryName);
		return JNI_ERR;
	}
	uResult = CreateBCIEngine_fn((pbcieng_t*)&pbcieng);
	if(0 != uResult)
	{
		logMessage(LOG_LEVEL_SEVERE, "IWAC0107S: failed to initialize BCI library", bciLibraryName);
		hProbeLib = NULL;
		free(bciLibraryName);
		return JNI_ERR;
	}
	SetCallback_fn(pbcieng, BCICallback, 0xFFFF);
	BCIEngineActive = TRUE;
	logMessage(LOG_LEVEL_INFO, "IWAC0108I: BCI engine initialized successfully", NULL);
	free(bciLibraryName);
	return JNI_OK;
}

/*******************************************************************************
 *
 * processProbeScript: when you use JVM command line options to drive
 * the probe engine in standalone mode (with no RAC), call this function
 * to process the engine script file named on the command line.
 *
 * This function does nothing if the BCI engine library has not been loaded.
 *
 * This function is only entered when we hold the lock.
 */
jint processProbeScript(char* i_szProbeName)
{
	jint nRet = JNI_OK;
	FILE* pfProbe;
	size_t cb = 0;
	char* pchProbe = NULL;
	size_t cbProbe = 0;	/* size of the file */
	size_t cbRead;		/* number of bytes actually read */
	unsigned error_code;

	if(hProbeLib = NULL)
	{
		return JNI_ERR;
	}

#if defined(MVS) || defined(__OS400__)
	__atoe(i_szProbeName);
#endif
	pfProbe = fopen(i_szProbeName, fopen_mode_read);
#if defined(MVS) || defined(__OS400__)
	__etoa(i_szProbeName);
#endif

	if (NULL == pfProbe)
	{
		logMessage(LOG_LEVEL_SEVERE, "IWAC0109S: can't open probe script file", i_szProbeName);
		return JNI_ERR;
	}
	fseek(pfProbe, 0, SEEK_END);
	cbProbe = ftell(pfProbe);
	if (cbProbe <= 0) {
		/* empty file or some error */
		logMessage(LOG_LEVEL_SEVERE, "IWAC0100S: probe script file is empty or error getting size", i_szProbeName);
		fclose(pfProbe);
		return JNI_ERR;
	}
	pchProbe = (char*)malloc(cbProbe+1);
	fseek(pfProbe, 0, SEEK_SET);
	cbRead = fread(pchProbe, 1, cbProbe, pfProbe);

	/* Check for errors, but allow for newline conversions changing the char count. */
	/* (Newline conversions apparently happen on OS400 at least) */
	if (cbRead <= 0) {
		logMessage(LOG_LEVEL_SEVERE, "IWAC0100S: error reading probe script file", i_szProbeName);
		fclose(pfProbe);
		free(pchProbe);
		return JNI_ERR;
	}
	pchProbe[cbProbe] = '\0';	/* null-terminate the buffer */

#if defined(MVS) || defined(__OS400__)
	/* convert the file contents to ASCII */
	__etoa(pchProbe);
#endif
	for(cb = 0; cb < cbProbe; cb++)
	{
		if(*(pchProbe + cb) == '\n')
			*(pchProbe + cb) = 0;
	}

	/* Note: we can only get here when we hold the lock */
	logMessage(LOG_LEVEL_FINE, "IWAC0110I: processing probe script file", i_szProbeName);
	error_code = Initialize_fn(pbcieng, pchProbe, cbProbe);

	if (error_code == 0)
		logMessage(LOG_LEVEL_INFO, "IWAC0111I: probe script file processed successfully", i_szProbeName);
	else
		logMessage(LOG_LEVEL_SEVERE, "IWAC0112S: error processing probe script file", i_szProbeName);

	free(pchProbe);
	fclose(pfProbe);
	return (error_code == 0 ? JNI_OK : JNI_ERR);
}

/*******************************************************************************
 * Process the probescript command coming from the Workbench through the RAC.
 * Turn every newline into a null byte.
 *
 * This method is only called when we already hold the lock.
 *
 * Returns JNI_OK or JNI_ERR.
 */
jint processRemoteProbeScript(unsigned int cbProbe, unsigned char* pbProbe)
{
	jint nRet = JNI_OK;
	unsigned int cb;

	if(!events_enabled)
	{
		InitializeEventHandlers(TRUE);
	}

	if (!BCIEngineActive)
		nRet = initializeProbeEngine();
	if(nRet == JNI_OK)
	{
		int error_code;
		for(cb = 0; cb < cbProbe; cb++)
		{
			unsigned char c = pbProbe[cb];
			if(c == '\n' || c == '\r')
				pbProbe[cb] = 0;
		}
		error_code = Initialize_fn(pbcieng, (char*)pbProbe, cbProbe);
		if (error_code == 0)
			logMessage(LOG_LEVEL_INFO, "IWAC0113I: remote probe script processed", NULL);
		else
			logMessage(LOG_LEVEL_SEVERE, "IWAC0114S: probe script received from remote has errors", NULL);
	}

	return nRet;
}

/*******************************************************************************
 * createClassFromBuffer: given a memory buffer full of bytes, instantiate
 * a class.
 *
 * Returns a global reference to the jclass object for this class.
 *
 * IMPORTANT NOTE about the global reference to the loaded jclass:
 * The return value from DefineClass is a local reference to the jclass
 * object for the loaded class. (It's an instance of java.lang.Class.)
 * We turn this into a global reference so the JVM won't think we're
 * finished with the class as soon as we return from the event handler.
 *
 * THIS GLOBAL REFERENCE IS LEAKED: it is not kept anywhere and it
 * is never released. We want the JVM to keep the class loaded forever,
 * because we never know when we'll do BCI that causes it to be referenced.
 */
jclass createClassFromBuffer(JNIEnv* env,
							 unsigned int size,
							 unsigned char* data,
							 char* name)
{
	jint		result = JNI_OK;
	jclass		cls = (jclass)NULL;


	/* put this class on the DenyInstrList */
	struct _DenyInstrList* new_item = (struct _DenyInstrList*)malloc(sizeof(*DenyInstrList));
	new_item->className = strdup(name);
	new_item->next = DenyInstrList;
	DenyInstrList = new_item;

	if(NULL == _jvmpiAgent_jvm) {
		logMessage(LOG_LEVEL_SEVERE, "IWAC0127S: JNI DefineClass call failed for a deployed class", name);
		return (jclass)NULL;
	}
	cls = (jclass)(*env)->DefineClass(env, name, (jclass)NULL, (jbyte*)data, size);
	if((jclass)NULL != cls)
	{
		cls = (jclass)(*env)->NewGlobalRef(env, cls);
	}
	else {
		logMessage(LOG_LEVEL_SEVERE, "IWAC0127S: JNI DefineClass call failed for a deployed class", name);
	}
	return cls;
}

/*******************************************************************************
 * addClassToDelayList: given a buffer full of bytes and some metadata,
 * add the class to the delay load list. It will be loaded later, after
 * JVM_INIT_DONE.
 *
 * No need to acquire the lock; we're called while the lock is held.
 *
 * This function makes a copy of the name string and the buffer it gets.
 */
void addClassToDelayList(unsigned int size, unsigned char* data, char* name)
{
	DelayClassLoadList_t* scan = pDelayLoad;
	DelayClassLoadList_t* new_class = NULL;
	new_class = (DelayClassLoadList_t*)malloc(sizeof(DelayClassLoadList_t));
	new_class->Size = size;
	new_class->Data = malloc(size);
	memcpy(new_class->Data, data, size);
	new_class->Name = strdup(name);
	new_class->pNext = NULL;
	if(scan == NULL)
	{
		pDelayLoad = new_class;
	}
	else
	{
		while(scan->pNext != NULL)
		{
			scan = scan->pNext;
		}
		scan->pNext = new_class;
	}
}

/*******************************************************************************
 * loadDelayedClasses: process the delay load list and load the indicated classes.
 *
 * This function FREES the "name" and "data" parts of the delay class load list
 * elements that it instantiates, and sets the Loaded flag TRUE.
 *
 * No need to acquire a lock: we're called while the instance lock is held.
 *
 * Returns JNI_OK or JNI_ERR.
 */
jint loadDelayedClasses(DelayClassLoadList_t* list)
{
	DelayClassLoadList_t* next = list;

	JNIEnv*		env  = NULL;				/* JNI environment */
	jint		result = JNI_OK;

	FILE* pfProbe=NULL;

	if(!BCIEngineActive) {
		return result;
	}

	/* Call GetEnv to get the JNIEnv pointer. If we aren't attached,
	 * then call AttachCurrentThread to attach.
	 *
	 * (Calling AttachCurrentThread when you're already attached is supposed
	 * to be harmless, but on AS400 it throws some kind of exception.)
	 *
	 * Don't ever detach: this thread is already attached, because this
	 * code is called from the JVM_INIT_DONE event handler. When already attached,
	 * AttachCurrentThread is a no-op as far as the JVM is concerned.
	 * (For example, there is not a counter inside the JVM to match attaches and detaches.)
	 */
	result = (*_jvmpiAgent_jvm)->GetEnv(_jvmpiAgent_jvm, &env, JNI_VERSION_1_2);
	if (result != JNI_OK) {
		result = ATTACH_THREAD(env);
	}
	else {
		logMessage(LOG_LEVEL_FINE, "In loadDelayedClasses was already attached", NULL);
	}
	if(result != JNI_OK) {
		/* todo: log a real error here */
		logMessage(LOG_LEVEL_SEVERE, "IWAC0128S: Error getting JNI environment in loadDelayedClasses", NULL);
		return result;
	}

	while(next != NULL)
	{
		jclass newClass;
		newClass = createClassFromBuffer(env, next->Size, next->Data, next->Name);
		if((jclass)NULL != newClass)
			next->Loaded = TRUE;
		/* see comment about "the global reference to the loaded jclass" */
		free(next->Data);
		free(next->Name);
		next = next->pNext;
	}

	return result;
}

/*******************************************************************************
 * instantiateRemoteClass: called by the command processor when it gets
 * a class over the wire. This function stashes the class on the delay list
 * if we haven't seen JVM_INIT_DONE yet, otherwise it instantiates the
 * class immediately.
 *
 * No need to acquire a lock: we're called when the lock is held.
 */
static jint instantiateRemoteClass(unsigned int size, unsigned char* data, char* name)
{
	jint	nRet = JNI_OK;
	jint	jni_result = JNI_OK;
	BOOL	was_attached;
	void*	jenv  = NULL;				/* JNI environment */

	logMessage(LOG_LEVEL_FINE, "IWAC0117I: deployed class received", name);

	if(is_init_done)
	{
		/* Make sure thread is attached to JVM before making JNI calls
		 * If thread wasn't attached, then attach it and detach before
		 * exit */
		jni_result = (*_jvmpiAgent_jvm)->GetEnv(_jvmpiAgent_jvm, &jenv, JNI_VERSION_1_2);
		if(jni_result == JNI_OK)
		{
			was_attached = TRUE;
		}
		else if(jni_result == JNI_EDETACHED)
		{
			was_attached = FALSE;
			ATTACH_THREAD(jenv);
		}

		nRet = ATTACH_THREAD(jenv);
		createClassFromBuffer(jenv, size, data, name);
		(*_jvmpiAgent_jvm)->DetachCurrentThread(_jvmpiAgent_jvm);

		if(!was_attached)
		{
			DETACH_THREAD();
		}
	}
	else
	{
		addClassToDelayList(size, data, name);
	}
	return nRet;
}

/*******************************************************************************
 * processClassLoadHookEvent: our handler for class load hook.
 * This function actually calls the BCI instrumentation engine
 * in response to the JVM's class load hook notification,
 * but only if BCIEngineActive.
 */
void processClassLoadHookEvent(JVMPI_Event *event, void **thread_local)
{
	void* pInClass = event->u.class_load_hook.class_data;
	unsigned cbInClass = event->u.class_load_hook.class_data_len;
	void* outData;
	size_t outSize;
	unsigned uResult = 0;

	if(BCIEngineActive)
	{
		if (!inside_init_done_handler) {
			/*
			 * Don't try to acquire the lock if we're inside the init done
			 * handler. (We get here from inside the init done handler
			 * when it processes the deferred class list: it defines the
			 * classes and we get classLoadHook for each one.)
			 */
			ra_mutexEnter(&engineInstanceLock);
		}
		/* Set the allocator function every time. */
		/* It might seem wasteful but it's safe against JVM oddities. */
		pfnMalloc = event->u.class_load_hook.malloc_f;
		SetAllocator_fn(pbcieng, pfnMalloc);

		outData = NULL;
		Instrument_fn(pbcieng, pInClass, cbInClass, &outData, &outSize);

		if (outData == NULL) {
			/* Guess the instrumenter did not do anything - maybe it threw an exception. */
			/* Recover gracefully: feed the input class back to the JVM */
			event->u.class_load_hook.new_class_data = pInClass;
			event->u.class_load_hook.new_class_data_len = cbInClass;
		}
		else {
			event->u.class_load_hook.new_class_data = outData;
			event->u.class_load_hook.new_class_data_len = outSize;
		}
		if (!inside_init_done_handler) {
			ra_mutexExit(&engineInstanceLock);
		}
	}
	else
	{
		event->u.class_load_hook.new_class_data = pInClass;
		event->u.class_load_hook.new_class_data_len = cbInClass;
	}
}


/*******************************************************************************
 */
void init_done_handler(JVMPI_Event *event, void **thread_local)
{
	ra_mutexEnter(&engineInstanceLock);
	is_init_done = TRUE;
	inside_init_done_handler = TRUE;	/* prevent reentering the mutex */
	loadDelayedClasses(pDelayLoad);
	pDelayLoad = NULL;
	inside_init_done_handler = FALSE;
	ra_mutexExit(&engineInstanceLock);
}

/*******************************************************************************
 */

void command_handler(ra_command_t *command)
{
	switch (command->tag) {
		case RA_BINARY_CUSTOM_COMMAND:
		{
			unsigned int	cbHeader = 0;
			unsigned int	cbData = 0;
			unsigned char*	pbHeader;
			unsigned char*	pbData;
			char*			szClassName = NULL;

			unsigned char* pScan = command->info.custom_command.message.data;

			/* Process command header */
			pbHeader = pScan;
			cbHeader = strlen(pScan);
			pScan += cbHeader;
			pScan++;

			/* Process command data
			 * Workbench always sends four bytes big-endian
			 * (because it's a Java operation on a ByteBuffer).
			 * Combine them into an int, even if this machine is different.
			 * (Waste of time if this machine is also big-endian. Sue me.)
			 */
			cbData = *pScan++;
			cbData = (cbData << 8) | *pScan++;
			cbData = (cbData << 8) | *pScan++;
			cbData = (cbData << 8) | *pScan++;

			pbData = pScan;
			/* Execute custom command (so far only BCI engine is using this feature). */
			if(strcmp((char*)pbHeader, "PROBE_SCRIPT") == 0)
			{
				jint nRet = JNI_OK;

				logMessage(LOG_LEVEL_FINE, "IWAC0118I: PROBE_SCRIPT command received", NULL);
				if (logLevel <= LOG_LEVEL_FINER) {
					/* If we are logging at "finer" level then show contents of engine scripts */
					/* Make a copy of the engine script and turn nulls into newlines for printing */
					/* ASCII/EBCDIC note: because the script came through in a binary custom command, */
					/* the script is ASCII, not EBCDIC. That's what logMessage expects. */
					unsigned char* strbuf = malloc(cbData + 1);
					unsigned char* p;
					memcpy(strbuf, pbData, cbData);
					strbuf[cbData] = '\0';
					for (p = strbuf; p < strbuf + cbData; p++) {
						if (*p == '\0') *p = '\n';
					}
					logMessage(LOG_LEVEL_FINER, "IWAC0119I: PROBE_SCRIPT contents (multiple lines)", strbuf);
					free(strbuf);
				}

				ra_mutexEnter(&engineInstanceLock);
				if(!BCIEngineActive)
				{
					nRet = initializeProbeEngine();
				}
				if(nRet == JNI_OK)
				{
					processRemoteProbeScript(cbData, pbData);
				}
				ra_mutexExit(&engineInstanceLock);
			}
			else if(strcmp((char*)pbHeader, "REMOTE_CLASS") == 0)
			{
				pScan += cbData;
				szClassName = pScan;
				logMessage(LOG_LEVEL_FINE, "IWAC0120I: REMOTE_CLASS command received:", szClassName);
				ra_mutexEnter(&engineInstanceLock);
				instantiateRemoteClass(cbData, pbData, szClassName);
				ra_mutexExit(&engineInstanceLock);
			}
			break;
		}
	}
}

static void
initializeLogLevel(void)
{
	int i;
	char* logLevelString = getConfigurationOption("ProbeAgentExtension", "logLevel");
	if (logLevelString == NULL) return;
	for (i = 0; logLevelStrings[i] != NULL; i++) {
		if (STRICOLL(logLevelStrings[i], logLevelString) == 0) {
			logLevel = i;
			free(logLevelString);
			return;
		}
	}
	logMessage(LOG_LEVEL_WARNING, "IWAC0121W: invalid log level string in configuration", logLevelString);
	free(logLevelString);
}


/*******************************************************************************
 *
 */
static void InitializeEventHandlers(BOOL i_bDoAttach)
{
	void*	jenv = NULL;
	BOOL	was_attached = FALSE;
	jint	jni_result = JNI_OK;

	if(events_enabled)
		return;
	/*
	 * Some of the functions that are going to be invoked below
	 * (set_command_handler, set_event_handler) are using JNI calls.
	 * Make sure the current thead is attached or attach it.
	 * It is important to detach the thread if it was not attached before.
	 * Otherwise the VM will never exit.
	 * If event_enable_safety_flag is set then the attach is not necessary.
	 */
	if(i_bDoAttach)
	{
		jni_result = (*_jvmpiAgent_jvm)->GetEnv(_jvmpiAgent_jvm, &jenv, JNI_VERSION_1_2);
		if(jni_result == JNI_OK)
		{
			was_attached = TRUE;
		}
		else if(jni_result == JNI_EDETACHED)
		{
			was_attached = FALSE;
			ATTACH_THREAD(jenv);
			logMessage(LOG_LEVEL_FINEST, "IWAC0123I: JNI thread attached", NULL);
		}
	}
	else
	{
		was_attached = TRUE;
	}

	/*= JVM Scope Begin ======================================================*/
	/*= Attention! All JVM operations should be invoked between this line and */
	/*= the 'JVM Scope End comment                                            */
	(*set_event_handler)(JVMPI_EVENT_JVM_INIT_DONE, init_done_handler);
	(*set_event_handler)(JVMPI_EVENT_CLASS_LOAD_HOOK, processClassLoadHookEvent);
	events_enabled = TRUE;

	/*= JVM Scope End =========================================================*/
	if (!was_attached)
	{
		DETACH_THREAD();
	}
}

/*******************************************************************************
 * agent_extension_init: the one exported function from this library, called by
 * the main piAgent at initialization time, during JVM_OnLoad.
 *
 * Register the command handler and our two event handlers.
 * We don't need to do anything else: everything we do is based on
 * reacting to commands and JVMPI events.
 *
 * Also, to support debugging and stand-alone operation, look for and
 * process a JVM property called "ProbeScript." If it's set, it's the
 * full path to an engine script file that we should use. In that case
 * there won't be any dynamic class loading - the classes should already
 * be in CLASSPATH or BOOTCLASSPATH or something.
 */
JNIEXPORT void agent_extension_init(AgentExtensionInitArgs *args)
{
	FILE *f;
	if(NULL != _jvmpiAgent_jvm)
	{
		/* Protection from reentering */
		return;
	}

#ifdef _WIN32
	if ((f = fopen("c:\\PI_AGENT_DEBUG_BREAK", "r")) != 0) {
		fprintf(stderr, "delete c:\\PI_AGENT_DEBUG_BREAK to stop this breakpoint\n");
		fclose(f);
		DebugBreak();
	}
#endif

	/*
	 * Initialize the lock.
	 * Important: this is a non-reentrant lock, so be sure no
	 * code path that holds the lock can reach an "ra_mutexEnter" call!
	 */
	ra_mutexCreate(&engineInstanceLock);

	_jvmpiAgent_jvm = args->jvm;
	rac_agent_handle = args->agent_handle;
	option_string = strdup(args->options);

	initializeLogLevel();
	logMessage(LOG_LEVEL_INFO, "IWAC0122I: initializing", NULL);

	(*args->set_command_handler)(command_handler);
	set_event_handler = args->set_event_handler;
	/* HACK!!! HACK!!! HACK!!!
	 * If we have !event_enable_safety_flag then we can't be sure if the
	 * init_done notification was processed by the agent. This is most likely
	 * the attach scenario and we assume that the next remote class loading
	 * request will be the result of a human action (e.g. launch with probes request)
	 * By that time the VM is done initializing except some corner cases when
	 * a Java application is driven by a test script.
	 *
	 * (Refresher: safety flag is true in standalone mode and controlled mode,
	 * false in enabled mode and application mode. In those modes, RAC commands
	 * are coming asynchronously from JVM events like init done.)
	 *
	 */
	if (!args->event_enable_safety_flag) {
		logMessage(LOG_LEVEL_INFO, "event_enable_safety_flag not set", NULL);
		is_init_done = TRUE;
	}
	else {
		InitializeEventHandlers(FALSE);
	}

#if 0

NOT YET - ENABLE THIS CODE
WHEN HYADES HAS JVM_INIT_DONE_PTR
IN THE ARGS STRUCT, AND USE IT
TO MANAGE THE ATTACH SCENARIO.

	/* If api_version >= 2 then we can
	 * get a pointer to a BOOL which will go TRUE when jvm_init_done has happened.
	 */
	if (args->api_version >= 2)
		jvm_init_done_ptr = args->jvm_init_done_ptr;
	else
		jvm_init_done_ptr = NULL;
#endif

	/*
	 * See if there's a configuration option for "probescript" and process it if so.
	 * This will generally only happen in standalone mode - people won't
	 * generally hard-code a probescript into their config file.
	 * But hey, they might!
	 */

	{
		char* engine_script_file_name = getConfigurationOption("ProbeAgentExtension", "probescript");
		if (engine_script_file_name != NULL) {
			int nRet;
			logMessage(LOG_LEVEL_CONFIG, "IWAC0125I: processing probescript config option", engine_script_file_name);

			ra_mutexEnter(&engineInstanceLock);
			nRet = initializeProbeEngine();
			if(nRet == JNI_OK) {
				processProbeScript(engine_script_file_name);
			}
			else {
				logMessage(LOG_LEVEL_SEVERE, "IWAC0126S: failed to initialize probe engine (tried because of probescript config option)", engine_script_file_name);
			}
			ra_mutexExit(&engineInstanceLock);
			free(engine_script_file_name);
		}
	}
}

