/********************************************************************** 
 * Copyright (c) 2005, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: porting.c,v 1.11 2009/11/21 22:19:34 jwest Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 

/*******************************************************************************
 * platform-indepdendent locking logic
 * copied from lock3.c / lock3.h from the Eclipse Hyades RAC project
 * org.eclipse.hyades.datacollection
 */

#include <stddef.h>
#include "porting.h"

BOOL initializeLock(Lock_t *lock) {
	BOOL fOk;
#ifdef _WIN32

	// Initialize all data members to NULL so that we can
	// accurately check whether an error has occurred.
	lock->hEventNoReaders = NULL;

	// This critical section guards access to the other objects
	// managed by this data structure and also indicates
	// whether there are any writer threads writing.
	InitializeCriticalSection(&lock->cs);
	
	// Create the manual-reset event that is signalled when
	// no reader threads are reading.
	// Initially no reader threads are reading.
	lock->hEventNoReaders = CreateEvent(NULL,
										TRUE,
										TRUE,
										NULL);


	// Initialize the variable that indicates the number of
	// reader threads that are reading.
	lock->numReaders=0;

	if ((NULL == lock->hEventNoReaders)) {
		// If a synchronization object could not be created,
		// destroy any created objects and return failure.
		deleteLock(lock);
		fOk = FALSE;
	} else {
		fOk = TRUE;
	}

	// Return TRUE upon success, FALSE upon failure.
	return(fOk);

#else
#if defined _LINUX_X86 || defined _LINUX_X86_64 || defined _LINUX_390 || defined _HPUX || (defined(__s390__) && defined(__linux__)) || defined _SOLARIS || defined _SOLARISX86
 pthread_mutex_init(&(lock->mutex), /* Create mutex */
					0);				/* Default behaviour (unlocked) */

 pthread_cond_init(&(lock->rcond),	/* Create read condition variable */
				   0);				/* Default behaviour */

 pthread_cond_init(&(lock->wcond),	/* Create write condition variable */
				   0);				/* Default behaviour */

 lock->waiting_writers=0;
 lock->readCount = 0;
 return TRUE;
#else
	pthread_rwlock_init(lock, NULL);
	return TRUE;
#endif

#endif
}



void deleteLock(Lock_t *lock) {
#ifdef _WIN32
	if (NULL != lock->hEventNoReaders)
		CloseHandle(lock->hEventNoReaders);
#elif defined _LINUX_X86 || defined _LINUX_X86_64 || defined _LINUX_390 || defined _HPUX || (defined(__s390__) && defined(__linux__)) || defined _SOLARIS || defined _SOLARISX86
  /* do nothing */
#else
	pthread_rwlock_destroy(lock);
#endif
}



int getWriteLock(Lock_t *lock) {
#ifdef _WIN32
	WaitForSingleObject(lock->hEventNoReaders, INFINITE);
	EnterCriticalSection(&lock->cs);
	return 0;
#elif defined _LINUX_X86 || defined _LINUX_X86_64 || defined _LINUX_390 || defined _HPUX || (defined(__s390__) && defined(__linux__)) || defined _SOLARIS || defined _SOLARISX86
	pthread_mutex_lock(&(lock->mutex));
	lock->waiting_writers++;
	while(lock->readCount || lock->active_writers ) {
		pthread_cond_wait(&(lock->wcond), &(lock->mutex));
	}
	lock->active_writers++;
	lock->waiting_writers--;
	pthread_mutex_unlock(&(lock->mutex));
#else
	pthread_rwlock_wrlock(lock);
	return 0;
#endif
}



void releaseWriteLock(Lock_t* lock) {
#ifdef _WIN32
	LeaveCriticalSection(&lock->cs);
#elif defined _LINUX_X86 || defined _LINUX_X86_64 || defined _LINUX_390 || defined _HPUX || (defined(__s390__) && defined(__linux__)) || defined _SOLARIS || defined _SOLARISX86
	pthread_mutex_lock(&(lock->mutex));
	if(!--lock->active_writers) {
		pthread_cond_broadcast(&lock->rcond);
	}
	else {
		pthread_cond_signal(&lock->wcond);
	}
	pthread_mutex_unlock(&(lock->mutex));
#else
	pthread_rwlock_unlock(lock);
#endif
}



int getReadLock(Lock_t *lock) {
#ifdef _WIN32
	EnterCriticalSection(&lock->cs);
	if (lock->numReaders++==1) {
		// If this is the first reader thread,
		// set our event to reflect this.
		ResetEvent(lock->hEventNoReaders);
	}
	LeaveCriticalSection(&lock->cs);
	return 0;
#elif defined _LINUX_X86 || defined _LINUX_X86_64 || defined _LINUX_390 || defined _HPUX || (defined(__s390__) && defined(__linux__)) || defined _SOLARIS || defined _SOLARISX86
	pthread_mutex_lock(&(lock->mutex));
	if(lock->waiting_writers || lock->active_writers) {
		pthread_cond_wait(&(lock->rcond), &(lock->mutex));
	}
	lock->readCount++;
	pthread_mutex_unlock(&(lock->mutex));
	return 0;
#else
	pthread_rwlock_rdlock(lock);
	return 0;
#endif
}




void releaseReadLock (Lock_t *lock) {
#ifdef _WIN32

	EnterCriticalSection(&lock->cs);
	if (!--lock->numReaders) {
		SetEvent(lock->hEventNoReaders);
	}
	LeaveCriticalSection(&lock->cs);
#elif defined _LINUX_X86 || defined _LINUX_X86_64 || defined _LINUX_390 || defined _HPUX || (defined(__s390__) && defined(__linux__)) || defined _SOLARIS || defined _SOLARISX86
	pthread_mutex_lock(&(lock->mutex));
	if (!--lock->readCount) {
		pthread_cond_signal(&(lock->wcond));
	}
	pthread_mutex_unlock(&(lock->mutex));
#else
	pthread_rwlock_unlock(lock);
#endif
}


