/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse
 * Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionHarnessPlugin.java,v 1.9 2007/05/03 16:53:36 paules Exp $
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/
package org.eclipse.hyades.execution.harness;

import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.runtime.ILog;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.execution.harness.util.ISystemUtility;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;

/**
 * Plugin class for the execution harness plug-in; also provides other utilities
 * exposed through its implementation of the system utility interface. System
 * utilities are provided through the standalone system utility class when in
 * standalone mode.
 * 
 * @author Scott E. Schneider 
 * @see ISystemUtility
 * @since 3.2
 */
public class ExecutionHarnessPlugin extends Plugin implements ISystemUtility {

    /**
     * A singleton instance of the plugin class is stored for access by the get
     * default method.
     */
    private static ExecutionHarnessPlugin singleton;

    /**
     * Provides public static access to the singleton instance of this plug-in
     * class.
     * 
     * @return the singleton instance stored in this class
     */
    public static ExecutionHarnessPlugin getDefault() {
        return ExecutionHarnessPlugin.singleton;
    }

    /**
     * Provides public static access to the plugin id for this plug-in, this is
     * defined in the plugin.xml file.
     * 
     * @return the plugin id for this plug-in
     * @deprecated this method is just a convenience method to access the plugin
     *             id and should eventually be changed to private access since
     *             it is assumed to be used internally only
     */
    public static String getPluginId() {
        return ExecutionHarnessPlugin.getDefault().getPlugId();
    }

    /**
     * Retrieve the plugin id for this plugin, the corresponding public static
     * version of this method has been deprecated.
     * 
     * @return the plugin id is returned
     * @see #getPluginId()
     */
    private String getPlugId() {
        return ExecutionHarnessPlugin.getDefault().getBundle()
                .getSymbolicName();
    }

    /**
     * The OSGi bundle that this plug-in is associated with, needed for various
     * information at run-time such as plugin id and resource string retrieval.
     */
    private Bundle bundle;

    /**
     * The resource bundle associated with this plug-in for the appropriate
     * locale at run-time.
     */
    private ResourceBundle resourceBundle;

    /**
     * Default constructor for the execution harness plugin, state is passed in
     * via the lifecycle-related methods start/stop providing the bundle
     * context. This class is meant to be a singleton but is not enforced by
     * this class, it is enforced by the caller that must only create us once.
     */
    public ExecutionHarnessPlugin() {
        super();
        ExecutionHarnessPlugin.singleton = this;
    }

    /**
     * Resolves the plugin's resource bundle.
     * <p>
     * If the plugin's resource bundle can not be resolved, <code>null</code> is returned.
     * <p>
     * IMPORTANT: Do not use this method to retrieve values from the resource bundle.  This method 
     * is provided so this plugin's resource bundle can be used as the parent of another plugin's 
     * resource bundle.
     * <p>
     * 
     * @return The plugin's resource bundle, otherwise <code>null</code>.
     * @deprecated As of TPTP V4.4, use the plug-in's resource bundle (e.g. {@link org.eclipse.hyades.execution.harness.internal.resources.ExecutionHarnessPluginResourceBundle} for this plug-in) for resolving resources.  The plugin.properties property file ONLY contains messages for the this plug-in's MANIFEST.MF and plugin.xml files.
     */
    public ResourceBundle getResourceBundle() {
        return this.resourceBundle;
    }

    /**
     * Resolves the localized message associated with the parameter <code>key</code>
     * based on the default locale from the plugin's resource bundle.
     * <p>
     * If the <code>key</code> does not exist in the plugin's resource bundle, the 
     * <code>key</code> is returned.
     * <p>
     * Noteworthy, the resultant message is not formatted (e.g. no message parameter substitution). 
     * <p>
     * 
     * @param key The <code>key</code> of the message in the plugin's resource bundle.
     * @return The localized message associated with the parameter <code>key</code> from the plugin's resource bundle, otherwise the <code>key</code>.
     * @deprecated As of TPTP V4.4, use the plug-in's resource bundle (e.g. {@link org.eclipse.hyades.execution.harness.internal.resources.ExecutionHarnessPluginResourceBundle} for this plug-in) for resolving resources.  The plugin.properties property file ONLY contains messages for the this plug-in's MANIFEST.MF and plugin.xml files.
     */
    public String getString(String key) {
    	ResourceBundle bundle = ExecutionHarnessPlugin.getDefault().getResourceBundle();
        try {
            return (bundle != null) ? bundle.getString(key) : key;
        } catch (MissingResourceException e) {
            return key;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.harness.util.ISystemUtility#logError(java.lang.Throwable)
     */
    public void logError(Throwable exc) {
        Status status = new Status(1, ExecutionHarnessPlugin.getPluginId(), 0,
                exc.toString(), exc);
		try {
			ILog log = this.getLog();
			log.log(status);
			exc.printStackTrace();
		} catch (Throwable t) {
			System.out.println("Unable to log exception from " + this);
			t.printStackTrace();
		}
    }

    /*
     * Stores the OSGi bundle and appropriate resource bundles for this plug-in
     * while first calling the super implementation of the start method.
     * 
     * (non-Javadoc)
     * 
     * @see org.osgi.framework.BundleActivator#start(org.osgi.framework.BundleContext)
     */
    public void start(BundleContext context) throws Exception {
        super.start(context);
        this.bundle = context.getBundle();
        this.resourceBundle = Platform.getResourceBundle(this.bundle);
    }

    /*
     * As instructed by the base class, a call to the super stop method is
     * required. This method is only overrided to provide a spot for potential
     * clean-up that is needed in the future.
     * 
     * (non-Javadoc)
     * 
     * @see org.osgi.framework.BundleActivator#stop(org.osgi.framework.BundleContext)
     */
    public void stop(BundleContext context) throws Exception {
        super.stop(context);
    }

}