/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IDataProcessorObservable.java,v 1.4 2007/05/02 19:36:14 paules Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.harness;


/**
 * An extended interface introduced to avoid breaking clients but also add new
 * functionality -- this interface adds the ability to listen for data processor
 * events. This is very helpful for learning when execution is truly done
 * digesting events and cleaning up the models.
 * 
 * Observers add them self to the observable waiting for the observed to notify
 * back via the observer callback interface. The context is passed back to the
 * various event calls to give the callback target the context of the change,
 * basically it indicates which observable this event concerns.
 * 
 * The state of a data processor goes started->stopped->cleaned, and this is the
 * expected order of the event calls.
 *  
 * @author Scott E. Schneider
 */
public interface IDataProcessorObservable {

	/**
	 * Observer nested interface, used as the callback for when observed events
	 * take place and notification is sent out, used in conjunction with the add
	 * and remove observer methods
	 * 
	 * @see IDataProcessorObservable#addObserver(IDataProcessorObservable.Observer)
	 * @see IDataProcessorObservable#removeObserver(IDataProcessorObservable.Observer)
	 * 
	 * @author Scott E. Schneider
	 */
	public interface Observer {
		/**
		 * Notification that the observed cleaned up their state
		 * 
		 * @param context
		 *            the data processor that is cleaning itself
		 */
		public void clean(IDataProcessorObservable context);

		/**
		 * Notification that the observed started its function
		 * 
		 * @param context
		 *            the data processor that is starting up
		 */
		public void start(IDataProcessorObservable context);

		/**
		 * Notification that the observed stopped its function
		 * 
		 * @param context
		 *            the data processor that has stopped its function, occurs
		 *            before the clean event, but after the start event
		 */
		public void stop(IDataProcessorObservable context);
	}

	/**
	 * Add an observer to this data processor, ensure that a matching remove
	 * observer call is used when observation is no longer needed
	 * 
	 * @param observer
	 *            the callback interface instance
	 * 
	 * @see #removeObserver(IDataProcessorObservable.Observer)
	 */
	public void addObserver(IDataProcessorObservable.Observer observer);

	/**
	 * Removes an observer to this data processor, use the add observer method
	 * to add observers for observation of this data processor
	 * 
	 * @param observer
	 *            the callback interface instance
	 * 
	 * @see #addObserver(IDataProcessorObservable.Observer)
	 */
	public void removeObserver(IDataProcessorObservable.Observer observer);
}
