/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others. All rights reserved. This program
 * and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * available at $Id: JavaExecutionDeploymentAdapter.java,v 1.26 2010/04/12 12:38:44 paules Exp $ Contributors: IBM -
 * Initial API and implementation
 ******************************************************************************/
package org.eclipse.hyades.execution.harness;

import java.io.File;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.execution.core.IExecutionEnvironment;
import org.eclipse.hyades.execution.core.INode;
import org.eclipse.hyades.execution.core.IOrderedProperty;
import org.eclipse.hyades.execution.core.file.IFileManager;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList;
import org.eclipse.hyades.execution.harness.internal.resources.ExecutionHarnessPluginResourceBundle;
import org.eclipse.hyades.execution.local.NodeImpl;
import org.eclipse.hyades.internal.execution.core.file.ServerNotAvailableException;
import org.eclipse.hyades.models.common.configuration.CFGArtifact;
import org.eclipse.hyades.models.common.configuration.CFGArtifactLocationPair;
import org.eclipse.hyades.models.common.configuration.CFGClass;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGMachineConstraint;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationFactory;
import org.eclipse.hyades.models.common.configuration.util.ConfigurationUtil;
import org.eclipse.hyades.models.common.interactions.BVRProperty;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.osgi.util.NLS;

/**
 * <p>Implementation of {@link JavaExecutionDeploymentAdapter} to automatically deploy 
 * test assets using the Agent Controller's file transfer service.</p>
 * 
 * 
 * @author  Bianca Xue Jiang
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version April 12, 2010
 * @since   January 21, 2005
 */
public class JavaExecutionDeploymentAdapter implements IExecutionDeploymentAdapter {
	
	/**
	 * Slash ("/") will be used in all path names processed by this class.
	 * Reason for this is when this class collects CLASSPATH and gets ready for
	 * file transfer, it has no knowledge of the target machine, whether it is a
	 * Windows or UNIX platform. Since either slash("/") or back slash("\")
	 * works on Windows and only slash("/") works on UNIX platform, slash is the
	 * only choice to ensure path name gets recognized on both type of
	 * platforms.
	 */
	public static final String PATH_NAME_SEPARATOR = "/"; //$NON-NLS-1$

	/**
	 * List of environment variables, which specify a temp directory, here
	 */
	public static final String ROOTDIR_ENV_VARIABLE1 = "LOCAL_AGENT_TEMP_DIR"; //$NON-NLS-1$

	public static final String ROOTDIR_ENV_VARIABLE2 = "TEMP"; //$NON-NLS-1$

	public static final String ROOTDIR_ENV_VARIABLE3 = "TMP"; //$NON-NLS-1$

	public JavaExecutionDeploymentAdapter() {
	}

	protected void cleanFromNode(INode node, String[] remoteFiles, StringBuffer bufError) {

		// File manager is not exposed through node interface yet
		if (node instanceof NodeImpl) {

			// Retrieve the extended file manager interface
			IFileManagerExtended fileManager = (IFileManagerExtended) ((NodeImpl) node).getFileManager();

			// Continue if extended file manager available
			if (fileManager != null) {

				// Delete remote files in bulk
				try {
					fileManager.deleteFile(FileIdentifierList.create(remoteFiles));
				} catch (ServerNotAvailableException e) {
					bufError.append(e.getMessage());
				} catch (IOException e) {
					bufError.append(e.getMessage());
				}

			}

		}

	}
	
	/**
	 * Remove the specified directory on the remote host, and any file or sub-directory
	 * it contains.
	 * @author jcanches
	 */
	protected void removeDirectory(INode node, String remoteDirectory, StringBuffer bufError) {
		// File manager is not exposed through node interface yet
		if (node instanceof NodeImpl) {

			// Retrieve the extended file manager interface
			IFileManagerExtended fileManager = (IFileManagerExtended) ((NodeImpl) node).getFileManager();

			// Continue if extended file manager available
			if (fileManager != null) {
				try {
					fileManager.deleteDirectory(FileIdentifierList.create(remoteDirectory));
				} catch (ServerNotAvailableException e) {
					bufError.append(e.getMessage());
				} catch (IOException e) {
					bufError.append(e.getMessage());
				}
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.harness.IExecutionDeploymentAdapter
	 *      #cleanUpTestAssets(org.eclipse.hyades.execution.core.INode,
	 *      org.eclipse.hyades.models.common.testprofile.TPFDeployment, boolean,
	 *      java.lang.StringBuffer)
	 */
	public void cleanUpTestAssets(INode node, TPFDeployment deployment, boolean isStandalone, StringBuffer bufError,
			IExecutionEnvironment exeEnvironment) {
		if (node == null || deployment == null)
			return;

		CFGArtifactLocationPair[] pairs = collectPairsByNode(deployment, node);
		if (pairs == null || pairs.length == 0)
			return;

		HashMap deployableFiles = collectDeployableFiles(pairs, isStandalone, exeEnvironment);
		if (deployableFiles != null && deployableFiles.size() > 0) {
			String[] deleteFiles = (String[]) deployableFiles.values().toArray();
			cleanFromNode(node, deleteFiles, bufError);
		}
	}

	/**
	 * Collect class files now collects all files (filtering those prefixed with
	 * a dot and other model files), class files as well as the peripheral files
	 * around the class files such as properties files and other files that
	 * would be copied from the src to the bin directory during the normal build
	 * process.
	 * 
	 * Model files (except for datapools) will also be filtered out in cases
	 * where the bin directory is not separate from the src directory.
	 * 
	 * @param dir
	 *            the directory containing files to collect
	 * @param clsFilePaths
	 *            list to accumulate (basically an in and out parameter)
	 */
	protected void collectClassFiles(File dir, Vector clsFilePaths) {

		// Retrieve all files in the directory specified
		File[] files = dir.listFiles();

		// Iterate over the files in the specified directory
		for (int i = 0; i < files.length; i++) {

			// Candidate file to be collected
			File candidate = files[i];
			String candidateName = candidate.getName();

			// Exclude names starting with a dot (system files)
			if (!candidateName.startsWith(".")) {

				// Take different actions if candidate is file versus directory
				if (candidate.isFile()) {

					/*
					 * If the file is collectable and interesting then normalize
					 * its path (to negate any platform dependencies), filter
					 * out model files except for datapools, if the binaries are
					 * not separated from the source, these types of files would
					 * be collected if not filtered.
					 */
					if (!candidateName.endsWith(".execution") && !candidateName.endsWith(".testsuite")
							&& !candidateName.endsWith(".deploy") && !candidateName.endsWith(".location")
							&& !candidateName.endsWith(".artifact") && !candidateName.endsWith(".java")) {
						clsFilePaths.add(uniformPath(candidate.getPath()));
					}

				} else {

					// If a directory then recurse down and collect
					collectClassFiles(candidate, clsFilePaths);

				}

			}

		}
	}

	protected HashMap collectDeployableFiles(CFGArtifactLocationPair[] pairs, boolean isStandalone,
			IExecutionEnvironment exeEnvironment) {
		HashMap deployableFiles = new HashMap();
		CFGArtifactLocationPair pair = null;
		CFGArtifact artifact = null;
		CFGLocation location = null;
		String deployRootDir = null;

		for (int i = 0; i < pairs.length; i++) {
			pair = pairs[i];
			location = pair.getLocation();
			artifact = pair.getArtifact();
			if (artifact == null || location == null || !(location instanceof CFGMachineConstraint))
				continue;

			// bugzilla 61137 enhancement, get the default root directory
			// from the temp directory environment variable in
			// IExecutionEnvironment
			String defaultRootDir = getDefaultRootDir(exeEnvironment);

			// will use defaultRootDir if not specified as a location property
			deployRootDir = getDeployRootDir(location, defaultRootDir);

			collectClasspathFiles(isStandalone, deployableFiles, artifact, location, deployRootDir);

			// deploy resources from artifact deployables.
			collectDeployableArtifacts(isStandalone, deployableFiles, artifact, location, deployRootDir);
		}

		return deployableFiles;
	}

	/**
	 * @param isStandalone
	 * @param deployableFiles
	 * @param artifact
	 * @param location
	 * @param deployRootDir
	 */
	protected void collectClasspathFiles(boolean isStandalone, HashMap deployableFiles, CFGArtifact artifact, CFGLocation location, String deployRootDir) {
		String srcAbsolutePath;
		String destAbsolutePath;
		CFGPropertyGroup propertyGroup = ConfigurationUtil.searchPropertyGroupById(artifact.getPropertyGroups(),
				ConfigurationUtil.ATTRS_PROP_GROUP_ID_ARTIFACT);
		if (propertyGroup != null) {
			// deploy files from classpath
			Vector targetClasspaths = new Vector();
			BVRProperty[] properties = ConfigurationUtil.searchPropertiesByName(propertyGroup.getProperties(),
					ConfigurationUtil.CLASSPATH, false);
			for (int j = 0; j < properties.length; j++) {
				StringTokenizer tokenizer = new StringTokenizer(properties[j].getValue(), File.pathSeparator);
				while (tokenizer.hasMoreTokens()) {
					srcAbsolutePath = uniformPath(tokenizer.nextToken());
					if (excludeFromClasspath(srcAbsolutePath, artifact, location))
						continue;
					
					destAbsolutePath = getDeployFilePath(srcAbsolutePath, deployRootDir, isStandalone);
					targetClasspaths.add(destAbsolutePath);

					// no deployment of classpath files if the source and
					// destination is the same
					if (isSourceDestinationSame((CFGMachineConstraint) location, srcAbsolutePath, destAbsolutePath))
						continue;

					File file = new File(srcAbsolutePath);
					if (file != null && file.isDirectory()) {
						String[] files = getClasspathFiles(file);
						for (int indx = 0; indx < files.length; indx++) {
							srcAbsolutePath = files[indx];
							destAbsolutePath = getDeployFilePath(srcAbsolutePath, deployRootDir, isStandalone);
							deployableFiles.put(srcAbsolutePath, destAbsolutePath);
						}
					} else
						deployableFiles.put(srcAbsolutePath, destAbsolutePath);
				}
			}

			// save the destination classpath in location property group
			// for(int x = 0; x < targetClasspaths.size(); x++) {
			// System.out.println(targetClasspaths.get(x)); }
			setLocationClasspath(location, targetClasspaths);
		}
	}
	
	/**
	 * Allows to exclude an entry declared in an artifact classpath, from the deployed
	 * classpath entries to the specified location. The base imlementation always returns
	 * <code>false</code>.
	 * @return <code>true</code> if the entry should be excluded from the deployed entries,
	 * <code>false</code> otherwise.
	 */
	protected boolean excludeFromClasspath(String localAbsolutePath, CFGArtifact artifact, CFGLocation location) {
		return false;
	}

	/**
	 * @param isStandalone
	 * @param deployableFiles
	 * @param artifact
	 * @param location
	 * @param deployRootDir
	 */
	protected void collectDeployableArtifacts(boolean isStandalone, HashMap deployableFiles, CFGArtifact artifact, CFGLocation location, String deployRootDir) {
		CFGClass deployable;
		String srcAbsolutePath;
		String destAbsolutePath;
		List deployables = artifact.getDeployableInstances();
		for (Iterator it = deployables.iterator(); it.hasNext();) {
			deployable = (CFGClass) it.next();
			if (deployable instanceof TPFTestSuite)
				collectTestSuiteFiles(isStandalone, deployableFiles, (TPFTestSuite)deployable, location, deployRootDir);

			srcAbsolutePath = getDeployableFilePath(deployable, isStandalone);
			if (srcAbsolutePath == null)
				continue;
			destAbsolutePath = getDeployFilePath(srcAbsolutePath, deployRootDir, isStandalone);
			if (!isSourceDestinationSame((CFGMachineConstraint) location, srcAbsolutePath, destAbsolutePath))
				deployableFiles.put(srcAbsolutePath, destAbsolutePath);
		}
	}
	
	protected void collectTestSuiteFiles(boolean isStandalone, HashMap deployableFiles, TPFTestSuite testSuite, CFGLocation location, String deployRootDir) {
		// NOP
	}

	protected CFGArtifactLocationPair[] collectPairsByNode(TPFDeployment deployment, INode node) {
		CFGArtifactLocationPair[] sameNodePairs = new CFGArtifactLocationPair[0];
		String nodeName = node.getName();
		if (nodeName == null || nodeName.length() < 1)
			return sameNodePairs;
		EList pairs = deployment.getArtifactLocations();
		if (pairs == null || pairs.isEmpty())
			return sameNodePairs;

		CFGArtifactLocationPair pair = null;
		String hostname = null;
		for (Iterator it = pairs.iterator(); it.hasNext();) {
			pair = (CFGArtifactLocationPair) it.next();
			if (!isValidPair(pair))
				continue;

			CFGLocation location = pair.getLocation();
			if (location instanceof CFGMachineConstraint)
				hostname = ((CFGMachineConstraint) pair.getLocation()).getHostname();
			else
				hostname = pair.getLocation().getName();

			if (hostname == null || hostname.length() < 1)
				continue;

			if (nodeName.equals(hostname)) {
				if (sameNodePairs.length == 0) {
					sameNodePairs = new CFGArtifactLocationPair[] { pair };
				} else {
					CFGArtifactLocationPair[] temp = sameNodePairs;
					sameNodePairs = new CFGArtifactLocationPair[temp.length + 1];
					System.arraycopy(temp, 0, sameNodePairs, 0, temp.length);
					sameNodePairs[temp.length] = pair;
				}
			}
		}

		return sameNodePairs;
	}

	protected boolean deployFile(IFileManager fileManager, String srcAbsolutePath, String destAbsolutePath, StringBuffer bufError) {

		try {

			fileManager.putFile(srcAbsolutePath, destAbsolutePath);

			return true;
		} 
		catch(IOException i) {

			String message = i.getMessage();

			if (message != null){
				bufError.append(message);
			} 
			else {
				bufError.append(i.toString());
			}
		} 
		catch (RuntimeException r) {
			
			//Add a file deployment error message to the error buffer:
			if (r.getCause() instanceof IOException) {
				bufError.append(NLS.bind(ExecutionHarnessPluginResourceBundle.FILE_DEPLOY_ERR_, destAbsolutePath));
			}

			throw r;
		}

		return false;
	}

	/**
	 * New method to perform bulk uploads of files to the server
	 * 
	 * @param fileManager
	 *            the extended file manager to use
	 * @param sourceFiles
	 *            the source files to transfer
	 * @param destinationFiles
	 *            the destination files to copy into
	 * @param bufError
	 *            the error buffer
	 * @return indicates if the transfer was successful
	 */
	protected boolean deployFile(IFileManagerExtended fileManager, FileIdentifierList sourceFiles, FileIdentifierList destinationFiles, StringBuffer bufError) {

		try {
			
			fileManager.putFile(sourceFiles, destinationFiles);
			
			return true;
		} 
		catch(IOException i) {
			
			String message = i.getMessage();
			
			if (message != null){
				bufError.append(message);
			} 
			else {
				bufError.append(i.toString());
			}
		} 
		catch (ServerNotAvailableException s) {
			
			String message = s.getMessage();
			
			if (message != null){
				bufError.append(message);
			} 
			else {
				bufError.append(s.toString());
			}

			throw new RuntimeException(s);
		} 
		catch (RuntimeException r) {
			
			//Add a file deployment error message to the error buffer:
			if (r.getCause() instanceof IOException) {
				bufError.append(NLS.bind(ExecutionHarnessPluginResourceBundle.FILE_DEPLOY_ERR_, destinationFiles.toString()));
			}

			throw r;
		}

		return false;
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.harness.IExecutionDeploymentAdapter
	 *      #deployTestAssets(org.eclipse.hyades.execution.core.INode,
	 *      org.eclipse.hyades.models.common.testprofile.TPFDeployment, boolean,
	 *      java.lang.StringBuffer)
	 */
	public void deployTestAssets(INode node, TPFDeployment deployment, boolean isStandalone, StringBuffer bufError,
			IExecutionEnvironment exeEnvironment) {
		if (node == null || deployment == null)
			return;

		CFGArtifactLocationPair[] pairs = collectPairsByNode(deployment, node);
		if (pairs == null || pairs.length == 0)
			return;

		HashMap deployableFiles = collectDeployableFiles(pairs, isStandalone, exeEnvironment);
		this.deployToNode(node, deployableFiles, bufError);
	}

	protected HashMap deployToNode(INode node, HashMap deployableFiles, StringBuffer bufError) {

		// The file manager is not exposed on the node interface
		if (node instanceof NodeImpl) {

			// Retrieve the new extended file manager interface
			IFileManagerExtended fileManager = (IFileManagerExtended) ((NodeImpl) node).getFileManager();

			// If extended file manager access is successful continue
			if (fileManager != null) {

				// Build the lists of files to deploy to the server
				FileIdentifierList sourceFileList = FileIdentifierList.create();
				FileIdentifierList destinationFileList = FileIdentifierList.create();

				// Step through each source and destination pair
				for (Iterator i = deployableFiles.keySet().iterator(); i.hasNext();) {
					String sourceFile = (String) i.next();
					String destinationFile = (String) deployableFiles.get(sourceFile);
					sourceFileList.add(sourceFile);
					destinationFileList.add(destinationFile);
				}

				// Deploy the files in bulk using the file manager
				this.deployFile(fileManager, sourceFileList, destinationFileList, bufError);

				return deployableFiles; // if no exceptions are thrown, then
				// return all

			}
		}

		return null; // if anything fails, return null
	}

	protected String[] getClasspathFiles(File clspathDirectory) {
		Vector clsFiles = new Vector();
		collectClassFiles(clspathDirectory, clsFiles);

		String[] files = new String[clsFiles.size()];
		for (int i = 0; i < clsFiles.size(); i++) {
			files[i] = (String) clsFiles.get(i);
		}
		return files;
	}

	protected String getDefaultRootDir(IExecutionEnvironment exeEnvironment) {
		String defaultRootDir = null;
		IOrderedProperty[] properties = exeEnvironment.getEnv();
		int x = properties.length;

		for (int i = 0; i < x; i++) {
			String propName = properties[i].getName();

			if (isRootEnvironmentVariable(propName)) {
				defaultRootDir = ((String) properties[i].getValues()[0]);
				break;
			}
		}
		/*
		 * to do: validate that defaultRootDir is properly formatted file path
		 */
		return defaultRootDir;
	}

	protected String getDefaultUsername() {
		String defaultUserName = null;
		try {
			defaultUserName = System.getProperty("user.name");
		} catch (SecurityException e) {
			// TODO The system does not allow us to get the user name
			// do we need to report this?
		}
		return defaultUserName;
	}

	protected String getDeployableFilePath(CFGClass deployable, boolean isStandalone) {

		// Retrieve resource for model element
		Resource resource = deployable.eResource();

		// If the resource is null then return null
		if (resource == null)
			return null;

		// Retrieve model resource URI
		URI uri = resource.getURI();

		// If running in legacy standalone mode use file string
		if (isStandalone) {
			return uri.toFileString();
		}

		// Retrieve URI in string form
		String path = uri.toString();

		// Prefix for platform resources
		final String PLATFORM_RESOURCE = "platform:/resource/";

		// If path is a platform URI then resolve to actual local location
		if (path.startsWith(PLATFORM_RESOURCE)) { //$NON-NLS-1$

			// Find member in workspace to get resource handle
			IResource fileResource = ResourcesPlugin.getWorkspace().getRoot().findMember(
					path.substring((PLATFORM_RESOURCE.length())));
			path = fileResource.getLocation().toString();

		}
		else {
			path = uniformPath(uri.toFileString());
		}

		return path;

	}

	protected String getDeployFilePath(String sourceFile, String deployRootDir, boolean isStandalone) {
		String deployFile;
		String sourceRootDir = getSourceRootDir(new File(sourceFile), isStandalone);
		deployRootDir = (deployRootDir == null || deployRootDir.length() < 1) ? sourceRootDir : deployRootDir;
		if (sourceFile.startsWith(deployRootDir))
			deployFile = sourceFile;
		else
			deployFile = (deployRootDir + sourceFile.substring(sourceRootDir.length() - 1, sourceFile.length()));

		// bugzilla 67677 fix, use name separator "/" to ensure path name on
		// Windows and UNIX platforms.
		deployFile = uniformPath(deployFile);
		return deployFile;
	}

	/*
	 * Build deployment root directory
	 */
	protected String getDeployRootDir(CFGLocation location, String defaultDir) {
		String deployRootDir = null;

		CFGPropertyGroup propGroup = ConfigurationUtil.searchPropertyGroupById(location.getPropertyGroups(),
				ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);
		if (propGroup == null) {
			propGroup = Common_ConfigurationFactory.eINSTANCE.createCFGPropertyGroup();
			propGroup.setPropertyGroupID(ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);
			location.getPropertyGroups().add(propGroup);
		}

		// get ROOTDIR from location properties
		BVRProperty[] props = ConfigurationUtil.searchPropertiesByName(propGroup.getProperties(),
				ConfigurationUtil.ROOTDIR, false);
		if (props == null || props.length == 0) {
			// if ROOTDIR not specified, use the default (passed in)
			deployRootDir = defaultDir;

			// no need to set this, just use the default each time in this case
			// CFGComparableProperty property =
			// Common_ConfigurationFactory.eINSTANCE.createCFGComparableProperty();
			// property.setName(ConfigurationUtil.ROOTDIR);
			// property.setOperator("="); //$NON-NLS-1$
			// property.setValue(deployRootDir);
			// propGroup.getProperties().add(property);
		} else
			deployRootDir = props[0].getValue();

		if (deployRootDir != null) {
			String userName = null;
			// bugzilla 83247 fix, append direcotry "usename" as as the
			// deployment directory
			// first attempt to get USR_NAME from location properties (user may
			// have set this)
			props = null;
			props = ConfigurationUtil.searchPropertiesByName(propGroup.getProperties(), ConfigurationUtil.USR_NAME, false);
			if (props == null || props.length == 0) {
				// if USR_NAME not specified, use the default (System property
				// user.name)
				userName = getDefaultUsername();

				// no need to set this, just use the default each time in this
				// case
				// CFGComparableProperty property =
				// Common_ConfigurationFactory.eINSTANCE.createCFGComparableProperty();
				// property.setName(ConfigurationUtil.USR_NAME);
				// property.setOperator("="); //$NON-NLS-1$
				// property.setValue(userName);
				// propGroup.getProperties().add(property);
			} else
				userName = props[0].getValue();

			if (userName != null) {
				deployRootDir += PATH_NAME_SEPARATOR + userName;
			}

			// bugzilla 67677 fix, use name separator "/" to ensure path name
			// works on Windows and UNIX platforms.
			deployRootDir = uniformPath(deployRootDir);
			
			// Set ROOTDIR_USED to track which root directory was used
			props = ConfigurationUtil.searchPropertiesByName(propGroup.getProperties(), ConfigurationUtil.USED_ROOTDIR, false);
			if (props == null || props.length == 0) {
				// Create and set it
				CFGComparableProperty property = Common_ConfigurationFactory.eINSTANCE.createCFGComparableProperty();
				property.setName(ConfigurationUtil.USED_ROOTDIR);
				property.setOperator("=");
				property.setValue(deployRootDir);
				propGroup.getProperties().add(property);
			} else {
				// Just update it
				props[0].setValue(deployRootDir);
			}
			
		}

		return deployRootDir;
	}

	protected String getSourceRootDir(File source, boolean isStandalone) {
		// bugzilla 67677 fix, use name separator "/" to ensure path name works
		// on Windows and UNIX platforms.
		String sourcePath = uniformPath(source.getPath());
		if (!isStandalone) {
			// get workspace root
			String workspaceRoot = getWorkspaceRoot();
			if (workspaceRoot != null && workspaceRoot.length() > 0 && sourcePath.startsWith(workspaceRoot)) {
				return workspaceRoot;
			}
		}

		// if it is not from workspace or this is a standalone execution
		// return the root directory of the source
		URI sourceURI = URI.createFileURI(source.getPath());
		if (sourceURI.hasDevice()) {
			return sourceURI.device() + PATH_NAME_SEPARATOR;
		}
		return PATH_NAME_SEPARATOR;
	}

	protected String getWorkspaceRoot() {
		if (ResourcesPlugin.getWorkspace() != null) {
			String workspaceRoot = ResourcesPlugin.getWorkspace().getRoot().getLocation().toString();
			workspaceRoot = uniformPath(workspaceRoot);
			if (workspaceRoot != null && !workspaceRoot.endsWith(PATH_NAME_SEPARATOR))
				workspaceRoot += PATH_NAME_SEPARATOR;
			return workspaceRoot;
		}

		return ""; //$NON-NLS-1$
	}

	protected boolean isRootEnvironmentVariable(String envVar) {
		return (envVar.equalsIgnoreCase(ROOTDIR_ENV_VARIABLE1) || envVar.equalsIgnoreCase(ROOTDIR_ENV_VARIABLE2) || envVar
				.equalsIgnoreCase(ROOTDIR_ENV_VARIABLE3));
	}

	protected boolean isSourceDestinationSame(CFGMachineConstraint location, String source, String destination) {
		String hostname = location.getHostname();
		try {
			if (!(InetAddress.getLocalHost().equals(InetAddress.getByName(hostname)) || InetAddress.getByName(hostname)
					.isLoopbackAddress()))
				return false;
		} catch (UnknownHostException e) {
		}

		File src = new File(source);
		File des = new File(destination);
		if (src.compareTo(des) == 0)
			return true;

		return false;
	}

	protected boolean isValidPair(CFGArtifactLocationPair pair) {
		CFGLocation location = pair.getLocation();
		CFGArtifact artifact = pair.getArtifact();
		if (location == null || artifact == null)
			return false;

		return true;
	}

	protected void setLocationClasspath(CFGLocation location, Vector classpaths) {
		if (classpaths == null || classpaths.size() < 1)
			return;

		CFGPropertyGroup propGroup = ConfigurationUtil.searchPropertyGroupById(location.getPropertyGroups(),
				ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);
		if (propGroup == null) {
			propGroup = Common_ConfigurationFactory.eINSTANCE.createCFGPropertyGroup();
			propGroup.setPropertyGroupID(ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);
			location.getPropertyGroups().add(propGroup);
		}

		String classpath = ""; //$NON-NLS-1$
		BVRProperty[] existingProps = ConfigurationUtil.searchPropertiesByName(propGroup.getProperties(),
				ConfigurationUtil.CLASSPATH, false);
		for (int i = 0; i < classpaths.size(); i++) {
			classpath = (String) classpaths.get(i);
			boolean existed = false;
			for (int propIndex = 0; propIndex < existingProps.length; propIndex++) {
				if (existingProps[propIndex].getValue().indexOf(classpath) > -1) {
					existed = true;
					break;
				}
			}

			if (existed == false) {
				CFGComparableProperty property = Common_ConfigurationFactory.eINSTANCE.createCFGComparableProperty();
				property.setName(ConfigurationUtil.CLASSPATH);
				property.setOperator("="); //$NON-NLS-1$
				property.setValue(classpath);
				propGroup.getProperties().add(property);
			}
		}
	}

	/**
	 * This method is used to make sure name separator used in <i>pathname</i>
	 * is slash ("/").
	 * <P>
	 * Since either slash("/") or back slash("\") works on Windows and only
	 * slash("/") works on UNIX platform, slash is the choice to ensure path
	 * name gets recognized on both type of platforms.
	 * 
	 * @param pathname
	 *            the path name to be reformed.
	 * @return a pathname separated by slashes("/").
	 */
	protected String uniformPath(String pathname) {
		if (pathname == null) return null;
		return pathname.replaceAll("\\\\", PATH_NAME_SEPARATOR); //$NON-NLS-1$
		// URI pathURI = URI.createFileURI(pathname);
		// String path = pathURI.devicePath();
		// if(pathURI.hasAuthority())
		// path = path.substring(pathURI.authority().length(), path.length());

		// return path;
	}
}