/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestExecutionHarness.java,v 1.59 2009/12/17 15:27:51 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.execution.harness;

import java.io.FileNotFoundException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.IDataProcessor;
import org.eclipse.hyades.execution.core.IEclipseExecutionComponentFactory;
import org.eclipse.hyades.execution.core.IExecutableObject;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutionComponentFactory;
import org.eclipse.hyades.execution.core.IExecutionComponentStateChangeListener;
import org.eclipse.hyades.execution.core.IExecutionEnvironment;
import org.eclipse.hyades.execution.core.IExecutor;
import org.eclipse.hyades.execution.core.IExecutorWithProgressMonitorSupport;
import org.eclipse.hyades.execution.core.INodeExtended;
import org.eclipse.hyades.execution.core.IRemoteHyadesComponent;
import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.core.UnknownDaemonException;
import org.eclipse.hyades.execution.core.util.ConnectionSpecifier;
import org.eclipse.hyades.execution.core.util.MutableBoolean;
import org.eclipse.hyades.execution.core.util.MutableObject;
import org.eclipse.hyades.execution.harness.internal.resources.ExecutionHarnessPluginResourceBundle;
import org.eclipse.hyades.execution.harness.provisional.ITestExecutionHarnessListener;
import org.eclipse.hyades.execution.harness.provisional.TestExecutionHarnessListenerNotifier;
import org.eclipse.hyades.execution.harness.util.ExecutionAdapterUtilities;
import org.eclipse.hyades.execution.harness.util.ISystemUtility;
import org.eclipse.hyades.execution.harness.util.StandaloneExecutionUtilities;
import org.eclipse.hyades.execution.harness.util.StandaloneSystemUtility;
import org.eclipse.hyades.execution.local.EclipseExecutionComponentFactoryImpl;
import org.eclipse.hyades.execution.local.ExecutableObjectStub;
import org.eclipse.hyades.execution.local.ExecutionComponentFactoryImpl;
import org.eclipse.hyades.execution.local.ExecutionComponentStub;
import org.eclipse.hyades.execution.local.ExecutorStub;
import org.eclipse.hyades.execution.local.JavaProcessExecutableObjectStub;
import org.eclipse.hyades.execution.local.NodeImpl;
import org.eclipse.hyades.execution.local.SessionContext;
import org.eclipse.hyades.execution.local.SessionContextToSessionMap;
import org.eclipse.hyades.execution.local.SessionStub;
import org.eclipse.hyades.internal.execution.local.control.AgentControllerUnavailableException;
import org.eclipse.hyades.loaders.util.RegistryReader;
import org.eclipse.hyades.models.common.configuration.CFGArtifactLocationPair;
import org.eclipse.hyades.models.common.configuration.CFGClass;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGMachineConstraint;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationFactory;
import org.eclipse.hyades.models.common.configuration.util.ConfigurationUtil;
import org.eclipse.hyades.models.common.facades.behavioral.IImplementor;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.facades.behavioral.impl.DatapoolFacadeResourceFactoryImpl;
import org.eclipse.hyades.models.common.facades.behavioral.impl.FacadeResourceFactoryImpl;
import org.eclipse.hyades.models.common.facades.behavioral.impl.HyadesFactory;
import org.eclipse.hyades.models.common.interactions.BVRProperty;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofileFactory;
import org.eclipse.hyades.models.common.testprofile.TPFBehavior;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionType;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.models.common.testprofile.TPFTypedEvent;
import org.eclipse.hyades.models.common.testprofile.TPFVerdict;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictEvent;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictReason;
import org.eclipse.hyades.models.common.testprofile.impl.Common_TestprofilePackageImpl;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.models.hierarchy.util.HyadesExtendedResourceFactory;
import org.eclipse.hyades.models.hierarchy.util.StringUtil;
import org.eclipse.hyades.models.util.ModelDebugger;
import org.eclipse.osgi.util.NLS;

/**
 * <p>This class provides the mechanism for launching Hyades tests.</p>
 * 
 * <p>The flow of the <code>launchTest()</code> method is depicted in the 
 * following execution environment use case:</p>
 * 
 * <p>
 * <a href="http://www.eclipse.org/tptp/test/documents/design/usecases.html#2">
 * Launch a single test instance from workbench on a single specified node and 
 * show control events to the running test</a></p>
 * 
 * <p>This class may used to launch tests from within the workbench (from code
 * running inside the eclipse shell), or it may be used to launch tests from
 * outside the workbench (referred to as "stand-alone".) When running stand-alone,
 * you must use the constructor that accepts a string argument, and pass in a
 * configuration directory as that argument. The configuration directory should
 * be structured like, and may be, an eclipse directory. The directory is used
 * to determine information such as the execution components and their mappings
 * to test types, the location of resource bundles, etc. Also remember that when
 * running stand-alone, you are responsible for properly setting the classpath of
 * your JVM so that all necessary classes can be loaded.</p>
 * 
 * <p>An example of how to launch a test stand-alone is shown below. Be sure to
 * substitute the correct values for standaloneConfigurationFiles and test suite
 * path.</p>
 * 
 * <code><pre>
 * public static void main(String args[]) {
 * 	TestExecutionHarness harness = new TestExecutionHarness(&quot;C:\Program Files\eclipse&quot;);
 *	harness.launchTest(&quot;C:\Program Files\eclipse\runtime-workspace\test\src\first.testsuite&quot;, null, &quot;localhost&quot;, &quot;10002&quot;, &quot;c:\temp&quot;, &quot;executionresult.execution&quot;, true, true);
 * }
 * </code></pre>
 * 
 * 
 * @author  Joseph P. Toomey
 * @author  Paul E. Slauenwhite
 * @version December 17, 2009
 * @since   January 31, 2005
 */
public class TestExecutionHarness {

	private boolean bAgentReady = false;

	private boolean bFactoryInitialized = false;

	private ExecutionHarnessDataProcessorFactory dpFactory = null;

	private IExecutionComponentFactory factory = null;

	private String[] standaloneConfigurationFiles = null;

	private ISystemUtility systemUtility = null;

	private final HashMap testTypeMap = new HashMap();

	private final Object agentLock = new Object();

	private final Object instanceLock = new Object();
	
	private final Object executorLock = new Object();

	/**
	 * Timeout duration for connecting to the node.
	 */
	private static final int NODE_CONNECT_TIMEOUT_SECONDS = 180;	

	/**
	 * Constructs a test execution harness (standalone or non-standalone mode is
	 * automatically determined based on invoking the is running standalone
	 * method)
	 */
	public TestExecutionHarness() {
		super();
		if (!this.isRunningStandalone()) {
			this.systemUtility = ExecutionHarnessPlugin.getDefault();
		} else {
			throw new IllegalArgumentException(
					"Standalone execution requires that "
							+ "a configuration file directory is passed to the TestExecutionHarness constructor!");
		}
	}

	private static IRemoteHyadesComponent findRemoteHyadesComponent(
			final IExecutor executor) {
		IRemoteHyadesComponent comp = null;
		final IExecutionComponent[] children = executor.getChildren();
		for (int i = 0; i < children.length; i++) {
			if (children[i] instanceof IRemoteHyadesComponent) {
				comp = (IRemoteHyadesComponent) children[i];
			}
		}
		return comp;
	}

	public static void waitForDataProcessorExit(final IExecutor executor) {
		final IRemoteHyadesComponent remoteComponent = TestExecutionHarness
				.findRemoteHyadesComponent(executor);
		if (remoteComponent != null) {
			while (remoteComponent.isActive()) {
				try {
					Thread.sleep(1000);
				} catch (final InterruptedException ex) {
				}
			}
		}
	}
	
	/**
	 * Constructs a test execution harness given a configuration file directory
	 * and a version string (standalone or non-standalone mode is automatically
	 * determined based on invoking the is running standalone method); this
	 * constructor is the same as calling the default constructor when not in
	 * standalone mode -- typically this would be used for standalone mode only
	 * 
	 * @param configurationFileDir
	 *            configuration file directory
	 * @param versionString
	 *            version string
	 */
	public TestExecutionHarness(final String configurationFileDir,
			final String versionString) {
		super();
		if (!this.isRunningStandalone()) {
			this.systemUtility = ExecutionHarnessPlugin.getDefault();
		} else {
			final String[] plugins = StandaloneExecutionUtilities
					.getConfigFilesFromConfigDir(configurationFileDir);
			this.setStandaloneConfigurationFiles(plugins);
			final String harnessDir = StandaloneExecutionUtilities
					.getHarnessDir(configurationFileDir, versionString);
			this.systemUtility = new StandaloneSystemUtility(harnessDir);
			RegistryReader.standaloneConfiguration = new RegistryReader.StandaloneConfiguration();
			RegistryReader.standaloneConfiguration
					.setEclipseFolder(configurationFileDir);
			this.init();
		}
	}

	/**
	 * Returns true if the particular control event can be performed on test
	 * specified.
	 * 
	 * @param suite
	 *            the test suite to be controlled.
	 * @param theTest
	 *            the test to be controlled (may be the same as suite, or may be
	 *            a test case contained within suite.
	 * @param location
	 *            on which the test will be controlled (CGFLocation or
	 *            CFGMachine)
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param execResult
	 *            the execution Result for the test run
	 * @param controlEvent
	 *            the control event
	 * @return a string containing an error message, or null if no error
	 *         occurred
	 */
	public synchronized boolean canSupportControlEvent(final ITestSuite suite,
			final ITest theTest, final CFGLocation location, final String port,
			final TPFExecutionResult execResult, final String controlEvent) {

		final String machine = ((CFGMachineConstraint) location).getHostname();
		final StringBuffer bufError = new StringBuffer();

		// Get a session for this Node. This is returning an session that was
		// created during launch test method
		final ISession session = this.getSession(port, null, machine, bufError);
		if (session == null) {
			return false;
		}

		boolean canSupport = false;
		final IExecutionComponent[] exeEnvironments = session.getChildren();
		// Navigate to the IExecutor find out if this can be stopped.
		for (int i = 0; i < exeEnvironments.length; i++) {
			if (exeEnvironments[i] instanceof IExecutionEnvironment) {
				final IExecutionComponent[] executors = exeEnvironments[i]
						.getChildren();
				for (int j = 0; j < executors.length; j++) {
					if (executors[j] instanceof ExecutorStub) {
						final ExecutableObjectStub exeobj = (ExecutableObjectStub) ((IExecutor) executors[j])
								.getExecutableObject();
						if (exeobj.getTestID().equals(theTest.getId())
								&& exeobj.getSuiteID().equals(suite.getId())
								&& ((ExecutorStub) executors[j])
										.getExecutionResultName().equals(
												execResult.getName())) {
							if (((IExecutor) executors[j])
									.supportsControlEvent(controlEvent)) {
								canSupport = true;
							} else {
								return false;
							}

						}
					}
				}
			}
		}
		return canSupport;
	}

	private IExecutionHarnessDataProcessor[] convertListToArray(
			final ArrayList registeredDataProcessorsForTestType) {
		final IExecutionHarnessDataProcessor dp[] = new IExecutionHarnessDataProcessor[registeredDataProcessorsForTestType
				.size()];
		for (int i = 0; i < registeredDataProcessorsForTestType.size(); i++) {
			dp[i] = (IExecutionHarnessDataProcessor) registeredDataProcessorsForTestType
					.get(i);
		}
		return dp;
	}

	/**
	 * Factory method to create a null progress monitor, this method could
	 * return any object that implements the progress monitor interface for
	 * debugging purposes in tracking calls into the concrete progress monitor
	 * instance. The proper object to return for release purposes is a null
	 * progress monitor instance. This method is typically called when a "real"
	 * progress monitor is not available.
	 * 
	 * @return a null object progress monitor (instead of performing continuous
	 *         null checks)
	 */
	private IProgressMonitor createNullProgressMonitor() {
		return new NullProgressMonitor();
	}

	/**
	 * This method queries the testTypeMap and determines which execution
	 * component should be used for a given test type and execution component
	 * category.
	 * 
	 * @param executionComponentType
	 *            the category of execution component being queried
	 * @param testType
	 *            the test type being queried
	 * @param bStandalone
	 *            are we running outside the workbench
	 * @return the type of execution component that should be used (which can be
	 *         passed to the factory)
	 * @throws ClassNotFoundException
	 */
	protected synchronized String getExecutionComponentForTestType(
			final String executionComponentType, final String testType,
			final boolean bStandalone) throws ClassNotFoundException {
		Object temp = this.testTypeMap.get(executionComponentType);
		if ((temp != null) && (temp instanceof HashMap)) {
			final HashMap execCompMap = (HashMap) temp;
			temp = execCompMap.get(testType);
			if ((temp != null) && (temp instanceof String)) {
				return (String) temp;
			}
		}

		String msg = ExecutionHarnessPluginResourceBundle.EXEC_NOT_FOUND_ERR_;
		msg = StringUtil.change(msg, "%1", executionComponentType); //$NON-NLS-1$
		msg = StringUtil.change(msg, "%2", testType); //$NON-NLS-1$

		// We couldn't find the right execution component
		throw new ClassNotFoundException(msg);

	}

	/**
	 * @param suite
	 * @param theTest
	 * @return
	 */
	private IImplementor getImplementor(final ITestSuite suite,
			final ITest theTest) {
		IImplementor theImplementor = null;

		if (theTest != null) {
			theImplementor = theTest.getImplementor();
		} else {
			theImplementor = suite.getImplementor();
		}

		return theImplementor;
	}

	private ISession getSession(final String port, final ISession session,
			final String machine, final StringBuffer bufError) {
		bufError.setLength(0);
		final SessionContext ctx = new SessionContext(machine, port, null);
		return SessionContextToSessionMap.getInstance().getSessionEntry(ctx);
	}

	private void init() {
		// Initialize our EMF models.
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				ICommonConstants.TEST_SUITE_FILE_EXTENSION,
				new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				ICommonConstants.DEPLOYMENT_FILE_EXTENSION,
				new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				ICommonConstants.EXECUTION_FILE_EXTENSION,
				new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				ICommonConstants.ARTIFACT_FILE_EXTENSION,
				new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				ICommonConstants.LOCATION_FILE_EXTENSION,
				new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				ICommonConstants.DATAPOOL_FILE_EXTENSION,
				new DatapoolFacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap()
				.put(ICommonConstants.SUT_EXTENSION,
						new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				ICommonConstants.TEST_COMPONENT_EXTENSION,
				new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				"xmi", new FacadeResourceFactoryImpl()); //$NON-NLS-1$
		if(ModelDebugger.INSTANCE.debugDatabaseResourcePostfix!=null && ModelDebugger.INSTANCE.debugDatabaseResourcePostfix.length()>0)
		{
			//add support for RDB backed execution resources
			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("executiondb", new HyadesExtendedResourceFactory());//$NON-NLS-1$
		}

		Common_TestprofilePackageImpl.init();
	}

	/**
	 * This method initializes the ExecutionComponentFactory with all registered
	 * execution component types. It also populates a testTypeMap to allow
	 * determination of which execution components should be used by which test
	 * types.
	 * 
	 * @param factory
	 *            the factory to be initialized
	 * @param bStandalone
	 *            whether we are running outside the workbench. If we are, this
	 *            method delegates to a method in StandaloneExecutionUtilities
	 * @throws ClassNotFoundException
	 * @throws FileNotFoundException
	 */
	protected void initializeRegisteredExecutionComponents(
			final IExecutionComponentFactory factory, boolean bStandalone,
			final ExecutionHarnessDataProcessorFactory dpFactory)
			throws ClassNotFoundException, FileNotFoundException {

		// Need to ensure exclusive access while initializing registered
		// execution components across
		// test execution harness instances. Without this synchronization race
		// conditions appear
		// when launching concurrent test executions.
		synchronized (TestExecutionHarness.class) {

			if (!bStandalone) {
				final IEclipseExecutionComponentFactory eclipseFactory = (IEclipseExecutionComponentFactory) factory;
				final IExtensionRegistry reg = Platform.getExtensionRegistry();

				final IConfigurationElement[] execComps = reg
						.getConfigurationElementsFor("org.eclipse.hyades.test.core.RegisteredExecutionComponentImpl"); //$NON-NLS-1$

				for (int i = 0; i < execComps.length; i++) {
					final String type = execComps[i].getAttribute("type"); //$NON-NLS-1$

					final String executionComponentName = execComps[i]
							.getAttribute("name"); //$NON-NLS-1$
					eclipseFactory.addExecutionComponent(execComps[i]);

					Object map = this.testTypeMap.get(type);
					if (map == null) {
						map = new HashMap();
						this.testTypeMap.put(type, map);
					}
					final HashMap execCompMap = (HashMap) map;

					final IConfigurationElement[] supportedTestTypes = execComps[i]
							.getChildren("SupportedTestType"); //$NON-NLS-1$
					for (int j = 0; j < supportedTestTypes.length; j++) {
						execCompMap.put(supportedTestTypes[j]
								.getAttribute("name"), executionComponentName); //$NON-NLS-1$
					}
				}

				this
						.mapAdapterClasses(
								"org.eclipse.hyades.test.core.ExecutionEnvironmentAdapter", "EXECUTION_ENVIRONMENT_ADAPTER", bStandalone); //$NON-NLS-1$ //$NON-NLS-2$
				this
						.mapAdapterClasses(
								"org.eclipse.hyades.test.core.ExecutableObjectAdapter", "EXECUTABLE_OBJECT_ADAPTER", bStandalone); //$NON-NLS-1$ //$NON-NLS-2$
				this
						.mapAdapterClasses(
								"org.eclipse.hyades.test.core.ExecutionDeploymentAdapter", "EXECUTION_DEPLOYMENT_ADAPTER", bStandalone); //$NON-NLS-1$ //$NON-NLS-2$

			} else {
				StandaloneExecutionUtilities
						.initializeRegisteredExecutionComponents(
								this.standaloneConfigurationFiles,
								this.testTypeMap, factory, dpFactory);
			}
		}
	}

	private void initializeRemoteHyadesComponent(final String agentType,
			final IExecutionHarnessDataProcessor dp, final IExecutor executor) {
		final IRemoteHyadesComponent remoteComponent = (IRemoteHyadesComponent) this.factory
				.createExecutionComponentByType(agentType);

		executor.addChild(remoteComponent);

		// Create a state change listener on the Agent.
		remoteComponent
				.addExecutionComponentStateChangeListener(new IExecutionComponentStateChangeListener() {
					public void stateChanged(
							final ExecutionComponentStateChangeEvent event) {
						// When the agent is ready, release the agent lock
						if (event.getState() == IExecutionComponent.READY) {
							releaseAgentLock();
						}
					}
				});

		if (dp != null) {
			remoteComponent.startMonitoring(dp);
		}

		// Wait for the agent to become active
		this.waitForActiveAgent(remoteComponent);
	}

	/**
	 * initializes dataprocessors that are valid for this run.
	 * 
	 * @param theTest
	 * @param hostName
	 * @param agentType
	 * @param executionResultName
	 * @param executionResultLocation
	 * @param port
	 * @param validDPIds
	 * @return
	 */
	private ArrayList initValidDataProcessors(final ITest theTest,
			final String hostName, final String agentType,
			final String executionResultName,
			final String executionResultLocation,
			final boolean overrideExisting, final String port,
			final ArrayList validDPIds, final IExecutor executor,
			final TPFDeployment deployment, final HashMap optionsMap) {

		final ArrayList retList = new ArrayList();
		final ArrayList dpList = this.dpFactory.getDataProcessors(validDPIds);
		for (int i = 0; i < dpList.size(); i++) {
			final IExecutionHarnessDataProcessor dataProcessor = (IExecutionHarnessDataProcessor) dpList
					.get(i);
			if (validDPIds.contains(dataProcessor.getID())) {
				if (dataProcessor instanceof IExecutionHarnessDataProcessor3) {
					((IExecutionHarnessDataProcessor3)dataProcessor).setInitData(theTest, hostName,
							executionResultName, executionResultLocation,
							overrideExisting, port, deployment, optionsMap);
				}
				else if (dataProcessor instanceof IExecutionHarnessDataProcessor2) {
					((IExecutionHarnessDataProcessor2) dataProcessor)
							.setInitData(theTest, hostName,
									executionResultName,
									executionResultLocation, overrideExisting,
									port);
				} else {
					dataProcessor.setInitData(theTest, hostName,
							executionResultName, executionResultLocation,
							overrideExisting, port, deployment);
				}

				// this should ensure that the executionhistory
				// dataprocessor is
				// first in the list
				if (dataProcessor.getID().equalsIgnoreCase(
						XMLExecutionDataProcessor.IID)) {
					retList.add(0, dataProcessor);
				} else {
					retList.add(dataProcessor);
				}
			}
		}

		// now lets go through the collection again and init them all starting
		// with the execution history dp if it's present
		// if it is not present, we will send a dummy control agent in it's
		// place.
		for (int i = 0; i < retList.size(); i++) {
			final IExecutionHarnessDataProcessor dp = (IExecutionHarnessDataProcessor) retList
					.get(i);
			if ((i == 0)
					&& (dp.getID().compareToIgnoreCase(
							XMLExecutionDataProcessor.IID) != 0)) {
				this.initializeRemoteHyadesComponent(agentType, null, executor);
			} else {
				this.initializeRemoteHyadesComponent(agentType, dp, executor);
			}
		}

		return retList;

	}

	/**
	 * Determines if the harness is running in standalone mode, using the
	 * platform is running method, switched from the method used in 3.3 due to
	 * the get product being null in cases where a run-time instance is launched
	 * in application mode instead of product mode
	 * 
	 * @return true if in standalone mode, false otherwise
	 */
	private boolean isRunningStandalone() {
		return (!Platform.isRunning());
	}

	/**
	 * @param theImplementor
	 * @param overrideTestType
	 * @param machineName
	 * @param deployment
	 * @param port
	 * @param executionResultLocation
	 * @param executionResultName
	 * @param bStandalone
	 * @param activeDataProcessorIDs
	 * @param bufError
	 * @param progressMonitor
	 * @param progressVisibility
	 *            determines if progress is shown in the user interface for this
	 *            test execution
	 */
	private synchronized IExecutor launchTest(
			final IImplementor theImplementor, final String overrideTestType,
			final String machineName, final TPFDeployment deployment,
			final String port, final String executionResultLocation,
			final String executionResultName, final boolean bStandalone,
			final ArrayList activeDataProcessorIDs,
			final StringBuffer bufError,
			final IProgressMonitor progressMonitor,
			final TestExecutionHarness.ProgressVisibility progressVisibility) {
		HashMap optionsMap = new HashMap();
		return this.launchTest(theImplementor, overrideTestType, machineName,
				deployment, port, executionResultLocation, executionResultName,
				false, bStandalone, activeDataProcessorIDs, optionsMap, bufError,
				progressMonitor, progressVisibility);
	}

	/**
	 * @param theImplementor
	 * @param overrideTestType
	 * @param machineName
	 * @param deployment
	 * @param port
	 * @param executionResultLocation
	 * @param executionResultName
	 * @param overrideExistingResult
	 * @param bStandalone
	 * @param activeDataProcessorIDs
	 * @param bufError
	 * @param progressMonitor
	 * @param progressVisibility
	 *            indicates if test execution progress is visible in the user
	 *            interface
	 * @return
	 */
	private IExecutor launchTest(
			final IImplementor theImplementor, final String overrideTestType,
			final String machineName, final TPFDeployment deployment,
			final String port, final String executionResultLocation,
			final String executionResultName,
			final boolean overrideExistingResult, final boolean bStandalone,
			final ArrayList activeDataProcessorIDs,
			final HashMap optionsMap,
			final StringBuffer bufError,
			final IProgressMonitor progressMonitor,
			final ProgressVisibility progressVisibility) {

		//Holds the executor to return back to caller of method:
		final MutableObject mutableExecutor = new MutableObject();

		//Kick off a new job to execute the test:
		final Job monitor = new Job(NLS.bind(ExecutionHarnessPluginResourceBundle.TEST_SUITE_JOB_NAME, theImplementor.getOwner().getName())) {

			/*
			 * Job run method that launches and waits for test execution to
			 * complete
			 */
			public IStatus run(IProgressMonitor monitor) {

				/*
				 * Create a mutable boolean to monitor the completeness of the
				 * execution
				 */
				final MutableBoolean complete = new MutableBoolean(false);

				/*
				 * Lock to use for launching and execution 
				 */
				final Object launchLock = TestExecutionHarness.this.instanceLock;
				
			
				// For serialized requests, task name to reflect wait
				monitor.setTaskName(ExecutionHarnessPluginResourceBundle.WAITING_TO_EXECUTE);

				// Need to obtain lock (either instance of class-based)
				synchronized (launchLock) {

					// Rename task to give better description
					monitor.setTaskName(ExecutionHarnessPluginResourceBundle.PREPARING_AND_DEPLOYING);
					monitor.worked(10);

					// Launch the test execution
					IExecutor executor = TestExecutionHarness.this
							.launchTestExecution(theImplementor,
									overrideTestType, machineName, deployment,
									port, executionResultLocation,
									executionResultName,
									overrideExistingResult, bStandalone,
									activeDataProcessorIDs, optionsMap, bufError,
									progressMonitor);

					// Set the mutable executor to the initialized executor
					monitor.worked(10);
					mutableExecutor.set(executor);

					if (executor != null) {

						// Rename task to give better description
						monitor.setTaskName(ExecutionHarnessPluginResourceBundle.RUNNING_AND_COLLECTING_RESULTS);

						// Monitor the test execution
						executor.addExecutionComponentStateChangeListener(new IExecutionComponentStateChangeListener() {

							public void stateChanged(ExecutionComponentStateChangeEvent state) {

								//Clean-up after first notification that the executor has died:
								if ((state.getState() == IExecutionComponent.DEAD) && (complete.isFalse())) {

									//Mark state as complete:
									complete.setValue(true);

									cleanup(((IExecutor)(mutableExecutor.get())));											

									//Notify waiting threads that test execution is complete:
									synchronized (complete) {

										complete.notifyAll();

										mutableExecutor.clear();
									}
								}
							}
						});							
					}

					//Notify waiting threads that executor is set-up:
					synchronized (executorLock) {
						executorLock.notifyAll();
					}

					//Mark state as complete if there is no executor or its state is dead:
					complete.setValue((mutableExecutor.isNull()) || ((executor != null) && (executor.getState() == IExecutionComponent.DEAD)));

					// Ready to report progress
					int workInstallment = 0;
					int workCompleted = 0;
					
					//Wait for the test execution to complete or be canceled:
					while ((complete.isFalse()) && (!monitor.isCanceled())) {

						//Calculate a unit of work:
						workInstallment = ((80 - workCompleted)/8);
						workCompleted = workCompleted + workInstallment;

						//Increment the progress monitor:
						monitor.worked(workInstallment);

						try {
							
							//Wait for 2.25 seconds:
							synchronized (complete) {
								complete.wait(2250);
							}
						} 
						catch (InterruptedException e) {
							//Ignore.
						}
					}
					
					if(monitor.isCanceled()){
						
						//Clean-up the executor:
						cleanup(executor);
					}

					//Finish the progress monitor:
					monitor.done();

					//Clear mutable executor to eliminate potential leaks:
					mutableExecutor.clear();

					return (Status.OK_STATUS);
				} 
			}

			/**
			 * Cleans the {@link IExecutor} after the test execution has terminated.
			 * 
			 * @param executor The {@link IExecutor} to be cleaned.
			 */
			private void cleanup(IExecutor executor){

				try {

					SessionStub sessionStub = ((SessionStub)((ExecutorStub)(executor)).getSessionContext());
					
					((ExecutorStub)(executor)).setDataProcessors(null);

					//Clean the agent:
					IExecutionComponent[] children = executor.getChildren();

					for (int counter = 0; counter < children.length; counter++) {
						executor.removeChild(((ExecutionComponentStub)(children[counter])));
					}

					IExecutableObject executableObject = executor.getExecutableObject();

					if (executableObject != null) {

						try {
							executableObject.setExecutor(null);
						} 
						catch (Exception e) {
							//Ignore.		
						}
						
						try {
							executor.setExecutableObject(null);
						} 
						catch (Exception e2) {
							//Ignore.		
						}
					}

					IExecutionEnvironment executionEnvironment = ((IExecutionEnvironment)(executor.getParent()));

					if (executionEnvironment != null) {

						//Clean the executor:
						ExecutionComponentStub executionComponentStub = ((ExecutionComponentStub)(executor));

						executionEnvironment.removeChild(executionComponentStub);

						executionComponentStub.setSessionContext(null);
					}

					//Sessions can only be used once with the code below, when session caching is put back into the product this
					//will have to be repaired.  Basically the session will only be "really" cleaned up when its last reference is
					//removed.
					if (sessionStub != null) {

						//Remove the agent listener:
						sessionStub.removeListener();

						// Reverse cleanup of execution environment:
						children = sessionStub.getChildren();

						for (int counter = 0; counter < children.length; counter++) {
							sessionStub.removeChild(((ExecutionComponentStub)(children[counter])));
						}
					}
				} 
				catch (Throwable t) {
					//Ignore.
				}
			}
		};

		//Set test launch and execution priority and schedule the task:
		monitor.setUser(progressVisibility == TestExecutionHarness.ProgressVisibility.ENABLED);
		monitor.setSystem(progressVisibility == TestExecutionHarness.ProgressVisibility.DISABLED);
		monitor.setPriority(Job.INTERACTIVE);
		
		final IProgressMonitor executionProgress = Job.getJobManager().createProgressGroup();
		executionProgress.beginTask(ExecutionHarnessPluginResourceBundle.EXECUTING_TESTS, 100);
		
		monitor.setProgressGroup(executionProgress, 100);

		//Schedule the job:
		monitor.schedule();

		//Return executor:
		IExecutor executor = null;

		while ((mutableExecutor.isNull()) && ((monitor.getState() == Job.RUNNING) || (monitor.getState() == Job.WAITING))) {
			
			try {
			
				synchronized (executorLock) {
				
					//Wait for the executor to be set, indicating the test execution process has started:
					executorLock.wait(3000);

					//Resolve the executor:
					executor = (IExecutor) mutableExecutor.get();
				}
			} 
			catch (final InterruptedException e) {
				//Ignore.
			}
		}

		//Executor may be null (for example, cancel or error):
		return executor;
	}

	/**
	 * Launch a test execution without a progress monitor
	 * 
	 * @param suite
	 *            the test suite to be launched
	 * @param theTest
	 *            the test to be launched (may be the same as suite, or may be a
	 *            test case contained within suite.
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param executionResultLocation
	 *            the directory in which to write the execution history file
	 * @param executionResultName
	 *            the filename for the execution history file
	 * @param bStandalone
	 *            are we not launching from the workbench
	 * @param activeDataProcessorIDs
	 *            a String[] of the dataprocessors that should be activated for
	 *            this launch. This ID should match the ID provided in the
	 *            extension.
	 * @param bufError
	 *            the accumulation of errors if encountered during launch
	 * @return an executor associated with the test execution, to control the
	 *         test -- at the time the executor returns, the test execution is
	 *         in most situations still running the test since the launch test
	 *         method does not block on the test execution
	 */
	public IExecutor launchTest(final ITestSuite suite, final ITest theTest,
			final String hostName, final String port,
			final String executionResultLocation,
			final String executionResultName, final boolean bStandalone,
			final ArrayList activeDataProcessorIDs, final StringBuffer bufError) {
		return this.launchTest(suite, theTest, hostName, port,
				executionResultLocation, executionResultName, bStandalone,
				activeDataProcessorIDs, bufError, this
						.createNullProgressMonitor());
	}

	/**
	 * Launch a test execution with a progress monitor
	 * 
	 * @param suite
	 *            the test suite to be launched
	 * @param theTest
	 *            the test to be launched (may be the same as suite, or may be a
	 *            test case contained within suite.
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param executionResultLocation
	 *            the directory in which to write the execution history file
	 * @param executionResultName
	 *            the filename for the execution history file
	 * @param bStandalone
	 *            are we not launching from the workbench
	 * @param activeDataProcessorIDs
	 *            a String[] of the dataprocessors that should be activated for
	 *            this launch. This ID should match the ID provided in the
	 *            extension.
	 * @param bufError
	 *            the accumulation of errors if encountered during launch
	 * @param progressMonitor
	 *            progress monitor to use to cancel or retrieve status of the
	 *            test execution
	 * @return an executor associated with the test execution, to control the
	 *         test -- at the time the executor returns, the test execution is
	 *         in most situations still running the test since the launch test
	 *         method does not block on the test execution
	 */
	public IExecutor launchTest(final ITestSuite suite, final ITest theTest,
			final String hostName, final String port,
			final String executionResultLocation,
			final String executionResultName, final boolean bStandalone,
			final ArrayList activeDataProcessorIDs,
			final StringBuffer bufError, final IProgressMonitor progressMonitor) {

		try {
			IExecutor executor = null;
			if (!progressMonitor.isCanceled()) {
				final IImplementor theImplementor = this.getImplementor(suite,
						theTest);
				if (!progressMonitor.isCanceled()) {
					executor = this.launchTest(theImplementor, null, hostName,
							null, port, executionResultLocation,
							executionResultName, bStandalone,
							activeDataProcessorIDs, bufError, progressMonitor,
							TestExecutionHarness.ProgressVisibility.ENABLED);
				}
			}
			return executor;
		} finally {
			progressMonitor.done();
		}

	}

	/**
	 * <p>Launches the specified test on the specified machine.</p>
	 * 
	 * <p>The flow of the <code>launchTest()</code> method is depicted in the 
	 * following execution environment use case:</p>
	 * 
	 * <p>
	 * <a href="http://www.eclipse.org/tptp/test/documents/design/usecases.html#2">
	 * Launch a single test instance from workbench on a single specified node and 
	 * show control events to the running test</a></p>
	 * 
	 * @param suite
	 *            the test suite to be launched
	 * @param theTest
	 *            the test to be launched (may be the same as suite, or may be a
	 *            test case contained within suite.
	 * @param machineName
	 *            the machine name on which the test will be launched (IP
	 *            address or host name)
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param executionResultLocation
	 *            the directory in which to write the execution history file
	 * @param executionResultName
	 *            the filename for the execution history file
	 * @param bMonitor
	 *            should we monitor the test
	 * @param bStandalone
	 *            are we not launching from the workbench
	 * @param bufError
	 *            the accumulation of errors if encountered during launch
	 * @return an executor associated with the test execution, to control the
	 *         test -- at the time the executor returns, the test execution is
	 *         in most situations still running the test since the launch test
	 *         method does not block on the test execution
	 */
	public synchronized IExecutor launchTest(final ITestSuite suite,
			final ITest theTest, final String machineName, final String port,
			final String executionResultLocation,
			final String executionResultName, final boolean bMonitor,
			final boolean bStandalone, final StringBuffer bufError) {
		return this.launchTest(suite, theTest, machineName, port,
				executionResultLocation, executionResultName, bMonitor,
				bStandalone, bufError, this.createNullProgressMonitor());
	}

	/**
	 * <p>Launches the specified test on the specified machine.</p>
	 * 
	 * <p>The flow of the <code>launchTest()</code> method is depicted in the 
	 * following execution environment use case:</p>
	 * 
	 * <p>
	 * <a href="http://www.eclipse.org/tptp/test/documents/design/usecases.html#2">
	 * Launch a single test instance from workbench on a single specified node and 
	 * show control events to the running test</a></p>
	 * 
	 * @param suite
	 *            the test suite to be launched
	 * @param theTest
	 *            the test to be launched (may be the same as suite, or may be a
	 *            test case contained within suite.
	 * @param machineName
	 *            the machine name on which the test will be launched (IP
	 *            address or host name)
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param executionResultLocation
	 *            the directory in which to write the execution history file
	 * @param executionResultName
	 *            the filename for the execution history file
	 * @param bMonitor
	 *            should we monitor the test
	 * @param bStandalone
	 *            are we not launching from the workbench
	 * @param progressMonitor
	 *            progress monitor to use
	 * @param bufError
	 *            the accumulation of errors if encountered during launch
	 * @return an executor associated with the test execution, to control the
	 *         test -- at the time the executor returns, the test execution is
	 *         in most situations still running the test since the launch test
	 *         method does not block on the test execution
	 */
	public synchronized IExecutor launchTest(final ITestSuite suite,
			final ITest theTest, final String machineName, final String port,
			final String executionResultLocation,
			final String executionResultName, final boolean bMonitor,
			final boolean bStandalone, final StringBuffer bufError,
			final IProgressMonitor progressMonitor) {
		ArrayList dpIDs = null;
		if (bMonitor) {
			dpIDs = new ArrayList();
			dpIDs.add(XMLExecutionDataProcessor.IID);
		}

		try {
			IExecutor executor = null;
			if (!progressMonitor.isCanceled()) {
				final IImplementor theImplementor = this.getImplementor(suite,
						theTest);
				if (!progressMonitor.isCanceled()) {
					executor = this.launchTest(theImplementor, null,
							machineName, null, port, executionResultLocation,
							executionResultName, bStandalone, dpIDs, bufError,
							progressMonitor,
							TestExecutionHarness.ProgressVisibility.ENABLED);
				}
			}
			return executor;
		} finally {
			progressMonitor.done();
		}

	}

	/**
	 * <p>Launches the specified test on the specified machine.</p>
	 * 
	 * <p>The flow of the <code>launchTest()</code> method is depicted in the 
	 * following execution environment use case:</p>
	 * 
	 * <p>
	 * <a href="http://www.eclipse.org/tptp/test/documents/design/usecases.html#2">
	 * Launch a single test instance from workbench on a single specified node and 
	 * show control events to the running test</a></p>
	 * 
	 * @param suite
	 *            the test suite to be launched
	 * @param theTest
	 *            the test to be launched (may be the same as suite, or may be a
	 *            test case contained within suite.
	 * @param overrideTestType
	 *            if specified, the test type to be used by the execution
	 *            harness (regardless of the test type specified by the test)
	 * @param deployment
	 *            the deployment with which to launch the test
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param executionResultLocation
	 *            the directory in which to write the execution history file
	 * @param executionResultName
	 *            the filename for the execution history file
	 * @param bStandalone
	 *            are we not launching from the workbench
	 * @param activeDataProcessorIDs
	 *            a String[] of the dataprocessors that should be activated for
	 *            this launch. This ID should match the ID provided in the
	 *            extension.
	 * @param bufError
	 *            the accumulation of errors if encountered during launch
	 * @return an executor associated with the test execution, to control the
	 *         test -- at the time the executor returns, the test execution is
	 *         in most situations still running the test since the launch test
	 *         method does not block on the test execution
	 */
	public IExecutor launchTest(final ITestSuite suite, final ITest theTest,
			final String overrideTestType, final TPFDeployment deployment,
			final String port, final String executionResultLocation,
			final String executionResultName, final boolean bStandalone,
			final ArrayList activeDataProcessorIDs, final StringBuffer bufError) {
		return this.launchTest(suite, theTest, overrideTestType, deployment,
				port, executionResultLocation, executionResultName,
				bStandalone, activeDataProcessorIDs, bufError, this
						.createNullProgressMonitor());
	}

	/**
	 * <p>Launches the specified test on the specified machine.</p>
	 * 
	 * <p>The flow of the <code>launchTest()</code> method is depicted in the 
	 * following execution environment use case:</p>
	 * 
	 * <p>
	 * <a href="http://www.eclipse.org/tptp/test/documents/design/usecases.html#2">
	 * Launch a single test instance from workbench on a single specified node and 
	 * show control events to the running test</a></p>
	 * 
	 * @param suite
	 *            the test suite to be launched
	 * @param theTest
	 *            the test to be launched (may be the same as suite, or may be a
	 *            test case contained within suite.
	 * @param overrideTestType
	 *            if specified, the test type to be used by the execution
	 *            harness (regardless of the test type specified by the test)
	 * @param deployment
	 *            the deployment with which to launch the test
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param executionResultLocation
	 *            the directory in which to write the execution history file
	 * @param executionResultName
	 *            the filename for the execution history file
	 * @param bStandalone
	 *            are we not launching from the workbench
	 * @param activeDataProcessorIDs
	 *            a String[] of the dataprocessors that should be activated for
	 *            this launch. This ID should match the ID provided in the
	 *            extension.
	 * @param bufError
	 *            the accumulation of errors if encountered during launch
	 * @param progressMonitor
	 *            progress monitor to use
	 * @return an executor associated with the test execution, to control the
	 *         test -- at the time the executor returns, the test execution is
	 *         in most situations still running the test since the launch test
	 *         method does not block on the test execution
	 *  @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public IExecutor launchTest(final ITestSuite suite, final ITest theTest,
			final String overrideTestType, final TPFDeployment deployment,
			final String port, final String executionResultLocation,
			final String executionResultName, final boolean bStandalone,
			final ArrayList activeDataProcessorIDs,
			final StringBuffer bufError, final IProgressMonitor progressMonitor) {

		return this.launchTest(suite, theTest, overrideTestType, deployment,
				port, executionResultLocation, executionResultName,
				/* overrideExistingResult */false, bStandalone,
				activeDataProcessorIDs, bufError, progressMonitor);
	}

	/**
	 * <p>Launches the specified test on the specified machine.</p>
	 * 
	 * <p>The flow of the <code>launchTest()</code> method is depicted in the 
	 * following execution environment use case:</p>
	 * 
	 * <p>
	 * <a href="http://www.eclipse.org/tptp/test/documents/design/usecases.html#2">
	 * Launch a single test instance from workbench on a single specified node and 
	 * show control events to the running test</a></p>
	 * 
	 * @param suite
	 *            the test suite to be launched
	 * @param theTest
	 *            the test to be launched (may be the same as suite, or may be a
	 *            test case contained within suite.
	 * @param overrideTestType
	 *            if specified, the test type to be used by the execution
	 *            harness (regardless of the test type specified by the test)
	 * @param deployment
	 *            the deployment with which to launch the test
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param executionResultLocation
	 *            the directory in which to write the execution history file
	 * @param executionResultName
	 *            the filename for the execution history file
	 * @param overrideExistingResult
	 *            whether to override existing result if such an execution
	 *            result already exists
	 * @param bStandalone
	 *            are we not launching from the workbench
	 * @param activeDataProcessorIDs
	 *            a String[] of the dataprocessors that should be activated for
	 *            this launch. This ID should match the ID provided in the
	 *            extension.
	 * @param bufError
	 *            the accumulation of errors if encountered during launch
	 * @param progressMonitor
	 *            progress monitor to use
	 * @return an executor associated with the test execution, to control the
	 *         test -- at the time the executor returns, the test execution is
	 *         in most situations still running the test since the launch test
	 *         method does not block on the test execution
	 */
	public IExecutor launchTest(final ITestSuite suite, final ITest theTest,
			final String overrideTestType, final TPFDeployment deployment,
			final String port, final String executionResultLocation,
			final String executionResultName,
			final boolean overrideExistingResult, final boolean bStandalone,
			final ArrayList activeDataProcessorIDs,
			final StringBuffer bufError, final IProgressMonitor progressMonitor) {

		try {
			IExecutor executor = null;
			if (!progressMonitor.isCanceled()) {
				final IImplementor theImplementor = this.getImplementor(suite,
						theTest);
				final HashMap optionsMap = new HashMap();
				if (!progressMonitor.isCanceled()) {
					executor = this.launchTest(theImplementor,
							overrideTestType, null, deployment, port,
							executionResultLocation, executionResultName,
							overrideExistingResult, bStandalone,
							activeDataProcessorIDs, optionsMap, bufError, 
							progressMonitor,
							TestExecutionHarness.ProgressVisibility.ENABLED);
				}
			}
			return executor;
		} finally {
			progressMonitor.done();
		}

	}

	
	/**
	 * <p>Launches the specified test on the specified machine.</p>
	 * 
	 * <p>The flow of the <code>launchTest()</code> method is depicted in the 
	 * following execution environment use case:</p>
	 * 
	 * <p>
	 * <a href="http://www.eclipse.org/tptp/test/documents/design/usecases.html#2">
	 * Launch a single test instance from workbench on a single specified node and 
	 * show control events to the running test</a></p>
	 * 
	 * @param suite
	 *            the test suite to be launched
	 * @param theTest
	 *            the test to be launched (may be the same as suite, or may be a
	 *            test case contained within suite.
	 * @param deployment
	 *            the deployment with which to launch the test
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param executionResultLocation
	 *            the directory in which to write the execution history file
	 * @param executionResultName
	 *            the filename for the execution history file
	 * @param bStandalone
	 *            are we not launching from the workbench
	 * @param activeDataProcessorIDs
	 *            a String[] of the dataprocessors that should be activated for
	 *            this launch. This ID should match the ID provided in the
	 *            extension.
	 * @param bufError
	 *            the accumulation of errors if encountered during launch
	 * @return an executor associated with the test execution, to control the
	 *         test -- at the time the executor returns, the test execution is
	 *         in most situations still running the test since the launch test
	 *         method does not block on the test execution
	 */
	public IExecutor launchTest(final ITestSuite suite, final ITest theTest,
			final TPFDeployment deployment, final String port,
			final String executionResultLocation,
			final String executionResultName, final boolean bStandalone,
			final ArrayList activeDataProcessorIDs, final StringBuffer bufError) {
		return this.launchTest(suite, theTest, deployment, port,
				executionResultLocation, executionResultName, bStandalone,
				activeDataProcessorIDs, bufError, this
						.createNullProgressMonitor());
	}

	/**
	 * <p>Launches the specified test on the specified machine.</p>
	 * 
	 * <p>The flow of the <code>launchTest()</code> method is depicted in the 
	 * following execution environment use case:</p>
	 * 
	 * <p>
	 * <a href="http://www.eclipse.org/tptp/test/documents/design/usecases.html#2">
	 * Launch a single test instance from workbench on a single specified node and 
	 * show control events to the running test</a></p>
	 * 
	 * @param suite
	 *            the test suite to be launched
	 * @param theTest
	 *            the test to be launched (may be the same as suite, or may be a
	 *            test case contained within suite.
	 * @param overrideTestType
	 *            if specified, the test type to be used by the execution
	 *            harness (regardless of the test type specified by the test)
	 * @param deployment
	 *            the deployment with which to launch the test
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param executionResultLocation
	 *            the directory in which to write the execution history file
	 * @param executionResultName
	 *            the filename for the execution history file
	 * @param overrideExistingResult
	 *            whether to override existing result if such an execution
	 *            result already exists
	 * @param bStandalone
	 *            are we not launching from the workbench
	 * @param activeDataProcessorIDs
	 *            a String[] of the dataprocessors that should be activated for
	 *            this launch. This ID should match the ID provided in the
	 *            extension.
	 * @param bufError
	 *            the accumulation of errors if encountered during launch
	 * @param progressMonitor
	 *            progress monitor to use
	 * @param progressVisibility
	 *            indicates if progress is shown in the user interface
	 * @return an executor associated with the test execution, to control the
	 *         test -- at the time the executor returns, the test execution is
	 *         in most situations still running the test since the launch test
	 *         method does not block on the test execution
	 * @provisional
	 */
	public IExecutor launchTest(final ITestSuite suite, final ITest theTest,
			final String overrideTestType, final TPFDeployment deployment,
			final String port, final String executionResultLocation,
			final String executionResultName,
			final boolean overrideExistingResult, final boolean bStandalone,
			final ArrayList activeDataProcessorIDs,
			final StringBuffer bufError, final IProgressMonitor progressMonitor,
			final TestExecutionHarness.ProgressVisibility progressVisibility) {

		try {
			IExecutor executor = null;
			if (!progressMonitor.isCanceled()) {
				final IImplementor theImplementor = this.getImplementor(suite,
						theTest);
				final HashMap optionsMap = new HashMap();
				
				if (!progressMonitor.isCanceled()) {
					executor = this.launchTest(theImplementor,
							overrideTestType, null, deployment, port,
							executionResultLocation, executionResultName,
							overrideExistingResult, bStandalone,
							activeDataProcessorIDs, optionsMap, bufError, 
							progressMonitor,
							progressVisibility);
				}
			}
			return executor;
		} finally {
			progressMonitor.done();
		}

	}
	

	/**
	 * <p>Launches the specified test on the specified machine.</p>
	 * 
	 * <p>The flow of the <code>launchTest()</code> method is depicted in the 
	 * following execution environment use case:</p>
	 * 
	 * <p>
	 * <a href="http://www.eclipse.org/tptp/test/documents/design/usecases.html#2">
	 * Launch a single test instance from workbench on a single specified node and 
	 * show control events to the running test</a></p>
	 * 
	 * @param suite
	 *            the test suite to be launched
	 * @param theTest
	 *            the test to be launched (may be the same as suite, or may be a
	 *            test case contained within suite.
	 * @param deployment
	 *            the deployment with which to launch the test
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param executionResultLocation
	 *            the directory in which to write the execution history file
	 * @param executionResultName
	 *            the filename for the execution history file
	 * @param bStandalone
	 *            are we not launching from the workbench
	 * @param activeDataProcessorIDs
	 *            a String[] of the dataprocessors that should be activated for
	 *            this launch. This ID should match the ID provided in the
	 *            extension.
	 * @param bufError
	 *            the accumulation of errors if encountered during launch
	 * @param progressMonitor
	 *            progress monitor to use
	 * @return an executor associated with the test execution, to control the
	 *         test -- at the time the executor returns, the test execution is
	 *         in most situations still running the test since the launch test
	 *         method does not block on the test execution
	 */
	public IExecutor launchTest(final ITestSuite suite, final ITest theTest,
			final TPFDeployment deployment, final String port,
			final String executionResultLocation,
			final String executionResultName, final boolean bStandalone,
			final ArrayList activeDataProcessorIDs,
			final StringBuffer bufError, final IProgressMonitor progressMonitor) {

		try {
			IExecutor executor = null;
			if (!progressMonitor.isCanceled()) {
				final IImplementor theImplementor = this.getImplementor(suite,
						theTest);
				if (!progressMonitor.isCanceled()) {
					executor = this.launchTest(theImplementor, null, null,
							deployment, port, executionResultLocation,
							executionResultName, bStandalone,
							activeDataProcessorIDs, bufError, progressMonitor,
							TestExecutionHarness.ProgressVisibility.ENABLED);
				}
			}
			return executor;
		} finally {
			progressMonitor.done();
		}

	}

	/**
	 * <p>Launches the specified test on the specified machine.</p>
	 * 
	 * <p>The flow of the <code>launchTest()</code> method is depicted in the 
	 * following execution environment use case:</p>
	 * 
	 * <p>
	 * <a href="http://www.eclipse.org/tptp/test/documents/design/usecases.html#2">
	 * Launch a single test instance from workbench on a single specified node and 
	 * show control events to the running test</a></p>
	 * 
	 * @param suite
	 *            the test suite to be launched
	 * @param theTest
	 *            the test to be launched (may be the same as suite, or may be a
	 *            test case contained within suite.
	 * @param machineName
	 *            the machine name on which the test will be launched (IP
	 *            address or host name)
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param executionResultLocation
	 *            the directory in which to write the execution history file
	 * @param executionResultName
	 *            the filename for the execution history file
	 * @param overrideExisting
	 *            whether override existing if there is such an execution result
	 *            already exist
	 * @param bMonitor
	 *            should we monitor the test
	 * @param bStandalone
	 *            are we not launching from the workbench
	 * @param bufError
	 *            the accumulation of errors if encountered during launch
	 * @return an executor associated with the test execution, to control the
	 *         test -- at the time the executor returns, the test execution is
	 *         in most situations still running the test since the launch test
	 *         method does not block on the test execution
	 */
	public synchronized IExecutor launchTest(final ITestSuite suite,
			final ITest theTest, final TPFDeployment deployment,
			final String port, final String executionResultLocation,
			final String executionResultName, final boolean overrideExisting,
			final boolean bMonitor, final boolean bStandalone,
			final StringBuffer bufError) {
		return this.launchTest(suite, theTest, deployment, port,
				executionResultLocation, executionResultName, overrideExisting,
				bMonitor, bStandalone, bufError, this
						.createNullProgressMonitor());
	}

	/**
	 * <p>Launches the specified test on the specified machine.</p>
	 * 
	 * <p>The flow of the <code>launchTest()</code> method is depicted in the 
	 * following execution environment use case:</p>
	 * 
	 * <p>
	 * <a href="http://www.eclipse.org/tptp/test/documents/design/usecases.html#2">
	 * Launch a single test instance from workbench on a single specified node and 
	 * show control events to the running test</a></p>
	 * 
	 * @param suite
	 *            the test suite to be launched
	 * @param theTest
	 *            the test to be launched (may be the same as suite, or may be a
	 *            test case contained within suite.
	 * @param machineName
	 *            the machine name on which the test will be launched (IP
	 *            address or host name)
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param executionResultLocation
	 *            the directory in which to write the execution history file
	 * @param executionResultName
	 *            the filename for the execution history file
	 * @param overrideExisting
	 *            whether override existing if there is such an execution result
	 *            already exist
	 * @param bMonitor
	 *            should we monitor the test
	 * @param bStandalone
	 *            are we not launching from the workbench
	 * @param progressMonitor
	 *            progress monitor to use
	 * @param bufError
	 *            the accumulation of errors if encountered during launch
	 * @return an executor associated with the test execution, to control the
	 *         test -- at the time the executor returns, the test execution is
	 *         in most situations still running the test since the launch test
	 *         method does not block on the test execution
	 */
	public synchronized IExecutor launchTest(final ITestSuite suite,
			final ITest theTest, final TPFDeployment deployment,
			final String port, final String executionResultLocation,
			final String executionResultName, final boolean overrideExisting,
			final boolean bMonitor, final boolean bStandalone,
			final StringBuffer bufError, final IProgressMonitor progressMonitor) {
		ArrayList dpIDs = null;
		if (bMonitor) {
			dpIDs = new ArrayList();
			dpIDs.add(XMLExecutionDataProcessor.IID);
		}

		try {
			IExecutor executor = null;
			if (!progressMonitor.isCanceled()) {
				final IImplementor theImplementor = this.getImplementor(suite,
						theTest);
				final HashMap optionsMap = new HashMap();
				
				if (!progressMonitor.isCanceled()) {
					executor = this.launchTest(theImplementor, null, null,
							deployment, port, executionResultLocation,
							executionResultName, overrideExisting, 
							bStandalone, dpIDs, optionsMap, 
							bufError, progressMonitor,
							TestExecutionHarness.ProgressVisibility.ENABLED);
				}
			}
			return executor;
		} finally {
			progressMonitor.done();
		}

	}

	/**
	 * <p>Launches the specified test on the specified machine.</p>
	 * 
	 * <p>The flow of the <code>launchTest()</code> method is depicted in the 
	 * following execution environment use case:</p>
	 * 
	 * <p>
	 * <a href="http://www.eclipse.org/tptp/test/documents/design/usecases.html#2">
	 * Launch a single test instance from workbench on a single specified node and 
	 * show control events to the running test</a></p>
	 * 
	 * @param rootSuitePath
	 *            the fully qualified path to the test suite to be launched
	 * @param deploymentPath
	 *            fully qualified path to the deployment to be used with
	 *            execution.
	 * @param testID
	 *            the ID of the test to be launched (may be null, or may be the
	 *            ID of a test case contained within suite.
	 * @param machineName
	 *            the machine name on which the test will be launched (IP
	 *            address or host name)
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param executionResultLocation
	 *            the directory in which to write the execution history file
	 * @param executionResultName
	 *            the filename for the execution history file
	 * @param bMonitor
	 *            should we monitor the test
	 * @param bStandalone
	 *            are we not launching from the workbench
	 * @return a string containing an error message, or null if no error
	 *         occurred
	 */
	public IExecutor launchTest(final String rootSuitePath,
			final String deploymentPath, final String testID,
			final String machineName, final String port,
			final String executionResultLocation,
			final String executionResultName,
			final boolean overwriteExistingResult, final boolean bMonitor,
			final boolean bStandalone, final StringBuffer bufError) {
		return this.launchTest(rootSuitePath, deploymentPath, testID,
				machineName, port, executionResultLocation,
				executionResultName, overwriteExistingResult, bMonitor,
				bStandalone, bufError, this.createNullProgressMonitor());
	}

	/**
	 * <p>Launches the specified test on the specified machine.</p>
	 * 
	 * <p>The flow of the <code>launchTest()</code> method is depicted in the 
	 * following execution environment use case:</p>
	 * 
	 * <p>
	 * <a href="http://www.eclipse.org/tptp/test/documents/design/usecases.html#2">
	 * Launch a single test instance from workbench on a single specified node and 
	 * show control events to the running test</a></p>
	 * 
	 * @param rootSuitePath
	 *            the fully qualified path to the test suite to be launched
	 * @param deploymentPath
	 *            fully qualified path to the deployment to be used with
	 *            execution.
	 * @param testID
	 *            the ID of the test to be launched (may be null, or may be the
	 *            ID of a test case contained within suite.
	 * @param machineName
	 *            the machine name on which the test will be launched (IP
	 *            address or host name)
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param executionResultLocation
	 *            the directory in which to write the execution history file
	 * @param executionResultName
	 *            the filename for the execution history file
	 * @param bMonitor
	 *            should we monitor the test
	 * @param bStandalone
	 *            are we not launching from the workbench
	 * @param progressMonitor
	 *            progress monitor to use
	 * @return a string containing an error message, or null if no error
	 *         occurred
	 */
	public IExecutor launchTest(final String rootSuitePath,
			final String deploymentPath, final String testID,
			final String machineName, final String port,
			final String executionResultLocation,
			final String executionResultName,
			final boolean overwriteExistingResult, final boolean bMonitor,
			final boolean bStandalone, final StringBuffer bufError,
			final IProgressMonitor progressMonitor) {

		ITest theTest = null;
		ITestSuite suite = null;
		TPFDeployment deployment = null;
		bufError.setLength(0);

		// Load the test suite.
		try {
			suite = HyadesFactory.INSTANCE.loadTestSuite(rootSuitePath);
		} catch (final Exception exc) {
			this.systemUtility.logError(exc);
			bufError.append(NLS.bind(ExecutionHarnessPluginResourceBundle.LOAD_SUITE_ERR_,rootSuitePath));
			return null;
		}

		// Load the deployment if it is specified.
		try {
			if ((deploymentPath != null) && (deploymentPath.length() > 0)) {
				deployment = HyadesFactory.INSTANCE
						.loadDeployment(deploymentPath);
			}
		} catch (final Exception exc) {
		}
		// Create a Deployment with the test suite and machine name
		if ((deployment == null) && (suite instanceof CFGClass)) {
			deployment = ConfigurationUtil.createDeploymentWithPair(
					machineName, null, (CFGClass) suite, machineName);
		}

		// If we're executing a test case and not a suite, get the test case.
		if ((testID != null) && (testID.length() != 0)) {
			theTest = (ITest) HyadesFactory.INSTANCE.getObjectByID(suite,
					testID);
		} else {
			theTest = suite;
		}

		return this.launchTest(suite, theTest, deployment, port,
				executionResultLocation, executionResultName,
				overwriteExistingResult, bMonitor, bStandalone, bufError,
				progressMonitor);
	}

	/**
	 * 
	 * @param theImplementor
	 * @param overrideTestType
	 * @param machineName
	 * @param deployment
	 * @param port
	 * @param executionResultLocation
	 * @param executionResultName
	 * @param overrideExistingResult
	 * @param bStandalone
	 * @param activeDataProcessorIDs
	 * @param bufError
	 * @param progressMonitor
	 * @return
	 */
	private IExecutor launchTestExecution(
			final IImplementor theImplementor, final String overrideTestType,
			final String machineName, TPFDeployment deployment,
			final String port, final String executionResultLocation,
			final String executionResultName,
			final boolean overrideExistingResult, final boolean bStandalone,
			final ArrayList activeDataProcessorIDs, final HashMap optionsMap,
			final StringBuffer bufError, final IProgressMonitor progressMonitor) {

		// Initialize variables
		IExecutor executor = null;
		ISession session = null;
		IExecutionEnvironment exeEnvironment = null;

		// Begin task with work units allocated
		progressMonitor.beginTask("", 100); //$NON-NLS-1$

		// Reinitialize all fields
		this.factory = null;

		// Empty the error buffer
		bufError.setLength(0);

		// Determine what type of implementor we're dealing with
		CFGClass rootResource = null;

		final ITest owningTest = theImplementor.getOwner();

		if (owningTest != null) {

			if (owningTest instanceof ITestSuite) {
				rootResource = (CFGClass) owningTest;
			} else {
				rootResource = (CFGClass) ((ITestCase) owningTest).getOwner();
			}

		} else {
			// If the implementor is not owned by a test, then it must
			// be owned by a test component
			final TPFTestComponent component = ((TPFBehavior) theImplementor)
					.getTestComponent();

			if (component.eContainer() == null) {
				// This is a standalone test component.
				rootResource = component;
			} else {
				// This test component is contained by a test suite
				rootResource = (CFGClass) component.eContainer();
			}
		}

		try {

			String machine = machineName;

			// Allow the test type to be overridden if the caller has specified.
			String testType;
			if ((overrideTestType != null) && (overrideTestType.length() > 0)) {
				testType = overrideTestType;
			} else {
				testType = owningTest.getType();
			}

			// Synchronize access to configuration utilities
			synchronized (TestExecutionHarness.class) {

				// determine deployment and hostname
				if (deployment == null) {
					deployment = ConfigurationUtil.createDeploymentWithPair(
							null, null, rootResource, machineName);
				} else {
					machine = ConfigurationUtil.getHostNameForTestAsset(
							rootResource, deployment);
					if (machine == null) {
						machine = machineName;
					}
				}

			}

			if (machine == null) {
				bufError.append(ExecutionHarnessPluginResourceBundle.NULL_HOST_NAME_ERR);
				return null;
			}

			// Instantiate any registered listeners
			final List listeners = this.createListeners(testType);

			TestExecutionHarnessListenerNotifier.notifyLaunchStarted(
					theImplementor, deployment, port, executionResultLocation,
					executionResultName, listeners);

			/*
			 * Get a session for this Node; allocate work units to session
			 * connect sub progress monitor
			 */
			session = this.sessionConnect(deployment, port, session, machine,
					bufError, new SubProgressMonitor(progressMonitor, 25));

			TestExecutionHarnessListenerNotifier.notifySessionCreated(session,
					listeners);

			// poll for cancel status
			if (progressMonitor.isCanceled()) {
				return executor;
			}

			if (session == null) {
				return null;
			}

			if (bStandalone) {
				this.factory = ExecutionComponentFactoryImpl
						.getInstance(session);
			} else {
				this.factory = EclipseExecutionComponentFactoryImpl
						.getInstance(session);
			}

			// poll for cancel status and track progress
			if (progressMonitor.isCanceled()) {
				return executor;
			}
			progressMonitor.worked(10);

			if (!this.bFactoryInitialized) {
				// initialize the dataPRocessor factory
				this.dpFactory = new ExecutionHarnessDataProcessorFactory(
						bStandalone);

				this.initializeRegisteredExecutionComponents(this.factory,
						bStandalone, this.dpFactory);
				this.bFactoryInitialized = true;
			}

			// poll for cancel status and track progress
			if (progressMonitor.isCanceled()) {
				return executor;
			}
			progressMonitor.worked(10);

			// Create the appropriate type of execution environment
			final String envType = this.getExecutionComponentForTestType(
					"ENVIRONMENT", testType, bStandalone); //$NON-NLS-1$
			exeEnvironment = (IExecutionEnvironment) this.factory
					.createExecutionComponentByType(envType);

			session.addChild(exeEnvironment);

			// bugzilla 61137 enhancement, dynamically determine the root
			// directory for file deployment
			// using the temp directory. the IExecutionEnvironment is passed
			// into the JavaExecutionDeploymentAdapter
			// here so to supply the temp directory environment variable
			// property for this purpose.
			if (!ExecutionAdapterUtilities.adaptExecutionDeployment(session
					.getNode(), deployment, bStandalone, testType, bufError,
					this.systemUtility, this.testTypeMap, exeEnvironment)) {
				return null;
			}

			TestExecutionHarnessListenerNotifier
					.notifyTestArtifactsDeployed(listeners);

			// Call out into an adapter class to configure this execution
			// environment for this test
			if (!ExecutionAdapterUtilities.adaptExecutionEnvironment(
					exeEnvironment, rootResource, theImplementor, bStandalone,
					testType, deployment, bufError, this.systemUtility,
					this.testTypeMap)) {
				return null;
			}

			TestExecutionHarnessListenerNotifier
					.notifyExecutionEnvironmentAdapted(exeEnvironment,
							listeners);

			// Create the appropriate type of executor
			final String executorType = this.getExecutionComponentForTestType(
					"EXECUTOR", testType, bStandalone);
			executor = (IExecutor) this.factory
					.createExecutionComponentByType(executorType);
			exeEnvironment.addChild(executor);

			final String execObjType = this.getExecutionComponentForTestType(""
					+ "EXECUTABLEOBJECT", testType, bStandalone); //$NON-NLS-1$
			final IExecutableObject executableObject = executor
					.getCompatibleExecutableObject(execObjType);

			// Call out into an adapter class to configure this execution
			// environment for this test
			if (!ExecutionAdapterUtilities.adaptExecutableObject(
					executableObject, rootResource, theImplementor,
					bStandalone, testType, deployment, bufError,
					this.systemUtility, this.testTypeMap)) {
				return null;
			}

			TestExecutionHarnessListenerNotifier.notifyExecutableObjectAdapted(
					executableObject, listeners);

			if (executableObject != null) {
				// Set our executable object -- this will move it to the remote
				// side
				executor.setExecutableObject(executableObject);

				if (executor instanceof ExecutorStub) {
					((ExecutorStub) executor)
							.setExecutionResultLocation(executionResultLocation);
					((ExecutorStub) executor)
							.setExecutionResultName(executionResultName);
					((ExecutorStub) executor).setCommunicationPort(port);
				}
			}

			// get the agentType that we need to create
			final String agentType = this.getExecutionComponentForTestType(
					"AGENT", testType, bStandalone); //$NON-NLS-1$

			// poll for cancel status and track progress
			if (progressMonitor.isCanceled()) {
				return executor;
			}
			progressMonitor.worked(10);

			IExecutionHarnessDataProcessor execHistProcessor = null;

			if ((activeDataProcessorIDs != null)
					&& (activeDataProcessorIDs.size() > 0)
					&& (executor instanceof ExecutorStub)) {
				final ArrayList activeDataProcessors = this
						.initValidDataProcessors(owningTest, machine,
								agentType, executionResultName,
								executionResultLocation,
								overrideExistingResult, port,
								activeDataProcessorIDs, executor, deployment,
								optionsMap);

				TestExecutionHarnessListenerNotifier
						.notifyDataProcessorsInitialized(activeDataProcessors,
								listeners);

				if (!activeDataProcessors.isEmpty()) {
					execHistProcessor = (IExecutionHarnessDataProcessor) activeDataProcessors
							.get(0);
				}

				// now we need to set the agent data which will be intercepted
				// by the exec history dp
				// this data will enable it to bootstrap the others.
				final IExecutionHarnessDataProcessor[] dpArray = this
						.convertListToArray(activeDataProcessors);
				this.setAgentData(rootResource, executableObject, dpArray);
				((ExecutorStub) executor).setDataProcessors(dpArray);
			} else {
				this.initializeRemoteHyadesComponent(agentType, null, executor);
			}

			// poll for cancel status and track progress
			if (progressMonitor.isCanceled()) {
				return executor;
			}
			progressMonitor.worked(10);

			// if this executor supports progress monitor support, pass some
			// work units in to the launch
			if (executor instanceof IExecutorWithProgressMonitorSupport) {
				((IExecutorWithProgressMonitorSupport) executor)
						.launch(new SubProgressMonitor(progressMonitor, 25));
			} else {
				executor.launch();
			}

			TestExecutionHarnessListenerNotifier.notifyLaunchCompleted(
					execHistProcessor, listeners);

			// poll for cancel status and track progress
			if (progressMonitor.isCanceled()) {
				return executor;
			}
			progressMonitor.worked(10);

		} catch (final ClassNotFoundException e) {

			this.systemUtility.logError(e);
			String msg = e.getMessage();
			if (msg == null) {
				msg = e.toString();
			}

			bufError.append(NLS.bind(ExecutionHarnessPluginResourceBundle.CLASS_NOT_FOUND_ERR_, msg));
		} catch (final Throwable t) {
			this.systemUtility.logError(t);
			final String msg = t.getMessage();
			if (msg != null) {
				bufError.append(msg);
			} else {
				bufError.append(t.toString());
			}
		} finally {
			progressMonitor.done();
			if (session != null) {
				session.release();
			}
			this.factory = null;
		}

		return executor;
	}

	/**
	 * @param testType
	 * @return
	 */
	private List createListeners(final String testType) {
		final IExtensionPoint extp = Platform
				.getExtensionRegistry()
				.getExtensionPoint(
						"org.eclipse.hyades.test.core.executionHarnessListener"); //$NON-NLS-1$
		final List listeners = new ArrayList();
		if (extp != null) {
			final IExtension[] extensions = extp.getExtensions();
			// TODO Filter on test type

			for (int i = 0; i < extensions.length; i++) {
				final IExtension extension = extensions[i];
				// Retrieve the extension point configuration elements
				final IConfigurationElement[] elements = extension
						.getConfigurationElements();

				// Find the executable extensions for listener
				try {
					for (int j = 0; j < elements.length; j++) {
						final IConfigurationElement element = elements[j];
						if (element.getName().equalsIgnoreCase(
								"executionHarnessListener")) { //$NON-NLS-1$
							final String extensionType = element
									.getAttribute("supportedTestType");
							if ((extensionType == null)
									|| extensionType.equalsIgnoreCase(testType)) {
								final Object executableExtension = element
										.createExecutableExtension("class"); //$NON-NLS-1$
								if (executableExtension instanceof ITestExecutionHarnessListener) {
									listeners.add(executableExtension);
									break;
								}
							}
						}
					}
				} catch (final CoreException e) {
					// Continue with next extension
				}
			}
		}

		return listeners;
	}

	private void logStopEvents(final TPFExecutionResult result) {
		final String msg = ExecutionHarnessPluginResourceBundle.TEST_ABORTED_MESSAGE_;
		// verdict
		final TPFVerdictEvent verdict = Common_TestprofileFactory.eINSTANCE
				.createTPFVerdictEvent();
		verdict.setVerdict(TPFVerdict.INCONCLUSIVE_LITERAL);
		verdict.setReason(TPFVerdictReason.ABORT_LITERAL);
		verdict.setText(msg);
		verdict.setExecutionHistory(result.getExecutionHistory());

		// stop
		final TPFTypedEvent stopEvent = Common_TestprofileFactory.eINSTANCE
				.createTPFTypedEvent();
		stopEvent.setType(TPFExecutionType.STOP_LITERAL);
		stopEvent.setText(msg);
		stopEvent.setTimestamp(System.currentTimeMillis());
		stopEvent.setExecutionHistory(result.getExecutionHistory());

		// promote verdict to result
		result.setVerdict(TPFVerdict.INCONCLUSIVE_LITERAL);

	}

	private void mapAdapterClasses(final String configElements,
			final String type, boolean bStandalone) {

		if (!bStandalone) {
			final IConfigurationElement[] execEnvAdapters = Platform
					.getExtensionRegistry().getConfigurationElementsFor(
							configElements);

			for (int i = 0; i < execEnvAdapters.length; i++) {
				Object map = this.testTypeMap.get(type);
				if (map == null) {
					map = new HashMap();
					this.testTypeMap.put(type, map);
				}
				final HashMap execCompMap = (HashMap) map;

				final IConfigurationElement[] supportedTestTypes = execEnvAdapters[i]
						.getChildren("SupportedTestType"); //$NON-NLS-1$
				for (int j = 0; j < supportedTestTypes.length; j++) {
					execCompMap
							.put(
									supportedTestTypes[j].getAttribute("name"), execEnvAdapters[i]); //$NON-NLS-1$
				}
			}
		}
	}

	/**
	 * Perfromed the control event on the specified test on the specified
	 * machine.
	 * 
	 * @param suite
	 *            the test suite to be controlled
	 * @param theTest
	 *            the test to be suspended (may be the same as suite, or may be
	 *            a test case contained within suite.
	 * @param location
	 *            on which the test will be controlled (CGFLocation or
	 *            CFGMachine)
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param execResult
	 *            execution Result
	 * @param controlEvent
	 *            the control event
	 * @return a string containing an error message, or null if no error
	 *         occurred
	 */
	public synchronized String performControlEvent(final ITestSuite suite,
			final ITest theTest, final CFGLocation location, final String port,
			final TPFExecutionResult execResult, final String controlEvent,
			final String[] params) {

		final String machine = ((CFGMachineConstraint) location).getHostname();
		final StringBuffer bufError = new StringBuffer();

		if (this.canSupportControlEvent(suite, theTest, location, port,
				execResult, controlEvent) == false) {
			return NLS.bind(ExecutionHarnessPluginResourceBundle.CANNOT_SUPPORT_MSG_, controlEvent); 
		}

		// Get a session for this Node. This is returning an session that was
		// created during launch test method
		final ISession session = this.getSession(port, null, machine, bufError);

		if (session == null) {
			return NLS.bind(ExecutionHarnessPluginResourceBundle.NO_SESSION_FOUND_ERROR, machine); 		
		}

		String error = null;
		try {
			boolean controlEventPerformed = false;

			final IExecutionComponent[] exeEnvironments = session.getChildren();
			// Navigate to the IExecutor and invoke kill method to stop the
			// running test.
			for (int i = 0; i < exeEnvironments.length; i++) {
				if (exeEnvironments[i] instanceof IExecutionEnvironment) {
					final IExecutionComponent[] executors = exeEnvironments[i]
							.getChildren();
					for (int j = 0; j < executors.length; j++) {
						if (executors[j] instanceof ExecutorStub) {
							final ExecutableObjectStub exeobj = (ExecutableObjectStub) ((IExecutor) executors[j])
									.getExecutableObject();
							if (exeobj.getTestID().equals(theTest.getId())
									&& exeobj.getSuiteID()
											.equals(suite.getId())
									&& ((ExecutorStub) executors[j])
											.getExecutionResultName().equals(
													execResult.getName())) {

								error = ((IExecutor) executors[j])
										.performControlEvent(
												IExecutor.CONTROL_EVENT_SUSPEND,
												params);
								if (error == null) {
									controlEventPerformed = true;
								}
							}
						}
					}
				}
			}
			if (!controlEventPerformed) {
				if (error == null) {
					return NLS.bind(ExecutionHarnessPluginResourceBundle.CANNOT_PERFORM_MSG_, controlEvent); 
				}
				return error;

			}

		} catch (final Throwable t) {
			this.systemUtility.logError(t);
			return t.getMessage();
		}
		return null;
	}

	protected void releaseAgentLock() {
		synchronized (this.agentLock) {
			this.bAgentReady = true;
			this.agentLock.notify();
		}
	}

	private void removeChildComponents(final ArrayList children,
			final IExecutionComponent parent) {
		for (int i = 0; i < children.size(); i++) {
			parent.removeChild((IExecutionComponent) children.get(i));
		}
	}

	/**
	 * Given a port, session, machine, error buffer and progress monitor,
	 * connect to the session.
	 * 
	 * @param port
	 *            the port to connect to
	 * @param machine
	 *            the machine to connect to
	 * @param bufError
	 *            a buffer for errors to accumulate
	 * @param progressMonitor
	 *            monitors progress and is polled to check cancel status
	 * @return the
	 */
	private ISession sessionConnect(final TPFDeployment deployment,
			final String port, final ISession session, final String machine,
			final StringBuffer bufError, final IProgressMonitor progressMonitor) {

		// Optional connection specifier can be used
		final MutableObject mutableSpecifier = new MutableObject();

		// If deployment specifies connection specifier, use it
		CFGLocation location = ConfigurationUtil.getDefaultLocation(deployment);
		if (location == null) {
			final CFGArtifactLocationPair artifactLocationPair = (CFGArtifactLocationPair) deployment
					.getArtifactLocations().get(0);
			if (artifactLocationPair != null) {
				location = artifactLocationPair.getLocation();
			}
		}

		// Find optional attached attribute for connection specifier
		if (location != null) {

			// Retrieve primary location attribute group
			CFGPropertyGroup group = ConfigurationUtil.searchPropertyGroupById(
					location.getPropertyGroups(),
					ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);

			// If non-existent create the property group
			if (group == null) {
				group = Common_ConfigurationFactory.eINSTANCE
						.createCFGPropertyGroup();
				group
						.setPropertyGroupID(ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);
				location.getPropertyGroups().add(group);
			}

			// Obtain the property name for connection specifiers
			final String propertyName = "org.eclipse.hyades.execution.core.util.ConnectionSpecifier";

			// Find or create location
			final BVRProperty[] properties = ConfigurationUtil
					.searchPropertiesByName(group.getProperties(),
							propertyName, false);
			if ((properties != null) && (properties.length >= 1)) {
				final CFGComparableProperty property = (CFGComparableProperty) properties[0];
				mutableSpecifier.set(new ConnectionSpecifier(property
						.getValue()));
			}

		}

		// If specifier not given, compute it based on method arguments
		if (mutableSpecifier.isNull()) {
			mutableSpecifier.set(new ConnectionSpecifier(machine, Integer
					.parseInt(port)));
		}

		// Begin the task of session connect
		progressMonitor.beginTask("", 20); //$NON-NLS-1$

		// Initialize error buffer
		bufError.setLength(0);

		// Use a mutable object to wrap session so it can be set within connect
		// thread
		final MutableObject mutableSession = new MutableObject(session);

		try {

			// Use another thread to increase responsiveness of cancel when
			// connecting
			final Runnable connectRunnable = new Runnable() {
				public void run() {
					try {
						ConnectionSpecifier specifier = (ConnectionSpecifier) mutableSpecifier
								.get();
						INodeExtended node = new NodeImpl(specifier.getHost());
						progressMonitor.worked(5);
						mutableSession.set(node.connect(specifier));
						progressMonitor.worked(5);
					} catch (UnknownHostException e) {
						bufError
								.append(NLS.bind(ExecutionHarnessPluginResourceBundle.TIMEOUT_NODE_ERROR_, machine)); 
					} catch (AgentControllerUnavailableException e) {
						bufError
								.append(NLS.bind(ExecutionHarnessPluginResourceBundle.AGENT_UNAV_ERROR_, machine)); 
					} catch (UnknownDaemonException e) {
						bufError
								.append(NLS.bind(ExecutionHarnessPluginResourceBundle.AGENT_UNAV_ERROR_, machine)); 
					} catch (Exception e) {
						TestExecutionHarness.this.systemUtility.logError(e);
						bufError
								.append(NLS.bind(ExecutionHarnessPluginResourceBundle.RUN_FAILED_ERR_, machine)); 
					}
				}
			};

			/*
			 * Create a new thread hooked up with the connect runnable and start
			 * the thread; creates as a daemon thread in case interrupt isn't
			 * effective and a cancel causes execution to continue without the
			 * connect thread ending. Basically exiting the VM is okay even if
			 * the connect thread is still alive.
			 */
			final Thread connectThread = new Thread(connectRunnable,
					"Hyades Session Connect"); //$NON-NLS-1$
			connectThread.setDaemon(true);
			connectThread.start();
			// add a timeout for connecting to agent
			int timeoutLoopCount = 0;
			

			// Primary thread blocks at intervals and then polls the cancel
			// status of the progress monitor
			try {
				for (boolean isCanceled = false; connectThread.isAlive()
						&& !isCanceled;) {
					progressMonitor.worked(1);
					if (progressMonitor.isCanceled()) {
						isCanceled = true;
						connectThread.interrupt();
					} else {
						connectThread.join(1000);
					}
					// add a timeout for connecting to agent
					// no matter what the error is that has the session connect 
					// in a hang, this should break us out
					// this timeout is reached when the session jvm is created,
					// but never reaches the ready state
					
					timeoutLoopCount++;
					if (timeoutLoopCount >= NODE_CONNECT_TIMEOUT_SECONDS){
						bufError.append(NLS.bind(ExecutionHarnessPluginResourceBundle.TIMEOUT_NODE_ERROR_, machine));
						// interrupt threads that have timed out so they don't keep running 
						connectThread.interrupt();
						break;
						
					}
				}

			} catch (final InterruptedException e) {
			}

		} finally {
			progressMonitor.done();
		}

		// Return session then clear to prevent memory leaks
		return (ISession) mutableSession.getAndClear();

	}

	private void setAgentData(final CFGClass rootResource,
			final IExecutableObject executableObject,
			final IDataProcessor[] dataProcessors) {
		if ((executableObject instanceof JavaProcessExecutableObjectStub)
				&& (dataProcessors != null) && (dataProcessors.length > 1)) {
			// since the xmlexecutionhistorydataprocessor is initialized from
			// main(), it does not need to be passed in the
			// agent data String. Therefore, will skip it in the collection as
			// we build the list
			// it is always first in the dataProcessors collection regardless of
			// when it was sent in the spec ArrayList.
			String dpString = "AgentIDs:"; //$NON-NLS-1$
			for (int i = 1; i < dataProcessors.length; i++) {
				dpString += ((IExecutionHarnessDataProcessor) dataProcessors[i])
						.getID();
				if (i < dataProcessors.length - 1) {
					dpString += ","; //$NON-NLS-1$
				}

			}
			((JavaProcessExecutableObjectStub) executableObject)
					.setAgentData(dpString);

		}

	}

	/**
	 * Sets the standaloneConfigurationFiles member to an array of strings
	 * containing all registered execution components and adapters.
	 * 
	 * @param standaloneConfigurationFiles
	 */

	private void setStandaloneConfigurationFiles(
			final String[] standaloneConfigurationFiles) {
		this.standaloneConfigurationFiles = standaloneConfigurationFiles;
	}

	private void stopMonitoringAndWait(final IRemoteHyadesComponent agent)
			throws InterruptedException {
		// State does not change when an agent goes from Monitoring to
		// Not monitoring state. This bug need to get fixed.
		// Until then just sleeping the thread for 3 secs.
		/*
		 * ((IRemoteHyadesComponent)
		 * agent).addExecutionComponentStateChangeListener(new
		 * IExecutionComponentStateChangeListener() { public void
		 * stateChanged(ExecutionComponentStateChangeEvent event) { // When the
		 * agent is dead, release the agent lock
		 * if(event.getState()==IExecutionComponent.DEAD) { releaseAgentLock(); } }
		 * });
		 */
		agent.stopMonitoring();
		Thread.sleep(3000);
		// waitForInactiveAgent((IRemoteHyadesComponent) agent);
	}

	private boolean stopTest(final IExecutor executor) {
		final IExecutionComponent[] agents = executor.getChildren();
		// Stop monitoring call to the agent is necessary for clean-up including
		// saving of the
		// execution results to occur.
		final ArrayList componentsToRemove = new ArrayList();
		for (int i = 0; i < agents.length; i++) {
			if (agents[i] instanceof IRemoteHyadesComponent) {
				try {
					this
							.stopMonitoringAndWait((IRemoteHyadesComponent) agents[i]);
					componentsToRemove.add(agents[i]);
				} catch (final InterruptedException e) {
					this.systemUtility.logError(e);
				}
			}
		}
		this.removeChildComponents(componentsToRemove, executor);
		componentsToRemove.clear();
		executor.kill();
		return true;
	}

	/**
	 * <p>Stops the specified test on the specified machine.</p> 
	 * 
	 * @param executor
	 *            the executor being used to execute the test.
	 * @param execResult
	 *            the execution result
	 * @return a string containing an error message, or null if no error
	 *         occurred
	 */
	public synchronized String stopTest(final IExecutor executor,
			final TPFExecutionResult execResult) {
		try {
			if (!this.stopTest(executor)) {
				return ExecutionHarnessPluginResourceBundle.TEST_STOP_ERROR_;
			}
			this.logStopEvents(execResult);
			final Resource resource = execResult.eResource();
			if (resource != null) {
				resource.save(Collections.EMPTY_MAP);
			}
		} catch (final Throwable t) {
			this.systemUtility.logError(t);
			return t.getMessage();
		}

		return null;
	}

	private void waitForActiveAgent(final IRemoteHyadesComponent agent) {
		// TODO: JPT -- There are other states that the agent may be in. For
		// instance, in the Java case, if the test JVM fails to launch, the
		// returned state will be NOT_CONFIGURED. We should do something
		// to handle that case, either here or earlier.
		if (agent.getState() == IExecutionComponent.SUSPENDED) {
			// Block and wait for the agent to come up.
			while (!this.bAgentReady) {
				synchronized (this.agentLock) {
					try {
						this.agentLock.wait();
					} catch (final InterruptedException e) {
						/* We can ignore this */
					}
				}
			}
		}
	}

	/**
	 * Provides the ability to specify whether progress visibility is desired or
	 * not when invoking the various launch test methods in this class.
	 * 
	 * @author Scott E. Schneider
	 */
	public static final class ProgressVisibility {
		/**
		 * Progress is visible in the user interface
		 */
		public static ProgressVisibility ENABLED = new ProgressVisibility();

		/**
		 * Progress is not visible in the user interface
		 */
		public static ProgressVisibility DISABLED = new ProgressVisibility();

		/**
		 * Restrict construction per type-safe enumeration idiom
		 */
		private ProgressVisibility() {
		}
	}


	/**
	 * <p>Launches the specified test on the specified machine.</p>
	 * 
	 * <p>The flow of the <code>launchTest()</code> method is depicted in the 
	 * following execution environment use case:</p>
	 * 
	 * <p>
	 * <a href="http://www.eclipse.org/tptp/test/documents/design/usecases.html#2">
	 * Launch a single test instance from workbench on a single specified node and 
	 * show control events to the running test</a></p>
	 * 
	 * @param suite
	 *            the test suite to be launched
	 * @param theTest
	 *            the test to be launched (may be the same as suite, or may be a
	 *            test case contained within suite.
	 * @param overrideTestType
	 *            if specified, the test type to be used by the execution
	 *            harness (regardless of the test type specified by the test)
	 * @param deployment
	 *            the deployment with which to launch the test
	 * @param port
	 *            the port on which the Agent Controller is running (usually
	 *            10002)
	 * @param executionResultLocation
	 *            the directory in which to write the execution history file
	 * @param executionResultName
	 *            the filename for the execution history file
	 * @param overrideExistingResult
	 *            whether to override existing result if such an execution
	 *            result already exists
	 * @param bStandalone
	 *            are we not launching from the workbench
	 * @param activeDataProcessorIDs
	 *            a String[] of the dataprocessors that should be activated for
	 *            this launch. This ID should match the ID provided in the
	 *            extension.
	 * @param bufError
	 *            the accumulation of errors if encountered during launch
	 * @param progressMonitor
	 *            progress monitor to use
	 * @return an executor associated with the test execution, to control the
	 *         test -- at the time the executor returns, the test execution is
	 *         in most situations still running the test since the launch test
	 *         method does not block on the test execution
	 */
	public IExecutor launchTest(final ITestSuite suite, final ITest theTest,
			final String overrideTestType, final TPFDeployment deployment,
			final String port, final String executionResultLocation,
			final String executionResultName,
			final boolean overrideExistingResult, final boolean bStandalone,
			final ArrayList activeDataProcessorIDs, final HashMap optionsMap,
			final StringBuffer bufError, final IProgressMonitor progressMonitor) {

		try {
			IExecutor executor = null;
			if (!progressMonitor.isCanceled()) {
				final IImplementor theImplementor = this.getImplementor(suite,
						theTest);
				if (!progressMonitor.isCanceled()) {
					executor = this.launchTest(theImplementor,
							overrideTestType, null, deployment, port,
							executionResultLocation, executionResultName,
							overrideExistingResult, bStandalone,
							activeDataProcessorIDs, optionsMap, bufError, progressMonitor,
							TestExecutionHarness.ProgressVisibility.ENABLED);
				}
			}
			return executor;
		} finally {
			progressMonitor.done();
		}

	}

}
