/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestExecutionHarnessExecutorStub.java,v 1.20 2008/04/22 17:36:33 jkubasta Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.execution.harness;

import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.ExecutionComponentStateException;
import org.eclipse.hyades.execution.core.IControlMessage;
import org.eclipse.hyades.execution.core.IDataProcessor;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.local.ExecutorStub;
import org.eclipse.hyades.execution.local.JavaProcessExecutableObjectStub;
import org.eclipse.hyades.execution.local.SessionStub;
import org.eclipse.hyades.execution.local.testservices.TestServiceAgentListener;
import org.eclipse.hyades.internal.execution.local.common.BinaryCustomCommand;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentListener;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NotConnectedException;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.jdt.launching.IVMConnector;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.jdt.launching.SocketUtil;

/**
 * The test execution harness executor stub is used by the test execution
 * harness to launch tests.
 * 
 * @author Ernest Jessee
 * @author Scott E. Schneider
 * @author jcanches
 */
public class TestExecutionHarnessExecutorStub extends ExecutorStub implements IDataProcessorObservable.Observer {

	/**
	 * Agent type used for executing tests.
	 */
	private static final String AGENT_TYPE = "tester";

	/**
	 * Default timeout for the process launch
	 */
	private static final int DEFAULT_LAUNCH_TIMEOUT = 60000;

	private final static int MAX_ATTACH_ATTEMPTS = 5;

	private final static int TIME_BETWEEN_ATTACH_ATTEMPTS = 1000;
	
	private boolean fatalErrorEncountered = false;

	/**
	 * The number of data processors currently active for this executor, this
	 * number will be less than or equal to the steady state length of the data
	 * processors array, this was introduced so the execution component state
	 * listener DEAD event will only be fired when there are zero data
	 * processors active and the associated agents is inactive.
	 */
	private int activeDataProcessors = 0;

	/**
	 * Indicates if agent is active or not
	 */
	private boolean isAgentActive = false;

	/**
	 * Attach the debugger to the executing process. Since this method may be
	 * invoked before the process is ready to be attached by a debugger, its
	 * implementation makes several attempts, waiting one second between each
	 * attempt.
	 * 
	 * @param host
	 *            The host IP address.
	 * @param port
	 *            The debug port to attach to.
	 * @param monitor
	 *            A progress monitor.
	 */
	private void attachDebugger(String host, int port, IProgressMonitor monitor) {
		monitor.beginTask("", MAX_ATTACH_ATTEMPTS); //$NON-NLS-1$
		try {
			CoreException e = null;
			for (int i = 0; i < MAX_ATTACH_ATTEMPTS; i++) {
				e = attemptDebuggerAttachment(host, port, getLaunch(), new SubProgressMonitor(monitor, 1,
						SubProgressMonitor.PREPEND_MAIN_LABEL_TO_SUBTASK));
				if (e == null)
					break;
				else {
					synchronized (this) {
						try {
							this.wait(TIME_BETWEEN_ATTACH_ATTEMPTS);
						} catch (InterruptedException e1) {
							// NOP
						}
					}
				}
			}
			if (e != null) {
				ExecutionHarnessPlugin.getDefault().logError(e);
			}
		} finally {
			monitor.done();
		}
	}

	/**
	 * Attempts to attach the debugger to the specified port.
	 * 
	 * @param host
	 *            The host IP address.
	 * @param port
	 *            The debug port to attach to.
	 * @param launch
	 *            A launch object that will host the debug stack.
	 * @param monitor
	 *            A progress monitor.
	 * @return A CoreException if the attachments failed, or <code>null</code>
	 *         if the attachment succeeded.
	 */
	private CoreException attemptDebuggerAttachment(String host, int port, ILaunch launch, IProgressMonitor monitor) {
		try {
			IVMConnector vmConnector = JavaRuntime.getDefaultVMConnector();
			Map arguments = new HashMap(vmConnector.getDefaultArguments());
			arguments.put("hostname", host); //$NON-NLS-1$
			arguments.put("port", Integer.toString(port)); //$NON-NLS-1$
			arguments.put("timeout", "0"); //$NON-NLS-1$//$NON-NLS-2$
			vmConnector.connect(arguments, new SubProgressMonitor(monitor, 1), launch);
			return null;
		} catch (CoreException e) {
			return e;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.harness.IDataProcessorObservable.Observer#clean(org.eclipse.hyades.execution.core.IDataProcessor)
	 */
	public void clean(IDataProcessorObservable context) {

		// Decrement the data processors active
		this.activeDataProcessors--;

		// If no more data processors are active, fire event
		if (this.activeDataProcessors == 0 && !this.isAgentActive) {

			// Notify execution component state listeners
			TestExecutionHarnessExecutorStub.this.fireStateChangeEvent(new ExecutionComponentStateChangeEvent(
					TestExecutionHarnessExecutorStub.this, IExecutionComponent.DEAD));

		}

		// Remove observer on data processor
		context.removeObserver(this);

	}

	/**
	 * Creates a custom command which is sent to the skeleton to start it.
	 * 
	 * @return a resume custom command
	 */
	private CustomCommand createResumeCommand() {
		BinaryCustomCommand resumeCommand = new BinaryCustomCommand();
		resumeCommand.setData(IControlMessage.START);
		return resumeCommand;
	}

	/**
	 * Wraps the doLaunch() call in debugger-attachment sequence: first the
	 * executable object arguments are augmented with debug arguments, then the
	 * process is launched, then the eclipse debugger is attached to the
	 * launched process. Sub-classes may override if an alternate
	 * debugger-attach method is desired. Implementers must make sure that
	 * doLaunch() is invoked by this method.
	 * 
	 * @param monitor
	 */
	protected void debugAndLaunch(IProgressMonitor monitor) {
		
		monitor.beginTask("", 2); //$NON-NLS-1$
		
		try {
			
			//Add the debug arguments to VM arguments:
			int port = SocketUtil.findFreePort();
			JavaProcessExecutableObjectStub exObject = (JavaProcessExecutableObjectStub) getExecutableObject();
			exObject.setArgs("-Xdebug -Xrunjdwp:transport=dt_socket,server=y,suspend=y,address=" + port + " " + exObject.getArgs()); //$NON-NLS-1$ //$NON-NLS-2$

			//Perform the launch:
			doLaunch(new SubProgressMonitor(monitor, 1, SubProgressMonitor.PREPEND_MAIN_LABEL_TO_SUBTASK));
			
			//Attach the Eclipse debugger to the launched process:
			try {
				attachDebugger(this.getAgentProcess().getNode().getInetAddress().getHostAddress(), port, new SubProgressMonitor(monitor, 1,	SubProgressMonitor.PREPEND_MAIN_LABEL_TO_SUBTASK));
			} 
			catch (Throwable t) {
				ExecutionHarnessPlugin.getDefault().logError(t);
			}
		} 
		finally {
			monitor.done();
		}
	}

	/**
	 * Performs the actual launch. After this method is called, a process is
	 * expected to be running, and getPid() should return a valid PID. This
	 * implementation invokes super's launch(IProgressMonitor) method.
	 * Subclasses may override this implementation (for instance, to allow
	 * attachment to an existing process, or to allow a debugger to attach to
	 * the launched process).
	 * 
	 * @param monitor
	 *            the monitor to use
	 */
	protected void doLaunch(IProgressMonitor monitor) {
		super.launch(monitor);
	}

	/**
	 * Retrieves the agent listener to be used to listen to the agent
	 * controller's control channel.
	 * 
	 * @return the agent listener
	 */
	protected AgentListener getAgentListener() {
		return new AgentListener() {

			public void agentActive(Agent agent) {

				// Sets the agent active status
				TestExecutionHarnessExecutorStub.this.isAgentActive = true;

				// Fire execution component READY event to listeners
				TestExecutionHarnessExecutorStub.this.fireStateChangeEvent(new ExecutionComponentStateChangeEvent(
						TestExecutionHarnessExecutorStub.this, IExecutionComponent.READY));

			}

			public void agentInactive(Agent agent) {

				/**
				 * For the test to be over, the agent is inactive and the
				 * execution context is done cleaning it up, basically the data
				 * processors active has to be at zero
				 */
				TestExecutionHarnessExecutorStub.this.isAgentActive = false;

				/*
				 * If the agent is found inactive with zero data processors,
				 * fire event, will be the case in exception situations or if
				 * the data processor is not observable (does not implement the
				 * data processor observable interface)
				 */
				if (TestExecutionHarnessExecutorStub.this.activeDataProcessors == 0) {

					// Fire event to execution component state listeners
					TestExecutionHarnessExecutorStub.this.fireStateChangeEvent(new ExecutionComponentStateChangeEvent(
							TestExecutionHarnessExecutorStub.this, IExecutionComponent.DEAD));

				}

			}

			public void error(Agent agent, String errorId, String errorMessage) {
				System.err.println(this + " " + errorMessage);
			}

			public void handleCommand(Agent agent, CommandElement command) {
			}

		};

	}

	/**
	 * Returns the process associated with the session's agent.
	 * 
	 * @return the agent's process
	 */
	protected Process getAgentProcess() {
		return ((SessionStub) this.getSessionContext()).getAgent().getProcess();
	}

	private ILaunch getLaunch() {
		// Temporary implementation: find the first launch that has no child
		// process
		ILaunch[] launches = DebugPlugin.getDefault().getLaunchManager().getLaunches();
		for (int i = 0; i < launches.length; i++) {
			if (launches[i].getProcesses().length == 0) {
				return launches[i];
			}
		}
		return null;
	}

	/**
	 * Override this method to change the timeout value to wait for the process
	 * to launch before throwing an InactiveProcessException and returning from
	 * the test execution harness executor stub.
	 * 
	 * The default timeout is set to 60000 which is 60 seconds -- this method
	 * returns timeout value in milliseconds.
	 * 
	 * @return the number of milliseconds to wait before aborting the launch
	 *         process and throwing an InactiveProcessException
	 */
	protected int getLaunchTimeout() {
		return TestExecutionHarnessExecutorStub.DEFAULT_LAUNCH_TIMEOUT;
	}

	/**
	 * Ensures consistent handling of caught exception; exceptions are logged.
	 * 
	 * @param exception
	 *            exception to handle
	 */
	private void handleException(Exception exception) {
		ExecutionHarnessPlugin.getDefault().logError(exception);
		this.fireStateChangeEvent(new ExecutionComponentStateChangeEvent(TestExecutionHarnessExecutorStub.this,
				IExecutionComponent.DEAD));
		fatalErrorEncountered = true;
		
		throw new RuntimeException(exception);
	}

	/**
	 * Returns whether the process should be launched in debug mode and the
	 * Eclipse debugger attached to it. This implementation always returns
	 * false. Sub-classes should override this method if debugging is needed.
	 */
	protected boolean isDebugMode() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.IExecutor#launch()
	 */
	public void launch() throws ExecutionComponentStateException {
		this.launch(new NullProgressMonitor());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.IExecutorWithProgressMonitorSupport
	 *      #launch(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void launch(IProgressMonitor progressMonitor) throws ExecutionComponentStateException {

		// Launches and then suspends execution component
		if (isDebugMode()) {
			debugAndLaunch(progressMonitor);
		} else {
			doLaunch(progressMonitor);
		}

		// Define a timeout so launch will not hang indefinitely
		int timeout = this.getLaunchTimeout();

		// Begin the launch task, allocating work units
		// TODO JPT: we can reach this point with a null dataprocessors 
		// collection.  Need to fail on this either here or earlier.
		progressMonitor.beginTask("", (timeout * 10) + ((this.getDataProcessors().length) * 12)); //$NON-NLS-1$

		try {

			// Resets agent counter, it is important to count initialized agents
			this.resetAgentInitCount();

			// Retrieve the node from the agent's process
			Node node = this.getAgentProcess().getNode();

			// Retrieve the active data processors, set by the execution harness
			IDataProcessor[] dataProcessors = this.getDataProcessors();

			/*
			 * Retrieves the specified process by identity on the given node;
			 * the wait for process method blocks until the process has been
			 * found, the timeout has been reached or the cancel is detected.
			 */
			Process process = this.waitForProcess(this.getPid(), node, timeout, progressMonitor);

			// If the process found is null then exit from launch
			if (process != null) {

				// Launch behavior determined by existence of data processors
				if (dataProcessors != null) {
					this.launch(process, progressMonitor, dataProcessors);
				} else {
					this.launch(process);
				}

			} else {

				// Handle the case of process retrieved as null
				throw new InactiveProcessException();

			}

		} catch (InactiveAgentException e) {

			// Thrown from launch
			this.handleException(e);

		} catch (InactiveProcessException e) {

			// Thrown from find process
			this.handleException(e);

		} catch (NotConnectedException e) {

			// Thrown from find process
			this.handleException(e);

		} catch (UnsupportedEncodingException e) {

			// Thrown from launch
			this.handleException(e);

		} finally {

			// Before this method returns, we must mark the task as completed
			progressMonitor.done();

		}

	}

	/**
	 * Launch without data processors
	 * 
	 */
	private void launch(Process process) throws InactiveAgentException, InactiveProcessException,
			UnsupportedEncodingException {

		Agent agent = findAgent(process, TestExecutionHarnessExecutorStub.AGENT_TYPE, XMLExecutionDataProcessor.IID);

		this.setupControlListener(this.getAgentListener(), agent);
		this.resume(agent);

	}

	/**
	 * Launch with data processors
	 */
	private void launch(Process process, IProgressMonitor progressMonitor, IDataProcessor[] dataProcessors)
			throws InactiveAgentException, InactiveProcessException, UnsupportedEncodingException {

		// Lock on process
		synchronized (process) {

			for (int i = 0, n = dataProcessors.length; i < n; i++) {

				if (progressMonitor.isCanceled()) {
					return;
				}
				progressMonitor.worked(4);

				IExecutionHarnessDataProcessor dataProcessor = (IExecutionHarnessDataProcessor) dataProcessors[i];

				// If the control agent is not null, look for the agent and
				// configure
				if (dataProcessor.getControlAgent() == null) {

					try {

						// Find the appropriate agent
						Agent agent = this.waitForAgent(TestExecutionHarnessExecutorStub.AGENT_TYPE, dataProcessor
								.getName(), process, this.getLaunchTimeout(), progressMonitor);

						if (progressMonitor.isCanceled()) {
							return;
						}
						progressMonitor.worked(4);

						// If the agent is found, configure data processor and
						// control listener
						if (agent != null) {

							this.setupDataProcessor(agent, process, dataProcessor);

							if (progressMonitor.isCanceled()) {
								return;
							}
							progressMonitor.worked(4);

							agent.addAgentListener(new TestServiceAgentListener());
							this.setupControlListener(this.getAgentListener(), agent);

						}

					} catch (NotConnectedException e) {

						// Thrown from find agent
						this.handleException(e);

					}

				}
			}

			// Resume the control agent
			IExecutionHarnessDataProcessor controller = (IExecutionHarnessDataProcessor) dataProcessors[0];
			if (controller != null) {
				this.resume(controller.getControlAgent());
			}

		}

	}

	private void observe(IDataProcessorObservable observable) {
		observable.addObserver(this);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.IExecutor
	 *      #performControlEvent(java.lang.String, java.lang.String[])
	 */
	public String performControlEvent(String controlEvent, String[] params) {
		return "";
	}

	/**
	 * Resumes execution component with the specified agent, if the agent is
	 * null then this method does nothing.
	 * 
	 * @param agent
	 *            the agent to send command to
	 */
	protected void resume(Agent agent) throws InactiveAgentException {
		if (agent != null) {
			agent.invokeCustomCommand(this.createResumeCommand());
		}
	}

	/**
	 * Associates specified data processor with the given agent and process and
	 * then intitializes the data processor.
	 * 
	 * @param agent
	 *            the agent to use
	 * @param process
	 *            the process to associate with this data processor
	 * @param dataProcessor
	 *            the data processor to associate with the agent and process
	 */
	protected void setupDataProcessor(Agent agent, Process process, IExecutionHarnessDataProcessor dataProcessor) {

		// Set up data processor
		dataProcessor.setControlAgent(agent);
		dataProcessor.setProcess(process);

		/*
		 * Set up to be observed by this executor, must be done before the data
		 * processor is inititalized
		 */
		if (dataProcessor instanceof IDataProcessorObservable) {
			this.observe((IDataProcessorObservable) dataProcessor);
		}

		// Init will trigger a start event
		dataProcessor.init();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.harness.IDataProcessorObservable.Observer#start(org.eclipse.hyades.execution.core.IDataProcessor)
	 */
	public void start(IDataProcessorObservable context) {

		// Increment the data processors that are active
		this.activeDataProcessors++;

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.harness.IDataProcessorObservable.Observer#stop(org.eclipse.hyades.execution.core.IDataProcessor)
	 */
	public void stop(IDataProcessorObservable context) {

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.IExecutor#supportsControlEvent(java.lang.String)
	 */
	public boolean supportsControlEvent(String controlEvent) {
		return true;
	}

	public int getState() {
		if (delegate == null) {
			// No remote component exists
			return fatalErrorEncountered ? IExecutionComponent.DEAD : IExecutionComponent.INACTIVE;
		}
		return super.getState();
	}

	
}