/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: XMLExecutionDataProcessor.java,v 1.18 2010/04/12 12:38:44 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.execution.harness;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.InetAddress;
import java.util.HashMap;
import java.util.Iterator;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.emf.common.util.URI;
import org.eclipse.hyades.internal.execution.local.common.DataServerListener;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.loaders.common.ExecutionContext;
import org.eclipse.hyades.loaders.hierarchy.IgnoredXMLFragmentLoader;
import org.eclipse.hyades.loaders.util.InvalidXMLException;
import org.eclipse.hyades.loaders.util.RegistryReader;
import org.eclipse.hyades.loaders.util.XMLLoader;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.util.internal.EMFWorkspaceUtil;

/**
 * <p>XMLExecutionDataProcessor.java</p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @author  Ashish Mathur
 * @author  Jospeh P. Toomey
 * @author  Ernst Jessee
 * @version April 12, 2010
 * @since   January 25, 2005
 */
public class XMLExecutionDataProcessor extends XMLLoader implements DataServerListener, IExecutionHarnessDataProcessor3,
		IDataProcessorObservable {

	// This map is never accessed other than for writing, is this needed?
	private static final HashMap contextMap = new HashMap();

	public static final String IID = "org.eclipse.hyades.execution.harness.XMLExecutionDataProcessor"; //$NON-NLS-1$

	private Agent controlAgent = null;

	private TPFDeployment deployment;

	private String executionResultLocation;

	private String executionResultName;

	/**
	 * Observers of this data processor, notified of various events
	 */
	private final HashMap observers = new HashMap();

	private boolean overrideExistingExcResult = false;

	private Process process = null;

	private final String START_TAG = "<EXECUTION>"; //$NON-NLS-1$

	private ITest test = null;

	private final String XML_VERSION_TAG = "<?xml version=\"1.0\"?>"; //$NON-NLS-1$

	private boolean databaseResource = false;

	private boolean testLogFile = false;
	
	private URI testLogURI = null;

	/**
     * Current thread lock for synchronization.
     */
    private final static Object LOCK = new Object();

	/**
	 * XMLDataProcessor constructor comment.
	 */
	public XMLExecutionDataProcessor() {

		super((TRCMonitor) null);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.harness.IDataProcessorObservable#addObserver(org.eclipse.hyades.execution.harness.IDataProcessorObservable.Observer)
	 */
	public void addObserver(IDataProcessorObservable.Observer observer) {
		this.observers.put(observer, observer);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.local.common.DataServerListener#dataServerExited()
	 */
	public void dataServerExited() {
		// Clean up the ExecutionContext
		final ExecutionContext eContext = (ExecutionContext) this.context.getCustomData().get(ExecutionContext.root);
		synchronized (eContext) {
			// if we are the last agent out, then we'll clean up the context.
			if (eContext.decrementAgentCount() == 0) {

				contextMap.remove(XMLExecutionDataProcessor.this.executionResultLocation + "/" + //$NON-NLS-1$ 
						XMLExecutionDataProcessor.this.executionResultName);

				// here we have a problem. Because sometimes, there are still
				// message streaming when the
				// dataserver exits, we cannot clean up yet. WE need to
				// synchronized this.
				// for now, we will just put in this less than optimal wait. I
				// think 3 seconds should be adequate time.
				Thread contextCleaner = new Thread() {
					public void run() {

						// Cleanup now
						eContext.cleanUp();
						
						if (testLogURI != null) {
							try {
								EMFWorkspaceUtil.refreshLocal(testLogURI);
								testLogURI = null;
							} catch (IOException e) {
								// If we can't refresh the file in the 
								// workspace, this is okay.
							}
						}
						
						// Fire cleaned event at this point
						XMLExecutionDataProcessor.this.fireClean();

					}

				};
				contextCleaner.setName("Execution Context Cleaner"); //$NON-NLS-1$
				contextCleaner.setPriority(Thread.NORM_PRIORITY);
				contextCleaner.start();

			}
		}
		cleanUp();

		// Fire complete event at this point
		this.fireStop();

	}

	/**
	 * Fire clean event to listeners
	 */
	private void fireClean() {
		for (Iterator observers = this.observers.values().iterator(); observers.hasNext();) {
			IDataProcessorObservable.Observer observer = (IDataProcessorObservable.Observer) observers.next();
			observer.clean(this);
		}
	}

	/**
	 * Fire start event to listeners
	 */
	private void fireStart() {
		for (Iterator observers = this.observers.values().iterator(); observers.hasNext();) {
			IDataProcessorObservable.Observer observer = (IDataProcessorObservable.Observer) observers.next();
			observer.start(this);
		}
	}

	/**
	 * Fire stop event to listeners
	 */
	private void fireStop() {
		for (Iterator observers = this.observers.values().iterator(); observers.hasNext();) {
			IDataProcessorObservable.Observer observer = (IDataProcessorObservable.Observer) observers.next();
			observer.stop(this);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.harness.IExecutionDataProcessor#getConfigElement()
	 */
	public IConfigurationElement getConfigElement() {
		return null;
	}

	/**
	 * @return
	 */
	public Agent getControlAgent() {
		return this.controlAgent;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.harness.IExecutionDataProcessor#getID()
	 */
	public String getID() {
		return IID;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.harness.IExecutionDataProcessor#getName()
	 */
	public String getName() {
		return getClass().getName();
	}

	/**
	 * @return
	 */
	public Process getProcess() {
		return this.process;
	}

	/**
	 * @return
	 */
	public ITest getTest() {
		return this.test;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.IDataProcessor#incommingData(byte[], int, java.net.InetAddress)
	 */
	public void incommingData(byte[] buffer, int length, InetAddress peer) {

		try {

			//Synchronize loading or writing the test execution events to 
			//handle when multiple test agents emit test execution events 
			//to the same test log:
			synchronized (LOCK) {
				
				if (!testLogFile){
					super.loadEvent(buffer, length);
				}
				else{
					writeToBinaryOutputFile(buffer, 0, length);
				}
			}
		} 
		catch (InvalidXMLException e) {
			//Ignore and omit invalid XML events.
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.IDataProcessor#incommingData(char[],
	 *      int, java.net.InetAddress)
	 */
	public void incommingData(char[] buffer, int length, InetAddress peer) {
		byte[] newBuffer = new byte[length];
		for (int i = 0; i < length; i++) {
			newBuffer[i] = (byte) buffer[i];
		}
		this.incommingData(newBuffer, newBuffer.length, peer);
	}

	/**
	 * @see org.eclipse.hyades.loaders.util.XMLLoader#createOutputFile()
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	protected boolean createOutputFile() {
		try {
			if (file == null) {
				
				String testLogName = executionResultName;
				
				if(testLogName.trim().endsWith("." + ICommonConstants.EXECUTION_FILE_EXTENSION)){
					testLogName = testLogName.substring(0, (testLogName.lastIndexOf('.')));			
				}
				
				String uriString = executionResultLocation + "/" + testLogName + (testLogName.trim().endsWith(".testlog") ? "" : ".testlog"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				File outputFile;
				
				// If we're not in both platform and workspace modes, then the 
				// path we have is a file system path, and we don't want to access 
				// any classes in org.eclipse.core.resources.
				if ( RegistryReader.isPlatformMode() && RegistryReader.isWorkspaceMode() ) {
					testLogURI = URI.createPlatformResourceURI(uriString, false);
					outputFile = EMFWorkspaceUtil.getFileFromURI(testLogURI);
				}
				else {
					outputFile = new File(uriString);
				}
				
				file = new BufferedOutputStream(new FileOutputStream(outputFile));
			}
		} catch (Exception e) {
			return false;
		}
		return true;
	}

//	private URI createURI(String filePath)
//	{
//		if ( RegistryReader.isPlatformMode() && RegistryReader.isWorkspaceMode() )
//			return URI.createPlatformResourceURI(filePath);
//		else
//			return URI.createFileURI(filePath);
//	}


	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.harness.TestExecutionHarnessDataProcessor#init(org.eclipse.hyades.models.common.facades.behavioral.ITest,
	 *      org.eclipse.hyades.models.common.configuration.CFGLocation,
	 *      java.lang.String, java.lang.String)
	 */
	public void init() {

		// Specify the ignore for an unexpected event
		this.defaultLoader = new IgnoredXMLFragmentLoader();

		if (!(getTest() instanceof TPFTestSuite))
			return;

		// Create an execution context & add it to the Hierarchy Context
		// This will create the root execution result/history for this execution
		ExecutionContext eContext = null;

		/*
		 * Lock down context map for gets and puts since it is static and shared
		 * amongst all class instances
		 */
		synchronized (XMLExecutionDataProcessor.contextMap) {

			eContext = (ExecutionContext) contextMap.get(this.executionResultLocation + "/" + this.executionResultName); //$NON-NLS-1$

			TPFTestSuite suite = (TPFTestSuite) getTest();

			if (eContext == null) {
				eContext = new ExecutionContext(suite, suite.getId(), this.executionResultLocation,
						this.executionResultName, this.overrideExistingExcResult, this.deployment, databaseResource, testLogFile);
				contextMap.put(this.executionResultLocation + "/" + this.executionResultName, eContext); //$NON-NLS-1$
			}
		}
		eContext.incrementAgentCount();
		// Add the ExecutionContext to the HierarchyContext
		this.context.getCustomData().put(ExecutionContext.root, eContext);

		// Fire start event at this point
		this.fireStart();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.IDataProcessor#invalidDataType(byte[],
	 *      int, java.net.InetAddress)
	 */
	public void invalidDataType(byte[] data, int length, InetAddress peer) {
		// No use of this method
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.loaders.util.XMLLoader#isIgnoredElement()
	 */
	protected boolean isIgnoredElement() {
		if (this.startDocument) {
			if (this.currentElementName == null) {
				return true;
			}
			if (this.currentElementName.equals("EXECUTION")) { //$NON-NLS-1$
				return true;
			}
			this.startDocument = false;
		}
		return false;
	}

	protected boolean isValidTag(String buf) {
		return (!(buf.startsWith(this.START_TAG) || buf.startsWith(this.XML_VERSION_TAG)));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.harness.IDataProcessorObservable#removeObserver(org.eclipse.hyades.execution.harness.IDataProcessorObservable.Observer)
	 */
	public void removeObserver(IDataProcessorObservable.Observer observer) {
		this.observers.remove(observer);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.harness.IExecutionDataProcessor#setConfigElement(org.eclipse.core.runtime.IConfigurationElement)
	 */
	public void setConfigElement(IConfigurationElement theElement) {

	}

	/**
	 * @param agent
	 */
	public void setControlAgent(Agent agent) {
		this.controlAgent = agent;
	}

	/**
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public void setInitData(ITest theTest, String hostName, String executionResultName, String executionResultLocation,
			boolean overwriteExistingResults, String portNumber, TPFDeployment deployment, HashMap optionsMap) {
		this.test = theTest;
		this.executionResultName = executionResultName;
		this.overrideExistingExcResult = overwriteExistingResults;
		this.executionResultLocation = executionResultLocation;
		this.deployment = deployment;
		
		String db = (String) optionsMap.get(ITestExecutionHarnessOptions.DATABASE);
		if (db != null) 
			this.databaseResource = Boolean.valueOf(db).booleanValue();
		
		String testLog = (String) optionsMap.get(ITestExecutionHarnessOptions.TEST_LOG_FILE);
		if (testLog != null)
			this.testLogFile  = Boolean.valueOf(testLog).booleanValue();
	}
	
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.harness.IExecutionHarnessDataProcessor#setInitData(org.eclipse.hyades.models.common.facades.behavioral.ITest,
	 *      java.lang.String, java.lang.String, boolean, java.lang.String,
	 *      java.lang.String)
	 */
	public void setInitData(ITest theTest, String hostName, String executionResultName, String executionResultLocation,
			boolean overwriteExistingResults, String portNumber, TPFDeployment deployment) {
		
		HashMap optionsMap = new HashMap();
		this.setInitData(theTest, hostName, executionResultName, executionResultLocation, false, portNumber, null, optionsMap);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.harness.IExecutionDataProcessor#setInitData(java.lang.String,
	 *      java.lang.String, java.lang.String, java.lang.String)
	 */
	public void setInitData(ITest test, String hostName, String executionResultName, String executionResultLocation,
			String portNumber) {
		this.setInitData(test, hostName, executionResultName, executionResultLocation, false, portNumber, null);

	}

	/**
	 * @param process
	 */
	public void setProcess(Process process) {
		this.process = process;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.IDataProcessor#waitingForData()
	 */
	public void waitingForData() {
		// No use of this method
	}

}
