/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionAdapterUtilities.java,v 1.10 2008/04/10 00:52:46 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.execution.harness.util;

import java.util.HashMap;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.hyades.execution.core.IExecutableObject;
import org.eclipse.hyades.execution.core.IExecutionEnvironment;
import org.eclipse.hyades.execution.core.INode;
import org.eclipse.hyades.execution.harness.IExecutableObjectAdapter;
import org.eclipse.hyades.execution.harness.IExecutionDeploymentAdapter;
import org.eclipse.hyades.execution.harness.IExecutionEnvironmentAdapter;
import org.eclipse.hyades.execution.harness.XMLExecutionDataProcessor;
import org.eclipse.hyades.execution.harness.internal.resources.ExecutionHarnessPluginResourceBundle;
import org.eclipse.hyades.execution.local.JavaProcessExecutableObjectStub;
import org.eclipse.hyades.models.common.configuration.CFGClass;
import org.eclipse.hyades.models.common.facades.behavioral.IImplementor;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.osgi.util.NLS;

/**
 * Execution adapter utilities.
 * 
 * 
 * @author  Joseph P. Toomey
 * @author  Paul E. Slauenwhite
 * @version April 3, 2008
 * @since   January 31, 2005
 */
public class ExecutionAdapterUtilities {
	
	// bugzilla 84790: set this to false to create command line args
	// the old way - position of the args is assumed by the runner code
	protected final static boolean USE_NAMED_PARAMETERS = true;
	
	public static Object getAdapterInstance(String executionComponentType, String testType, 
			boolean bStandalone, HashMap testTypemap, ISystemUtility systemUtility ) throws ClassNotFoundException
		{
			Object temp = testTypemap.get(executionComponentType);
			if ( temp != null && temp instanceof HashMap )
			{
				HashMap execCompMap = (HashMap) temp;
				temp = execCompMap.get(testType);
				try {
					if ( bStandalone )
					{
						if ( temp != null && temp instanceof String )
						{
							return Class.forName((String)temp).newInstance();
						}
					}
					else
					{
						if ( temp != null && temp instanceof IConfigurationElement)
						{
							return ((IConfigurationElement)temp).createExecutableExtension("class"); //$NON-NLS-1$
						}
					}
				}
				catch ( Exception accessExc )
				{
				}
				
			}
			String msg = NLS.bind(ExecutionHarnessPluginResourceBundle.EXEC_NOT_FOUND_ERR_, executionComponentType, testType); 
			
			// We couldn't find the right execution component
			throw new ClassNotFoundException(msg);
		}

		public static boolean adaptExecutableObject(IExecutableObject executableObject, CFGClass rootResource, IImplementor theImplementor, boolean bStandalone, String testType, TPFDeployment deployment, StringBuffer bufError, ISystemUtility systemUtility, HashMap testTypeMap) {
			
			// Call out into an adapter class to configure this execution 
			// environment for this test
			IExecutableObjectAdapter execObjAdapter = null;
			try {
				execObjAdapter = (IExecutableObjectAdapter) getAdapterInstance("EXECUTABLE_OBJECT_ADAPTER", testType, bStandalone, testTypeMap, systemUtility); //$NON-NLS-1$
			}
			catch ( Exception accessExc )
			{
				systemUtility.logError(accessExc);			
				bufError.append(NLS.bind(ExecutionHarnessPluginResourceBundle.EXEC_ENV_ERR_, testType)); 
				return false;					
			}
			if ( execObjAdapter != null )
				execObjAdapter.setupExecutableObject(executableObject, rootResource, theImplementor, deployment);
			
			if(executableObject instanceof JavaProcessExecutableObjectStub)
			 {
				JavaProcessExecutableObjectStub stub = (JavaProcessExecutableObjectStub)executableObject;
				String args = stub.getArgs();
				
				// bugzilla 84790: set jvmArgs using new named parameter 
				if ( useNamedParameters() )
				{
					// using new way of passing named parameters to the runner 
					String jvmArgs = ""; //$NON-NLS-1$
					jvmArgs += "-Dhyades.runner.agentName=" + XMLExecutionDataProcessor.IID + " "; //$NON-NLS-1$ //$NON-NLS-2$
					jvmArgs += args;
					stub.setArgs(jvmArgs);
				}
				else
				{
					// using old way of passing args to the runner
					// runner assumes a particular position for the args
					args+=" "+XMLExecutionDataProcessor.IID; //$NON-NLS-1$
					stub.setArgs(args);
				}
			 }
				
			return true;
		}

		// bugzilla 84790: returns true if passing named parameters to the runner
		public static boolean useNamedParameters()
		{
			return USE_NAMED_PARAMETERS;
		}
		
		public static boolean adaptExecutionEnvironment(IExecutionEnvironment exeEnvironment, CFGClass rootResource, IImplementor theImplementor, boolean bStandalone, String testType, TPFDeployment deployment, StringBuffer bufError, ISystemUtility systemUtility, HashMap testTypeMap) {
			// Call out into an adapter class to configure this execution 
			// environment for this test
			
			IExecutionEnvironmentAdapter envAdapter = null;
			try {
				envAdapter = (IExecutionEnvironmentAdapter)getAdapterInstance("EXECUTION_ENVIRONMENT_ADAPTER", testType, bStandalone, testTypeMap, systemUtility); //$NON-NLS-1$
			}
			catch ( Exception accessExc )
			{
				systemUtility.logError(accessExc);			
				bufError.append(NLS.bind(ExecutionHarnessPluginResourceBundle.EXEC_ENV_ERR_, testType));
				return false;				
			}
			if ( envAdapter != null )
				envAdapter.setupExecutionEnvironment(exeEnvironment, rootResource, theImplementor, deployment);
				
			return true;
		}

		public static boolean adaptExecutionDeployment(INode node, TPFDeployment deployment, boolean bStandalone, String testType, StringBuffer bufError, ISystemUtility systemUtility, HashMap testTypeMap, IExecutionEnvironment exeEnvironment)
		{
		    IExecutionDeploymentAdapter deployAdapter = null;
			try {
				deployAdapter = (IExecutionDeploymentAdapter)getAdapterInstance("EXECUTION_DEPLOYMENT_ADAPTER", testType, bStandalone, testTypeMap, systemUtility); //$NON-NLS-1$
			}
			catch ( Exception accessExc )
			{
			    systemUtility.logError(accessExc);			
				return false;				
			}
			if ( deployAdapter != null ) {
				deployAdapter.deployTestAssets(node, deployment, bStandalone, bufError, exeEnvironment);
			    return (bufError.length() == 0); // an exception was accumulated while deploying test assets
			}
			return true;
		}	
		
		public static boolean adaptExecutionDeploymentCleanup(INode node, TPFDeployment deployment, boolean bStandalone, String testType, StringBuffer bufError, ISystemUtility systemUtility, HashMap testTypeMap, IExecutionEnvironment exeEnvironment)
		{
		    IExecutionDeploymentAdapter deployAdapter = null;
			try {
				deployAdapter = (IExecutionDeploymentAdapter)getAdapterInstance("EXECUTION_DEPLOYMENT_ADAPTER", testType, bStandalone, testTypeMap, systemUtility); //$NON-NLS-1$
			}
			catch ( Exception accessExc )
			{
			    systemUtility.logError(accessExc);			
				return false;				
			}
			if ( deployAdapter != null ) {
				deployAdapter.cleanUpTestAssets(node, deployment, bStandalone, bufError, exeEnvironment);
			        return (bufError.length() == 0); // an exception was accumulated while cleaning up test assets
			}
			return true;
		}	
}
