/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: StandaloneExecutionUtilities.java,v 1.6 2007/05/03 16:54:09 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.harness.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.hyades.execution.core.IExecutionComponentFactory;
import org.eclipse.hyades.execution.harness.ExecutionHarnessDataProcessorFactory;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * This class provides utility methods for standalone test execution. 
 * 
 * @author Joseph P. Toomey
 * @author Paul E. Slauenwhite
 */
public class StandaloneExecutionUtilities {


	/**
	 * <p>This method parses all of the plugin.xml files specified in the 
	 * standaloneConfigurationFiles parameter searching for the following three
	 * extension points:</p>
	 * <ul><li>org.eclipse.hyades.execution.harness.RegisteredExecutionComponentImpl</li>
	 * <li>org.eclipse.hyades.execution.harness.ExecutionEnvironmentAdapter</li>
	 * <li>org.eclipse.hyades.execution.harness.ExecutableObjectAdapter</li></ul>
	 * 
	 * <p>For all found implementations of these extension points, the factory and
	 * testTypeMap are appropriately populated.</p>
	 *   
	 * @param standaloneConfigurationFiles
	 * @param testTypeMap
	 * @param factory
	 * @throws ClassNotFoundException
	 * @throws FileNotFoundException
	 */
	public static void initializeRegisteredExecutionComponents(String[] standaloneConfigurationFiles, 
															   HashMap testTypeMap,
															   IExecutionComponentFactory factory, 
															   ExecutionHarnessDataProcessorFactory dpFactory) throws ClassNotFoundException, FileNotFoundException
	{
		ArrayList dataProcessorCollection = new ArrayList();
			
		for ( int i=0; i < standaloneConfigurationFiles.length; i++)
		{
			parseConfigFile(standaloneConfigurationFiles[i], testTypeMap, factory, dataProcessorCollection);
		}
		
		ExecutionHarnessDataProcessorFactory.setStandAloneClassObjects(dataProcessorCollection);
	}
	
	
	private static void parseConfigFile( String configFile, HashMap testTypeMap, 
		IExecutionComponentFactory factory, ArrayList dataProcessors) throws ClassNotFoundException, FileNotFoundException
	{
		FileInputStream stream = new FileInputStream(configFile);
		Element plugin = XMLUtil.loadDom(stream, "plugin"); //$NON-NLS-1$
		
		NodeList nodeList = XMLUtil.getChildrenByName(plugin, "extension"); //$NON-NLS-1$
		if(nodeList != null)
		{
			for(int i=0; i<nodeList.getLength(); i++)
			{
				if(nodeList.item(i) instanceof Element)
				{
					Element extensionElement = (Element)nodeList.item(i);
					if(extensionElement != null)
					{
						if ( extensionElement.getAttribute("point").equals("org.eclipse.hyades.execution.harness.RegisteredExecutionComponentImpl")) //$NON-NLS-1$ //$NON-NLS-2$
						{
							NodeList execComps = XMLUtil.getChildrenByName(extensionElement, "RegisteredExecutionComponentImpl"); //$NON-NLS-1$
							for ( int j=0; j<execComps.getLength(); j++ )
							{
								if ( execComps.item(j) instanceof Element )
									parseRegisteredExecutionComponents((Element)execComps.item(j), testTypeMap, factory);
							}
						}

						if ( extensionElement.getAttribute("point").equals("org.eclipse.hyades.execution.harness.ExecutionEnvironmentAdapter")) //$NON-NLS-1$ //$NON-NLS-2$
						{
							NodeList envAdapters = XMLUtil.getChildrenByName(extensionElement, "ExecutionEnvironmentAdapter"); //$NON-NLS-1$
							for ( int j=0; j<envAdapters.getLength(); j++ )
							{
								if ( envAdapters.item(j) instanceof Element )
									parseAdapters((Element)envAdapters.item(j), testTypeMap, "EXECUTION_ENVIRONMENT_ADAPTER"); //$NON-NLS-1$
							}
						}

						if ( extensionElement.getAttribute("point").equals("org.eclipse.hyades.execution.harness.ExecutableObjectAdapter")) //$NON-NLS-1$ //$NON-NLS-2$
						{
							NodeList execObjAdapters = XMLUtil.getChildrenByName(extensionElement, "ExecutableObjectAdapter"); //$NON-NLS-1$
							for ( int j=0; j<execObjAdapters.getLength(); j++ )
							{
								if ( execObjAdapters.item(j) instanceof Element )
									parseAdapters((Element)execObjAdapters.item(j), testTypeMap, "EXECUTABLE_OBJECT_ADAPTER"); //$NON-NLS-1$
							}
						}
						
						if ( extensionElement.getAttribute("point").equals("org.eclipse.hyades.execution.harness.ExecutionDeploymentAdapter")) //$NON-NLS-1$ //$NON-NLS-2$
						{
							NodeList envAdapters = XMLUtil.getChildrenByName(extensionElement, "ExecutionDeploymentAdapter"); //$NON-NLS-1$
							for ( int j=0; j<envAdapters.getLength(); j++ )
							{
								if ( envAdapters.item(j) instanceof Element )
									parseAdapters((Element)envAdapters.item(j), testTypeMap, "EXECUTION_DEPLOYMENT_ADAPTER"); //$NON-NLS-1$
							}
						}
						
						if ( extensionElement.getAttribute("point").equals("org.eclipse.hyades.execution.harness.Dataprocessor")) //$NON-NLS-1$ //$NON-NLS-2$
						{
							NodeList dataProcessorNodeList = XMLUtil.getChildrenByName(extensionElement, "dataprocessor"); //$NON-NLS-1$
							for ( int j=0; j<dataProcessorNodeList.getLength(); j++ )
							{
								if ( dataProcessorNodeList.item(j) instanceof Element )
									parseDataProcessors((Element)dataProcessorNodeList.item(j),dataProcessors);
							}
						}
					}
				}
			}
		}
		
	}

	private static void parseRegisteredExecutionComponents(Element extensionElement, HashMap testTypeMap, 
		IExecutionComponentFactory factory) throws ClassNotFoundException
	{
		String type = extensionElement.getAttribute("type"); //$NON-NLS-1$
		String executionComponentName = extensionElement.getAttribute("name"); //$NON-NLS-1$
		factory.addExecutionComponent(executionComponentName, extensionElement.getAttribute("implClass")); //$NON-NLS-1$
		factory.addStub(executionComponentName, extensionElement.getAttribute("stubClass")); //$NON-NLS-1$
		factory.addSkeleton(executionComponentName, extensionElement.getAttribute("skeletonClass")); //$NON-NLS-1$
					
		Object map = testTypeMap.get(type);
		if ( map == null )
		{
			map = new HashMap();
			testTypeMap.put(type, map);
		}
		HashMap execCompMap = (HashMap)map;
		
		NodeList supportedTestTypes = XMLUtil.getChildrenByName(extensionElement, "SupportedTestType"); //$NON-NLS-1$
		for ( int j=0; j<supportedTestTypes.getLength(); j++ )
		{
			if (supportedTestTypes.item(j) instanceof Element )
			{
				Element supportedType = (Element) supportedTestTypes.item(j);
				execCompMap.put(supportedType.getAttribute("name"), executionComponentName); //$NON-NLS-1$
			}
		}
	}


	private static void parseAdapters(Element adapterElement, HashMap testTypeMap, String type) 
	{

		String adapterClass = adapterElement.getAttribute("class"); //$NON-NLS-1$
		
		Object map = testTypeMap.get(type);
		if ( map == null )
		{
			map = new HashMap();
			testTypeMap.put(type, map);
		}
		HashMap execCompMap = (HashMap)map;
			
		NodeList supportedTestTypes = XMLUtil.getChildrenByName(adapterElement, "SupportedTestType"); //$NON-NLS-1$
		for ( int j=0; j<supportedTestTypes.getLength(); j++ )
		{
			if (supportedTestTypes.item(j) instanceof Element )
			{
				Element supportedType = (Element) supportedTestTypes.item(j);
				execCompMap.put(supportedType.getAttribute("name"), adapterClass); //$NON-NLS-1$
			}
		}
	}
	
	
	/**
	 * instantiates dataprocessors in standalone mode
	 * @param dataProcessorElement
	 * @param dataProcessorCollection
	 * @throws ClassNotFoundException
	 */
	private static void parseDataProcessors(Element dataProcessorElement, ArrayList dataProcessorCollection)
	{

		String adapterClass = dataProcessorElement.getAttribute("class"); //$NON-NLS-1$
		
		try
		{
			dataProcessorCollection.add(Class.forName(adapterClass));
		}
		catch(Exception e)
		{
			e.printStackTrace();
		}
		

	}

	/**
	 * This method takes a directory which is structured as (and may actually be)
	 * an Eclipse directory.  This method will walk all subdirectories in the plugins subdir of
	 * the specified configDir, and will return an array containing the fully 
	 * qualified paths to any plugin.xml files found in those subdirectories.  If
	 * no plugin.xml files are found, or if the configDir does not exist, this method
	 * will return null. 
	 * 
	 * @param configDir
	 * @return
	 */
	public static String[] getConfigFilesFromConfigDir(String configDir)
	{
		File configDirFile = new File(configDir);
		if ( !configDirFile.isDirectory() )
			return null;

		configDir = configDirFile.getAbsolutePath();
		configDir += File.separator + "plugins"; //$NON-NLS-1$
		configDirFile = new File(configDir);
		if ( !configDirFile.isDirectory() )
			return null;

		ArrayList pluginPaths = new ArrayList(); 
			 
		// Get the list of all elements of configDir's plugin directory
		String[] elements = configDirFile.list();
		String element;
		File elementFile;
		
		for ( int i=0; i<elements.length; i++ )
		{
			element = configDir + File.separator + elements[i];
			elementFile = new File(element);
			if ( !elementFile.isDirectory())
				continue;
			
			// Found a subdirectory -- check to see if it contains
			// a file called plugin.xml
			element = element + File.separator + "plugin.xml"; //$NON-NLS-1$
			elementFile = new File(element);
			if ( elementFile.exists() )
				pluginPaths.add(element);
		}
		
		return (String[])pluginPaths.toArray(new String[0]);
		
	}

	/**
	 * This method takes a directory which is structured as (and may actually be)
	 * an Eclipse plugins directory.  This method attempts to locate the 
	 * execution harness plugin directory (org.eclipse.hyades.execution.harness)
	 * within the specified directory.  If the execution harness plugin 
	 * directory can not be found in the specified directory, or if the 
	 * specified directory does not exist, this method will return null. 
	 * 
	 * @param configDir
	 * @return 
     * @deprecated As of TPTP V4.4, use the plug-in's resource bundle (e.g. {@link org.eclipse.hyades.execution.harness.internal.resources.ExecutionHarnessPluginResourceBundle} for this plug-in) for resolving resources in a stand-alone environment.  
	 */
	public static String getHarnessDir(String configDir, String version)
	{
		File configDirFile = new File(configDir);
		if ( !configDirFile.isDirectory() )
			return null;
			
		configDir = configDirFile.getAbsolutePath();
		
		String harnessDir = configDir + File.separator + "plugins" + File.separator + "org.eclipse.hyades.test.core_"+version; //$NON-NLS-1$ //$NON-NLS-2$ 
		
		File harnessFile = new File(harnessDir);
		
		if ( harnessFile.isDirectory()) {
			return harnessDir;
		}
		
		return null;
		
	}
	
}

