/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: StandaloneSystemUtility.java,v 1.8 2008/12/23 17:59:04 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.harness.util;

import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import com.ibm.icu.util.ULocale;

/**
 * This class provides a small set of services for stand-alone execution that
 * are otherwise provided by the Eclipse plug-in.  Among these services are
 * locating and retrieving strings from the correct resource bundle for the 
 * current locale, and logging errors (which currently simply writes to 
 * System.out.)
 * 
 *  
 * @author  Joseph Toomey
 * @author  Paul Slauenwhite
 * @version December 23, 2008
 * @since   January 31, 2005
 */
public class StandaloneSystemUtility implements ISystemUtility {

	private ResourceBundle bundle;
	private String harnessDir;
	private String localizedBundleJar;
	
	/**
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public StandaloneSystemUtility() {
		//No-operation.
	}
		
	/**
	 * Single argument constructor.
	 * 
	 * @param harnessDir The directory containing this plug-in.
	 * @deprecated As of TPTP V4.4, use the plug-in's resource bundle (e.g. {@link org.eclipse.hyades.execution.harness.internal.resources.ExecutionHarnessPluginResourceBundle} for this plug-in) for resolving resources in a stand-alone environment.  
	 */
	public StandaloneSystemUtility(String harnessDir) {
		super();

		// Make sure the harness dir ends in a '/' character, or the 
		// URLClassLoader will assume it is a jar file. (!!)
		this.harnessDir = harnessDir;
		if ( harnessDir.charAt(harnessDir.length()-1) == '/')
		{
			this.localizedBundleJar = harnessDir.substring(0,harnessDir.length()-2);
		}
		else
		{
			this.harnessDir += '/';
			this.localizedBundleJar = harnessDir;
		}
		
		// Build path to localized bundle jar
		this.localizedBundleJar += ".nl1/nl1.jar"; //$NON-NLS-1$
	}


	/**
     * Resolves the plugin's resource bundle.
     * <p>
     * If the plugin's resource bundle can not be resolved, <code>null</code> is returned.
     * <p>
     * IMPORTANT: Do not use this method to retrieve values from the resource bundle.  This method 
     * is provided so this plugin's resource bundle can be used as the parent of another plugin's 
     * resource bundle.
     * <p>
     * 
     * @return The plugin's resource bundle, otherwise <code>null</code>.
	 * @deprecated As of TPTP V4.4, use the plug-in's resource bundle (e.g. {@link org.eclipse.hyades.execution.harness.internal.resources.ExecutionHarnessPluginResourceBundle} for this plug-in) for resolving resources in a stand-alone environment.  
	 */
	public ResourceBundle getResourceBundle() {
		return getResourceBundle(ULocale.getDefault());
	}
	
	/**
     * Resolves the plugin's resource bundle for a specific locale.
     * <p>
     * If the plugin's resource bundle can not be resolved, <code>null</code> is returned.
     * <p>
     * IMPORTANT: Do not use this method to retrieve values from the resource bundle.  This method 
     * is provided so this plugin's resource bundle can be used as the parent of another plugin's 
     * resource bundle.
     * <p>
     * 
     * @param targetLocale The specific locale of the returned resource bundle.
     * @return The plugin's resource bundle for a specific locale, otherwise <code>null</code>. 
	 * @deprecated As of TPTP V4.4, use the plug-in's resource bundle (e.g. {@link org.eclipse.hyades.execution.harness.internal.resources.ExecutionHarnessPluginResourceBundle} for this plug-in) for resolving resources in a stand-alone environment.  
	 */
	public ResourceBundle getResourceBundle(ULocale targetLocale) {
		
		// return the cached bundle, if we've already located it.
		if ( this.bundle != null )
			return this.bundle;
			
		ResourceBundle newBundle = null;

		try {
			URL[] newcp = {
			        new URL("file", "", this.harnessDir), //$NON-NLS-1$ //$NON-NLS-2$
			        new URL("file", "", this.localizedBundleJar) //$NON-NLS-1$ //$NON-NLS-2$
			};  
			ClassLoader resourceLoader = new URLClassLoader(newcp, null);
	
			newBundle = ResourceBundle.getBundle("plugin", targetLocale.toLocale(), resourceLoader); //$NON-NLS-1$
			this.bundle = newBundle;
		}
		// If we throw an exception above, we'll log the error and return null. 
		catch (MalformedURLException exc)
		{
			logError(exc);
			return null;
		}
		catch (MissingResourceException exc)
		{
			logError(exc);
			return null;
		}

		return newBundle;
	}

   /**
    * Resolves the localized message associated with the parameter <code>key</code>
    * based on the default locale from the plugin's resource bundle.
    * <p>
    * If the <code>key</code> does not exist in the plugin's resource bundle, the 
    * <code>key</code> is returned.
    * <p>
    * Noteworthy, the resultant message is not formatted (e.g. no message parameter substitution). 
    * <p>
    * 
    * @param key The <code>key</code> of the message in the plugin's resource bundle.
    * @return The localized message associated with the parameter <code>key</code> from the plugin's resource bundle, otherwise the <code>key</code>.
	* @deprecated As of TPTP V4.4, use the plug-in's resource bundle (e.g. {@link org.eclipse.hyades.execution.harness.internal.resources.ExecutionHarnessPluginResourceBundle} for this plug-in) for resolving resources in a stand-alone environment.  
	*/
	public String getString(String key) {
		ResourceBundle bundle = getResourceBundle();
		if ( bundle == null )
			return key;

		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			return key;
		}
	}

	/**
	 * This method logs the message of the specified Throwable to System.out.
	 * @see org.eclipse.hyades.execution.harness.util.ISystemUtility#logError(java.lang.Throwable)
	 */
	public void logError(Throwable exc) {
		System.out.println(exc.getMessage());
	}

}
