/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RecorderFactory.java,v 1.10 2007/05/03 01:23:58 paules Exp $
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.recorder;

import java.io.IOException;
import java.net.URL;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle;
import org.eclipse.osgi.util.NLS;
import org.eclipse.tptp.test.provisional.recorder.framework.IRecorderClientHelper;
import org.eclipse.tptp.test.provisional.recorder.framework.IRecorderExecOptionsProvider;
import org.eclipse.tptp.test.provisional.recorder.framework.IRecorderMessageProvider;
import org.eclipse.tptp.test.provisional.recorder.framework.IRecorderProvider;

/**
 * Factory which manages recorders 
 * This object is a singleton which should be accessed via getInstance().
 * @author Ernest Jessee
 *
 */
public class RecorderFactory
{
	/**
	 * stores default instance
	 */
	private static RecorderFactory instance = null;
	
	public static final String RECORDER_EXTENSION_ID = "org.eclipse.hyades.test.core.Recorder";
	/**
	 * HashMap for storing recorder reference with ID key
	 */
	private HashMap recorderMap = new HashMap();
	
	/**
	 * Hashmap for storing actions to listen to the active state of the recorder 
	 */
	private HashMap listenerMap = new HashMap();
			
	//TODO: Handle listeners for multiple recorders
	
	private HashMap recorderProviderMap = new HashMap();
	/**
	 * This is the method called to acquire a reference to the RecorderFactory
	 * 
	 */
	public static RecorderFactory getInstance()
	{
		
		if(instance == null)
		{
			instance = new RecorderFactory();
		}	
		
		return instance;
	}
	
	
	/**
	 * called to add a recorder to the factory.  This should not be called directly
	 * by any object other than the recorder object itself.  When a recorder is 
	 * constructed, it automatically adds itself to the factory
	 * 
	 * @param Recorder recorder
	 */
	protected void addRecorder(Recorder recorder)
	{
		recorderMap.put(recorder.getId(), recorder);
	}

	
	/**
	 * private constructor
	 * @see getInstance
	 */
	private RecorderFactory()
	{	
	}
	
	
	/**
	 * called to get a reference to a recorder with a specific ID.  If a recorder with that ID does not exist,
	 * one is created.
	 * @param String ID
	 * @return Recorder
	 */
	public Recorder getRecorderWithID(String ID)
	{
		Object rec = recorderMap.get(ID);
		if (rec == null)
		{

			try {
				rec = new Recorder(ID);
			}
			catch (RecorderException e) {
				//TODO: Log error
			}
			
			recorderMap.put(ID, rec);
		}
		return (Recorder) rec;
	}
	
	public boolean doesRecorderExist(String id)
	{
		return (recorderMap.get(id) != null);
	}
	
	/**
	 * returns the currently active recorder
	 * @return Recorder
	 */
	public Recorder getActiveRecorder()
	{
		Collection recorders = recorderMap.values();
		Iterator recorderIterator = recorders.iterator();
		while(recorderIterator.hasNext())
		{
			Recorder recorder = (Recorder)recorderIterator.next();
			if(recorder.isActive())
				return recorder;
		}
		
		return null;
		
	}	

	/**
	 * called to add a listening action to the factory.  This should not be called directly
	 * by any object other than the action object itself.  When an action is 
	 * constructed, it automatically adds itself to the factory
	 * 
	 * @param IRecorderAction action
	 */
	public void addListener(IRecorderListener action)
	{
		listenerMap.put(action.toString(), action);
		if (getActiveRecorder() == null)
			action.updateRecorderActive(false);
		else 
			action.updateRecorderActive(getActiveRecorder().isActive());
	}

	/**
	 * Removes a listener added via addListener
	 * returns: false if listener was not previously added
	 */
	public boolean removeListener(Recorder recorder, IRecorderListener listener)
	{
		return listenerMap.remove(listener.toString())!=null;
	}
	
	public Object[] getListenersForRecorder(Recorder recorder)
	{
		//TODO: map for multiple recorders
		//if (recorder == getActiveRecorder())
		//{
			Object[] actions = listenerMap.values().toArray();
			return actions;
		//}
		//return new Object[0];
	}
	public void updateActiveState(Recorder recorder, boolean bActive)
	{
		//if (recorder == getActiveRecorder())
		//{
			Object[] actions = getListenersForRecorder(recorder);
			for (int i=0; i < actions.length; i++)
			{
				if (actions[i] instanceof IRecorderListener)
				{
					((IRecorderListener)actions[i]).updateRecorderActive(bActive);
				}
			}
		//}
	}
	
	public void controlMessage(Recorder recorder, String key, String msg)
	{
		Object[] actions = getListenersForRecorder(recorder);
		for (int i=0; i < actions.length; i++)
		{
			if (actions[i] instanceof IRecorderListenerFullFeedback)
			{
				((IRecorderListenerFullFeedback)actions[i]).controlMessage(key, msg);
			}
		}
	}
	
	public void updateStatus(Recorder recorder, String msg)
	{
		Object[] actions = getListenersForRecorder(recorder);
		for (int i=0; i < actions.length; i++)
		{
			if (actions[i] instanceof IRecorderListener)
			{
				((IRecorderListener)actions[i]).updateStatus(msg);
			}
		}
		
	}
	
	public void reportExceptionToUser(Recorder recorder, final Exception e1,
		     final String consequenceDescription,
		     final String cause,
		     final String dlgTitle) 
	{
		Object[] actions = getListenersForRecorder(recorder);
		for (int i=0; i < actions.length; i++)
		{
			if (actions[i] instanceof MasterRecorderListener)
			{
				((MasterRecorderListener)actions[i]).reportExceptionToUser(e1, consequenceDescription,
						cause, dlgTitle);
			}
		}
		//TODO: log error
	}
	
   public String[] getRegisteredRecorderID()
   {
	   IConfigurationElement[] elements = Platform.getExtensionRegistry().getConfigurationElementsFor(RECORDER_EXTENSION_ID);
	   String[] ids = new String[elements.length];
	   for (int i = 0; i < elements.length; i++)
	   {
		   ids[i] = elements[i].getAttribute("id");
	   }
	   return ids;
   }
   
   public IRecorderClientHelper getRecorderClientHelper(String recID, boolean bForceReload)
   {	   
	   return (IRecorderClientHelper) loadRecorderObject(recID, "recorderClientHelper", bForceReload);
   }

   public IRecorderExecOptionsProvider getRecorderExecOptions(String recID, boolean bForceReload)
   {
	   return (IRecorderExecOptionsProvider) loadRecorderObject(recID, "execOptionsProvider", bForceReload);
   }
   
   public IRecorderMessageProvider getMessageProvider(String recID, boolean bForceReload)
   {
	   return (IRecorderMessageProvider) loadRecorderObject(recID, "recorderMessageHandlers", bForceReload);
   }
   
   public Object getWizardPageProvider(String recID, boolean bForceReload)
   {
	   return  loadRecorderObject(recID, "wizardPageProvider", bForceReload);

   }
   
   public String getRecorderFileExtension(String recID)
   {
	   IConfigurationElement element = findRecorder(recID);
	   if (element == null)
		   return null;
	   return element.getAttribute("fileExtension");
	   
   }
   public String getRecorderAgent(String recID)
   {
	   IConfigurationElement element = findRecorder(recID);
	   if (element == null)
		   return null;
	   return element.getAttribute("recorderAgent");
   }
   
   public String getRecorderName(String recID)
   {
	   IConfigurationElement element = findRecorder(recID);
	   if (element == null)
		   return null;
	   return element.getAttribute("name");

   }
   
   public boolean requiresIntermediateFile(String recID)
   {
	   IConfigurationElement element = findRecorder(recID);
	   if (element == null)
		   return true;
	   return !element.getAttribute("requiresIntermediateFile").equals("false");

   }
   
   public URL getRecorderIconURL(String recID)
   {
	   IConfigurationElement element = findRecorder(recID);
	   if (element == null)
		   return null;	   
	   String strIcon = element.getAttribute("icon");
	   if (strIcon == null || strIcon.equals(""))
		   return null;
	   
	   URL url = Platform.getBundle(element.getNamespaceIdentifier()).getEntry(strIcon);
	   if (url == null)
		   return null;
	   try {
		   return FileLocator.resolve(url);
	   }
	   catch (IOException ioe) {
		   return null;
	   }
   }
   
   public String getRecorderDescription(String recID)
   {
	   IConfigurationElement element = findRecorder(recID);
	   if (element == null)
		   return null;
	   return element.getAttribute("description");	   
   }
   /**
    * Convenience method to find a recorder with a particular id
    * @param recID
    * @return  IConfigurationElement
    */
   private IConfigurationElement findRecorder(String recID)
   {
	   IConfigurationElement[] elements = Platform.getExtensionRegistry().getConfigurationElementsFor(RECORDER_EXTENSION_ID);
	   for (int i = 0; i < elements.length; i++)
	   {
		   if (elements[i].getAttribute("id").equals(recID))
				   return elements[i];
	   }
	   return null;
   }
   
   private Object loadRecorderObject(String recID, String attribute, boolean bForceReload)
   {
	   IConfigurationElement element = findRecorder(recID);
	   if (element == null)
		   return null;
	   Object recMap = recorderProviderMap.get(recID);
	   if (recMap == null || !(recMap instanceof HashMap))
	   {
		   recMap = new HashMap();
		   recorderProviderMap.put(recID, recMap);
		   
	   }
	   Object obj = ((HashMap)recMap).get(attribute);
	   if (obj != null && !bForceReload)
		   return obj;
	   if (element.getAttribute(attribute) == null)
		   return null;
	   try {
		   obj = element.createExecutableExtension(attribute);
		   ((HashMap)recMap).put(attribute, obj);
		   if (obj instanceof IRecorderProvider)
		   {
			   ((IRecorderProvider)obj).setRecorder(recID);
		   }
		   return obj;
	   }
	   catch (CoreException e)
	   {
		   reportExceptionToUser(null, e, TestCorePluginResourceBundle.RecorderFactory_ERROR_LOADING_RECORDER_DATA,
				   NLS.bind(TestCorePluginResourceBundle.RecorderFactory_ERROR_LOADING_ATTRIBUTE, attribute),
				   TestCorePluginResourceBundle.RecorderFactory_ERROR_LOADING_RECORDER_DATA_TITLE);
		   return null;
	   }
	   
   }
}
