/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: InternetExplorerAdapter.java,v 1.19 2010/02/22 16:00:05 paules Exp $
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.recorder.local.appadapters;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle;

/**
 * <p>Internet Explorer client for the <code>org.eclipse.hyades.execution.recorder.RecorderApplication</code> 
 * extension point implementation.</p>
 * 
 * <p>Theis client handle Internet Explorer initializing, starting, stopping, and cleanup.  For initialization, 
 * this client modifies the Windows registry to configure Internet Explorer to use a particular host and port for 
 * proxy recording.</p>
 * 
 * 
 * @author  Ernest Jessee
 * @author  Paul E. Slauenwhite
 * @author  Duwayne Morris
 * @version February 22, 2010
 * @since   April 22, 2005
 * @deprecated  This is internal API.
 */
public class InternetExplorerAdapter implements IRecorderApplicationAdapter
{	
	/**
	 * the Internet Explorer Process
	 */
	private Process ieProcess;
		
	/**
	 * filename for storing the unmodified registry settings 
	 */
	private String originalSettingsFileName = "IEConfig.reg";//$NON-NLS-1$
	// defect 78895 below - also enable java console
	private String consoleSettingsFileName = "IEConfigConsole.reg";//$NON-NLS-1$

	/**
	 * filname for storing the registry settings to be used while recording
	 */
	private String tempSettingsFileName = "tempIEConfig.reg";//$NON-NLS-1$
	
	private String startupURL = "about:blank";
	private String launchPath = "C:\\Program Files\\Internet Explorer\\iexplore.exe";
	private String portnum = null;
	/**
	 * init routine called by the recorder to have internet explorer prepared for recording
	 * @param String initInfo -- this is a comma delimeted string indicating which port to point IE at. (ie. port,450)
	 */
	// defect 78895 mdd 12/8/2004
	final String ProxySrvStr = "\"ProxyServer";
	final String BadCertStr = "\"WarnonBadCertRecving";
	final String BadCertStr2 = "\"WarnOnBadCertRecving";
	final String ZoneCrossStr = "\"WarnonZoneCrossing";
	final String ZoneCrossStr2 = "\"WarnOnZoneCrossing";
	final String JavaConsoleStr = "\"EnableJavaConsole";
	final String ZonesMixedContentStr = "\"1609";
	
	// bugzilla 114253 mdd 1/6/2006
	private String originalSettingsFilePath = "";
	private String tempSettingsFilePath = "";
	private String consoleSettingsFilePath = "";
	
	// bugzilla 165310 mdd
	final String mylocalhost = "127.0.0.1:";
	
	// bugzilla 168597
	/* Later we may want to change this to always use native code
	 * to get and set Internet Explorer registry settings.
	 * 
	 * For this defect fix, we will determine that our attempt to run regedit.exe
	 * has failed running on Vista without using "Run as Administrator" elevated privileges
	 * 
	 * When this happens, we will set the boolean below true and create an instance of JNIWrapper class.
	 * The JNIWrapper class will load the native DLL and make the calls into native code.
	 * 
	 */
	private boolean bUseNativeRegistryInterface = false;
	private boolean bRunningVista = false;
	private JNIWrapper jniWrapper = null;
	
	public static final String PREFERENCE_PATH = "RECORDER_IE_PATH";
	public static final String REGISTRY_FILE_NEW_LINE = "\r\n"; //$NON-NLS-1$
	
	public void init(String initInfo) throws RecorderApplicationAdapterException
	{
		StringBuffer inputFileContents;
				
		// bugzilla 114253 set the path of temp files to the workspace path
		setTempFilePaths();
				
		parseInitInfo(initInfo);
		checkOSVersion();
		
		// Bugilla 168597		
		if (bUseNativeRegistryInterface)
			inputFileContents = getRegistryInfoUsingNativeAPI();
		else
			inputFileContents = getRegistryInfo();
		
		createOriginalRegSettingsFile(inputFileContents);
		
		StringBuffer outputFileContents = new StringBuffer();
		// new mdd hyades 1.3 check for http
		boolean isProxyEnabled = checkProxyEnabled(inputFileContents);

		Iterator<String> registryFileEntriesIterator = parseRegistryFileContents(inputFileContents).iterator();
		
		while(registryFileEntriesIterator.hasNext()){
			
			String line = registryFileEntriesIterator.next();
			String newLine = line;
			if(line.startsWith("\"ProxyEnable"))//$NON-NLS-1$
			{
				StringTokenizer lineTokenizer = new StringTokenizer(line,":");//$NON-NLS-1$
				newLine=lineTokenizer.nextToken();
				newLine+=":00000001";//$NON-NLS-1$
				setNativeRegistryValue(newLine);
				outputFileContents.append(newLine + REGISTRY_FILE_NEW_LINE);
				if (isProxyEnabled){
					// newLine+="\r\n\"ProxyServer\"=\"http=localhost:" + portnum;//$NON-NLS-1$
					newLine="\"ProxyServer\"=\"http=" + mylocalhost+  portnum;//$NON-NLS-1$
					newLine += ";https=" + mylocalhost + portnum + "\"";
				}
				else {
					// newLine+="\r\n\"ProxyServer\"=\"socks=localhost:";//$NON-NLS-1$
					newLine="\"ProxyServer\"=\"socks=" + mylocalhost;//$NON-NLS-1$
					newLine+=portnum+"\"";//$NON-NLS-1$
				}								
				
			}
			else if(line.startsWith("\"ProxyHttp1.1"))//$NON-NLS-1$
			{
				StringTokenizer lineTokenizer = new StringTokenizer(line,":");//$NON-NLS-1$
				newLine=lineTokenizer.nextToken();
				newLine+=":00000001";//$NON-NLS-1$
								
			}
			else if(line.startsWith("\"ProxyServer"))//$NON-NLS-1$
			{
				
				//it has been found that if the target computer has never been pointed to a proxy.  This 
				//entry may not be present.  Therefore, we will ignore it if it is there and always create it from scratch.
				newLine=null;				
			}
			else if (line.startsWith("\"ProxyOverride"))
			{
				//don't want to bypass the proxy for any hosts.  Leave empty
				newLine= "\"ProxyOverride\"=\"\"";
		
			}

			else if ( (line.startsWith(BadCertStr)) || (line.startsWith(BadCertStr2)) ) {
				newLine = disableLine(line);
				
			}
			else if ((line.startsWith(ZoneCrossStr))||(line.startsWith(ZoneCrossStr2))){
				newLine = disableLine(line);
			}
			else if (line.startsWith(JavaConsoleStr)){
				newLine = enableConsoleLine(line);
			} 
			else if (line.startsWith(ZonesMixedContentStr))  {
				newLine = enableZonesLine(line);
			}
			else if (line.startsWith("[HKEY_CLASSES_ROOT\\Applications\\iexplore.exe\\shell\\open\\command]"))
			{ //find ie path
				line = registryFileEntriesIterator.next();
				if (line.startsWith("@="))
				{
					launchPath = line.substring(2);
					if (launchPath.charAt(0) == '"' && launchPath.charAt(launchPath.length() - 1) == '"')
						launchPath = launchPath.substring(1, launchPath.length()-1);

					launchPath = launchPath.replaceAll("%1", "");
					int n = launchPath.indexOf("\\\\");  //replaceAll doesn't work for these cases
					int m = launchPath.indexOf("\\\"");
					while (n > -1 || m > -1)
					{						
						if (n > -1)
						{
							launchPath = launchPath.substring(0, n) + launchPath.substring(n+1, launchPath.length());
							
						}
						else if (m > -1) //wait until next loop (have to reassign indeces) 
						{ 
							launchPath = launchPath.substring(0, m) + launchPath.substring(m+1, launchPath.length());
							
						}	
						n = launchPath.indexOf("\\\\");
						m = launchPath.indexOf("\\\"");
					}
				}
				newLine = null; //don't update this
			}
			if(newLine!=null){
				if (bUseNativeRegistryInterface)
					setNativeRegistryValue(newLine);
				else
					outputFileContents.append(newLine + REGISTRY_FILE_NEW_LINE);
			}
		}

			
		
		if (bUseNativeRegistryInterface){
			launchPath = launchPath.replaceAll("%1", "");
			int n = launchPath.indexOf("\\\\");  //replaceAll doesn't work for these cases
			int m = launchPath.indexOf("\\\"");
			while (n > -1 || m > -1)
			{						
				if (n > -1)
				{
					launchPath = launchPath.substring(0, n) + launchPath.substring(n+1, launchPath.length());
					
				}
				else if (m > -1) //wait until next loop (have to reassign indeces) 
				{ 
					launchPath = launchPath.substring(0, m) + launchPath.substring(m+1, launchPath.length());
					
				}	
				n = launchPath.indexOf("\\\\");
				m = launchPath.indexOf("\\\"");
			}
		}
	
		
		try
		{
			if (!bUseNativeRegistryInterface){
				createRegSettingsFile(outputFileContents);
				pushRegFile(tempSettingsFilePath);
			}
		}
		catch (Exception e)
		{
				String myError = TestCorePluginResourceBundle.RecorderClient_INTERNET_EXPLORER_ERROR;
				throw new RecorderApplicationAdapterException(myError + "\n" + e.getMessage());
		}
		
	}
	
	private void checkOSVersion(){
		String currentOS = Platform.getOS();
		if (currentOS.equals("win32")) {
			String myVersion = System.getProperty("os.version");
			if (myVersion.startsWith("6")) {
				try{
					// Bugilla 168597, use native registry calls 
					// instead of displaying an error message
					bRunningVista = true;
					jniWrapper = new JNIWrapper();
					if ((jniWrapper.isProcessRunningElevated()) == false)
						bUseNativeRegistryInterface = true;
				}
				// so, if we can't load our DLL, we will throw the old
				// error message out so the user can perform a recording by
				// running eclipse using "Run as Administrator
				// this should never happen 
				catch (UnsatisfiedLinkError e2){
					StringBuffer inputFileContents = new StringBuffer();					
					inputFileContents.append("VISTA_ADMIN_ERROR");
				}
			}
		}
		
	}	
	
	private void setNativeRegistryValue(String newLine){
		if (bUseNativeRegistryInterface){
			int index1 = 1;  // next character after beginning quote
			int index2 = newLine.indexOf("=");
			String keyname = newLine.substring(index1, index2 - 1);
			String value = newLine.substring(index2 + 1);
			if (value.charAt(0) == '\"')
				value = value.substring(1);
			if (value.endsWith("\""))
				value = value.substring(0, value.length() - 1);

	        //The DefaultConnectionSettings and SavedLegacySettings registry keys 
			//(HKEY_USERS\<UUID>\Software\Microsoft\Windows\CurrentVersion\Internet Settings\Connections\) 
			//cache the connection configuration used by Internet Explorer (DefaultConnectionSettings) and 
			//other network connections (SavedLegacySettings) on Windows Vista/7.   
			//NOTE: The below keys must be set AFTER the above keys since their values are derived from the 
			//values of the above keys.  
			if((keyname.contains("DefaultConnectionSettings")) || (keyname.contains("SavedLegacySettings"))){ //$NON-NLS-1$ //$NON-NLS-2$
				jniWrapper.setInternetConnectionsSetting(keyname, value);
			}
			else{
				jniWrapper.setInternetSetting(keyname, value);				
			}
		}
	}

	/*
	 * Bugzilla 114253  - move location of temp files
	 * set the Paths for the temporary files used when manipulating the Windows
	 *  registry settings
	 *  Mark D. Dunn  1/6/2006
	 *  
	 */
	private void setTempFilePaths() {
		
		String tempPath = getTempPath();
		originalSettingsFilePath = tempPath + "\\" + originalSettingsFileName;
		tempSettingsFilePath = tempPath + "\\" + tempSettingsFileName;
		consoleSettingsFilePath = tempPath + "\\" + consoleSettingsFileName;
	}

	/*
	 * Use the location of the workspace to place the temporary files needed to manipulate
	 * the Windows registry elements for internet explorer
	 * 
	 * Mark D. Dunn 1/6/2006
	 * @return String that is the path to the workspace
	 */
	public String getTempPath()
	{
		String theTmpPath = "";
		IPath baseLoc = Platform.getLocation();
		String newtpath = baseLoc.toOSString();
				
		theTmpPath = newtpath;
		return theTmpPath;
	}

	/**
	 * causes a registry file to be pushed into the registry using regedit.
	 * @param String fileName
	 */
	private void pushRegFile(String fileName) throws InterruptedException, IOException
	{
		File file = new File(fileName);
		if (!file.exists())
			return;
		// bugzilla 127536 mark d dunn put quotes around file name due to spaces in workspace path
		String pushFileCommandLine = "regedit /s "+"\"" + fileName + "\"";//$NON-NLS-1$
		try {
		Process pusher = Runtime.getRuntime().exec(pushFileCommandLine,null);
		pusher.waitFor();
		}
		catch (InterruptedException e) {
			//Ignore.
		}
		
	}

	/**
	 * method which creates a registry settings file (*.reg)
	 * @param StringBuffer ouputFileContents
	 */
	private void createRegSettingsFile(StringBuffer outputFileContents)
	{
		// Bugzilla 114253 - use new path for temp files to put them in workspace
		File newFile = new File(tempSettingsFilePath);
				
		try
		{
			OutputStreamWriter outputWriter = new OutputStreamWriter(new FileOutputStream(newFile),"UnicodeLittle");//$NON-NLS-1$
			outputWriter.write(outputFileContents.toString());
			outputWriter.close();
		}
		catch (UnsupportedEncodingException e1)
		{
			e1.printStackTrace();
		}
		catch (FileNotFoundException e1)
		{
			e1.printStackTrace();
		} 
		catch (IOException e)
		{
			e.printStackTrace();
		}
		
	}

	/**
	 * method which creates a registry settings file (*.reg)
	 * @param StringBuffer ouputFileContents
	 */
	private void createOriginalRegSettingsFile(StringBuffer inputFileContents)
	{
		// Bugzilla 114253 - use new path for temp files to put them in workspace
		File newFile = new File(originalSettingsFilePath);
				
		try
		{
			OutputStreamWriter outputWriter = new OutputStreamWriter(new FileOutputStream(newFile),"UnicodeLittle");//$NON-NLS-1$
			outputWriter.write(inputFileContents.toString());
			outputWriter.close();
		}
		catch (UnsupportedEncodingException e1)
		{
			e1.printStackTrace();
		}
		catch (FileNotFoundException e1)
		{
			e1.printStackTrace();
		} 
		catch (IOException e)
		{
			e.printStackTrace();
		}
		
	}
	

	/**
	 * method which exports the registry keys of interested into a file using regedit and then reads the file
	 * @return StringBuffer
	 */
	private StringBuffer getRegistryInfo()
	{	
		

		String[] regInfo = {"\"HKEY_CLASSES_ROOT\\Applications\\iexplore.exe\\shell\\open\\command\"",
				"\"HKEY_CURRENT_USER\\Software\\Microsoft\\Windows\\CurrentVersion\\Internet Settings\"",
				"\"HKEY_CURRENT_USER\\Software\\Microsoft\\Java VM\""};
		// bugzilla 114253 - write temp files in workspace instead of eclipse instance location
		// use the new path to the temp files 
		//	 bugzilla 127536 mark d dunn put quotes around file name due to spaces in workspace path
		String cmdLine = "regedit /e "+"\"" + originalSettingsFilePath+ "\" ";//$NON-NLS-1$//$NON-NLS-2$
		String consolecmdLine = "regedit /e "+"\""+consoleSettingsFilePath+ "\" ";
		
		jniWrapper = new JNIWrapper();

		InputStreamReader inputReader=null;
		File file = null;
		File consoleFile = null;
		Process registryChange = null;
		StringBuffer inputFileContents = new StringBuffer();
		
		try
		{
			// bugzilla 114253 - write temp files in workspace instead of eclipse instance location
			// use the new path for the temp files 
			file = new File(originalSettingsFilePath);
			if (file.exists())
				file.delete();
			consoleFile = new File(consoleSettingsFilePath);
			if (consoleFile.exists())
				consoleFile.delete();

			for (int i = 0; i < regInfo.length; i++)
			{
				if (i == 2  && bRunningVista == false){  // the Java command console registry setting is not available on Vista
					
					registryChange = Runtime.getRuntime().exec(consolecmdLine + regInfo[i],null);	
					
					registryChange.waitFor();
					
					//Since the regedit process may terminate before the exported registry file is created,
					//check if the file is created every 200 ms to a maximum of 1 second:
					for (int counter = 0; counter < 5; counter++) {

						if(consoleFile.exists()){
							break;
						}
						else{
							
							try {
								Thread.sleep(200);
							} 
							catch (InterruptedException e) {
								//Ignore and continue.
							}
						}
					}
				} 
				else {
				
					registryChange = Runtime.getRuntime().exec(cmdLine + regInfo[i],null);

					registryChange.waitFor();
					
					//Since the regedit process may terminate before the exported registry file is created,
					//check if the file is created every 200 ms to a maximum of 1 second:
					for (int counter = 0; counter < 5; counter++) {

						if(file.exists()){
							break;
						}
						else{
							
							try {
								Thread.sleep(200);
							} 
							catch (InterruptedException e) {
								//Ignore and continue.
							}
						}
					}
				}
							
				if (i == 2  && bRunningVista == false){
					// the Java command console registry setting is not available on Vista
					// so this was throwing a File I/O exception
					inputReader= new InputStreamReader(new FileInputStream(consoleFile),"UTF-16");//$NON-NLS-1$
				} else 
				{
					inputReader= new InputStreamReader(new FileInputStream(file),"UTF-16");//$NON-NLS-1$
				}
				char[] inputBuf = new char[1000];
				int bytesRead = inputReader.read(inputBuf);
									
				while(bytesRead!=-1)
				{				
					inputFileContents.append(inputBuf,0,bytesRead);
					bytesRead = inputReader.read(inputBuf);
					
				}	
				inputReader.close();
			}
		}
		catch (IOException e)
		{
			TestCorePlugin.getDefault().logError(e);
		} 
		catch (InterruptedException e)
		{
			TestCorePlugin.getDefault().logError(e);
		}
		finally
		{
			if(inputReader!=null)
				try
				{
					inputReader.close();
				}
				catch (IOException e1)
				{
					//Ignore since closing input stream reader.
				}
		}
		return inputFileContents;
	}
	//Bugilla 168597
	StringBuffer getRegistryInfoUsingNativeAPI(){
		bUseNativeRegistryInterface = true;
		
		StringBuffer inputFileContents = new StringBuffer();		
		// fire up an instance of the JNI layer to start using native code
		launchPath = jniWrapper.getIEPath();
		
		AddSettingToBuffer(jniWrapper, inputFileContents, "ProxyEnable"); //$NON-NLS-1$
		AddSettingToBuffer(jniWrapper, inputFileContents, "AutoConfigURL"); //$NON-NLS-1$
		AddSettingToBuffer(jniWrapper, inputFileContents, "ProxyServer"); //$NON-NLS-1$
		AddSettingToBuffer(jniWrapper, inputFileContents, "ProxyOverride"); //$NON-NLS-1$
		AddSettingToBuffer(jniWrapper, inputFileContents, "ProxyHttp1.1"); //$NON-NLS-1$
		AddSettingToBuffer(jniWrapper, inputFileContents, "WarnOnBadCertRecving"); //$NON-NLS-1$
		AddSettingToBuffer(jniWrapper, inputFileContents, "WarnOnZoneCrossing"); //$NON-NLS-1$

        //The DefaultConnectionSettings and SavedLegacySettings registry keys 
		//(HKEY_USERS\<UUID>\Software\Microsoft\Windows\CurrentVersion\Internet Settings\Connections\) 
		//cache the connection configuration used by Internet Explorer (DefaultConnectionSettings) and 
		//other network connections (SavedLegacySettings) on Windows Vista/7.   
		//NOTE: The below keys must be set AFTER the above keys since their values are derived from the 
		//values of the above keys.  
		AddConnectionsSettingToBuffer(jniWrapper, inputFileContents, "DefaultConnectionSettings"); //$NON-NLS-1$
		AddConnectionsSettingToBuffer(jniWrapper, inputFileContents, "SavedLegacySettings"); //$NON-NLS-1$
		
		// left out ZonesMixedContent for now, that would require reading many more keys
		// to set all that
		
		return inputFileContents;
	}
	
	private void AddSettingToBuffer(JNIWrapper jniWrapper, StringBuffer inputFileContents, String name){
		String setting = jniWrapper.getInternetSetting(name);
		if (setting != null  && setting.length() > 0){
			inputFileContents.append("\"" + name + "\"=" + setting + REGISTRY_FILE_NEW_LINE); //$NON-NLS-1$ //$NON-NLS-2$
		}
		return;
	}
			
	private void AddConnectionsSettingToBuffer(JNIWrapper jniWrapper, StringBuffer inputFileContents, String name){
		String setting = jniWrapper.getInternetConnectionsSetting(name);
		if (setting != null  && setting.length() > 0){
			inputFileContents.append("\"" + name + "\"=" + setting + REGISTRY_FILE_NEW_LINE); //$NON-NLS-1$ //$NON-NLS-2$
		}
		return;
	}
	
	/**
	 * method called by the recorder to start the target application (Internet Explorer)
	 * @return Process
	 */
	public Process start() throws RecorderApplicationAdapterException 	
	{
		ieProcess = null;
		try
		{
			String strStart = launchPath + " " + startupURL;//$NON-NLS-1$
			//TODO derive path to iexplore.exe using registry.
			if (bUseNativeRegistryInterface == true)
				jniWrapper.startIE(startupURL);
			else
				ieProcess = Runtime.getRuntime().exec(strStart);
		}
		catch (IOException e)
		{
			throw new RecorderApplicationAdapterException("unable to start Internet Explorer Process: "+e.getMessage());//$NON-NLS-1$
		}
		return ieProcess;
	}

	
	/**
	 * method called to restore the process back to its original state
	 */
	public void cleanup()throws RecorderApplicationAdapterException
	{
		try
		{
			// bugzilla 114253 - write temp files in workspace instead of eclipse instance location
			// use the new path for the temp files
			if (bUseNativeRegistryInterface){
				try {
				StringBuffer inputFileContents = new StringBuffer();
				String str;
				byte [] buffer = new byte[4096];
				int bytes_read;
				File newFile = new File(originalSettingsFilePath);				
				FileInputStream inputStream = new FileInputStream(newFile);//$NON-NLS-1$
				while ((bytes_read = inputStream.read(buffer)) != -1){
					str = new String(buffer, 0, bytes_read, "UnicodeLittle");
					inputFileContents.append(str);
				}
				inputStream.close();				
				Iterator<String> registryFileEntriesIterator = parseRegistryFileContents(inputFileContents).iterator();

				while(registryFileEntriesIterator.hasNext()){
					
					String line = registryFileEntriesIterator.next();
					// check to see if it is a real setting or a blank line
					int index = line.indexOf('=');
					if (index > 0)
						setNativeRegistryValue(line);
				}
				}
				catch (IOException eio){
					// file not found is ok because cleanup gets called twice if stopping from the GUI
					// rather than by closing IE.
				}
								
			}
			else{
				pushRegFile(originalSettingsFilePath);
				
				pushRegFile(consoleSettingsFilePath);
			}
		}
		catch (Exception e)
		{
			throw new RecorderApplicationAdapterException(e.getMessage());
		}
		// bugzilla 114253 - write temp files in workspace instead of eclipse instance location
		// use the new path for temp files 
		File origfile = new File(originalSettingsFilePath);
		if (origfile.exists())
			origfile.delete();
		File regfile = new File(tempSettingsFilePath);
		if (regfile.exists()) 
			regfile.delete();
		File consolefile = new File(consoleSettingsFilePath);
		if (consolefile.exists())
			consolefile.delete();
	}

	/**
	 * method called to stop the target process.
	 */
	public void stop() throws RecorderApplicationAdapterException
	{
		if (ieProcess == null){
			if (jniWrapper != null)
				jniWrapper.terminateIE();
			
		}else{
			ieProcess.destroy();
		}
	}
	
	//init format:
	//param,value;param,value
	//Params:
	//  port - proxy port
	//  startupURL - page to launch IE with (file protocol only)
	private void parseInitInfo(String initInfo)
	{
		portnum = "";
		startupURL = "";
		StringTokenizer tok = new StringTokenizer(initInfo, ";");
		while (tok.hasMoreTokens())
		{
			String curTokn = tok.nextToken();
			StringTokenizer tok2 = new StringTokenizer(curTokn, ",");
			String key = tok2.nextToken();
			String val = tok2.nextToken();
			if (key.equals("port"))
			{
				portnum = val;
				try {
					Integer.parseInt(portnum);
				}
				catch (NumberFormatException e) {
					//error condition, should maybe just exit? 
					portnum = "443"; 
				}
			}
			else if (key.equals("startupURL"))
				startupURL = val; 
		}
	}
	private boolean checkProxyEnabled(StringBuffer inputBuffer) 
	{
			boolean comparesOK = false;
			Iterator<String> registryFileEntriesIterator = parseRegistryFileContents(inputBuffer).iterator();
			
			while(registryFileEntriesIterator.hasNext()){
				
				String line = registryFileEntriesIterator.next();
				String newLine = line;
				// example ProxyServer is key, "http=markdunn:1000;socks=localhost:450"
				if(line.startsWith("\"ProxyEnable"))//$NON-NLS-1$
				{
					StringTokenizer lineTokenizer = new StringTokenizer(line,":");//$NON-NLS-1$
					newLine=lineTokenizer.nextToken();  // get ProxyEnable
					newLine=lineTokenizer.nextToken();
					if (newLine.equalsIgnoreCase("00000001"))
						comparesOK = true;
				
					break;				
				}
			}
		return comparesOK;
	}
//	private String getProxyInfo(StringBuffer inputBuffer) 
//	{
//		String proxyString = "";
//		String httpString = "http=";
//	    Iterator<String> registryFileEntriesIterator = parseRegistryFileContents(inputBuffer).iterator();
//	    while(registryFileEntriesIterator.hasNext()){
//		
//		    String line = registryFileEntriesIterator.next();
//			// example ProxyServer is key, "http=markdunn:1000;socks=localhost:450"
//			if(line.startsWith("\"ProxyServer"))//$NON-NLS-1$
//			{
//				String tmp = "\"ProxyServer\"=";
//				String modifiedLine = line.substring(tmp.length());
//				String [] entireString = modifiedLine.split(";");
//				int numVars = entireString.length;
//				for (int i =0; i < numVars; i++){
//					String tmpEntireString = entireString[i].replaceAll("\"","");
//					int findIndex = tmpEntireString.indexOf(httpString);
//					if (findIndex != -1) {
//						// found the revisedString
//						String tmpVal = "";
//						tmpVal = tmpEntireString.substring(findIndex+httpString.length());
//						proxyString = tmpVal;
//					}
//				}
//			}
//		}
//		return proxyString;
//	}
	// defect 78895 mdd 12/08/2004
	private String disableLine(String origLine) {
		String returnLine = "";
		String newLine = "";
		StringTokenizer lineTokenizer = new StringTokenizer(origLine,":");//$NON-NLS-1$
		newLine=lineTokenizer.nextToken();
		newLine+=":00000000";//$NON-NLS-1$
		returnLine = newLine;
		
		return returnLine;
	}
	private String enableConsoleLine(String origLine) {
		String returnLine = "";
		String newLine = "";
		StringTokenizer lineTokenizer = new StringTokenizer(origLine,":");//$NON-NLS-1$
		newLine=lineTokenizer.nextToken();
		newLine+=":01,00,00,00";//$NON-NLS-1$
		returnLine = newLine;
		
		return returnLine;
	}
	private String enableZonesLine(String origLine) {
		String returnLine = "";
		String newLine = "";
		StringTokenizer lineTokenizer = new StringTokenizer(origLine,":");//$NON-NLS-1$
		newLine=lineTokenizer.nextToken();
		newLine+=":00000000";//$NON-NLS-1$
		returnLine = newLine;

		return returnLine;
	}

	/**
	 * stores the initialization string which will be passed to the agent side adapter object at runtime
	 */
	private String initString = null;
	
	
	/**
	 * called by the recorder to acquire the adapters init string for passing to the agent side of the rac
	 */
	public String getInitString()
	{
		return initString;
	}

	
	/**
	 * used for external objects to set the adapters initialization String
	 * @param String initString - a semi-colon delimeted collection of comma delimeted name-value pairs (host,localhost;port,450)
	 */
	public void setInitString(String initString)
	{
		this.initString = initString;
	}

	/* (non-Javadoc)
	* @see org.eclipse.hyades.execution.recorder.local.appadapters.IRecorderApplicationAdapter#getDefaultApplicationPath()
	*/
	public String getDefaultApplicationPath() {
		return launchPath;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.recorder.local.appadapters.IRecorderApplicationAdapter#getApplicationPath()
	 */
	public String getApplicationPath() {
		String str = TestCorePlugin.getDefault().getPluginPreferences().getString(PREFERENCE_PATH);
		if (str == null || str.length() == 0){
			return getDefaultApplicationPath();
		}
		else{
			return str;
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.recorder.local.appadapters.IRecorderApplicationAdapter#setApplicationPath(java.lang.String)
	 */
	public void setApplicationPath(String str) {
		
		if ( (str != null) && str.length() > 0) {
			TestCorePlugin.getDefault().getPluginPreferences().setValue(PREFERENCE_PATH, str);
		}
		else {
			String newPath = getDefaultApplicationPath();
			TestCorePlugin.getDefault().getPluginPreferences().setValue(PREFERENCE_PATH, newPath);
		}
	}
	
	/**
	 * <p>Called by {@link org.eclipse.hyades.internal.execution.recorder.local.RecorderDataProcessorHelper} when the client connects, 
     * we will look for a new instance of IE if we are using the Native DLL code 
     * to control the recording.</p>
	 *
	 * @deprecated  This is internal API.
	 */  
	public void clientConnectNotification(){
		
		if (bUseNativeRegistryInterface  && jniWrapper != null){
			jniWrapper.clientConnectNotification();
		}
		
	}
	
	/**
	 * <p>Parses the registry file contents delimited by the {@link #REGISTRY_FILE_NEW_LINE} 
	 * characters into a list of key/value pairs.</p>
	 * 
	 * @param registryFileContents The registry file contents.
	 * @return A list of key/value pairs, otherwise an empty list.
	 * @see #REGISTRY_FILE_NEW_LINE
	 */
	private List<String> parseRegistryFileContents(StringBuffer registryFileContents){

		List<String> registryFileEntries = new ArrayList<String>();

		int currentIndex = 0;

		while(currentIndex < registryFileContents.length()){

			int delimiterIndex = registryFileContents.indexOf(REGISTRY_FILE_NEW_LINE, currentIndex); 
			
			if (delimiterIndex < 0){
				
				registryFileEntries.add(registryFileContents.substring(currentIndex));
				
				break;
			}
			else{
				
				registryFileEntries.add(registryFileContents.substring(currentIndex, delimiterIndex));
				
				currentIndex = (delimiterIndex + REGISTRY_FILE_NEW_LINE.length());
			}
		}

		return registryFileEntries;
	}
}
