/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RecorderClient.java,v 1.40 2009/10/27 19:44:41 paules Exp $
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.internal.execution.recorder.local;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Enumeration;
import java.util.Iterator;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.hyades.execution.core.IExecutableObject;
import org.eclipse.hyades.execution.core.IExecutionComponentFactory;
import org.eclipse.hyades.execution.core.IExecutionEnvironment;
import org.eclipse.hyades.execution.core.IRemoteHyadesComponent;
import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList;
import org.eclipse.hyades.execution.local.ExecutionComponentFactoryImpl;
import org.eclipse.hyades.execution.local.JavaProcessExecutableObjectStub;
import org.eclipse.hyades.execution.local.JavaProcessExecutorStub;
import org.eclipse.hyades.execution.local.NodeImpl;
import org.eclipse.hyades.execution.local.SessionStub;
import org.eclipse.hyades.execution.recorder.IRecorderListenerFullFeedback;
import org.eclipse.hyades.execution.recorder.MasterRecorderListener;
import org.eclipse.hyades.execution.recorder.Recorder;
import org.eclipse.hyades.execution.recorder.RecorderFactory;
import org.eclipse.hyades.execution.recorder.local.appadapters.IRecorderApplicationAdapter;
import org.eclipse.hyades.execution.recorder.local.appadapters.RecorderAppAdapterFactory;
import org.eclipse.hyades.execution.recorder.local.appadapters.RecorderApplicationAdapterException;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentControllerUnavailableException;
import org.eclipse.hyades.internal.execution.local.control.AgentListener;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NotConnectedException;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.TestCorePreferences;
import org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle;
import org.eclipse.osgi.util.NLS;
import org.eclipse.tptp.test.internal.recorder.framework.RecorderLoggingUtil;
import org.eclipse.tptp.test.provisional.recorder.framework.IRecorderClientHelper;

/**
 * <p>Recorder client responsible for the following</p>:
 * 
 * <ul>
 * <li>Creating and initializing a recorder session.</li>
 * <li>Creating and initializing an Agent Controller session.</li>
 * <li>Stopping the recorder session.</li>
 * <li>Shut-down and clean-up of the Agent Controller session.</li>
 * <li>Shut-down and clean-up of the recording session.</li>
 * </ul>
 * 
 * 
 * @author  Ernest Jessee
 * @author  Paul E. Slauenwhite
 * @version October 27, 2009
 * @since   April 22, 2005
 */
public class RecorderClient 
{	
	/**
	 * This is our handle to the remote Java process.  It is our abstraction of the Java Process 
	 */	
	private JavaProcessExecutorStub executor;

	/**
	 * The the object used to setup the execution environment which will be used 
	 * on the Agent Controller to run the recorder
	 */
	private IExecutionEnvironment exeEnvironment = null;

	/**
	 * The factory which manages the instances of Agent Controller execution infrastructure
	 * objects.
	 */
	private IExecutionComponentFactory factory=null;

	/**
	 * The session in which the recorder resides and performs it's duties.
	 */
	private ISession session=null;

	/**
	 * this object is the a reference to the execution object which runs on
	 * the Agent Controller and facilitates communication between the remote process and Eclipse.
	 * This could be called an agent.
	 */
	private IRemoteHyadesComponent remoteHyadesComponent=null;

	/**
	 * This is the object that actually runs on the Agent Controller. Although this object does not contain the 
	 * desired functionality, it knows how to start the java object that does contain the desired
	 * functionality.
	 */
	private IExecutableObject executableObject=null;

	/**
	 * The recorder that this client will spin up
	 */
	private Recorder theRecorder;
	
	/**
	 * The Agent object used to send control messages from eclipse to the Recorder as it 
	 * runs on the Agent Controller
	 */
	private Agent controlAgent = null;
	
	/**
	 * This object is constructed with an IRecorderDataProcessor and contains methods
	 * to facilitate data routing as it comes from the RecorderAgent.
	 * This object is registered with the execution infrastructure as the dataprocessor 
	 * rather than the IRecorderDataProcessor
	 */
	private RecorderDataProcessorHelper dataProcessorHelper;
	
	public IRecorderApplicationAdapter appAdapter = null;	
	
	/**
	 * object which watches the automatically started application for shutdown and performs cleanup.
	 */
	private RecorderStopper recorderStopper = null;
	
	private IProgressMonitor progressMonitor = new NullProgressMonitor();
	
	private boolean bStopCalled = false;

	private IRecorderClientHelper helper = null;
	
	/**
	 * Recorder deployment directory
	 */
	public final String RECORDER_DESTINATION_PATH="Recorder_Deploy";
	
	/**
	 * JAR containing the recorder framework
	 */
	private final String RECORDER_JAR_NAME = "hexrecr.jar";
	
	/**
	 * This thread object is started by the RecorderClient and waits for a complete 
	 * message to come from the Agent Controller upon receiving this message, the 
	 * scripts are generated if a script generator was specified.
	 * <p/>
	 * 
	 * 
	 * @author  Ernest Jessee
	 * @author  Paul E. Slauenwhite
	 * @version May 6, 2008
	 * @since   April 22, 2005
	 */
	private class FinishCleanupAndGenTestThread extends Thread {
		
		/**
		 * the constructor
		 */
		public FinishCleanupAndGenTestThread()
		{
			setName("Recording completion listener");			 //$NON-NLS-1$
		}
		
		/**
		 * the main run method of the thread.
		 */
		public void run()
		{				
			
			try {
				
				//wait for the dataProcessor to get a complete call
				while(!dataProcessorHelper.isRecordingComplete())
				{
					delay(200);
				}		
				if (helper != null)
					helper.preStopRecording();
				try  {
					stopApplication();
				} catch (Exception e) {}
				if (helper != null)
					helper.postStopRecording();
				//since we have the completion notification, we'll de-activate this recorder
				theRecorder.setActive(false);
				
				//and tell the user that the recorder is done.
				RecorderFactory.getInstance().controlMessage(theRecorder, IRecorderListenerFullFeedback.VIEW_MESSAGE,
						TestCorePluginResourceBundle.RecorderClient_RECORDING_COMPLETED_MESSAGE); 
	
				//now we need tdo seeif the user wants to generate the suite.  If he does, there will be a scriptgenID in the recorder object.
				//if he does not, the scriptgenid wil be -1.
			
				
				// First, write all Unwritten "CLOSE"   mdd
				
				//if we don't have a -1, then we'll kick off scriptgen
				if(!theRecorder.getScriptgenID().equalsIgnoreCase("-1")) //$NON-NLS-1$
				{
					//Testgen will cause TestHttp to be initialized.
					//if we testgen in a worker thread, when TestHttp is initialized, it will have no
					//concept of the current display (which it needs to load images)
					//Thus, we'll run this asynchronously in the UI thread
				}

				//Update the recorder status (Stopped) in the Recorder Control view:
				updateViewStatus(TestCorePluginResourceBundle.RecorderClient_STATUS_STOPPED);
				
				// bugzilla 139992 . check state of session before releasing it.
				// bugzilla 144444. Put session.release() back in mdd 06/01/2006
				session.release();
	
				//notify appropriate listeners to start test gen.
				
				Object[] objs = RecorderFactory.getInstance().getListenersForRecorder(theRecorder);
				for (int i = 0; i < objs.length; i++)
				{
				    if (objs[i] instanceof MasterRecorderListener)
				    {
				        ((MasterRecorderListener)objs[i]).startTestGen();
				        break; //only start test gen once.  
				    }
				}
	
				if (helper != null)
					helper.cleanup();
			} 
			catch (Exception e) {
				TestCorePlugin.getDefault().logError(e);
			}
		}

		private void delay(int miliseconds)
		{
			try
			{
				Thread.sleep(miliseconds);
			}
			catch (InterruptedException e)
			{
				e.printStackTrace();
			}
		}

	}
	
	
	/**
	 * This object is an agent listener to intercept messages coming across the data channel from the remote side
	 * @author Ernest Jessee
	 */
	private class RecorderAgentListener implements AgentListener
	{
			
		public void agentActive(Agent agent)
		{
		}

		
		public void agentInactive(Agent agent)
		{

		}

		
		public void error(Agent agent, String errorId, String errorMessage)
		{
//			System.out.println("received Error: "+errorMessage);
		}

		
		public void handleCommand(Agent agent, CommandElement command)
		{
			if(command instanceof CustomCommand)
			{
				String errorMessage=((CustomCommand)command).getData();
				System.out.println("recieving message: "+errorMessage); //$NON-NLS-1$
			}	

		}

	}	
	
	/**
	 * This is where the RecorderClient actually instantiates all the objects it needs and initiates a recording on the Agent Controller
	 */
	public void run(final Recorder recorder, IProgressMonitor monitor) throws IllegalAccessException, InstantiationException, IOException 
	{					
		
		theRecorder = recorder;
		progressMonitor = monitor;
	    bStopCalled = false;

	    try{			
	    	
	    	//Reset the KBytes Recorded field in the Recorder Control View to '0':
	    	updateKBytesReceived(0);

			//Update the recorder status (Initializing...) in the Recorder Control view:
	    	updateViewStatus(TestCorePluginResourceBundle.RecorderClient_STATUS_INITIALIZING);

			//Set the task and number of work units (e.g. calls to tickProgressDialog(String) or sendStatusMessageToControlView(String)) until the progress monitor is terminated: 
	    	progressMonitor.beginTask(TestCorePluginResourceBundle.RecorderClient_PROGRESS_TITLE, 23);		

	    	//Step 1: Start the recording:
			//Add a status message (Recording started) to the status pane in the Recorder Control view:
	    	sendStatusMessageToControlView(TestCorePluginResourceBundle.RecorderClient_RECORDING_STARTED_STATUS_MESSAGE); 

	    	//Check if the recorder has been canceled:
	    	if(monitor.isCanceled()){

	    		cancelRecording();				

	    		return;
	    	}
	    	
			//Initialize the recorder helper:
	    	helper = RecorderFactory.getInstance().getRecorderClientHelper(theRecorder.getId(), false);

	    	//By default, DefaultRecorderClientHelper.okToLaunch() returns true:  
	    	if ((helper != null) && (!helper.okToLaunch())){
	    		return;
	    	}
	    	
	    	if (helper != null){
	    		
	    		helper.setRecorder(theRecorder);				
	    		helper.preSetup();
	    	}
	    	
	    	//Step 2: Connect to the Agent Controller:
			String raserverHost = getRAServerHost();
			String raserverPort = getRAServerPort();
			
			//Add a status message (Connecting to Agent Controller on host <host> and port <port>) to the status pane in the Recorder Control view:
			sendStatusMessageToControlView(NLS.bind(TestCorePluginResourceBundle.RecorderClient_CONNECTING_TO_RASERVER_STATUS_MESSAGE, raserverHost, raserverPort)); 
			
			//Check if the recorder has been canceled:
	    	if(monitor.isCanceled()){

	    		cancelRecording();				

	    		return;
	    	}
	    	
			session = new NodeImpl(raserverHost).connect(raserverPort, null);	

	    	//Resolve the session's ExecutionComponentFactory:
	    	factory = ExecutionComponentFactoryImpl.getInstance(session);

	    	//Step 3: Deploy the required recorder files:	    	
			//Add a status message (Deploying required recorder files) to the status pane in the Recorder Control view:
    		sendStatusMessageToControlView(TestCorePluginResourceBundle.RecorderClient_DEPLOYING_FILES);

    		//Check if the recorder has been canceled:
	    	if(monitor.isCanceled()){

	    		cancelRecording();				

	    		return;
	    	}
	    	
	    	try {
	    		
	    		//[0] = source files
	    		//[1] = destination files
	    		FileIdentifierList[] fileIdentifiers = getRecorderJars();
	    				    	
	    		IFileManagerExtended manager = ((IFileManagerExtended)(((NodeImpl)(session.getNode())).getFileManager()));
	    		manager.putFile(fileIdentifiers[0], fileIdentifiers[1]);

		    	//Add the deployment paths to the execution environment:
		    	exeEnvironment = setupJavaEnvironment(getDestinationAbsolutePath(fileIdentifiers[1]));
	    	}
	    	catch (NullPointerException e){
	    		throw new FileNotFoundException(TestCorePluginResourceBundle.RecorderClient_CANT_FIND_RECORDER_JARS_TO_DEPLOY);
	    	}
	    	
			//Increment the progress monitor:
	    	tickProgressDialog(null);

	    	//Add the execution environment to the Agent Controller session:
	    	session.addChild(exeEnvironment);

			//Increment the progress monitor:
	    	tickProgressDialog(TestCorePluginResourceBundle.RecorderClient_RECORDING_STARTED_STATUS_MESSAGE);

	    	//Create an executor:
	    	executor = setupExecutor();

			//Increment the progress monitor:
	    	tickProgressDialog(null);

	    	//Setup and start monitoring the recorder agent:
	    	remoteHyadesComponent = setupAgent();
	    	dataProcessorHelper = new RecorderDataProcessorHelper(theRecorder.getDataProcessor());
	    	remoteHyadesComponent.startMonitoring(dataProcessorHelper);

			//Increment the progress monitor:
	    	tickProgressDialog(null);

	    	//Setup the executable object:
	    	executableObject = setupHarnessAgentExecutableObject();

			//Increment the progress monitor:
	    	tickProgressDialog(null);

	    	//Add the executor to the execution environment:
	    	exeEnvironment.addChild(executor);

			//Increment the progress monitor:
	    	tickProgressDialog(null);

	    	//Initialize the recorder agent's data processor:
	    	theRecorder.getDataProcessor().initialize();

	    	//Set the executable object on the executor:	
	    	executor.setExecutableObject(executableObject);

	    	//Add the recorder agent to the executor:	
	    	executor.addChild(remoteHyadesComponent);

			//Increment the progress monitor:
	    	tickProgressDialog(null);
	    	
	    	//Launch the executor:
	    	executor.launch();

			//Increment the progress monitor:
	    	tickProgressDialog(null);			

			//Step 4: Start recording:
	    	//Check if the recorder has been canceled:
	    	if(monitor.isCanceled()){

	    		cancelRecording();				

	    		return;
	    	}

	    	//Pre-start the recording:
	    	if (helper != null){
	    		helper.preStartRecording();
	    	}
			
	    	//Initialize the recorder agent:
			sendStatusMessageToControlView(TestCorePluginResourceBundle.RecorderClient_SENDING_COMMAND_STATUS_MESSAGE + " Init"); //$NON-NLS-1$
			sendMessage("Init " + theRecorder.getConfigParams()); //$NON-NLS-1$
			
	    	//Start the recorder agent:
			sendStatusMessageToControlView(TestCorePluginResourceBundle.RecorderClient_SENDING_COMMAND_STATUS_MESSAGE + " StartRecording"); //$NON-NLS-1$
			sendMessage("StartRecording"); //$NON-NLS-1$	
			
			String appAdapterAgentClasspath = null;
			String appAdapterInitString = "-1";
			String appAdapter = theRecorder.getApplicationAdapterID();
			
			if (!(appAdapter.equalsIgnoreCase("-1"))){			
				
				appAdapterAgentClasspath = RecorderAppAdapterFactory.getInstance().getAgentClasspath(theRecorder.getApplicationAdapterID());
				appAdapterInitString = RecorderAppAdapterFactory.getInstance().getAppAdapter(theRecorder.getApplicationAdapterID()).getInitString();
			}

			//Start the recording:
			if((appAdapterAgentClasspath != null) && (!appAdapterAgentClasspath.equals("-1"))){ //$NON-NLS-1$			
			
				sendStatusMessageToControlView(TestCorePluginResourceBundle.RecorderClient_SENDING_COMMAND_STATUS_MESSAGE + " StartApp '" + RecorderAppAdapterFactory.getInstance().getAdapterName(appAdapter) + "'"); 
				startApplication(RecorderAppAdapterFactory.getInstance().getAdapterConfigElement(theRecorder.getApplicationAdapterID()), appAdapterInitString);
			}
			else{
				sendStatusMessageToControlView(TestCorePluginResourceBundle.RecorderClient_MANUAL_START_APP);
			}

			//Post-start the recording:
	    	if (helper != null){
	    		helper.postStartRecording();
	    	}
	    	
			//Terminate the process monitor:
	    	progressMonitor.done();
	    	
			//Update the recorder status (Recording...) in the Recorder Control view:
	    	updateViewStatus(TestCorePluginResourceBundle.RecorderClient_STATUS_RECORDING);

	    	//Start the clean-up and test generation thread:
	    	new FinishCleanupAndGenTestThread().start();

	    }
	    catch(Exception e){
	    	
	    	if ((e.getMessage() != null) && (e.getMessage().length() > 0)){
	    		sendStatusMessageToControlView(e.getMessage());
	    	}
	    	else{
	    		sendStatusMessageToControlView(TestCorePluginResourceBundle.RecorderClient_START_RECORDING_ABORTED_DUE_TO_EXCEPTION_STATUS_MESSAGE+e.getClass().getName()); 
	    	}
	    	
	    	//Terminate the process monitor:
	    	progressMonitor.done();

	    	//Update the recorder status (Stopped) in the Recorder Control view:
	    	updateViewStatus(TestCorePluginResourceBundle.RecorderClient_STATUS_STOPPED);
	    	
	    	String cause = e.getMessage();
	    	if (cause == null)
	    		cause = e.getClass().getName();

	    	//handle particular errors & give more detailed error messages
	    	if (e instanceof AgentControllerUnavailableException)
	    	{
	    		String[] strs = new String[2];
	    		strs[0] = getRAServerHost();
	    		strs[1] = getRAServerPort();
	    		cause = NLS.bind(TestCorePluginResourceBundle.RecorderClient_RAC_UNAVAILABLE_MESSAGE, strs);
	    		cause += "\r\n";
	    		cause += TestCorePluginResourceBundle.RecorderClient_RAC_UNAVAILABLE_MESSAGE_DESC1;
	    		sendStatusMessageToControlView(NLS.bind(TestCorePluginResourceBundle.RecorderClient_RAC_UNAVAILABLE_MESSAGE, strs)); 

	    	}
	    	// bugzilla 126979 mdd 04/25/2006
	    	if (cause.startsWith("Unable to find control agent")){
	    		cause += "\n ";
	    		cause += TestCorePluginResourceBundle.RecorderClient_RECORDER_JARS_NOT_DEPLOYED;
	    		sendStatusMessageToControlView(TestCorePluginResourceBundle.RecorderClient_RECORDER_JARS_NOT_DEPLOYED);
	    	}				
	    	RecorderFactory.getInstance().reportExceptionToUser(theRecorder, e,TestCorePluginResourceBundle.RecorderClient_START_RECORDING_ABORTED,cause,TestCorePluginResourceBundle.RecorderClient_RECORDER_ERROR); 
	    	RecorderLoggingUtil.severe(TestCorePluginResourceBundle.RecorderClient_START_RECORDING_ABORTED, this.getClass(), e);

	    	//TODO: Logging
	    	if(session!=null)
	    		session.release();

	    	theRecorder.setActive(false);
	    }
	}

	/**
	 * A method for sending a message to the Recorder control view.  Also updates progress dialog
	 * @param String message
	 */
	private void sendStatusMessageToControlView(final String message){
		
		RecorderFactory.getInstance().controlMessage(theRecorder, IRecorderListenerFullFeedback.VIEW_MESSAGE, message);

		//Increment the progress monitor:
		tickProgressDialog(message);
	}
		
	/**
	 * <p>Cancels the recording.</p>
	 * 
	 * <p>Note: This method should only be invoked when the recorder is initialzing.  That  
	 * is, the status of the recorder is "Initializing..." (see {@link TestCorePluginResourceBundle#RecorderClient_STATUS_INITIALIZING}).</p>
	 * 
	 * @see TestCorePluginResourceBundle#RecorderClient_STATUS_INITIALIZING
	 * @see RecorderFactory#updateStatus(Recorder, String)
	 */
	private void cancelRecording(){
		
		//Release the session:
		if(session != null){
			session.release();
		}

		//De-activate the recorder:
		theRecorder.setActive(false);

		//Terminate the process monitor:
		progressMonitor.done();
		
		//Add a status message (Recording canceled) to the status pane in the Recorder Control view:
		RecorderFactory.getInstance().controlMessage(theRecorder, IRecorderListenerFullFeedback.VIEW_MESSAGE,
				TestCorePluginResourceBundle.RecorderClient_RECORDING_CANCELED_MESSAGE);
		
		//Update the recorder status (Stopped) in the Recorder Control view:
		updateViewStatus(TestCorePluginResourceBundle.RecorderClient_STATUS_STOPPED);	
	}
	
	/**
	 * <p>Stops the recording.</p>
	 * 
	 * <p>Note: This method should only be invoked when the recorder is recording.  That  
	 * is, the status of the recorder is "Recording..." (see {@link TestCorePluginResourceBundle#RecorderClient_STATUS_RECORDING}).</p>
	 * 
	 * @see TestCorePluginResourceBundle#RecorderClient_STATUS_RECORDING
	 * @see RecorderFactory#updateStatus(Recorder, String)
	 */
	public void stopRecording(){

		try {
			
			// stop can be called by the recorder stopper
			// or by clicking the stop button
			// do not cleanup twice, set the boolean true until
			// another recording is started
			if (!bStopCalled){
				bStopCalled = true;
				stopApplication();
			}
		}
		catch(Exception e) {
									
			//Release the session:
			if(session != null){
				session.release();
			}

			//De-activate the recorder:
			theRecorder.setActive(false);			
			
			//Terminate the process monitor:
			progressMonitor.done();
				
			sendStatusMessageToControlView(TestCorePluginResourceBundle.RecorderClient_STOP_RECORDING_ABORTED_DUE_TO_EXCEPTION_STATUS_MESSAGE+e.getClass().getName()); 

			RecorderFactory.getInstance().reportExceptionToUser(theRecorder, e,TestCorePluginResourceBundle.RecorderClient_STOP_RECORDING_ABORTED,
					e.getClass().getName(),TestCorePluginResourceBundle.RecorderClient_RECORDER_ERROR); 
		
		}
		
		sendMessage("Stop"); //$NON-NLS-1$			
	}

	/**
	 *sends a message across the data channel
	 */
	private void sendMessage(String message)
	{
		try {
			Thread.sleep(500);
		}
		catch (Exception e) {}
		//TODO: solve circular stop issue that requires this sleep. 
		
		if(controlAgent == null)
			initControlAgent();	
		
		CustomCommand command=new CustomCommand();
		command.setData(message);

		try
		{
			controlAgent.invokeCustomCommand(command);	
		}
		catch (InactiveAgentException e1)
		{				
			e1.printStackTrace();
		}
	}


	/**
	 *This method initiates the control agent reference that will be used to send control messages across the control channel to the remote
	 *side of the Agent Controller.  To accomplish this initialization, it prompts the executor for all process that are currently active.  IT then
	 *must look at each until it finds a process whose ID matches our processes ID.  
	 *This should only be done once and the reference is reused after that.
	 */
	private void initControlAgent()
	{
		String pid=executor.getPid();
		//String pid = executor.getUniqueId().toString();
		int trialCount=100;
		
		try 
		{
			boolean found=false;
			Node node=((SessionStub)executor.getSessionContext()).getAgent().getProcess().getNode();
	
			while(!found && trialCount>0) 
			{
				Enumeration processes=node.listProcesses();
		
				while(processes.hasMoreElements()) 
				{
					Process proc=(Process)processes.nextElement();
					if(proc.getProcessId().equals(pid)) 
					{
						found=true;
						synchronized(proc) 
						{
							Enumeration agents=proc.getAgentsByType("tester"); //$NON-NLS-1$
							while(agents.hasMoreElements()) 
							{
								controlAgent = (Agent)agents.nextElement();
								try 
								{
									controlAgent.addAgentListener(new RecorderAgentListener());																				
									controlAgent.attach();
									break;								
								}
								catch(InactiveAgentException e) 
								{									
								}
							
							}
						}
					}						
					trialCount--;
				}
			}
		
		}
		catch(InactiveProcessException e) 
		{
			
		}
		catch(NotConnectedException e) {
			
		}
	}

	/**
	 * adds the appropriate Executable Object triple to the factory and then pulls an instantiation from the factory.
	 */
	private IExecutableObject setupHarnessAgentExecutableObject()throws ClassNotFoundException
	{
		factory.addExecutionComponent("JAVA_EXECUTABLE","org.eclipse.hyades.execution.core.impl.JavaProcessExecutableObjectImpl"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addStub("JAVA_EXECUTABLE", "org.eclipse.hyades.execution.local.JavaProcessExecutableObjectStub"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addSkeleton("JAVA_EXECUTABLE", "org.eclipse.hyades.execution.remote.JavaProcessExecutableObjectSkeleton"); //$NON-NLS-1$ //$NON-NLS-2$
					
		IExecutableObject execObject = (IExecutableObject)executor.getCompatibleExecutableObject("JAVA_EXECUTABLE"); //$NON-NLS-1$
		
		theRecorder.getExecutableObjectAdapter().setupExecutableObject((JavaProcessExecutableObjectStub) execObject);
		
		return execObject;
	}

	/**
	 * adds the approprate remote execution component triple to the factory and pulls and instantiation from the factory.
	 */
	private IRemoteHyadesComponent setupAgent()	throws ClassNotFoundException
	{
		factory.addExecutionComponent("AGENT", "org.eclipse.hyades.execution.core.impl.JavaTaskRemoteHyadesComponentImpl"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addStub("AGENT", "org.eclipse.hyades.execution.local.JavaTaskRemoteHyadesComponentStub"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addSkeleton("AGENT", "org.eclipse.hyades.execution.remote.JavaTaskRemoteHyadesComponentSkeleton"); //$NON-NLS-1$ //$NON-NLS-2$
		return (IRemoteHyadesComponent)factory.createExecutionComponentByType("AGENT"); //$NON-NLS-1$
		
	}

	/**
	 * adds the appropriate executor triple to the factory and pulls and instantiation from the factory.
	 */
	private JavaProcessExecutorStub setupExecutor()	throws ClassNotFoundException
	{
		
		factory.addExecutionComponent("EXECUTOR", "org.eclipse.hyades.execution.core.impl.ProcessExecutorImpl"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addStub("EXECUTOR", "org.eclipse.hyades.execution.local.JavaProcessExecutorStub"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addSkeleton("EXECUTOR", "org.eclipse.hyades.execution.remote.JavaProcessExecutorSkeleton"); //$NON-NLS-1$ //$NON-NLS-2$
		
		/* create our executor */
		return (JavaProcessExecutorStub)factory.createExecutionComponentByType("EXECUTOR"); //$NON-NLS-1$
	}

	private String getRAServerHost()
	{
		return TestCorePlugin.getDefault().getPluginPreferences().getString(TestCorePreferences.DEFAULT_HOST_NAME);
	}
	
	private String getRAServerPort()
	{
		return TestCorePlugin.getDefault().getPluginPreferences().getString(TestCorePreferences.DEFAULT_PORT);
	}
	
	/**
	 * adds the appropriate environment triple to the factory and pulls and instantation from the factory.
	 * @return IExecutionEnvironment
	 */
	private IExecutionEnvironment setupJavaEnvironment(String deployPath) throws ClassNotFoundException, IOException
	{
		IExecutionEnvironment environment = null;
		factory.addExecutionComponent("ENVIRONMENT", "org.eclipse.hyades.execution.core.impl.ExecutionEnvironmentImpl"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addStub("ENVIRONMENT", "org.eclipse.hyades.execution.local.ExecutionEnvironmentStub"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addSkeleton("ENVIRONMENT", "org.eclipse.hyades.execution.remote.ExecutionEnvironmentSkeleton"); //$NON-NLS-1$ //$NON-NLS-2$
				
		environment =(IExecutionEnvironment)factory.createExecutionComponentByType("ENVIRONMENT"); //$NON-NLS-1$
				
		theRecorder.getAgentEnvironmentAdapter().setupExecutionEnvironment(environment, deployPath);
				
		return environment;
		
	}

	private void tickProgressDialog(final String msg)
	{
	    if (msg != null && msg.length() > 0)
	    {
	        progressMonitor.subTask(msg);
	    }
		progressMonitor.worked(1);
	}
	
	private void startApplication(IConfigurationElement adapterElement, String adapterInitString) throws InstantiationException, IllegalAccessException, ClassNotFoundException, RecorderApplicationAdapterException, CoreException 
	{
		appAdapter = (IRecorderApplicationAdapter)adapterElement.createExecutableExtension("class");		

		appAdapter.init(adapterInitString);
		
		java.lang.Process process = appAdapter.start();

		
		recorderStopper = new RecorderStopper(process,this);
		recorderStopper.start();
	
		
	}
	
	private void stopApplication() throws RecorderApplicationAdapterException
	{
		if(appAdapter!=null )
		{
			appAdapter.stop();
			appAdapter.cleanup();
			
			//just in case the process stopper is somehow still running, we will interrupt.
			if (recorderStopper.isActive())
				recorderStopper.interrupt();			
		}
	}
	
	/**
	 * Updates the value of the KBytes Recorded field in the 
	 * Recorder Control View to zero or more kilobytes.
	 * 
	 * @param kBytesReceived Zero or more kilobyte.
	 */
	private void updateKBytesReceived(int kBytesReceived ){
		
		if(kBytesReceived >= 0){
			RecorderFactory.getInstance().controlMessage(theRecorder, IRecorderListenerFullFeedback.KB_RECEIVED, String.valueOf(kBytesReceived));
		}
	}
	
	private void updateViewStatus(final String status)
	{
		RecorderFactory.getInstance().updateStatus(theRecorder, status);
	}

	public void annotateRecording(String msg)
	{
	    sendMessage("Annotate " + msg); //$NON-NLS-1$
	}

	private FileIdentifierList[] getRecorderJars() {

		//get additional JARs from environment adapter
		java.io.File[] srcFiles = theRecorder.getAgentEnvironmentAdapter().getRecorderJARs();
		
		//create FileIdentifierList's for both the source & destination
		FileIdentifierList srcList = FileIdentifierList.create();
		FileIdentifierList destList = FileIdentifierList.create();
		String destPrefix = System.getProperty("user.name") + "/" + RECORDER_DESTINATION_PATH + "/";
		for (int i =0; i < srcFiles.length; i++)
		{
			srcList.add(srcFiles[i]);
			destList.add( destPrefix + srcFiles[i].getName());
		}
		// now add the recorder framework JAR
		URL resolvedRecJarPath= null;
		try {
			URL url = TestCorePlugin.getDefault().getBundle().getEntry(RECORDER_JAR_NAME);
			if (url != null)				
				resolvedRecJarPath = FileLocator.resolve(url);
			else
			{
				//cannot find recorder jar.  If running from a debug environment, this might not be an error
				//TODO log warning
			}
		} catch (MalformedURLException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

//		String hexrecrJarPath=resolvedHexrecrJarPath.toString();
//		String httpHexrecrJarPath=resolvedHttpHexrecrJarPath.toString();

		if (resolvedRecJarPath != null && resolvedRecJarPath.getProtocol().equals("file"))
			{
				String hjp = resolvedRecJarPath.getFile();
				srcList.add(hjp);
				destList.add(destPrefix + RECORDER_JAR_NAME);
			}

		
		return new FileIdentifierList[] {srcList, destList};
	}
	
	/**
	 * 
	 * Scans the desinationFileIdentifiers and adds the paths to returnedPath.
	 * returnedPath will be added to the classpath for deployed recorder jars.
	 * This method returs a string with a lit of jars separated and terminated with 
	 * a semi-colon (;)
	 * 
	 * @param destinationFileIdentifiers
	 * @return String with the paths to the deployed Recorder Jars to add to the CLASSPATH
	 */
		private String getDestinationAbsolutePath(FileIdentifierList destinationFileIdentifiers) {
			//pull paths out of identifier list
			String returnedPath = "";
			for (Iterator it = destinationFileIdentifiers.iterator(); it.hasNext();) {				
				String tmpStr = (String)it.next();
				//returnedPath += tmpStr + ";";
				// bugzilla 138636 mdd - wrong separator for Linux.
				returnedPath += tmpStr + File.pathSeparator;
			}
			return returnedPath;
		}


}
