/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RecorderFileUtil.java,v 1.9 2010/03/25 16:09:44 paules Exp $
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.internal.execution.recorder.utilities;

import java.io.File;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;

/**
 * <p>Utilities for recording files.</p>
 * 
 * 
 * @author  Ashish Mathur
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version March 25, 2010
 * @since   September 23, 2005
 */
public class RecorderFileUtil {

	/**
	 * <p>Determines if the {@link IFile} exists in the workspace or local file system.</p>
	 * 
	 * @param iFile The {@link IFile} to be checked.
	 * @return <code>true</code> if the {@link IFile} exists in the workspace or local file system, otherwise <code>false</code>.
	 */
	public static boolean fileExists(IFile iFile){

		//Workspace file check: 
		if (iFile.exists()){
			return true;
		}

		//File system file check: 
		else{

			IPath iFilePath = iFile.getLocation();

			if (iFilePath != null){
				return (iFilePath.toFile().exists());
			}
		}

		return false;
	}
	
	/**
	 * <p>Deletes the recording {@link IFile} if it exists in the workspace or local file system.</p>
	 * 
	 * <p>If the recording {@link IFile} was successfully deleted, the recording {@link IFile}'s parent container is refreshed.</p> 
	 * 
	 * <p>If the recording {@link IFile} ends with the <code>.recmodel</code> file extension, the recording model {@link IFile}
	 * (<code>*.recmodel</code>) is deleted and the recording {@link IFile}'s parent container is refreshed.</p> 
	 * 
	 * @param iFile The recording {@link IFile} to be deleted.
	 * @return <code>true</code> if the recording {@link IFile} and possible recording model {@link IFile} were successfully deleted, otherwise <code>false</code>.
	 */
	public static boolean deleteRecordingFile(IFile recordingFile){

		boolean isSuccessful = false;

		if(recordingFile != null){

			isSuccessful = deleteFile(recordingFile);

			if(isSuccessful){

				IPath iFilePath = recordingFile.getFullPath(); 

				if("rec".equals(iFilePath.getFileExtension())){ //$NON-NLS-1$					
					isSuccessful = deleteFile(ResourcesPlugin.getWorkspace().getRoot().getFile(iFilePath.removeFileExtension().addFileExtension("recmodel"))); //$NON-NLS-1$
				}
			}
		}

		return isSuccessful;
	}

	/**
	 * <p>Deletes the {@link IFile} if it exists in the workspace or local file system.</p>
	 * 
	 * <p>If the {@link IFile} was successfully deleted, the {@link IFile}'s parent container is refreshed.</p> 
	 * 
	 * @param iFile The {@link IFile} to be deleted.
	 * @return <code>true</code> if the {@link IFile} was successfully deleted, otherwise <code>false</code>.
	 */
	public static boolean deleteFile(IFile iFile){

		boolean isSuccessful = false;

		if(iFile != null){

			//Workspace file check: 
			if (iFile.exists()){

				try {					

					iFile.delete(true, new NullProgressMonitor());
					
					isSuccessful = true;
				} 
				catch (Exception e) {
					//Ignore and return false;
				}
			}

			//File system file check: 
			else{

				IPath iFilePath = iFile.getLocation();

				if (iFilePath != null){

					File file = iFilePath.toFile();

					if(file.exists()){
						isSuccessful = file.delete();
					}
				}
			}

			if(isSuccessful){

				//Attempt to refresh the file's container:
				try {
					iFile.getParent().refreshLocal(IResource.DEPTH_ONE, new NullProgressMonitor());
				}
				catch (Exception e) {
					//Ignore since the file's container could not be refreshed.
				}
			}
		}

		return isSuccessful;
	}
}
