/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse
 * Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestCorePlugin.java,v 1.24 2009/07/31 16:36:14 paules Exp $
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/

package org.eclipse.hyades.test.core;

import java.io.File;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.execution.harness.ExecutionHarnessPlugin;
import org.eclipse.hyades.execution.harness.util.ISystemUtility;
import org.eclipse.hyades.execution.recorder.RecorderFactory;
import org.eclipse.hyades.models.common.facades.behavioral.impl.DatapoolFacadeResourceFactoryImpl;
import org.eclipse.hyades.models.common.facades.behavioral.impl.FacadeResourceFactoryImpl;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.models.hierarchy.util.HyadesExtendedResourceFactory;
import org.eclipse.hyades.models.util.ModelDebugger;
import org.eclipse.hyades.test.core.internal.ITestCommonConstants;
import org.eclipse.hyades.test.core.util.JavaUtil;
import org.eclipse.osgi.util.NLS;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;

/**
 * <p>Plug-in class for the test core plug-in; also provides other utilities
 * exposed through its implementation of the system utility interface. System
 * utilities are provided through the stand-alone system utility class when in
 * stand-alone mode.</p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @author  Scott E. Schneider
 * @version July 31, 2009
 * @since   January 31, 2005
 * @see     ISystemUtility
 */
public class TestCorePlugin extends Plugin implements ISystemUtility {
	
    /**
     * A singleton instance of the plugin class is stored for access by the get
     * default method.
     */
    private static TestCorePlugin singleton;
    
	private ExecutionHarnessPlugin harnessPlugin;

    /**
     * Provides public static access to the singleton instance of this plug-in
     * class.
     * 
     * @return the singleton instance stored in this class
     */
    public static TestCorePlugin getDefault() {
        return TestCorePlugin.singleton;
    }

    /**
     * Provides public static access to the plugin id for this plug-in, this is
     * defined in the plugin.xml file.
     * 
     * @return the plugin id for this plug-in
     */
    public static String getPluginId() {
        return TestCorePlugin.getDefault().getPlugId();
    }

    /**
     * Retrieve the plugin id for this plugin, the corresponding public static
     * version of this method has been deprecated.
     * 
     * @return the plugin id is returned
     * @see #getPluginId()
     */
    private String getPlugId() {
        return TestCorePlugin.getDefault().getBundle()
                .getSymbolicName();
    }

    /**
     * The OSGi bundle that this plug-in is associated with, needed for various
     * information at run-time such as plugin id and resource string retrieval.
     */
    private Bundle bundle;

    /**
     * The resource bundle associated with this plug-in for the appropriate
     * locale at run-time.
     */
    private ResourceBundle resourceBundle;

    /**
     * Default constructor for the execution harness plugin, state is passed in
     * via the lifecycle-related methods start/stop providing the bundle
     * context. This class is meant to be a singleton but is not enforced by
     * this class, it is enforced by the caller that must only create us once.
     */
    public TestCorePlugin() {
        super();
        TestCorePlugin.singleton = this;
        try
        {
        	harnessPlugin = (ExecutionHarnessPlugin)ExecutionHarnessPlugin.class.newInstance();
        }
        catch(Exception e)
        {
        	logError(e);
        }
    }

    /**
     * Resolves the plugin's resource bundle.
     * <p>
     * If the plugin's resource bundle can not be resolved, <code>null</code> is returned.
     * <p>
     * IMPORTANT: Do not use this method to retrieve values from the resource bundle.  This method 
     * is provided so this plugin's resource bundle can be used as the parent of another plugin's 
     * resource bundle.
     * <p>
     * 
     * @return The plugin's resource bundle, otherwise <code>null</code>.
     * @deprecated As of TPTP V4.4, use {@link org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle} for resolving resources.  The plugin.properties property file ONLY contains messages for the this plug-in's MANIFEST.MF and plugin.xml files.
     */
    public ResourceBundle getResourceBundle() {
        try {
            if (resourceBundle == null)
                resourceBundle   = Platform.getResourceBundle(bundle);
        } catch (MissingResourceException x) {
            resourceBundle = null;
        }
        return resourceBundle;
    }

   
    /**
     * Resolves the localized message associated with the parameter <code>key</code>
     * based on the default locale from the plugin's resource bundle.
     * <p>
     * If the <code>key</code> does not exist in the plugin's resource bundle, the 
     * <code>key</code> is returned.
     * <p>
     * Noteworthy, the resultant message is not formatted (e.g. no message parameter substitution). 
     * <p>
     * 
     * @param key The <code>key</code> of the message in the plugin's resource bundle.
     * @return The localized message associated with the parameter <code>key</code> from the plugin's resource bundle, otherwise the <code>key</code>.
     * @deprecated As of TPTP V4.4, use {@link org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle} for resolving resources.  The plugin.properties property file ONLY contains messages for the this plug-in's MANIFEST.MF and plugin.xml files.
     */
    public String getString(String key) {
    	ResourceBundle bundle = TestCorePlugin.getDefault().getResourceBundle();
        try {
            return (bundle != null) ? bundle.getString(key) : key;
        } catch (MissingResourceException e) {
            return key;
        }
    }
    
    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * based on the default locale from the plugin's resource bundle.
     * <p>
     * Message formatting is done by the {@link org.eclipse.osgi.util.NLS} class using the 
     * default locale. 
     * <p>
     * If the <code>key</code> does not exist in the plugin's resource bundle, the 
     * <code>key</code> is returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the plugin's resource bundle.
     * @param argument The first message parameter for formatting in the localized and formatted message.
     * @return The localized message associated with the parameter <code>key</code> from the plugin's resource bundle, otherwise the <code>key</code>.
     * @deprecated As of TPTP V4.4, use {@link org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle} for resolving resources.  The plugin.properties property file ONLY contains messages for the this plug-in's MANIFEST.MF and plugin.xml files.
     */
    public static String getString(String key, String argument)
    {
        return getString(key, new String[] {argument});
    }   
    
    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * based on the default locale from the plugin's resource bundle.
     * <p>
     * Message formatting is done by the {@link org.eclipse.osgi.util.NLS} class using the 
     * default locale. 
     * <p>
     * If the <code>key</code> does not exist in the plugin's resource bundle, the 
     * <code>key</code> is returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the plugin's resource bundle.
     * @param arguments The array of message parameters for formatting in the localized and formatted message.
     * @return The localized message associated with the parameter <code>key</code> from the plugin's resource bundle, otherwise the <code>key</code>.
     * @deprecated As of TPTP V4.4, use {@link org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle} for resolving resources.  The plugin.properties property file ONLY contains messages for the this plug-in's MANIFEST.MF and plugin.xml files.
     */
    public static String getString(String key, String[] arguments)
    {
        return NLS.bind(TestCorePlugin.getDefault().getString(key), ((Object[])(arguments)));
    }
    
	/*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.harness.util.ISystemUtility#logError(java.lang.Throwable)
     */
    public void logError(Throwable throwable) {
        this.getLog().log(new Status(IStatus.ERROR, TestCorePlugin.getPluginId(), IStatus.OK, throwable.toString(), throwable));
    }

	/**
	 * Logs an error described by a text.
	 * 
	 * <p>This method should be used whenever a class in this plugin
	 * has to log an error since it adheres to the global logging
	 * strategy.</p>
	 * 
	 * @param text
	 */
	public void logError(String text){
        this.getLog().log(new Status(IStatus.ERROR, TestCorePlugin.getPluginId(), text));
	}
	
    /*
     * Stores the OSGi bundle and appropriate resource bundles for this plug-in
     * while first calling the super implementation of the start method.
     * 
     * (non-Javadoc)
     * 
     * @see org.osgi.framework.BundleActivator#start(org.osgi.framework.BundleContext)
     */
    public void start(BundleContext context) throws Exception {
        super.start(context);
        this.bundle = context.getBundle();
        this.resourceBundle = Platform.getResourceBundle(this.bundle);
		
		/* bugzilla_105982: Resource factories must be registered in core plugin, not UI. */
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.TEST_SUITE_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.DEPLOYMENT_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.EXECUTION_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.LOCATION_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.DATAPOOL_FILE_EXTENSION, new DatapoolFacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.ARTIFACT_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		registerRecorderExtensions();
		if(ModelDebugger.INSTANCE.debugDatabaseResourcePostfix!=null && ModelDebugger.INSTANCE.debugDatabaseResourcePostfix.length()>0)
		{
			//add support for RDB backed execution resources
			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("executiondb", new HyadesExtendedResourceFactory());//$NON-NLS-1$
		}
		
		//Filter the test asset resources so they are not copied to the output folder during a build:
		JavaUtil.addResourcePatternToNotCopyList("*." + ICommonConstants.TEST_SUITE_FILE_EXTENSION); //$NON-NLS-1$
		JavaUtil.addResourcePatternToNotCopyList("*." + ICommonConstants.DEPLOYMENT_FILE_EXTENSION); //$NON-NLS-1$
		JavaUtil.addResourcePatternToNotCopyList("*." + ICommonConstants.LOCATION_FILE_EXTENSION); //$NON-NLS-1$
		JavaUtil.addResourcePatternToNotCopyList("*." + ICommonConstants.EXECUTION_FILE_EXTENSION); //$NON-NLS-1$
		JavaUtil.addResourcePatternToNotCopyList("*." + ICommonConstants.DATAPOOL_FILE_EXTENSION); //$NON-NLS-1$
		JavaUtil.addResourcePatternToNotCopyList("*." + ICommonConstants.ARTIFACT_FILE_EXTENSION); //$NON-NLS-1$
		JavaUtil.addResourcePatternToNotCopyList("*." + ITestCommonConstants.TESTLOG_FILE_EXTENSION); //$NON-NLS-1$

		harnessPlugin.start(context);
    }

    private void registerRecorderExtensions() {
    	RecorderFactory factory = RecorderFactory.getInstance();
    	String ids[] = factory.getRegisteredRecorderID();
    	if (ids != null) {
    		for (int i=0; i<ids.length; i++) {
    			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(factory.getRecorderFileExtension(ids[i]), new FacadeResourceFactoryImpl());
    		}
    	}
    }
    
    /*
     * As instructed by the base class, a call to the super stop method is
     * required. This method is only overrided to provide a spot for potential
     * clean-up that is needed in the future.
     * 
     * (non-Javadoc)
     * 
     * @see org.osgi.framework.BundleActivator#stop(org.osgi.framework.BundleContext)
     */
    public void stop(BundleContext context) throws Exception {
        super.stop(context);
    }
    
    /**
     * <p>Resolves the absolute path to the <code>org.eclipse.hyades.test.core</code> plug-in. For example:</p>
     * 
     * <p><code>C:/eclipse/dropins/eclipse/plugins/org.eclipse.hyades.test.core_4.5.0.v200905180100/</code></p>
     * 
     * <p>This method returns <code>null</code> if the absolute path to the <code>org.eclipse.hyades.test.core</code> 
     * plug-in cannot be resolved.</p>
     *  
     * @return The absolute path to the <code>org.eclipse.hyades.test.core</code> plug-in, otherwise <code>null</code>.
     */
    public String getPluginPath(){
    	
    	try{			
			return (FileLocator.resolve(Platform.getBundle(getPlugId()).getEntry("/")).getPath().substring(1)); //$NON-NLS-1$
		}
		catch (Exception e){
			//Ignore and return null.
		}
		
    	return null;
    }
    
    /**
     * <p>Resolves the absolute path to the <code>TPTPRecUtil.dll</code> library
     * for the current operating system architecture.  For example:</p>
     * 
     * <p><code>C:/eclipse/dropins/eclipse/plugins/org.eclipse.hyades.test.core_4.5.0.v200905180100/x86/TPTPRecUtil.dll</code></p>
     * 
     * <p>Supported operating system architectures include:<p>
     * 
     * <ul>
     * <li><code>x86</code>: See <code>/org.eclipse.hyades.test.core/x86/TPTPRecUtil.dll</code>.</li>
     * <li><code>x86_64</code>: See <code>/org.eclipse.hyades.test.core/x86_64/TPTPRecUtil.dll</code>.</li>
     * </ul>
     * 
     * <p>This method returns <code>null</code> if the absolute path to the <code>TPTPRecUtil.dll</code> library 
     * cannot be resolved for the current operating system architecture.</p>
     * 
     * @return The absolute path to the <code>TPTPRecUtil.dll</code> library, otherwise <code>null</code>.
     * @see    Platform#getOSArch()
     */
    public String getRecordingUtilityLibraryPath(){
    			
    	String pluginPath = getPluginPath();

    	if(pluginPath != null){
    		return (pluginPath.concat(Platform.getOSArch()).concat(File.separator).concat("TPTPRecUtil.dll")); //$NON-NLS-1$    				
    	}
		
    	return null;
    }
}