/*******************************************************************************
 * Copyright (c) 2000, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.internal.changes;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jface.text.IDocument;
import org.eclipse.ltk.core.refactoring.TextFileChange;

/**
 * Change that modifies the content of a compilation unit.
 * Copied from org.eclipse.jdt.internal.corext.refactoring.changes.CompilationUnitChange.
 * @author jcanches
 * @since 4.3
 */
public class CompilationUnitChange extends TextFileChange {

	private ICompilationUnit fCUnit;
	private boolean destructive;
	
	/**
	 * Creates a new <code>CompilationUnitChange</code>.
	 * 
	 * @param name the change's name mainly used to render the change in the UI
	 * @param cunit the compilation unit this text change works on
	 * @param destructive Whether this change contains destructive changes (i.e. applying it
	 * will delete some existing code)
	 */
	public CompilationUnitChange(String name, ICompilationUnit cunit, boolean destructive) {
		super(name, getFile(cunit));
		Assert.isNotNull(cunit);
		fCUnit= cunit;
		setTextType("java"); //$NON-NLS-1$
		this.destructive = destructive;
	}
	
	private static IFile getFile(ICompilationUnit cunit) {
		return (IFile) cunit.getResource();
	}
	
	/* non java-doc
	 * Method declared in IChange.
	 */
	public Object getModifiedElement(){
		return fCUnit;
	}
	
	/**
	 * Returns the compilation unit this change works on.
	 * 
	 * @return the compilation unit this change works on
	 */
	public ICompilationUnit getCompilationUnit() {
		return fCUnit;
	}
	
	/**
	 * {@inheritDoc}
	 */
	protected IDocument acquireDocument(IProgressMonitor pm) throws CoreException {
		pm.beginTask("", 2); //$NON-NLS-1$
		fCUnit.becomeWorkingCopy(null, new SubProgressMonitor(pm, 1));
		return super.acquireDocument(new SubProgressMonitor(pm, 1));
	}
	
	/**
	 * {@inheritDoc}
	 */
	protected void releaseDocument(IDocument document, IProgressMonitor pm) throws CoreException {
		super.releaseDocument(document, pm);
		try {
			fCUnit.discardWorkingCopy();
		} finally {
			if (!isDocumentAcquired()) {
				if (fCUnit.isWorkingCopy())
					fCUnit.reconcile(
							ICompilationUnit.NO_AST, 
							false /* don't force problem detection */, 
							null /* use primary owner */, 
							null /* no progress monitor */);

				else
					fCUnit.makeConsistent(pm);
			}
		}
	}
	
//	/**
//	 * {@inheritDoc}
//	 */
//	protected Change createUndoChange(UndoEdit edit, ContentStamp stampToRestore) {
//		try {
//			return new UndoCompilationUnitChange(getName(), fCUnit, edit, stampToRestore, getSaveMode());
//		} catch (CoreException e) {
//			JavaPlugin.log(e);
//			return null;
//		}
//	}
	
	/**
	 * {@inheritDoc}
	 */
	public Object getAdapter(Class adapter) {
		if (ICompilationUnit.class.equals(adapter))
			return fCUnit;
		return super.getAdapter(adapter);
	}
	
	public boolean isDestructive() {
		return destructive;
	}
}

