/*******************************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.internal.changes;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jdt.core.IJavaModelStatusConstants;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;

/**
 * <p>Change that creates a text file with initial content.</p>
 * 
 * 
 * @author  Julien Canches
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version March 25, 2010
 * @since   August 29, 2006
 */
public class CreateFileChange extends Change {

	private IFile fFile;
	private byte[] fContents;
	private String fSource;
	private String fEncoding;
	private boolean fExplicitEncoding;

	private long fStampToRestore;

	public CreateFileChange(IFile file, String source, String encoding) {
		this(file, source, encoding, IResource.NULL_STAMP);
	}

	public CreateFileChange(IFile file, String source, String encoding, long stampToRestore) {
		Assert.isNotNull(file, "file"); //$NON-NLS-1$
		Assert.isNotNull(source, "source"); //$NON-NLS-1$
		fFile = file;
		fSource= source;
		fEncoding= encoding;
		fContents = null;
		fExplicitEncoding= fEncoding != null;
		fStampToRestore= stampToRestore;
	}
	
	public CreateFileChange(IFile file, byte[] contents) {
		this(file, contents, IResource.NULL_STAMP);
	}
	
	public CreateFileChange(IFile file, byte[] contents, long stampToRestore) {
		Assert.isNotNull(file, "file"); //$NON-NLS-1$
		Assert.isNotNull(contents, "contents"); //$NON-NLS-1$
		fFile = file;
		fContents = contents;
		fSource= null;
		fEncoding= null;
		fExplicitEncoding = false;
		fStampToRestore= stampToRestore;
	}

	protected void setEncoding(String encoding, boolean explicit) {
		Assert.isNotNull(encoding, "encoding"); //$NON-NLS-1$
		fEncoding= encoding;
		fExplicitEncoding= explicit;
	}
	
	public String getName() {
		return NLS.bind(ChangesMessages.GENERATE_FILE, fFile.getFullPath().toString());
	}

	protected void setSource(String source) {
		fSource= source;
	}

	protected String getSource() {
		return fSource;
	}
	
	protected void setContens(byte[] contens) {
		fContents = contens;
	}

	protected byte[] getContents() {
		return fContents;
	}

	protected IFile getFile() {
		return fFile;
	}

	public Object getModifiedElement() {
		return fFile;
	}

	public RefactoringStatus isValid(IProgressMonitor pm) throws CoreException {
		
		RefactoringStatus status = new RefactoringStatus();
		
		if (fFile.getLocation() == null) {
			status.addFatalError(NLS.bind(ChangesMessages.UNKNOWN_LOCATION, fFile.getFullPath().toString()));
		}
		
		return status;
	}

	public Change perform(IProgressMonitor pm) throws CoreException {

		InputStream is= null;
		try {
			pm.beginTask(ChangesMessages.GENERATING_FILE, 2); 
			try {
				if (fSource == null) { // contents defined by a byte array
					is= new ByteArrayInputStream(fContents);
				} else { // contents defined by source + encoding
					initializeEncoding();
					is= new ByteArrayInputStream(fSource.getBytes(fEncoding));
				}
				fFile.create(is, false, new SubProgressMonitor(pm, 1));
				if (fStampToRestore != IResource.NULL_STAMP) {
					fFile.revertModificationStamp(fStampToRestore);
				}
				if (fExplicitEncoding) {
					fFile.setCharset(fEncoding, new SubProgressMonitor(pm, 1));
				} else {
					pm.worked(1);
				}				
				return new DeleteFileChange(fFile);
			} catch (CoreException e) {
				throw new JavaModelException(e, IJavaModelStatusConstants.CORE_EXCEPTION);
			} catch (UnsupportedEncodingException e) {
				throw new JavaModelException(e, IJavaModelStatusConstants.IO_EXCEPTION);
			}
		} finally {
			try {
				if (is != null)
					is.close();
			} catch (IOException ioe) {
				throw new JavaModelException(ioe, IJavaModelStatusConstants.IO_EXCEPTION);
			} finally {
				pm.done();
			}
		}
	}
	
	private void initializeEncoding() {
		if (fEncoding == null) {
			fExplicitEncoding= false;
			if (fFile != null) {
				try {
					if (fFile.exists()) {
						fEncoding= fFile .getCharset(false);
						if (fEncoding == null) {
							fEncoding= fFile.getCharset(true);
						} else {
							fExplicitEncoding= true;
						}
					} else {
						fEncoding= fFile.getCharset(true);
					}
				} catch (CoreException e) {
					fEncoding= ResourcesPlugin.getEncoding();
					fExplicitEncoding= true;
				}
			} else {
				fEncoding= ResourcesPlugin.getEncoding();
				fExplicitEncoding= true;
			}
		}
		Assert.isNotNull(fEncoding);
	}

	public void initializeValidationData(IProgressMonitor pm) {
		// TODO Auto-generated method stub
		
	}
}
