/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.internal.changes;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.jdt.core.IJavaModelStatusConstants;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;

/**
 * Change that deletes an existing file.
 * <p/>
 * 
 * 
 * @author  Julien Canches
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @author  Paul E. Slauenwhite
 * @version July 25, 2008
 * @since   August 29, 2006
 */
public class DeleteFileChange extends Change {

	private IFile file;
	private byte[] fSource;
	private String name;
	
	public DeleteFileChange(IFile file){
		Assert.isNotNull(file, "file"); //$NON-NLS-1$
		this.file = file;
	}
	
	public DeleteFileChange(IFile file, String name){
		this(file);
		this.name = name;
	}
	
	public RefactoringStatus isValid(IProgressMonitor pm) throws CoreException {
		RefactoringStatus status = new RefactoringStatus();
		if (!file.exists()) {
			status.addError(ChangesMessages.NO_FILE);
		}
		if (file.isReadOnly()) {
			status.addWarning(ChangesMessages.READ_ONLY_FILE);
		}
		return status;
	}
	
	public Change perform(IProgressMonitor pm) throws CoreException {
		
		try {

			pm.beginTask(ChangesMessages.DELETING_FILE, 1); 

			CreateFileChange undo = null;
			
			if(file.exists()){
			
				file.refreshLocal(0, null);

				if (file.isAccessible() && (!file.isReadOnly())) {				
			
					try {
					
						fSource= getSource(file);
						
						undo= createUndoChange(file, file.getModificationStamp(), fSource);
					} 
					catch (RuntimeException e) {
						// something went bad on undo building, but do not let this break main functionality
						TestCorePlugin.getDefault().logError("Error while trying to read file contens : undo is disabled");
						TestCorePlugin.getDefault().logError(e);
					}			
					
					file.delete(true, true, pm);
				}
			}
			
			return undo;
		} 
		finally {
			pm.done();
		}
	}
	
	private byte[] getSource(IFile file) throws CoreException {
		BufferedInputStream br= null;
		InputStream in= null;
		byte[] ret = null;
		try {
			in= file.getContents();
			ret = new byte[in.available()];
			br= new BufferedInputStream(in);
			br.read(ret);
		} catch (IOException e){
			throw new JavaModelException(e, IJavaModelStatusConstants.IO_EXCEPTION);
		} finally {
			try{
				if (br != null) br.close();
				if (in != null) in.close();
			} catch (IOException e){
				throw new JavaModelException(e, IJavaModelStatusConstants.IO_EXCEPTION);
			}	
		}
		return ret;
	}
	
	private static CreateFileChange createUndoChange(IFile file, long stampToRestore, byte[] contens) {
		return new CreateFileChange(file, contens, stampToRestore);
	}

	public String getName() {
		if(name != null) {
			return name;
		} else {
			return NLS.bind(ChangesMessages.DELETE_FILE, file.getFullPath().toOSString());
		}
	}

	public Object getModifiedElement() {
		return file;
	}
	
	public Object[] getAffectedObjects() {
		Object [] res = new Object[1];
		res[0] = file;
		return res;
	}

	public void initializeValidationData(IProgressMonitor pm) {
		// TODO Auto-generated method stub
		
	}
}

