/*******************************************************************************
 * Copyright (c) 2000, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.internal.changes;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;

/**
 * Changes that moves a compilation unit (to a new package and/or a new name).
 * @author jcanches
 * @since 4.3
 */
public class MoveCompilationUnitChange extends Change {

	private boolean fUndoable;
	private long fStampToRestore;
	private ICompilationUnit compilationUnit;
	private IPackageFragment newPackage;
	private String newName;
	private boolean isPackageChanged;
	private boolean isClassnameChanged;
	
	public MoveCompilationUnitChange(ICompilationUnit cu, IPackageFragment newPackage, String newName){
		this(cu, newPackage, newName, IResource.NULL_STAMP);
	}
	
	private MoveCompilationUnitChange(ICompilationUnit cu, IPackageFragment newPackage, String newName, long stampToRestore) {
		this.compilationUnit = cu;
		if (newPackage == null) {
			newPackage = (IPackageFragment)compilationUnit.getParent();
		} else {
			this.newPackage = newPackage;
		}
		if (newName == null) {
			this.newName = compilationUnit.getElementName();
		} else {
			this.newName = newName;
		}
		fStampToRestore= stampToRestore;
		isPackageChanged = !compilationUnit.getParent().equals(newPackage);
		isClassnameChanged = !this.newName.equals(compilationUnit.getElementName());
	}
	
	public String getName() {
		if (isPackageChanged && isClassnameChanged) {
			return NLS.bind(ChangesMessages.MOVE_RENAME_COMPILATION_UNIT,
					new Object[] { compilationUnit.getElementName(), getPackageName(newPackage), newName });
		}
		if (isPackageChanged) {
			return NLS.bind(ChangesMessages.MOVE_COMPILATION_UNIT,
					new Object[] { compilationUnit.getElementName(), getPackageName(newPackage) });
		}
		if (isClassnameChanged) {
			return NLS.bind(ChangesMessages.RENAME_COMPILATION_UNIT,
					new Object[] { compilationUnit.getElementName(), newName});
		}
		return "Invalid change"; //$NON-NLS-1$
	}

	public RefactoringStatus isValid(IProgressMonitor pm) throws CoreException {
		RefactoringStatus status = new RefactoringStatus();
		if (compilationUnit.isReadOnly()) {
			status.addFatalError(NLS.bind(ChangesMessages.E_COMP_UNIT_READ_ONLY, compilationUnit.getElementName()));
		}
		if (newPackage.getCompilationUnit(newName).exists()) {
			status.addWarning(NLS.bind(ChangesMessages.W_COMP_UNIT_OVERWRITE, compilationUnit.getElementName()));
		}
		return status;
	}
	
	public Change perform(IProgressMonitor pm) throws CoreException {
		// get current modification stamp
		long currentStamp= IResource.NULL_STAMP;
		IResource resource= compilationUnit.getResource();
		if (resource != null) {
			currentStamp= resource.getModificationStamp();
		}
		
		fUndoable= ! newPackage.getCompilationUnit(newName).exists();
		String oldName = compilationUnit.getElementName();
		IPackageFragment oldPackage = (IPackageFragment) compilationUnit.getParent();
		
		// perform the move and restore modification stamp
		compilationUnit.move(newPackage, null, newName, true, pm);
		ICompilationUnit moved= newPackage.getCompilationUnit(newName);
		if (fStampToRestore != IResource.NULL_STAMP) {
			IResource movedResource= moved.getResource();
			if (movedResource != null) {
				movedResource.revertModificationStamp(fStampToRestore);
			}
		}
		
		if (fUndoable) {
			return new MoveCompilationUnitChange(moved, oldPackage, oldName, currentStamp);
		} else {
			return null;
		}
	}
	
	static String getPackageName(IPackageFragment pack) {
		if (pack.isDefaultPackage())
			return ChangesMessages.DEFAULT_PACKAGE; 
		else
			return pack.getElementName();
	}

	public Object getModifiedElement() {
		return compilationUnit;
	}

	public void initializeValidationData(IProgressMonitor pm) {
		// NOP
	}

}
