/********************************************************************** 
 * Copyright (c) 2008, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: LaunchDatapoolExtensionManager.java,v 1.2 2009/05/17 16:16:03 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.core.internal.launch.datapool.extensions;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.launch.extensions.ILaunchconfigDatapoolHandler;
import org.eclipse.hyades.test.core.launch.extensions.IPassInfoCollector;

/**
 * <p>Manages extensions for the <code>launchconfigDatapoolHandler</code> and 
 * <code>launchDatapoolCheckPass</code> extension points, based on the type of 
 * each test.</p>
 * 
 * 
 * @author  Huang Xin Ying
 * @author  Paul Slauenwhite
 * @version May 16, 2009 
 * @sine    March 20, 2008
 */
public class LaunchDatapoolExtensionManager {

	private static LaunchDatapoolExtensionManager instance;
	private Map datapoolHandlerExtensionMap = new HashMap();

    /**
     * Return instance of LaunchDatapoolExtensionManager .
	*/
	public static LaunchDatapoolExtensionManager getInstance() {
		if (instance == null) {
			instance = new LaunchDatapoolExtensionManager();
		}
		return instance;
	}
   
	/**
	 * Register LaunchconfigDatapoolHandler and LaunchDatapoolCheckPass extensions 
	 */
	private LaunchDatapoolExtensionManager() {
		registerDatapoolHandlers();
		registerCheckPass();
	}

	/**
	 * 
	 * @param testElement
	 * @return
	 */
    
	public ILaunchconfigDatapoolHandler getRunHandler(Object testElement) {
		LaunchDatapoolExtensionDescriper ed = getLaunchDatapoolDescriptor(testElement);
		if (ed != null) {
			return ed.getRunHandler();
		}
		return null;
	}
	
	public IPassInfoCollector getCollector(Object testElement) {
		LaunchDatapoolExtensionDescriper ed = getLaunchDatapoolDescriptor(testElement);
		if (ed != null && ed.getCollector()!= null) 
			return ed.getCollector();

		return getExtensionDescriptor("default").getCollector();
	}
   
	/**
	 * Register LaunchconfigDatapoolHandler extensions 
	 */
	private void registerDatapoolHandlers() {
		IExtensionPoint extPoint = Platform.getExtensionRegistry()
				.getExtensionPoint(
						TestCorePlugin.getPluginId()
								+ ".launchconfigDatapoolHandler");
		if (extPoint != null) {
			IConfigurationElement[] datapoolHandlers = extPoint
					.getConfigurationElements();
			for (int i = 0; i < datapoolHandlers.length; i++) {
				IConfigurationElement[] supportedTypes = datapoolHandlers[i]
						.getChildren();
				for (int j = 0; j < supportedTypes.length; j++) {
					String type = supportedTypes[j].getAttribute("name"); //$NON-NLS-1$					
					try {
						LaunchDatapoolExtensionDescriper ed = getExtensionDescriptor(type);
						ed.launchDatapoolConfigElem = datapoolHandlers[i];
					} catch (Exception e) {
						TestCorePlugin
								.getDefault()
								.logError(
										"Extension "
												+ datapoolHandlers[i].getName()
												+ " was ignored. See next messages for details.");
						TestCorePlugin.getDefault().logError(e);
					}
				}
			}
		}
	}
	
	/**
	 * Register LaunchconfigDatapoolHandler extensions 
	 */
	private void registerCheckPass() {
		IExtensionPoint extPoint = Platform.getExtensionRegistry()
				.getExtensionPoint(
						TestCorePlugin.getPluginId()
								+ ".launchDatapoolCheckPass");
		if (extPoint != null) {
			IConfigurationElement[] datapoolHandlers = extPoint
					.getConfigurationElements();
			for (int i = 0; i < datapoolHandlers.length; i++) {
				IConfigurationElement[] supportedTypes = datapoolHandlers[i]
						.getChildren();
				for (int j = 0; j < supportedTypes.length; j++) {
					String type = supportedTypes[j].getAttribute("name"); //$NON-NLS-1$	
					if(type == null || type.equals(""))
						type = "default";
					try {
						LaunchDatapoolExtensionDescriper ed = getExtensionDescriptor(type);
						ed.CheckPassConfigElem = datapoolHandlers[i];
					} catch (Exception e) {
						TestCorePlugin
								.getDefault()
								.logError(
										"Extension "
												+ datapoolHandlers[i].getName()
												+ " was ignored. See next messages for details.");
						TestCorePlugin.getDefault().logError(e);
					}
				}
			}
		}
	}

	private LaunchDatapoolExtensionDescriper getExtensionDescriptor(String type) {

		if (type == null) {
			throw new NullPointerException("Type must be non-null"); //$NON-NLS-1$
		}

		Object o = datapoolHandlerExtensionMap.get(type);
		if (o == null) {
			o = new LaunchDatapoolExtensionDescriper();
			datapoolHandlerExtensionMap.put(type, o);
		}
		return (LaunchDatapoolExtensionDescriper) o;
	}

	private LaunchDatapoolExtensionDescriper getLaunchDatapoolDescriptor(
			Object testElement) {
		if (testElement != null) {
			if (testElement instanceof TPFTest) {
				TPFTest test = (TPFTest) testElement;
				if (test.getType() != null) {
					return getExtensionDescriptor(test.getType());
				}
			}
		}
		return null;
	}

	private class LaunchDatapoolExtensionDescriper {
		public IConfigurationElement launchDatapoolConfigElem = null;
		public IConfigurationElement CheckPassConfigElem = null;
		private ILaunchconfigDatapoolHandler hanler = null;
		private IPassInfoCollector collector = null;

		public ILaunchconfigDatapoolHandler getRunHandler() {
			if (this.hanler == null) {
				if (this.launchDatapoolConfigElem != null) {
					try {
						this.hanler = (ILaunchconfigDatapoolHandler) this.launchDatapoolConfigElem
								.createExecutableExtension("class"); //$NON-NLS-1$
					} catch (CoreException e) {
						TestCorePlugin.getDefault().logError(e);
						this.launchDatapoolConfigElem = null;
					}
				}
			}
			return this.hanler;
		}
		
		public IPassInfoCollector getCollector() {
			if (this.collector == null) {
				if (this.CheckPassConfigElem != null) {
					try {
						this.collector = (IPassInfoCollector) this.CheckPassConfigElem
								.createExecutableExtension("uiClass"); //$NON-NLS-1$
					} catch (CoreException e) {
						TestCorePlugin.getDefault().logError(e);
						this.CheckPassConfigElem = null;
					}
				}
			}
			return this.collector;
		}
	}
}
