/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestExecutionProcess.java,v 1.13 2008/04/09 19:57:50 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.internal.launch.processes;

import java.util.Iterator;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.debug.core.model.IStreamsProxy;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.IDataProcessor;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutionComponentStateChangeListener;
import org.eclipse.hyades.execution.core.IExecutor;
import org.eclipse.hyades.execution.harness.TestExecutionHarness;
import org.eclipse.hyades.execution.harness.XMLExecutionDataProcessor;
import org.eclipse.hyades.execution.local.ExecutionComponentStub;
import org.eclipse.hyades.execution.local.ExecutorStub;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentListener;
import org.eclipse.hyades.loaders.common.ExecutionContext;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle;
import org.eclipse.hyades.test.core.launch.configurations.TestLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.extensions.IRunHandler;
import org.eclipse.hyades.test.core.testservices.resources.PasswordCollection;

/**
 * Fake process that represents a Hyades Execution Session. The lifecycle of
 * this process is mapped to the one of an IExecutor Execution Component.
 * 
 * @author  Julien Canches
 * @author  Paul E. Slauenwhite
 * @version April 3, 2008
 * @since   February 8, 2005
 */
public class TestExecutionProcess implements IProcess {

	private IExecutor executor;

	private ILaunch launch;

	private boolean terminated = false;

	private TPFTest test;

	private TPFExecutionResult executionResult;

	private AgentListener controlAgentListener;

	private Agent controlAgent;

	public TestExecutionProcess(IExecutor executor, ILaunch launch, ResourceSet resourceSet) throws CoreException {
		if (executor != null && launch != null) {
			this.executor = executor;
			this.launch = launch;
			this.test = TestLaunchConfigurationFacade.getTest(launch.getLaunchConfiguration(), resourceSet);
			synchronized (executor) {
				if (executor.getState() == IExecutionComponent.DEAD) {
					onTerminate();
				} else {
					this.executor.addExecutionComponentStateChangeListener(new IExecutionComponentStateChangeListener() {
						public void stateChanged(ExecutionComponentStateChangeEvent newState) {
							onTerminate();
						}
					});
				}
			}
			if (executor instanceof ExecutorStub) {
				ExecutorStub exStub = (ExecutorStub) executor;
				IDataProcessor[] dataProcessors = exStub.getDataProcessors();
				if (dataProcessors != null && dataProcessors.length > 0) {
					IDataProcessor dataProcessor = exStub.getDataProcessors()[0];
					this.executionResult = getExecutionResult(dataProcessor);
				}
			}
		}
	}

	public String getLabel() {
		return TestCorePluginResourceBundle.TestExecutionProcess_testProcess + " [" + this.test.getName() + "]"; 
	}

	public ILaunch getLaunch() {
		return launch;
	}

	public IStreamsProxy getStreamsProxy() {
		// TODO Consider supporting this feature ?
		// null means not supported
		return null;
	}

	public void setAttribute(String key, String value) {
		// We don't support attributes. We don't need them anyway.
	}

	public String getAttribute(String key) {
		return null;
	}

	public int getExitValue() /* throws DebugException */{
		// TODO Consider providing something else
		return 0;
	}

	public Object getAdapter(Class adapter) {
		return null;
	}

	public boolean canTerminate() {
		return executionResult != null;
	}

	public synchronized boolean isTerminated() {
		return this.terminated || executor.getState() == IExecutionComponent.DEAD;
	}

	public void terminate() throws DebugException {
		if (!isTerminated()) {
			TestExecutionHarness teh = new TestExecutionHarness();
			String errorMessage = teh.stopTest(this.executor, this.executionResult);
			if (errorMessage != null) {
				IStatus status = new Status(IStatus.ERROR, TestCorePlugin.getPluginId(), 0, errorMessage, null);
				throw new DebugException(status);
			}
		}
	}

	protected synchronized void onTerminate() {
		if (!terminated) {
			//remove the dps and passwords from passwordCollection
			PasswordCollection.getInstance().clear(test.getImplementor().getId());
			// Notify the run handler (if any)
			IRunHandler runHandler = LaunchConfigurationExtensionsManager.getInstance().getRunHandler(test);
			if (runHandler != null) {
				runHandler.postRun(launch);
			}
			this.terminated = true;
			if (executor instanceof ExecutionComponentStub) {
				ExecutionComponentStub component = (ExecutionComponentStub) this.executor;
				component.setDelegate(null);
			}
			this.executor = null;
			// Notify the event
			DebugPlugin.getDefault().fireDebugEventSet(new DebugEvent[] { new DebugEvent(this, DebugEvent.CHANGE) });
		}
	}

	/**
	 * Returns the TPFExecutionResult object that the execution of the test is
	 * producing. This method may return null in non-standard cases.
	 * 
	 * @return
	 */
	private TPFExecutionResult getExecutionResult(IDataProcessor dataProcessor) {
		if (dataProcessor instanceof XMLExecutionDataProcessor) {
			XMLExecutionDataProcessor dp = (XMLExecutionDataProcessor) dataProcessor;
			ExecutionContext ec = (ExecutionContext) dp.getContext().getCustomData().get(ExecutionContext.root);
			if (ec != null) {
				Resource res = ec.getResource();
				if (res != null) {
					Iterator it = res.getContents().iterator();
					while (it.hasNext()) {
						EObject obj = (EObject) it.next();
						if (obj instanceof TPFExecutionResult) {
							return (TPFExecutionResult) obj;
						}
					}
				}
			}
		}
		return null;
	}

	public TPFExecutionResult getExecutionResult() {
		return this.executionResult;
	}

	public IExecutor getExecutor() {
		return this.executor;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#finalize()
	 */
	protected void finalize() throws Throwable {
		if (this.controlAgent != null && this.controlAgentListener != null) {
			this.controlAgent.removeAgentListener(this.controlAgentListener);
			this.controlAgent = null;
			this.controlAgentListener = null;
		}
		super.finalize();
	}

}
