/*******************************************************************************
 * Copyright (c) 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EMFResourceReader.java,v 1.3 2010/02/23 14:12:15 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.internal.util;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import javax.xml.parsers.SAXParserFactory;

import org.eclipse.core.resources.IFile;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle;
import org.eclipse.osgi.util.NLS;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/** 
 * <p>Generic reader for resolving properties from serialized EMF models in EMF resource files.</p>
 * 
 * <p>For performance reasons, this reader does NOT load EMF model(s) into memory.</p>
 * 
 * 
 * @author  Paul Slauenwhite
 * @version February 23, 2010
 * @since   October 21, 2008
 */
public class EMFResourceReader {

	/**
	 * The message for {@link SAXException}s thrown to stop parsing.
	 */
	private static final String STOP_PARSING_SAX_EXCEPTION_MESSAGE = "STOP_PARSING_SAX_EXCEPTION_MESSAGE"; //$NON-NLS-1$
	private static final String PREMATURE_EOF_SAX_EXCEPION_MESSAGE = "Premature end of file."; //$NON-NLS-1$
	
	/**
	 * <p>Resolves the root properties from the serialized EMF model in the EMF resource file in the 
	 * Eclipse workspace.</p>
	 * 
	 * <p>Note, only the root properties from the first serialized EMF model, named 'ResourceContents', 
	 * are resolved from the EMF resource file in the Eclipse workspace.</p>
	 * 
	 * <p>Root properties are strings stored as attributes on the root element in the serialized 
	 * EMF model.</p>
	 * 
	 * @param file The EMF resource file in the Eclipse workspace containing a serialized EMF model.
	 * @return Root properties of the serialized EMF model, otherwise an empty {@link Map}.
	 */
	public static Map getRootProperties(IFile file){
		
		final Map rootAttributes = new HashMap();

		//Note: ZipInputStream used instead of ZipFile due to periodic ZipExceptions ("Error opening zip file") with some valid zip files.
		ZipInputStream zipInputStream = null;

		try {

			zipInputStream = new ZipInputStream(file.getContents());
			ZipEntry zipEntry = null;

			while ((zipEntry = zipInputStream.getNextEntry()) != null) {

				//Only process the first serialized EMF model, named 'ResourceContents':
				if (zipEntry.getName().trim().equals("ResourceContents")) { //$NON-NLS-1$

					//Only process the root element in the serialized EMF model:
					DefaultHandler rootElementHandler = new DefaultHandler(){

						/* (non-Javadoc)
						 * @see org.xml.sax.helpers.DefaultHandler#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
						 */
						public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {

							//Save the attributes of the root element:
							for (int counter = 0; counter < attributes.getLength(); counter++) {	    						
								rootAttributes.put(attributes.getQName(counter).trim(), attributes.getValue(counter).trim());
							}

							//Stop parsing after the root element:
							throw new SAXException(STOP_PARSING_SAX_EXCEPTION_MESSAGE);
						}
					};

					//Parse the serialized EMF model:
					SAXParserFactory.newInstance().newSAXParser().parse(zipInputStream, rootElementHandler);
				}
			}
		} 
		catch (SAXException s) {

			//Ignore SAXExceptions thrown to stop parsing:
			if (!STOP_PARSING_SAX_EXCEPTION_MESSAGE.equals(s.getMessage().trim())) {				
				TestCorePlugin.getDefault().logError(NLS.bind(TestCorePluginResourceBundle.EMFUtil_ERROR_LOADING_RESOURCE,  new String[] {file.getFullPath().toOSString(), s.getLocalizedMessage()}));
			}
		}
		catch (Exception e) {
			TestCorePlugin.getDefault().logError(e);
		}
		finally{

			//Close the zip input stream:
			try{

				if (zipInputStream != null) {
					zipInputStream.close();
				}
			}
			catch (IOException e1) {
				//Ignore since closing the zip input stream.
			}
		}

		return rootAttributes;
	}
}
