/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AbstractLaunchConfigurationFacade.java,v 1.8 2008/03/20 19:49:44 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.launch.configurations;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;

/**
 * This class provides common methods for Launch Configuration Facades. A Launch
 * Configuration Facade provides a set of helper methods for reading from and
 * writing to Launch Configurations.<br>
 * This class is meant to be extended by new Launch Configuration Facades.
 * 
 * @author jcanches
 */
public class AbstractLaunchConfigurationFacade {

	private static final String PROPERTY_SUFFIX_FRAGMENT = "URIFragment"; //$NON-NLS-1$

	private static final String PROPERTY_SUFFIX_URI = "URI"; //$NON-NLS-1$

	/**
	 * This prefix is used for all Hyades properties within a Launch
	 * Configuration.
	 */
	protected final static String PROPERTY_PREFIX = "org.eclipse.hyades.test.ui.launch"; //$NON-NLS-1$

	/**
	 * Stores in the provided Launch Configuration Working Copy the reference to
	 * an EMF Object.
	 * 
	 * @param configuration
	 *            The LaunchConfigurationWorkingCopy where the reference is
	 *            stored.
	 * @param property
	 *            The name of the property within the Launch Configuration that
	 *            receives the reference.
	 * @param eObject
	 *            The EMF Object. This object must belong to an EMF Resource.
	 *            <code>null</code> is accepted.
	 */
	protected static void storeEMFReference(
			ILaunchConfigurationWorkingCopy configuration, String property,
			EObject eObject) {
		String uri;
		String uriFragment;

		if (eObject == null) {
			uri = ""; //$NON-NLS-1$
			uriFragment = ""; //$NON-NLS-1$
		} else {
			if (eObject.eResource() == null) {
				throw new IllegalArgumentException(
						"eObject must belong to a resource"); //$NON-NLS-1$
			}
			uri = eObject.eResource().getURI().toString();
			uriFragment = eObject.eResource().getURIFragment(eObject);
		}
		configuration.setAttribute(property + PROPERTY_SUFFIX_URI, uri);
		configuration.setAttribute(property + PROPERTY_SUFFIX_FRAGMENT,
				uriFragment);
	}

	/**
	 * Retrieves from the provided Launch Configuration Working Copy an EMF
	 * Object.
	 * 
	 * @param configuration
	 *            The LaunchConfigurationWorkingCopy to load from.
	 * @param property
	 *            The name of the property within the Launch Configuration that
	 *            contains EMF Object.
	 * @param resourceSet
	 *            The Resource Set that should contain the returned object.
	 * @return The EMF Object (or null if the reference could not be resolved)
	 * @since 3.1
	 */
	protected static EObject resolveEMFReference(
			ILaunchConfiguration configuration, String property,
			ResourceSet resourceSet) throws CoreException {
		String uriString = configuration.getAttribute(property
				+ PROPERTY_SUFFIX_URI, (String) null);
		String uriFragment = configuration.getAttribute(property
				+ PROPERTY_SUFFIX_FRAGMENT, (String) null);
		if (uriString == null || uriString.equals("")) { //$NON-NLS-1$
			return null;
		}
		URI uri = URI.createURI(uriString);
		try {
			Resource res = resourceSet
					.getResource(uri, /* loadOnDemand */true);
			if (res == null) {
				return null;
			}
			return res.getEObject(uriFragment);
		} catch (Throwable t) {
			// This may happen if some elements referenced by the configuration
			// have been deleted. Failing to resolve just returns null.
			return null;
		}
	}

	/**
	 * Stores in the provided Launch Configuration Working Copy the reference to
	 * a set of EMF Objects.
	 * 
	 * @param configuration
	 *            The LaunchConfigurationWorkingCopy where the reference is
	 *            stored.
	 * @param property
	 *            The name of the property within the Launch Configuration that
	 *            receives the reference.
	 * @param list
	 *            The list of EMF Objects (subclasses of EObject). These objects
	 *            must belong to an EMF Resource. Any of them may be
	 *            <code>null</code>, but the list should not be
	 *            <code>null</code>.
	 */
	protected static void storeEMFReferences(
			ILaunchConfigurationWorkingCopy configuration, String property,
			List list) {
		List uriList = new ArrayList();
		List uriFragmentList = new ArrayList();
		for (Iterator it = list.iterator(); it.hasNext();) {
			Object obj = it.next();
			if (obj instanceof EObject) {
				EObject eObject = (EObject) obj;
				if (eObject == null) {
					uriList.add(""); //$NON-NLS-1$
					uriFragmentList.add(""); //$NON-NLS-1$
				} else {
					Resource resource = eObject.eResource();
					if (resource == null) {
						throw new NullPointerException(
								"Resource object must be non-null"); //$NON-NLS-1$
					}
					uriList.add(eObject.eResource().getURI().toString());
					uriFragmentList.add(eObject.eResource().getURIFragment(
							eObject));
				}
			}
		}
		configuration.setAttribute(property + PROPERTY_SUFFIX_URI, uriList);
		configuration.setAttribute(property + PROPERTY_SUFFIX_FRAGMENT,
				uriFragmentList);
	}

	/**
	 * Unsets any EMF Reference associated to the given property in the Launch
	 * Configuration.
	 * 
	 * @param configuration
	 * @param property
	 */
	protected static void unsetEMFReference(
			ILaunchConfigurationWorkingCopy configuration, String property) {
		configuration.setAttribute(property + PROPERTY_SUFFIX_URI,
				(String) null);
		configuration.setAttribute(property + PROPERTY_SUFFIX_FRAGMENT,
				(String) null);
	}

	/**
	 * Retrieves from the provided Launch Configuration Working Copy a set of
	 * EMF Object.
	 * 
	 * @param configuration
	 *            The LaunchConfigurationWorkingCopy to load from.
	 * @param property
	 *            The name of the property within the Launch Configuration that
	 *            contains the set of EMF Objects.
	 * @param resourceSet
	 *            The Resource Set that should contain the returned objects.
	 * @return The list of EMF Objects (subclasses of EObject). If any of the
	 *         EMF references that were previously stored cannot be resolved,
	 *         the associated object will not be returned.
	 * @since 3.1
	 */
	protected static List resolveEMFReferences(
			ILaunchConfiguration configuration, String property,
			ResourceSet resourceSet) throws CoreException {
		List ret = new ArrayList();
		List uriList = configuration.getAttribute(property
				+ PROPERTY_SUFFIX_URI, new ArrayList());
		List uriFragmentList = configuration.getAttribute(property
				+ PROPERTY_SUFFIX_FRAGMENT, new ArrayList());
		if (uriFragmentList.isEmpty() || uriList.isEmpty()) {
			return new ArrayList();
		}
		for (int i = 0, max = uriList.size(); i < max; ++i) {
			String uriString = (String) uriList.get(i);
			if (uriString != null && !uriString.equals("")) { //$NON-NLS-1$
				URI uri = URI.createURI((String) uriList.get(i));
				try {
					Resource res = resourceSet.getResource(uri, /* loadOnDemand */
					true);
					if (res != null) {
						EObject obj = res.getEObject((String) uriFragmentList
								.get(i));
						if (obj != null) {
							ret.add(obj);
						}
					}
				} catch (Throwable t) {
					// This may happen if some artifacts referenced by the
					// configuration
					// have been deleted. Keep on resolving other elements.
				}
			}
		}
		return ret;
	}

	/**
	 * Retrieves from the supplied Launch Configuration a container property.
	 * 
	 * @param configuration
	 *            The LaunchConfigurationWorkingCopy to load from.
	 * @param property
	 *            The name of the property within the Launch Configuration that
	 *            contains the container reference.
	 * @return A Workspace Container, or <code>null</code> if the property is
	 *         not set, or if the container does not exist.
	 * @throws CoreException
	 */
	protected static IContainer getContainerProperty(
			ILaunchConfiguration configuration, String property)
			throws CoreException {
		String location = configuration.getAttribute(property, (String) null);
		if (location != null) {
			IResource res = ResourcesPlugin.getWorkspace().getRoot()
					.findMember(location);
			if (res != null && res instanceof IContainer) {
				return (IContainer) res;
			}
		}
		return null;
	}

	/**
	 * Stores in the supplied Launch Configuration a container property.
	 * 
	 * @param configuration
	 *            The LaunchConfigurationWorkingCopy where the reference is
	 *            stored.
	 * @param property
	 *            The name of the property within the Launch Configuration that
	 *            receives the reference to the container.
	 * @param value
	 *            The container.
	 */
	protected static void setContainerProperty(
			ILaunchConfigurationWorkingCopy configuration, String property,
			IContainer value) {
		String container = null;
		if (value != null) {
			container = value.getFullPath().toString();
		}
		configuration.setAttribute(property, container);
	}
	
	/**
	 * Returns whether the given property in a launch configuration is an EMF reference
	 * to an object with the same URI as the given URI.
	 * @param configuration
	 * @param property Name of the property.
	 * @param uri The URI to compare to. null is not allowed.
	 * @return <code>true</code> if the launch configuration has a property with this name,
	 * and if it is an EMF reference to an object with the same URI as <code>uri</code>.
	 * @throws CoreException
	 */
	protected static boolean isSameURI(ILaunchConfiguration configuration, String property, URI uri) throws CoreException {
		String uriString = configuration.getAttribute(property
				+ PROPERTY_SUFFIX_URI, (String) null);
		String uriFragment = configuration.getAttribute(property
				+ PROPERTY_SUFFIX_FRAGMENT, (String) null);
		return uri.trimFragment().toString().equals(uriString)
			&& uri.fragment().equals(uriFragment);
	}

	/**
	 * Updates an EMF reference property within a launch configuration by changing its URI.
	 * This API is provided for refactor use cases when the URI of a referenced EMF object
	 * needs to be updated.
	 * @param configuration
	 * @param property Name of the property.
	 * @param newURI The new URI.
	 */
	protected static void updateURI(ILaunchConfigurationWorkingCopy configuration, String property, URI newURI) {
		configuration.setAttribute(property + PROPERTY_SUFFIX_URI, newURI.trimFragment().toString());
		configuration.setAttribute(property + PROPERTY_SUFFIX_FRAGMENT, newURI.fragment());
	}

}
