/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DeploymentLaunchConfigurationFacade.java,v 1.7 2005/10/28 20:11:45 sschneid Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.launch.configurations;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.execution.core.util.ConnectionSpecifier;
import org.eclipse.hyades.models.common.configuration.CFGArtifactLocationPair;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGMachineConstraint;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationFactory;
import org.eclipse.hyades.models.common.configuration.util.ConfigurationUtil;
import org.eclipse.hyades.models.common.interactions.BVRProperty;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.core.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.core.launch.extensions.IDeploymentsProvider;
import org.eclipse.hyades.test.core.util.DeploymentUtil;

/**
 * <p>
 * This class is a facade for accessing Deployment information in a TPTP launch
 * configuration.
 * </p>
 * <p>
 * This facade manages the following properties:
 * <ul>
 * <li>deployment (TPFDeployment): the deployment model used for specifying how
 * a test should be deployed.</li>
 * </ul>
 * </p>
 * 
 * @author jcanches
 */
public class DeploymentLaunchConfigurationFacade extends AbstractLaunchConfigurationFacade {

	private static final String DEFAULT_DEPLOYMENT_VALUE = "default";

	private final static String PROPERTY_DEPLOYMENT = PROPERTY_PREFIX + ".deployment"; //$NON-NLS-1$

	/**
	 * Returns the Deployment selected in a TPTP launch configuration.
	 * 
	 * @param configuration
	 *            A launch configuration.
	 * @param resourceSet
	 *            The Resource Set that should contain the returned Deployment.
	 * @return A Deployment loaded in the specified ResourceSet. If no
	 *         deployment is set, or if the deployment does not exist any more,
	 *         <code>null</code> is returned.
	 * @since 3.1
	 */
	public synchronized static TPFDeployment getDeployment(ILaunchConfiguration configuration, ResourceSet resourceSet)
			throws CoreException {
		TPFDeployment deployment = null;

		// Handle transitory deployment case
		List list = configuration.getAttribute("transitory", (List) null);
		if (list != null) {
			deployment = (TPFDeployment) list.get(0);
		} else {

			// Handle default deployment and non-default deployments
			if (DEFAULT_DEPLOYMENT_VALUE.equals(configuration.getAttribute(PROPERTY_DEPLOYMENT, (String) null))) {
				// This is the default deployment
				TPFTest test = TestLaunchConfigurationFacade.getTest(configuration, resourceSet);
				IDeploymentsProvider dProvider = LaunchConfigurationExtensionsManager.getInstance()
						.getDeploymentsProvider(test);
				deployment = dProvider.getDefaultDeployment(test);
			} else {
				EObject eObject = resolveEMFReference(configuration, PROPERTY_DEPLOYMENT, resourceSet);
				if (eObject != null && eObject instanceof TPFDeployment) {
					deployment = (TPFDeployment) eObject;
				}
			}

		}

		if (deployment != null) {
			// added to make sure deployment contains a valid pair of location
			// and artifact for this test suite
			// and automatically calculate CLASSPATH for this test
			// bugzilla# 58827. bjiang 4/21/2004
			TPFTest test = TestLaunchConfigurationFacade.getTest(configuration, resourceSet);
			TPFTestSuite testSuite = null;
			if (test instanceof TPFTestSuite) {
				testSuite = (TPFTestSuite) test;
			} else if (test instanceof TPFTestCase) {
				testSuite = ((TPFTestCase) test).getTestSuite();
			}

			if (testSuite != null) {
				DeploymentUtil.associateTestAsset(testSuite, deployment);
			}
		}
		return deployment;
	}

	/**
	 * Sets the Deployment in a TPTP launch configuration.
	 * 
	 * @param configuration
	 *            A Launch Configuration Working Copy.
	 * @param A
	 *            Deployment that belongs to a resource, or the default
	 *            deployment.
	 */
	public synchronized static void setDeployment(ILaunchConfigurationWorkingCopy configuration, TPFDeployment deployment) {
		if (deployment != null && ConfigurationUtil.isDefaultDeployment(deployment)) {
			// Remove any previous EMF Reference associated to the launch
			// configuration
			unsetEMFReference(configuration, PROPERTY_DEPLOYMENT);
			// Set the deployment property to "default"
			configuration.setAttribute(PROPERTY_DEPLOYMENT, DEFAULT_DEPLOYMENT_VALUE);
		} else {
			// Remove the "default deployment" attribute
			configuration.setAttribute(PROPERTY_DEPLOYMENT, (String) null);
			// Store the EMF reference to the deployment
			storeEMFReference(configuration, PROPERTY_DEPLOYMENT, deployment);
		}
	}

	/**
	 * Sets deployment to a deployment instance that is transitory, it will not
	 * be persisted across sessions. Used in situations where a deployment needs
	 * to be modified but the changes can not or are not desired to be
	 * persisted.
	 * 
	 * @param configuration
	 *            the configuration to tag with this transitory deployment
	 * @param deployment
	 *            the deployment that will be related to this configuration as a
	 *            transitory deployment
	 * @param connectionString
	 *            connection string used to construct a connection specifier,
	 *            will override setting specified by the deployment input to
	 *            this method (for example, it will adjust the locations to
	 *            match the connection string identified host name)
	 */
	public synchronized static void setTransitoryDeployment(ILaunchConfigurationWorkingCopy configuration,
			TPFDeployment deployment, String connectionString) {

		// Create connection specifier and retrieve deployment
		ConnectionSpecifier specifier = new ConnectionSpecifier(connectionString);

		// Retrieve connection string
		String canonicalString = specifier.getCanonicalString();

		// Obtain the location property name for connection specifiers
		String propertyName = specifier.getClass().getName();

		// Adjust deployment locations with specifier and host name updated
		EList artifactLocationPairs = deployment.getArtifactLocations();

		// If pairs is valid, iterate through each pair
		if (artifactLocationPairs != null && !artifactLocationPairs.isEmpty()) {

			// Iterate through each location pair
			for (Iterator pairs = artifactLocationPairs.iterator(); pairs.hasNext();) {

				// Retrieve associated location in pair
				CFGArtifactLocationPair pair = (CFGArtifactLocationPair) pairs.next();
				CFGLocation location = pair.getLocation();

				// Retrieve host name to use
				String hostName = specifier.getHost();

				/*
				 * Adjust location with connection string specified host name,
				 * without this inspectors of the location host names during
				 * deployment or other times in test execution might fail due to
				 * an incorrect host name being specified (since the node will
				 * be connected to the host name specified in the connection
				 * string given
				 */
				if (location instanceof CFGMachineConstraint) {
					((CFGMachineConstraint) location).setHostname(hostName);
				} else {
					location.setName(hostName);
				}

				// Retrieve primary location attribute group
				CFGPropertyGroup group = ConfigurationUtil.searchPropertyGroupById(location.getPropertyGroups(),
						ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);

				// If non-existent create the property group
				if (group == null) {
					group = Common_ConfigurationFactory.eINSTANCE.createCFGPropertyGroup();
					group.setPropertyGroupID(ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);
					location.getPropertyGroups().add(group);
				}

				// Find or create connection specifier property
				BVRProperty[] properties = ConfigurationUtil.searchPropertiesByName(group.getProperties(), propertyName,
						false);
				CFGComparableProperty property = null;
				if (properties == null || properties.length < 1) {
					property = Common_ConfigurationFactory.eINSTANCE.createCFGComparableProperty();
					property.setName(propertyName);
					property.setOperator("=");
					group.getProperties().add(property);
				} else {
					property = (CFGComparableProperty) properties[0];
				}

				// Attach value to connection specifier related property
				property.setValue(canonicalString);

			}
		}

		// Adjust configuration to reflect transitory deployment use
		List list = new ArrayList();

		// Stores the deployment in a list, cannot persist but it doesn't matter
		list.add(deployment);
		unsetEMFReference(configuration, PROPERTY_DEPLOYMENT);
		configuration.setAttribute(PROPERTY_DEPLOYMENT, (String) null);
		configuration.setAttribute("transitory", list);

	}
}
