/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionHistoryLaunchConfigurationFacade.java,v 1.8 2007/06/04 13:59:07 jptoomey Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.launch.configurations;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.core.launch.extensions.IExecutionHistoryDefaultsProvider;
import org.eclipse.hyades.test.core.launch.extensions.IExecutionHistoryDefaultsProvider2;
import org.eclipse.hyades.test.core.launch.extensions.IExecutionHistoryDefaultsProvider2Extended;
import org.eclipse.hyades.test.core.launch.extensions.IExecutionHistoryDefaultsProviderExtended;
import org.eclipse.hyades.test.core.util.EMFUtil;

/**
 * <p>This class is a facade for accessing Test Log (formerly known as Execution History)
 * information in a TPTP launch configuration.</p>
 * <p>This facade manages the following properties:
 * <ul>
 *   <li>useDefaults (boolean): Whether defaults values should be used for other properties</li>
 *   <li>executionHistoryName (String): the name of the Execution History file</li>
 *   <li>executionHistoryOverriden (boolean): whether an existing Execution History file
 *   with the same name should be overwritten or not.</li>
 *   <li>executionHistoryLocation (IContainer): a container in the workspace (project or folder)
 *   where the Test Log file should be produced.</li>
 * </ul></p>
 * <p> Note that if useDefault property is set to true, any getter will return the default
 * value, and any value set will be ignored. If useDefault is set to false, all other
 * properties are initialized to null.</p>
 * @author jcanches
 * @author Scott E. Schneider
 */
public class ExecutionHistoryLaunchConfigurationFacade extends AbstractLaunchConfigurationFacade {
	
	private static final String PROPERTY_USE_DEFAULTS = PROPERTY_PREFIX + ".useDefaults"; //$NON-NLS-1$
	private static final String PROPERTY_USER_TEH_NAME = PROPERTY_PREFIX + ".tehName"; //$NON-NLS-1$
	private static final String PROPERTY_USER_TEH_OVERRIDE = PROPERTY_PREFIX + ".tehOverride"; //$NON-NLS-1$
	private static final String PROPERTY_USER_TEH_LOCATION = PROPERTY_PREFIX + ".tehLocation"; //$NON-NLS-1$
	private static final String PROPERTY_USER_TEH_DATABASE = PROPERTY_PREFIX + ".tehDatabase"; //$NON-NLS-1$
	private static final String PROPERTY_USER_TEH_TEST_LOG_FILE = PROPERTY_PREFIX + ".tehFile"; //$NON-NLS-1$
	
	
	/**
	 * Returns the Test Log name in a TPTP launch configuration.
	 * @param configuration A Launch Configuration.
	 * @return The Test Log name, or <code>null</code> if this property was never set and
	 * <code>useDefaults</code> is <code>false</code>.
	 * @throws CoreException
	 */
	public synchronized static String getExecutionHistoryName(ILaunchConfiguration configuration) throws CoreException {
		if (isUseDefaults(configuration)) {
			return getDefaultExecutionHistoryName(configuration);
		} else {
			return configuration.getAttribute(PROPERTY_USER_TEH_NAME, (String)null);
		}
	}
	
	/**
	 * Sets the Test Log name in a launch configuration. This method has no effect is
	 * <code>useDefaults</code> is <code>true</code>.
	 * @param configuration A launch configuration working copy.
	 * @param name The name of the Test Log. A value of <code>null</code> unsets the 
	 * property.
	 * @see ExecutionHistoryLaunchConfigurationFacade#getExecutionHistoryName(ILaunchConfiguration)
	 */
	public synchronized static void setExecutionHistoryName(ILaunchConfigurationWorkingCopy configuration, String name) {
		configuration.setAttribute(PROPERTY_USER_TEH_NAME, name);
	}
	
	/**
	 * Returns whether the produced Test Log file is to be overwritten or not.
	 * If set to <code>false</code> the Test Execution Harness uses an incremental naming
	 * strategy to avoid name collisions.
	 * @param configuration A launch configuration.
	 * @return Whether the produced Test Log file is to be overwritten or not.
	 * @throws CoreException
	 */
	public synchronized static boolean isExecutionHistoryOverriden(ILaunchConfiguration configuration) throws CoreException {
		if (isUseDefaults(configuration)) {
			return getDefaultExecutionHistoryOverrideState(configuration);
		} else {
			return configuration.getAttribute(PROPERTY_USER_TEH_OVERRIDE, false);
		}
	}
	
	/**
	 * Sets whether the produced Test Log file is to be overwritten or not.
	 * @param configuration A launch configuration working copy.
	 * @param value <code>true</code> if the Test Log must be overwritten.
	 * @see ExecutionHistoryLaunchConfigurationFacade#isExecutionHistoryOverriden(ILaunchConfiguration)
	 */
	public synchronized static void setExecutionHistoryOverriden(ILaunchConfigurationWorkingCopy configuration, boolean value) {
		configuration.setAttribute(PROPERTY_USER_TEH_OVERRIDE, value);
	}
	
	/**
	 * Sets whether the produced Test Log file is to be overwritten or not.
	 * @param configuration A launch configuration working copy.
	 * @param value <code>true</code> if the Test Log must be overwritten.
	 * @see ExecutionHistoryLaunchConfigurationFacade#isExecutionHistoryDatabase(ILaunchConfiguration)
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public synchronized static void setExecutionHistoryDatabase(ILaunchConfigurationWorkingCopy configuration, boolean value) {
		configuration.setAttribute(PROPERTY_USER_TEH_DATABASE, value);
	}

	/**
	 * Returns whether the produced Test Log file is to be overwritten or not.
	 * If set to <code>false</code> the Test Execution Harness uses an incremental naming
	 * strategy to avoid name collisions.
	 * @param configuration A launch configuration.
	 * @return Whether the produced Test Log file is to be overwritten or not.
	 * @throws CoreException
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public synchronized static boolean isExecutionHistoryDatabase(ILaunchConfiguration configuration) throws CoreException {
		if (isUseDefaults(configuration)) {
			return getDefaultExecutionHistoryDatabaseState(configuration);
		} else {
			return configuration.getAttribute(PROPERTY_USER_TEH_DATABASE, false);
		}
	}

	/**
	 * Sets whether the produced Test Log file is to be overwritten or not.
	 * @param configuration A launch configuration working copy.
	 * @param value <code>true</code> if the Test Log must be overwritten.
	 * @see ExecutionHistoryLaunchConfigurationFacade#isExecutionHistoryTestLogFile(ILaunchConfiguration)
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public synchronized static void setExecutionHistoryTestLogFile(ILaunchConfigurationWorkingCopy configuration, boolean value) {
		configuration.setAttribute(PROPERTY_USER_TEH_TEST_LOG_FILE, value);
	}

	/**
	 * Returns whether the produced Test Log file is to be overwritten or not.
	 * If set to <code>false</code> the Test Execution Harness uses an incremental naming
	 * strategy to avoid name collisions.
	 * @param configuration A launch configuration.
	 * @return Whether the produced Test Log file is to be overwritten or not.
	 * @throws CoreException
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public synchronized static boolean isExecutionHistoryTestLogFile(ILaunchConfiguration configuration) throws CoreException {
		if (isUseDefaults(configuration)) {
			return getDefaultExecutionHistoryTestLogFileState(configuration);
		} else {
			return configuration.getAttribute(PROPERTY_USER_TEH_TEST_LOG_FILE, false);
		}
	}
	
	
	
	/**
	 * Returns the Test Log file location in a TPTP launch configuration.
	 * @param configuration A launch configuration.
	 * @return The Project or Folder of the Test Log file.<br>
	 * Note that the return value has changed in 4.0 version. If the launch configuration
	 * does not contain a valid value, and a default value cannot be computed,
	 * <code>null</code> is returned. In previous versions, an attempt was made to
	 * return a location derived from the current selection in the workspace.
	 * @throws CoreException
	 */
	public synchronized static IContainer getExecutionHistoryLocation(ILaunchConfiguration configuration) throws CoreException {
		if (isUseDefaults(configuration)) {
			return getDefaultExecutionHistoryLocation(configuration);
		} else {
			return getContainerProperty(configuration, PROPERTY_USER_TEH_LOCATION);
		}
	}
	
	/**
	 * Sets the Test Log file location in a TPTP launch configuration.
	 * @param configuration A launch configuration working copy.
	 * @param location This location must be either a project or a folder resource.
	 * The workspace root is not a valid argument.
	 */
	public synchronized static void setExecutionHistoryLocation(ILaunchConfigurationWorkingCopy configuration, IContainer location) {
		if (location != null && location.getType() == IResource.ROOT) {
			throw new IllegalArgumentException("location argument cannot be the Workspace Root"); //$NON-NLS-1$
		}
		setContainerProperty(configuration, PROPERTY_USER_TEH_LOCATION, location);
	}
	
	/**
	 * Returns whether the supplied Launch Configuration is using default values for
	 * the Test Log name, overwrite state, and location.
	 * @param configuration A TPTP launch configuration.
	 * @return <code>true</code> if the supplied launch configuration is using
	 * default values.
	 * @since 4.0
	 */
	public synchronized static boolean isUseDefaults(ILaunchConfiguration configuration) {
		try {
			return configuration.getAttribute(PROPERTY_USE_DEFAULTS, true);
		} catch (CoreException e) {
    		TestCorePlugin.getDefault().logError(e);
    		return true;
		}
	}
	
    /**
     * Sets whether the supplied Launch Configuration uses the default values for
     * the Test Log name, overwrite state, and location.
     * @param configuration A TPTP launch configuration working copy.
     * @param value If <code>true</code>, any value that was previously set is discarded
     * and any subsequent call to <code>getExecutionHistoryName</code>, 
     * <code>isExecutionHistoryOverriden</code> and
     * <code>getExecutionHistoryLocation</code> will return the default value.
	 * @since 4.0
     */
    public synchronized static void setUseDefaults(ILaunchConfigurationWorkingCopy configuration, boolean value) {
    	configuration.setAttribute(PROPERTY_USE_DEFAULTS, value);
    	// Clear other values from the launch configuration
    	configuration.setAttribute(PROPERTY_USER_TEH_NAME, (String)null);
    	configuration.setAttribute(PROPERTY_USER_TEH_OVERRIDE, (String)null);
    	configuration.setAttribute(PROPERTY_USER_TEH_LOCATION, (String)null);
    }

	private synchronized static IExecutionHistoryDefaultsProvider2 getExecutionHistoryDefaultsProvider2(ILaunchConfiguration configuration) {
		String launchConfigurationType = null;
		try {
			launchConfigurationType = configuration.getType().getIdentifier();
		} catch (CoreException e) {
			TestCorePlugin.getDefault().logError(e);
		}
		// Search an IExecutionHistoryDefaultsProvider2 for the launch configuration first
		if (launchConfigurationType != null) {
			IExecutionHistoryDefaultsProvider2 defaultsProvider2 = LaunchConfigurationExtensionsManager.getInstance().getExecutionHistoryDefaultsProvider2(launchConfigurationType);
			if (defaultsProvider2 != null) {
				return defaultsProvider2;
			}
		}
		return null;
	}

	private synchronized static IExecutionHistoryDefaultsProvider2Extended getExecutionHistoryDefaultsProvider2Extended(ILaunchConfiguration configuration) {
		IExecutionHistoryDefaultsProvider2 defaultsProvider2 = getExecutionHistoryDefaultsProvider2(configuration);
		if (defaultsProvider2 != null && defaultsProvider2 instanceof IExecutionHistoryDefaultsProvider2Extended)
			return (IExecutionHistoryDefaultsProvider2Extended) defaultsProvider2;
		return null;
	}
	
	/**
	 * Returns the default Test Log name for the specified launch configuration.
	 * This is the value that would be returned by <code>getExecutionHistoryName</code> if the
	 * <code>useDefaults</code> property was set to <code>true</code>.
	 * @return A non-null default name. Values returned by this method may be customized
	 * using one of the following extension points:
	 * <ul>
	 *   <li><code>launchconfigExecutionHistoryDefaults</code> if you are using the TPTP launch
	 *   configuration for launching your own types of tests</li>
	 *   <li><code>launchconfigExecutionHistoryDefaults2</code> if you are defining your own
	 *   launch configuration and re-using the Test Log tab.</li>
	 * </ul>
	 */
	public synchronized static String getDefaultExecutionHistoryName(ILaunchConfiguration configuration) {
		String name = null;
		IExecutionHistoryDefaultsProvider2 defaultsProvider2 = getExecutionHistoryDefaultsProvider2(configuration);
		if (defaultsProvider2 != null) {
			name = defaultsProvider2.getDefaultName(configuration);
		}
		
		if (name == null) {
			TPFTest test = null;
			try {
				test = TestLaunchConfigurationFacade.getTest(configuration, EMFUtil.getResourceSet());
			} catch (CoreException e) {
				TestCorePlugin.getDefault().logError(e);
			}
			IExecutionHistoryDefaultsProvider defaultsProvider = LaunchConfigurationExtensionsManager.getInstance().getExecutionHistoryDefaultsProvider(test);
			if (defaultsProvider != null) {
				name = defaultsProvider.getDefaultName(test);
			}
			if (name == null) {
				// Use the DefaultDeploymentsProvider suggestion
				name = LaunchConfigurationExtensionsManager.getInstance().getDefaultExecutionHistoryDefaultsProvider().getDefaultName(test);
			}
		}
		return name;
	}
	
	/**
	 * Returns the default Test Log Overwrite state for the specified launch configuration.
	 * This is the value that would be returned by <code>isExecutionHistoryOverriden</code> if the
	 * <code>useDefaults</code> property was set to <code>true</code>.
	 * @return The default value. If no launchconfigExecutionHistoryDefaults extension has been
	 * defined for the type of launch configuration, this value is <code>false</code>.
	 * Values returned by this method may be customized using one of the following extension points:
	 * <ul>
	 *   <li><code>launchconfigExecutionHistoryDefaults</code> if you are using the TPTP launch
	 *   configuration for launching your own types of tests</li>
	 *   <li><code>launchconfigExecutionHistoryDefaults2</code> if you are defining your own
	 *   launch configuration and re-using the Test Log tab.</li>
	 * </ul>
	 */
	public synchronized static boolean getDefaultExecutionHistoryOverrideState(ILaunchConfiguration configuration) {
		IExecutionHistoryDefaultsProvider2 defaultsProvider2 = getExecutionHistoryDefaultsProvider2(configuration);
		if (defaultsProvider2 != null) {
			return defaultsProvider2.getDefaultOverrideState(configuration);
		}
		
		TPFTest test = null;
		try {
			test = TestLaunchConfigurationFacade.getTest(configuration, EMFUtil.getResourceSet());
		} catch (CoreException e) {
			TestCorePlugin.getDefault().logError(e);
		}
		
		IExecutionHistoryDefaultsProvider defaultsProvider = LaunchConfigurationExtensionsManager.getInstance().getExecutionHistoryDefaultsProvider(test);
		// The return value is never null
		return defaultsProvider.getDefaultOverrideState(test);
	}

	/**
	 * Returns the default Test Log Database state for the specified launch configuration.
	 * This is the value that would be returned by <code>isExecutionHistoryDatabase</code> if the
	 * <code>useDefaults</code> property was set to <code>true</code>.
	 * @return The default value. If no launchconfigExecutionHistoryDefaults extension has been
	 * defined for the type of launch configuration, this value is <code>false</code>.
	 * Values returned by this method may be customized using one of the following extension points:
	 * <ul>
	 *   <li><code>launchconfigExecutionHistoryDefaults</code> if you are using the TPTP launch
	 *   configuration for launching your own types of tests</li>
	 *   <li><code>launchconfigExecutionHistoryDefaults2</code> if you are defining your own
	 *   launch configuration and re-using the Test Log tab.</li>
	 * </ul>
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public synchronized static boolean getDefaultExecutionHistoryDatabaseState(ILaunchConfiguration configuration) {
		IExecutionHistoryDefaultsProvider2Extended defaultsProvider2Extended = getExecutionHistoryDefaultsProvider2Extended(configuration);
		if (defaultsProvider2Extended != null) {
			return defaultsProvider2Extended.getDefaultDatabaseState(configuration);
		}
		TPFTest test = null;
		try {
			test = TestLaunchConfigurationFacade.getTest(configuration, EMFUtil.getResourceSet());
		} catch (CoreException e) {
			TestCorePlugin.getDefault().logError(e);
		}
		
		IExecutionHistoryDefaultsProviderExtended defaultsProvider = LaunchConfigurationExtensionsManager.getInstance().getExecutionHistoryDefaultsProviderExtended(test);
		// The return value is never null
		return defaultsProvider.getDefaultDatabaseState(test);
	}

	/**
	 * Returns the default Test Log File state for the specified launch configuration.
	 * This is the value that would be returned by <code>isExecutionHistoryTestLogFile</code> if the
	 * <code>useDefaults</code> property was set to <code>true</code>.
	 * @return The default value. If no launchconfigExecutionHistoryDefaults extension has been
	 * defined for the type of launch configuration, this value is <code>false</code>.
	 * Values returned by this method may be customized using one of the following extension points:
	 * <ul>
	 *   <li><code>launchconfigExecutionHistoryDefaults</code> if you are using the TPTP launch
	 *   configuration for launching your own types of tests</li>
	 *   <li><code>launchconfigExecutionHistoryDefaults2</code> if you are defining your own
	 *   launch configuration and re-using the Test Log tab.</li>
	 * </ul>
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public synchronized static boolean getDefaultExecutionHistoryTestLogFileState(ILaunchConfiguration configuration) {
		IExecutionHistoryDefaultsProvider2Extended defaultsProvider2Extended = getExecutionHistoryDefaultsProvider2Extended(configuration);
		if (defaultsProvider2Extended != null) {
			return defaultsProvider2Extended.getDefaultTestLogFileState(configuration);
		}
		TPFTest test = null;
		try {
			test = TestLaunchConfigurationFacade.getTest(configuration, EMFUtil.getResourceSet());
		} catch (CoreException e) {
			TestCorePlugin.getDefault().logError(e);
		}
		
		IExecutionHistoryDefaultsProviderExtended defaultsProvider = LaunchConfigurationExtensionsManager.getInstance().getExecutionHistoryDefaultsProviderExtended(test);
		// The return value is never null
		return defaultsProvider.getDefaultTestLogFileState(test);
	}
	
	
	/**
	 * Returns the default Test Log file location for the specified launch configuration.
	 * This is the value that would be returned by <code>getExecutionHistoryLocation</code> if the
	 * <code>useDefaults</code> property was set to <code>true</code>.
	 * @return A container that is not the workspace root, or null if no appropriate location
	 * could be computed from the launch configuration content.
	 * Values returned by this method may be customized using one of the following extension points:
	 * <ul>
	 *   <li><code>launchconfigExecutionHistoryDefaults</code> if you are using the TPTP launch
	 *   configuration for launching your own types of tests</li>
	 *   <li><code>launchconfigExecutionHistoryDefaults2</code> if you are defining your own
	 *   launch configuration and re-using the Test Log tab.</li>
	 * </ul>
	 */
	public synchronized static IContainer getDefaultExecutionHistoryLocation(ILaunchConfiguration configuration) {
		IContainer location = null;
		IExecutionHistoryDefaultsProvider2 defaultsProvider2 = getExecutionHistoryDefaultsProvider2(configuration);
		if (defaultsProvider2 != null) {
			location = defaultsProvider2.getDefaultLocation(configuration);
		}
		
		if (location == null) {
			TPFTest test = null;
			try {
				test = TestLaunchConfigurationFacade.getTest(configuration, EMFUtil.getResourceSet());
			} catch (CoreException e) {
				TestCorePlugin.getDefault().logError(e);
			}
			
			IExecutionHistoryDefaultsProvider defaultsProvider = LaunchConfigurationExtensionsManager.getInstance().getExecutionHistoryDefaultsProvider(test);
			// The return value (above) is never null
			location = defaultsProvider.getDefaultLocation(test);
			if (location == null) {
				// Use the DefaultDeploymentsProvider suggestion
				location = LaunchConfigurationExtensionsManager.getInstance().getDefaultExecutionHistoryDefaultsProvider().getDefaultLocation(test);
			}
		}
	
		return location;
	}

}
