/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestLaunchConfigurationFacade.java,v 1.11 2008/03/20 19:49:44 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.launch.configurations;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.test.core.internal.launch.debug.TestSourcePathProvider;
import org.eclipse.hyades.test.core.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle;
import org.eclipse.hyades.test.core.launch.extensions.ITestLaunchConfigurationValidator;
import org.eclipse.hyades.test.core.launch.extensions.ITestLaunchConfigurationValidator.Diagnostic;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;

/**
 * <p>This class is a facade for accessing Test information in a TPTP launch configuration.</p>
 * <p>This facade manages the following properties:
 * <ul>
 *   <li>test (TPFTest): the test that should be deployed and executed when launching
 *   a TPTP launch configuration.</li>
 * </ul></p>
 * @author jcanches
 */
public class TestLaunchConfigurationFacade extends AbstractLaunchConfigurationFacade {
	
	/**
	 * The id of the TPTP Test launch configuration.
	 */
	public final static String LAUNCH_CONFIGURATION_TYPE =  "org.eclipse.hyades.test.ui.launch.basicTest"; //$NON-NLS-1$
	
	public static ILaunchConfigurationType getLaunchConfigurationType() {
		return DebugPlugin.getDefault().getLaunchManager().getLaunchConfigurationType(LAUNCH_CONFIGURATION_TYPE);
	}
	
	private final static String PROPERTY_TEST = PROPERTY_PREFIX + ".test"; //$NON-NLS-1$
	
	/**
	 * Returns the Test selected in a TPTP launch configuration.
	 * @param resourceSet The Resource Set that should contain the returned object.
	 * @return A Test. If no test was previously set in the launch configuration,
	 * or if the test does not exist any more, <code>null</code> is returned.
	 * @since 3.1
	 */
	public synchronized static TPFTest getTest(ILaunchConfiguration configuration, ResourceSet resourceSet) throws CoreException {
		EObject eObject = resolveEMFReference(configuration, PROPERTY_TEST, resourceSet);
		if (eObject != null && eObject instanceof TPFTest) {
			return (TPFTest)eObject;
		}
		return null;
	}
	
	/**
	 * Sets the Test property in a TPTP Launch Configuration.
	 * @param configuration A launch configuration.
	 * @param test A test that belongs to a resource.
	 */	
	public synchronized static void setTest(ILaunchConfigurationWorkingCopy configuration, TPFTest test) {
		storeEMFReference(configuration, PROPERTY_TEST, test);
	}
	
	/**
	 * Sets the specified Launch Configuration's source path provider to TPTP default.
	 * A launch configuration's source path provider is used by the Source Lookup tab, in
	 * the launch configuration dialog, and by the debugger, during the execution,
	 * for defining the domain of code source lookup. TPTP provides a default source
	 * path provider for Test launch configurations. Invoking this method sets the
	 * launch configuration's source path provider to the TPTP default source path
	 * provider.
	 * @since 4.0
	 */
	public static void setDefaultSourcePathProvider(ILaunchConfigurationWorkingCopy configuration) {
		configuration.setAttribute(IJavaLaunchConfigurationConstants.ATTR_SOURCE_PATH_PROVIDER, TestSourcePathProvider.ID);
	}
	
	private static Diagnostic createOKDiagnostic() {
		return new Diagnostic() {
			public int getSeverity() {
				return OK;
			}
			public String getMessage() {
				return ""; //$NON-NLS-1$
			}
		};
	}
	
	/**
	 * Validates a Test launch configuration.
	 * The validation is performed by the validator registered in the extension registry
	 * using the extension point <code>org.eclipse.hyades.test.core.launchConfigValidator</code>.
	 * @param configuration A launch configuration.
	 * @param resourceSet The resource set that should be used for loading EMF objects,
	 * if required.
	 * @return The Diagnostic resulting of the validation.
	 * @since 4.2
	 */
	public static Diagnostic validate(ILaunchConfiguration configuration, ResourceSet resourceSet) throws CoreException {
		TPFTest test = getTest(configuration, resourceSet);
		if (test == null) {
			return new Diagnostic() {
				public int getSeverity() {
					return ERROR;
				}
				public String getMessage() {
					return TestCorePluginResourceBundle._EXC_BasicTestLaunchConfigurationDelegate2_nolaunchable; 
				}
			};
		}
        ITestLaunchConfigurationValidator validator = LaunchConfigurationExtensionsManager.getInstance().getLaunchConfigurationValidator(test);
        if (validator != null) {
            Diagnostic diag = validator.validate(configuration, resourceSet);
            if (diag == null) {
            	// The extension point implementor is allowed to return null if the validation
            	// returns no problem. To enforce better API pratices, we translate this into
            	// an OK diagnostic.
            	return createOKDiagnostic();
            }
            return diag;
        }
        // If no validator was registered for the type of test, then we assume that these
        // tests do not need any validation and are always OK
        return createOKDiagnostic();
	}
	
	/**
	 * Returns whether a test launch configuration references a test that has a given URI.
	 * @param configuration A test launch configuration.
	 * @param uri A URI to compare to.
	 * @return <code>true</code> if the test launch configuration references a test with the
	 * same URI as <code>uri</code>.
	 * @throws CoreException
	 */
	public static boolean isTestURI(ILaunchConfiguration configuration, URI uri) throws CoreException {
		return isSameURI(configuration, PROPERTY_TEST, uri);
	}
	
	/**
	 * Updates a test launch configuration by changing its test URI.
	 * This API is provided for refactor use cases when a test URI is changed. If the
	 * launch configuration's test must be changed, use {@link #setTest(ILaunchConfigurationWorkingCopy, TPFTest)}
	 * instead.
	 * @param configuration A test launch configuration.
	 * @param newURI The new URI of the test.
	 * @throws CoreException
	 */
	public static void updateTestURI(ILaunchConfigurationWorkingCopy configuration, URI newURI) throws CoreException {
		updateURI(configuration, PROPERTY_TEST, newURI);
	}
}
