/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AbstractLaunchConfigurationDelegate2.java,v 1.15 2008/04/10 00:59:02 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.launch.delegates;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.model.ILaunchConfigurationDelegate;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.execution.core.IExecutor;
import org.eclipse.hyades.models.common.datapool.DPLDatapool;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.internal.launch.datapool.extensions.LaunchDatapoolExtensionManager;
import org.eclipse.hyades.test.core.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.core.internal.launch.processes.TestExecutionProcess;
import org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle;
import org.eclipse.hyades.test.core.launch.configurations.DeploymentLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.configurations.TestLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.extensions.ILaunchconfigDatapoolHandler;
import org.eclipse.hyades.test.core.launch.extensions.IPassInfoCollector;
import org.eclipse.hyades.test.core.launch.extensions.IRunHandler;
import org.eclipse.hyades.test.core.launch.extensions.IRunHandler2;
import org.eclipse.hyades.test.core.launch.extensions.ITestLaunchConfigurationValidator;
import org.eclipse.hyades.test.core.testservices.resources.PasswordCollection;
import org.eclipse.osgi.util.NLS;

/**
 * This class is an abstract LaunchConfigurationDelegate that all TPTP launch configuration 
 * delegates extend. It handles the common properties and behaviors for launching
 * a TPTP launch configuration.
 * Sub-classes should implement <code>getLaunchedElement(ILaunchConfiguration)</code>,
 * <code>invokeTestExecutionHarness(ILaunchConfiguration, String, StringBuffer)</code>.
 * They should usually also override <code>validate(ILaunchConfiguration)</code>.
 * 
 * 
 * @author  Julien Canches
 * @author  Paul E. Slauenwhite
 * @version April 3, 2008
 * @since   February 8, 2005
 */
public abstract class AbstractLaunchConfigurationDelegate2 implements ILaunchConfigurationDelegate {

	private List problems;
	
	private ResourceSet resourceSet;
	
	/**
	 * Get the Test Element that the configuration will launch. This Test Element is used to
	 * determine which runHandler will be invoked before the Execution Harness is called.
	 * @param configuration A Hyades Launch Configuration
	 * @return A typed Test Element (instance of TPFTest or TPFTestComponent)
	 * @throws CoreException
	 */
	protected abstract Object getLaunchedElement(ILaunchConfiguration configuration) throws CoreException;
	
	/**
	 * Validate a given configuration. This method checks that the configuration contains the
	 * necessary information for running a test.
	 * Sub-classes should add more checks by overriding this method, and invoking the protected
	 * method <code>reportProblem(String)</code> (one or more times) for reporting any problems
	 * found in the configuration. If this method is not invoked, the configuration is considered
	 * as valid. Overriding methods should invoke the parent method using
	 * <code>super.validate(configuration, mode)</code>.<br>
	 * This implementation checks the following points:
	 * <ul>
	 *   <li>The configuration specifies an element to launch</li>
	 *   <li>The element to launch can be launched in the specified mode</li>
	 *   <li>The validator registered for the test element type successfully validates the configuration</li>
	 * </ul>
	 * @param configuration A Hyades Launch Configuration
	 * @param mode The run mode the Launch Configuration should be running in.
	 * @throws CoreException
	 */
	protected void validate(ILaunchConfiguration configuration, String mode) throws CoreException {
		Object launchedElement = getLaunchedElement(configuration);
		if (launchedElement == null) {
			reportProblem(TestCorePluginResourceBundle._EXC_BasicTestLaunchConfigurationDelegate2_nolaunchable); 
		}
		
		if (!LaunchConfigurationExtensionsManager.getInstance().isLaunchableElement(launchedElement, mode, configuration.getType().getIdentifier())) {
			reportProblem(NLS.bind(TestCorePluginResourceBundle._EXC_AbstractLaunchConfigurationDelegate_modeNotSupported, mode)); 
		}
		
        // Invoke an extension point's validator
        ITestLaunchConfigurationValidator validator = LaunchConfigurationExtensionsManager.getInstance().getLaunchConfigurationValidator(launchedElement);
        if (validator != null) {
            ITestLaunchConfigurationValidator.Diagnostic diag = validator.validate(configuration, getResourceSet());
            if (diag != null && diag.getSeverity() == ITestLaunchConfigurationValidator.Diagnostic.ERROR) {
                reportProblem(diag.getMessage());
            }
        }

	}
	
	/**
	 * Invokes the Test Execution Harness with the appropriate information from the
	 * given Launch Configuration, in the specified mode.
	 * @param configuration A Hyades Launch Configuration
	 * @param mode The run mode the Launch Configuration should be running in.
	 * @param errorMessages This is initially an empty buffer that this method should fill
	 * with the errors reported by the Test Execution Harness. The Test Execution Harness will be
	 * considered as failed if this buffer is not empty after the call to this method.
	 * @return An IExecutor instance. This instance represents the executor of the test. Its state
	 * will be monitored to display its state in the Debug View. null is accepted: in that case,
	 * the Test Execution Progress will not be monitored.
	 * @throws CoreException
	 */
	protected abstract IExecutor invokeTestExecutionHarness(ILaunchConfiguration configuration, String mode, StringBuffer errorMessages, IProgressMonitor monitor) throws CoreException;
	
	/**
	 * This method is used when validating a configuration for reporting problems.
	 * This method may be invoked several times by <code>validate</code> for reporting
	 * several problems.
	 * Invoking this method from another method than validate has no effect.
	 * @param description A description of the problem to report.
	 */
	protected void reportProblem(String description) {
		problems.add(description);
	}
	
	/**
	 * Allows to reset any private field. This method may be overrided, but sub-classes should not
	 * omit the call to super.resetFields().
	 */
	protected void resetFields() {
		this.problems = new ArrayList();
		this.resourceSet = null;
	}
	
	/**
	 * Returns the Delegate's private Resource Set. A delegate instantiates its own 
	 * ResourceSet at each launch. The Resource Set is unique during the launch process.
	 * @return
	 */
	protected ResourceSet getResourceSet() {
		if (this.resourceSet == null) {
			this.resourceSet = new ResourceSetImpl();
		}
		return this.resourceSet;
	}
	
	/**
	 * This method is not meant to be overrided. It is invoked by the Debug UI Plugin.
	 */
	public void launch(ILaunchConfiguration configuration, String mode,	ILaunch launch, IProgressMonitor monitor) throws CoreException {
		monitor.beginTask(TestCorePluginResourceBundle.BasicTestLaunchConfigurationDelegate_monitorLaunchingTest, 3); 
		try {
			// Initialize all fields
			this.resetFields();
			
			// Validate the configuration
			this.validate(configuration, mode);
			if (problems.size() == 1) {
				String problem = (String)problems.get(0);
				IStatus status = new Status(IStatus.ERROR, TestCorePlugin.getPluginId(), 0, problem, null);
				throw new CoreException(status);
			} else if (problems.size() > 1) {
				MultiStatus status = new MultiStatus(TestCorePlugin.getPluginId(), 0, TestCorePluginResourceBundle._EXC_AbstractLaunchConfigurationDelegate_configurationProblems, null); 
				Iterator it = problems.iterator();
				while (it.hasNext()) {
					String problem = (String) it.next();
					IStatus subStatus = new Status(IStatus.ERROR, TestCorePlugin.getPluginId(), 0, problem, null);
					status.add(subStatus);
				}
				throw new CoreException(status);
			}
			
			// Pre-run extension
	    	IRunHandler runHandler = LaunchConfigurationExtensionsManager.getInstance().getRunHandler(getLaunchedElement(configuration));
	    	if (runHandler != null) {
	    		try {
	    			runHandler.preRun(configuration, mode, launch, getResourceSet(), new SubProgressMonitor(monitor, 1));
	    		} catch (Throwable t) {
	    			if (t instanceof CoreException) throw (CoreException)t;
	    			IStatus status = new Status(IStatus.ERROR, TestCorePlugin.getPluginId(), 0, TestCorePluginResourceBundle._EXC_AbstractLaunchConfigurationDelegate_preRunHandlerException, t); 
	    			throw new CoreException(status);
	    		}
	    	}
			
	    	// find the datapools which is encrypted   	
	    	ILaunchconfigDatapoolHandler datapoolHandler = LaunchDatapoolExtensionManager.getInstance().getRunHandler(getLaunchedElement(configuration));
			DPLDatapool[] dp = null;
			if(datapoolHandler != null){
				dp = datapoolHandler.getAllDatapools(TestLaunchConfigurationFacade.getTest(configuration, getResourceSet()), 
						DeploymentLaunchConfigurationFacade.getDeployment(configuration, getResourceSet()));				
			}
			HashMap pass = PasswordCollection.getInstance().getDatapoolPassword();
			
			// show a dialog to collect the passwords
			if(dp!=null && dp.length > 0){
				IPassInfoCollector collector = LaunchDatapoolExtensionManager.getInstance().getCollector(getLaunchedElement(configuration));
				List isContinue = new ArrayList();
				isContinue.add(TestLaunchConfigurationFacade.getTest(configuration, getResourceSet()).getImplementor().getId());
				collector.execute(dp, pass, isContinue);
				if(isContinue.size()==0)				
					return;
			}				
			// Invoke the Test Execution Harness
			StringBuffer errorMessages = new StringBuffer();
			IExecutor executor = null;
			try {				
				executor = invokeTestExecutionHarness(configuration, mode, errorMessages, new SubProgressMonitor(monitor, 1, SubProgressMonitor.PREPEND_MAIN_LABEL_TO_SUBTASK));
				if (errorMessages.length() > 0) {
					IStatus status = new Status(IStatus.ERROR, TestCorePlugin.getPluginId(), 0, TestCorePluginResourceBundle._EXC_AbstractLaunchConfigurationDelegate_testHarnessProblems.trim() + " " + errorMessages, null); 
		    		throw new CoreException(status);
				}
			} catch (Throwable t) {
				if (t instanceof CoreException) throw (CoreException)t;
				IStatus status = new Status(IStatus.ERROR, TestCorePlugin.getPluginId(), 0, TestCorePluginResourceBundle._EXC_AbstractLaunchConfigurationDelegate_testHarnessProblems.trim() + " " + t.getClass().getName(), t); 
	    		throw new CoreException(status);
			}
			
			if (executor != null) {
				launch.addProcess(new TestExecutionProcess(executor, launch, getResourceSet()));
				
				if (runHandler != null && runHandler instanceof IRunHandler2) {
					((IRunHandler2)runHandler).postLaunch(launch, executor, new SubProgressMonitor(monitor, 1, SubProgressMonitor.PREPEND_MAIN_LABEL_TO_SUBTASK));
				}
			}
			
		} finally {
			monitor.done();
		}
	}
	
}
