/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BasicTestLaunchConfigurationDelegate2.java,v 1.18 2008/04/18 20:14:54 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.launch.delegates;

import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.hyades.execution.core.IExecutor;
import org.eclipse.hyades.execution.harness.ITestExecutionHarnessOptions;
import org.eclipse.hyades.execution.harness.TestExecutionHarness;
import org.eclipse.hyades.execution.harness.XMLExecutionDataProcessor;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.util.SaveManager;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.TestCorePreferences;
import org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle;
import org.eclipse.hyades.test.core.launch.configurations.DeploymentLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.configurations.ExecutionHistoryLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.configurations.TestLaunchConfigurationFacade;

/**
 * This class is the delegate for the Test launch configuration.
 * It handles the launching of a TPTP Test according to a Test Launch Configuration.
 * @see org.eclipse.debug.core.model.ILaunchConfigurationDelegate
 * @author jcanches
 */
public class BasicTestLaunchConfigurationDelegate2 extends AbstractLaunchConfigurationDelegate2 {
	
    /**
     * Create a Test launch configuration delegate.
     */
    public BasicTestLaunchConfigurationDelegate2() {
    }

    /**
     * Returns the test launched by the Test launch configuration. This is a conveniency method
     * for sub-classes.
     * @throws CoreException
     * @see TestLaunchConfigurationFacade#getTest(ILaunchConfiguration, org.eclipse.emf.ecore.resource.ResourceSet)
     */
    protected TPFTest getLaunchedTest(ILaunchConfiguration configuration) throws CoreException {
    	return TestLaunchConfigurationFacade.getTest(configuration, getResourceSet());
    }
    
     /**
     * Returns the deployment used for launching the test. This is a conveniency method for
     * sub-classes.
     * @throws CoreException
     * @see DeploymentLaunchConfigurationFacade#getDeployment(ILaunchConfiguration, org.eclipse.emf.ecore.resource.ResourceSet)
     */
    protected TPFDeployment getDeployment(ILaunchConfiguration configuration) throws CoreException {
    	return DeploymentLaunchConfigurationFacade.getDeployment(configuration, getResourceSet());
    }
    
    /**
     * Returns the Test Log name. This is a conveniency method for sub-classes.
     * @throws CoreException
     * @see ExecutionHistoryLaunchConfigurationFacade#getExecutionHistoryName(ILaunchConfiguration)
     */
    protected String getTestExecutionHistoryName(ILaunchConfiguration configuration) throws CoreException {
    	return ExecutionHistoryLaunchConfigurationFacade.getExecutionHistoryName(configuration);
    }
    
    /**
     * Returns the Test Log file location. This is a conveniency method for sub-classes.
     * @throws CoreException
     * @see ExecutionHistoryLaunchConfigurationFacade#getExecutionHistoryLocation(ILaunchConfiguration)
     */
    protected IContainer getTestExecutionHistoryLocation(ILaunchConfiguration configuration) throws CoreException {
    	return ExecutionHistoryLaunchConfigurationFacade.getExecutionHistoryLocation(configuration);
    }   
    
    /**
     * Returns the Test Log "overwrite" property. This is a convenience method for sub-classes.
     * @throws CoreException
     * @see ExecutionHistoryLaunchConfigurationFacade#isExecutionHistoryOverriden(ILaunchConfiguration)
     */
    protected boolean overrideExistingTestExecutionHistory(ILaunchConfiguration configuration) throws CoreException {
    	return ExecutionHistoryLaunchConfigurationFacade.isExecutionHistoryOverriden(configuration);
    }

    /**
     * Returns the Test Log "database" property. This is a convenience method for sub-classes.
     * @throws CoreException
     * @see ExecutionHistoryLaunchConfigurationFacade#isExecutionHistoryDatabase(ILaunchConfiguration)
     * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
     */
    protected boolean isTestExecutionHistoryDatabase(ILaunchConfiguration configuration) throws CoreException {
    	return ExecutionHistoryLaunchConfigurationFacade.isExecutionHistoryDatabase(configuration);
    }

    /**
     * Returns the "Test Log File" property. This is a convenience method for sub-classes.
     * @throws CoreException
     * @see ExecutionHistoryLaunchConfigurationFacade#isExecutionHistoryTestLogFile(ILaunchConfiguration)
     * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
     */
    protected boolean isTestExecutionHistoryTestLogFile(ILaunchConfiguration configuration) throws CoreException {
    	return ExecutionHistoryLaunchConfigurationFacade.isExecutionHistoryTestLogFile(configuration);
    }

    /**
     * This implementation returns the test launched by the specified launch configuration.
     */
	protected Object getLaunchedElement(ILaunchConfiguration configuration)	throws CoreException {
		return getLaunchedTest(configuration);
	}
	
	/**
	 * In addition to the points checked by the super class, this implementation
	 * checks the following points:
	 * <ul>
	 *   <li>The deployment is not <code>null</code></li>
	 *   <li>The Test Log name is neither empty nor <code>null</code></li>
	 *   <li>The Test Log file location is not <code>null</code></li>
	 * </ul>
	 */
    protected void validate(ILaunchConfiguration configuration, String mode) throws CoreException {
    	super.validate(configuration, mode);
   	
    	if (getDeployment(configuration) == null) {
			reportProblem(TestCorePluginResourceBundle._EXC_BasicTestLaunchConfigurationDelegate2_noDeployment); 
		}
    	
    	String tehName = getTestExecutionHistoryName(configuration);
    	if (tehName == null || tehName.equals("")) { //$NON-NLS-1$
    		reportProblem(TestCorePluginResourceBundle._EXC_BasicTestLaunchConfigurationDelegate2_noExecName); 
    	}
    	
    	IContainer tehLocation = getTestExecutionHistoryLocation(configuration);
    	if (tehLocation == null) {
    		reportProblem(TestCorePluginResourceBundle._EXC_BasicTestLaunchConfigurationDelegate2_noExecLocation); 
    	} else if (!tehLocation.exists()) {
    		reportProblem(TestCorePluginResourceBundle._EXC_BasicTestLaunchConfigurationDelegate2_locationDNE); 
    	}
    }
	
    
    /**
     * <p>This implementation invokes the Test Execution Harness to launch the test specified
     * by a Test launch configuration, using the Test launch configuration properties.</p>
     * <p>In run mode, the type of test passed to the Execution Harness is the type of test test. In
     * debug or profile mode, the type passed to the Execution Harness is the type of the, appended
     * with ".debug" or ".profile" respectively.<br>
     * For instance, if the type of the test "org.myorg.mytest" and the delegate is invoked
     * in debug mode, the type passed to the execution harness is "org.myorg.mytest.debug". 
     * @see org.eclipse.debug.core.model.ILaunchConfigurationDelegate#launch
     */
    public IExecutor invokeTestExecutionHarness(ILaunchConfiguration configuration, String mode, StringBuffer errorMessages, IProgressMonitor monitor) throws CoreException {
		// Compute parameters
    	TPFTest test = getLaunchedTest(configuration);
    	TPFTestSuite testSuite = null;
    	TPFTest subTest = null;
    	if (test instanceof TPFTestCase) {
    		subTest = test;
    		testSuite = ((TPFTestCase)subTest).getTestSuite();
    	} else if (test instanceof TPFTestSuite) {
    		testSuite = (TPFTestSuite)test;
    		subTest = test;
    	}
		String testType = testSuite.getType();
		if (ILaunchManager.DEBUG_MODE.equals(mode)) {
			testType += ".debug"; //$NON-NLS-1$
		} else if (ILaunchManager.PROFILE_MODE.equals(mode)) {
			testType += ".profile"; //$NON-NLS-1$
		}
		
    	TPFDeployment deployment = getDeployment(configuration);	    	
    	String tehName = getTestExecutionHistoryName(configuration);
    	String tehLocation = getTestExecutionHistoryLocation(configuration).getFullPath().toString();
    	boolean overrideExistingExecHistory = overrideExistingTestExecutionHistory(configuration);
    	boolean database = isTestExecutionHistoryDatabase(configuration);
    	boolean testLogFile = isTestExecutionHistoryTestLogFile(configuration);
    	String port = TestCorePlugin.getDefault().getPluginPreferences().getString(TestCorePreferences.DEFAULT_PORT);
    	
		TestExecutionHarness teh = new TestExecutionHarness();
		ArrayList activeDataProcessorIDs = new ArrayList();
		activeDataProcessorIDs.add(XMLExecutionDataProcessor.IID);
		
		HashMap map = new HashMap();
		map.put(ITestExecutionHarnessOptions.DATABASE, Boolean.toString(database));
		map.put(ITestExecutionHarnessOptions.TEST_LOG_FILE, Boolean.toString(testLogFile));
		
		IExecutor executor = teh.launchTest(
			(ITestSuite)testSuite,
			(ITest)subTest,
			testType,
			deployment,
			port,
			tehLocation,
			tehName,
			overrideExistingExecHistory,
			/*standalone*/false,
			activeDataProcessorIDs,
			map,
			errorMessages,
			monitor);
		
		if (!monitor.isCanceled() && !testLogFile) {
			SaveManager.getInstance().start(TestCorePlugin.getDefault().getPluginPreferences().getLong(TestCorePreferences.SAVE_INTERVAL));
		}
		return executor;
    }

}
