/********************************************************************** 
 * Copyright (c) 2005, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AbstractAgentControllerVerificationService.java,v 1.10 2009/11/23 20:40:45 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.core.services;

import java.util.Properties;

import org.eclipse.core.runtime.IPlatformRunnable;
import org.eclipse.hyades.automation.client.adapters.java.AutomationClientAdapter;
import org.eclipse.hyades.automation.server.AbstractRelaunchableService;
import org.eclipse.hyades.execution.core.util.ConnectionSpecifier;
import org.eclipse.hyades.execution.core.util.MutableObject;

/**
 * The abstract agent controller verification service provides a base
 * implementation of verifying that the agent controller is and and running
 * 
 * One input property of 'connection' and one output property of 'verified', the
 * 'verified' property is set to true if the agent controller is available and
 * also the file system services file server is running
 * 
 * The service identity is 'org.eclipse.hyades.test.core.verify'
 * 
 * @author  Scott E. Schneider
 * @author  Paul E. Slauenwhite
 * @version November 23, 2009
 * @since   March 13, 2006
 */
public abstract class AbstractAgentControllerVerificationService extends
		AbstractRelaunchableService {

	private static final long serialVersionUID = 5655532375993506028L;
	
	/**
	 * Specifies the connection string to use to attempt connection with the
	 * agent controller
	 */
	private String connectionSpecifier;

	/**
	 * Default constructor invokes up hierarchy for any initialization required
	 */
	protected AbstractAgentControllerVerificationService() {
		super();
	}

	/**
	 * Cleanup after service executes
	 */
	protected void cleanup() {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service#execute()
	 */
	public final Object execute() {

		// Entire execution in throwable catch block
		try {

			// Execute common execute behavior before service specific
			Object object = super.execute();

			// If base class decides to abort this execution early (to restart)
			//NOTE: When this is changed to IApplication.EXIT_OK, we will need to change it everywhere otherwise this condition may return false since the objects are not the same:
			if (object != IPlatformRunnable.EXIT_OK) {
				return object;
			}

			// Verification started
			this.println("Verification started!");

			// Store connection specifier
			this.connectionSpecifier = this.getProperty("connection");

			// Verify agent controller is up and running as instructed
			boolean verified = this.verify(this.connectionSpecifier);

			// Output verification findings
			this
					.println("The specified agent controller '"
							+ this.connectionSpecifier
							+ "' "
							+ (verified ? "is up and running!"
									: "The specified agent controller cannot be contacted!"));

			// Verification completed
			this.println("Verification completed!");

			// Set return value verified in properties
			this.getProperties().setProperty("verified",
					String.valueOf(verified));

		} catch (Throwable t) {

			// Handle throwable during execution
			this.handleThrowable(t);

			// If a throwable is caught, set verified to false
			this.getProperties().setProperty("verified", String.valueOf(false));

		} finally {

			// Invoke cleanup
			this.cleanup();

		}

		// Prepare and set return state
		return this.returnResult();

	}

	/**
	 * Default outputs throwable to standard console out
	 * 
	 * @param t
	 *            the throwable caught to be handled
	 */
	protected void handleThrowable(Throwable t) {
		t.printStackTrace();
	}

	/**
	 * Derive result to return
	 * 
	 * @return the result to return, service-level not semantic-level per
	 *         service domain
	 */
	protected Object returnResult() {
		return this;
	}

	/**
	 * Verify the agent controller specified by the given connection specifier
	 * is up and running
	 */
	protected boolean verify(String connection) {

		/*
		 * Prepare to use the headless Agent Controller connector service, an automation
		 * service used to connect to the Agent Controller (secure or unsecure)
		 * and returns back the node once connected.
		 */
		AutomationClientAdapter client = new AutomationClientAdapter();

		// Create a specifier to parse the connection string
		ConnectionSpecifier specifier = new ConnectionSpecifier(connection);
		this.println("Verifying connection:");
		this.println(specifier.toString());

		// Configure state to pass into the connect service
		Properties properties = new Properties();
		properties.setProperty("host", specifier.getHost());
		properties.setProperty("port", String.valueOf(specifier.getPort()));
		if (specifier.getUser() != null) {
			properties.setProperty("user", specifier.getUser());
		}
		if (specifier.getPassword() != null) {
			properties.setProperty("password", specifier.getPassword());
		}

		MutableObject mutableNode = new MutableObject();
		properties.put("mutableNode", mutableNode);

		// Execute the headless Agent Controller connector service to obtain node
		try {

			// Connect to the headless Agent controller and store the node returned
			client.execute("org.eclipse.tptp.platform.common.headlessConnector", properties);

		} catch (Throwable t) {

			// If exception, output to console
			t.printStackTrace();
			return false;

		}

		// If node is connected and not null, its verified functional
		return !mutableNode.isNull();

	}

}
