/********************************************************************** 
 * Copyright (c) 2005, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AbstractTestExecutionService.java,v 1.38 2010/04/12 12:38:44 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.core.services;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPlatformRunnable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.automation.client.adapters.java.AutomationClientAdapter;
import org.eclipse.hyades.automation.server.AbstractProjectSensitiveService;
import org.eclipse.hyades.execution.core.util.ConnectionSpecifier;
import org.eclipse.hyades.execution.core.util.MutableObject;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.configuration.util.ConfigurationUtil;
import org.eclipse.hyades.models.common.facades.behavioral.impl.DatapoolFacadeResourceFactoryImpl;
import org.eclipse.hyades.models.common.facades.behavioral.impl.FacadeResourceFactoryImpl;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.testprofile.impl.Common_TestprofilePackageImpl;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.models.hierarchy.util.HyadesExtendedResourceFactory;
import org.eclipse.hyades.models.util.ModelDebugger;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.core.internal.launch.processes.TestExecutionProcess;
import org.eclipse.hyades.test.core.launch.configurations.DeploymentLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.configurations.ExecutionHistoryLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.configurations.TestLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.services.internal.util.ServiceUtil;
import org.eclipse.hyades.test.core.util.EMFUtil;

/**
 * <p>Abstract class that all test core execution related services should extend,
 * common execution related methods and state are located in this class. The
 * idea is that this class will define a template method with associated
 * abstract or default implementation concrete methods that will be extended by
 * the various test types and situations. It might very well be the case that
 * the default is fine for most test types without having to extend this class.</p>
 * 
 * <p>For the service properties, variables named specifiers indicate zero to one
 * values whereas a selector is interpreted as a variable that can have zero to
 * many values.</p>
 * 
 * <p>The template method is the execute method and is marked final, for subclasses
 * to participate as test execution services they must override the template
 * method associate methods in a way to receive the desired results. The
 * associated template methods are listed below and referenced in order that they
 * are invoked, for the latest order please verify with the code in this
 * classes' execute method.</p>
 * 
 * <p>Note that the suite property takes either a string or a list of strings to
 * specify the tests to execute and that the results property is also used as an
 * list out value (a list of strings naming the results)</p>
 * 
 * <p>Template method methods:</p>
 *  
 * <ul>
 * <li>{@link #initialize()}</li>
 * <li>{@link #beforeLaunch(String)}</li>
 * <li>{@link #launch(ILaunchConfiguration, String, IProgressMonitor)}</li>
 * <li>{@link #afterLaunch(ILaunch)}</li>
 * <li>{@link #afterExecution()}</li>
 * <li>{@link #handleThrowable(Throwable)}</li>
 * <li>{@link #cleanup()}</li>
 * <li>{@link #returnResult()}</li>
 * </ul>
 * 
 * <p>Auxiliary supporting associated template method methods:</p>
 * 
 * <li>{@link #waitForCompletion(ILaunch, int)}</li>
 * <li>{@link #loadDeployment(IProject, String)}</li>
 * <li>{@link #loadModelElement(IProject, String)}</li>
 * <li>{@link #loadTestSuite(IProject, String)}</li>
 * <li>{@link #getProgressMonitor()}</li>
 * <li>{@link #getWaitForCompletionInterval()}</li>
 * <li>{@link #createDerivedTestSuiteFile(IProject, String, boolean)}</li>
 * 
 * 
 * @author  Scott E. Schneider
 * @author  Paul E. Slauenwhite
 * @author	Jerome Bozier
 * @version April 12, 2010
 * @since   October 12, 2005
 */
public abstract class AbstractTestExecutionService extends
		AbstractProjectSensitiveService {

	private static final long serialVersionUID = -7354518997777478197L;

	/**
	 * Time to wait between checks on the test execution processes is terminated
	 * method, this should be switched to a notify rather than polling strategy
	 * in the test execution process implementation
	 */
	private static final int DEFAULT_POLL_INTERVAL = 1000;

	/**
	 * The classpath specifier, if specified, specifies the classpath that will
	 * be prepended to any previously associated classpath with the test
	 * execution of interest.
	 */
	protected String classpathSpecifier;

	/**
	 * The configuration selector, if specified, specifies the launch
	 * configuration to be used as the starting point for this test execution.
	 * It is possible that this selector identifies more than one configuration.
	 * The concrete services interpret how to use the selector assigned.
	 */
	protected String configurationSelector;

	/**
	 * The connection specifier, if specified, shadows the previously associated
	 * and store connection-related state, this specifier can suggest scheme,
	 * server class, host, port and server instance. Scheme is almost always set
	 * to "TPTP" whereas, server class is "RAC, IAC, etc" and host is the
	 * network host name and port is the port number to connect to the agent
	 * controller on, the server instance is "default"
	 */
	protected String connectionSpecifier;

	/**
	 * The deployment selector, if specified, identifies one to potentially many
	 * deployments involved in this test execution, the concrete service
	 * determines how to use the selector specified.
	 */
	protected String deploymentSelector;

	/**
	 * Result list to return to callers of the service
	 */
	protected List results;
	
	
	/**
	 * Result string containing file system path to project for locating
	 * project relative result files in results member
	 * 
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	protected String resultProjectDir;

	/**
	 * The overwrite specifier, indicates if results should be overwritten each
	 * time or if result names will be generated in order to preserve the
	 * original results and not overwrite them.
	 */
	protected boolean overwriteSpecifier;

	/**
	 * <p>Optional <code>results</code> property representing the workspace-relative 
	 * (prefixed with the project name) or project-relative (relative to a project) 
	 * path to the execution histories.  Possible path values include:</p>
	 * 
	 * <ul>
	 * <li>Directory:
	 * <ul>
	 * <li>Directory or path to a directory with a trailing path separator character representing 
	 * the execution history location. Execution histories are named using the default execution 
	 * history name (see {@link ExecutionHistoryLaunchConfigurationFacade.getDefaultExecutionHistoryName(ILaunchConfiguration)})
	 * for all test executions.  For example:</li>
	 * 
	 * <ul>
	 * <li><code>[\]&lt;empty&gt;</code></li>
	 * <li><code>[\]&lt;execution history name&gt;[.execution]</code></li>
	 * <li><code>[\]&lt;directory&gt;\[&lt;execution history name&gt;[.execution]]</code></li>
	 * <li><code>[\]&lt;project name&gt;\[&lt;execution history name&gt;[.execution]]</code></li>
	 * <li><code>[\]&lt;project name&gt;\&lt;directory&gt;\[&lt;execution history name&gt;[.execution]]</code></li>
	 * </ul>
	 * </ul>
	 * 
	 * <li>File:
	 * <ul>
	 * <li>File name or path to a file name with an optional <code>.execution</code> file extension 
	 * (no trailing path separator character) representing the execution history location
	 * and execution history name for all test executions.  For example:</li>
	 * <ul>
	 * 
	 * <li><code>[\]&lt;execution history name&gt;[.execution]</code></li>
	 * <li><code>[\]&lt;directory&gt;\&lt;execution history name&gt;[.execution]</code></li>
	 * <li><code>[\]&lt;project name&gt;\&lt;execution history name&gt;[.execution]</code></li>
	 * <li><code>[\]&lt;project name&gt;\&lt;directory&gt;\&lt;execution history name&gt;[.execution]</code></li>
	 * </ul>
	 * </ul>
	 * </ul>
	 * 
	 * <p><b>Notes:</b></p>
	 * 
	 * <ol>
	 * <li>The project must exist in the workspace.</li>
	 * <li>The project-relative directory structure must exist in the workspace.</li>
	 * <li>Path separator characters are platform-independent (slash ('/') or backslash ('\') characters are interchangeable).</li>
	 * <li>The path may contain an optional leading path separator character.</li>
	 * </ol>
	 */
	protected String resultsSpecifier;

	/**
	 * The test suite selector, if specified, identifies one to potentially many
	 * test suites involved in this test execution, the concrete service
	 * determines how to use the selector specified. The suite selector is a
	 * list of strings.
	 */
	protected List suiteSelector;

	/**
	 * Default constructor invokes up hierarchy for any initialization required
	 */
	protected AbstractTestExecutionService() {
		super();
	}

	/**
	 * Executed after the execution completion has been waited on and it is
	 * found to be complete. This might also be reached in cases where
	 * subclasses override the behavior of the after launch method -- making
	 * that particular method not block for the entire length of the test. In
	 * this case, this method is most appropriately during execution.
	 */
	protected EObject afterExecution(IProcess process) {

		// Output end of test information
		if (process != null) {
			if (process instanceof TestExecutionProcess) {
				if (!this.isQuiet()) {
					TPFExecutionResult result = ((TestExecutionProcess) process)
							.getExecutionResult();
					if (result != null) {
						this.println("Test result '" + result.getName() + ".execution' collected.");
					} else {
						this.println("No results collected!");
					}

					// Return execution results
					return result;
				}
			}
		}

		// For exceptional situations
		return null;

	}

	/**
	 * Invoke after the launch but before the execution is waited on and
	 * completed.
	 */
	protected IProcess afterLaunch(ILaunch launch) {

		// Retrieve test execution process
		IProcess process = this.getProcessOfInterest(launch);

		// Wait for test to complete and results to collect
		this.waitForCompletion(process, this.getWaitForCompletionInterval());

		// Return process for use by the after execution method
		return process;

	}

	/**
	 * Before the execution has launched but after the initialize method sets
	 * up, could be used to select the actual instance objects to use based on
	 * the various selector and specifier variables before a launch.
	 */
	protected ILaunchConfiguration beforeLaunch(String suiteSpecifier)
			throws CoreException {

		ILaunchManager manager = DebugPlugin.getDefault().getLaunchManager();
		ILaunchConfigurationWorkingCopy configuration = null;
		TPFTestSuite suite = null;
		IProject theProject = this.project;
		
		// If we are executing with a launch configuration, set up the
		// necessary values now.
		if (this.configurationSelector != null) {
			if (theProject == null) {
				// Here we assume that since the project is not specified, the path 
				// passed in for the launch configuration is a relative path that
				// begins with the project name as the first segment.
				File configFile = new File(this.configurationSelector);
				if (!configFile.isAbsolute()) {
					theProject = getProjectFromWorkspaceRelativePath(this.configurationSelector);
					String launchFile = getFilePathFromWorkspaceRelativePath(this.configurationSelector);
					
					// Create a temporary launch configuration
					configuration = manager.getLaunchConfiguration(
							theProject.getFile(launchFile))
							.getWorkingCopy();
					suite = (TPFTestSuite) TestLaunchConfigurationFacade.getTest(configuration, new ResourceSetImpl());
				}
				else {
					// throw an exception
				}
					
			}
		}
		else {
			// Load the suite to use in this launch
			suite = this.loadTestSuite(theProject, suiteSpecifier);
			
		}
		
		// Ensure project is synchronized with file system
		this.refreshProject(theProject);

		// Retrieve the test type
		String launchConfigTypeID = LaunchConfigurationExtensionsManager.getInstance().getLaunchConfigurationType(suite);

		// Based on type, retrieve correct launch configuration type
		ILaunchConfigurationType type = manager
				.getLaunchConfigurationType(launchConfigTypeID);

		// If we didn't a launch config as a parameter, create a temporary launch configuration
		if (configuration == null) {
			configuration = type.newInstance(null, manager.generateUniqueLaunchConfigurationNameFrom("automation"));

			// Write test identifier to launch configuration
			if (suite != null) {
				TestLaunchConfigurationFacade.setTest(configuration, suite);
			}
		}

		// Use default deployment if not overridden
		if (this.deploymentSelector != null) {
			DeploymentLaunchConfigurationFacade.setDeployment(configuration,
					this.loadDeployment(theProject, this.deploymentSelector));
		} else {

			// If configuration option not give, deployment needs default
			if (this.configurationSelector == null) {
				DeploymentLaunchConfigurationFacade.setDeployment(
						configuration, ConfigurationUtil
								.createDefaultDeployment());
			}

		}

		// Take optional connection string and apply to deployment locations
		if (this.connectionSpecifier != null) {

			// Retrieve deployment in effect for the specified configuration
			TPFDeployment deployment = DeploymentLaunchConfigurationFacade
					.getDeployment(configuration, EMFUtil.getResourceSet());

			/*
			 * Set transitory deployment accounting for connection specifier,
			 * this will take the input deployment and adjust it for the given
			 * connection string, such as change the location host names to
			 * match the desired host names in the connection string
			 */
			DeploymentLaunchConfigurationFacade.setTransitoryDeployment(
					configuration, deployment, this.connectionSpecifier);

			// Create connection specifier to inspect parameters
			ConnectionSpecifier specifier = new ConnectionSpecifier(this.connectionSpecifier);
			
			// If username and password are specified, then we are probably
			// trying to connect to a secure agent controller.  The normal 
			// execution code path will not provide the username and password
			// and instead will try to prompt with an input dialog.
			// We will bypass that by pre-connecting with the username and
			// password, and the connection will be re-used for test execution.
			if (specifier.getUser() != null && specifier.getPassword() != null) {
				AutomationClientAdapter client = new AutomationClientAdapter();
	
				// Configure state to pass into the connect service
				Properties properties = new Properties();
				properties.setProperty("host", specifier.getHost());
				properties.setProperty("port", Integer.toString(specifier.getPort()));
				properties.setProperty("user", specifier.getUser());
				properties.setProperty("password", specifier.getPassword());
				MutableObject mutableNode = new MutableObject();
				properties.put("mutableNode", mutableNode);
	
				// Execute the headless Agent Controller connector service to obtain node
				// Connect to the headless Agent Controller and store the node returned
				client.execute("org.eclipse.tptp.platform.common.headlessConnector", properties);
				mutableNode.clear();
			}

		}

		//Set the use defaults property on the execution history launch configuration:
		ExecutionHistoryLaunchConfigurationFacade.setUseDefaults(configuration, false);

		//Set the user-specified execution history overridden property on the execution history launch configuration:
		ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryOverriden(configuration, this.overwriteSpecifier);
       
		//Set the user-specified execution history location and name on the execution history launch configuration:
		if (resultsSpecifier != null) {

			IContainer executionHistoryLocation = ServiceUtil.getContainer(theProject, resultsSpecifier);

			//Set the user-specified execution history location on the execution history launch configuration:
			if(executionHistoryLocation != null){				
				ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryLocation(configuration, executionHistoryLocation);
			}

			//Throw an exception if the user-specified execution history location is invalid:
			else{				
				throw new CoreException(new Status(IStatus.ERROR, TestCorePlugin.getPluginId(), "The 'results' property '"  + resultsSpecifier + "' is invalid."));
			}

			String executionHistoryName = ServiceUtil.getFileName(resultsSpecifier);

			//Set the user-specified execution history name on the execution history launch configuration:
			if(executionHistoryName != null){				
				ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryName(configuration, executionHistoryName);
			}

			//Set the default execution history name on the execution history launch configuration:
			else{			
				ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryName(configuration, ExecutionHistoryLaunchConfigurationFacade.getDefaultExecutionHistoryName(configuration));
			}			
		}
		else{
			
			//Set the default execution history location on the execution history launch configuration:
			ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryLocation(configuration, ExecutionHistoryLaunchConfigurationFacade.getDefaultExecutionHistoryLocation(configuration));

			//Set the default execution history name on the execution history launch configuration:
			ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryName(configuration, ExecutionHistoryLaunchConfigurationFacade.getDefaultExecutionHistoryName(configuration));
		}

		// Return configuration to next step of launch
		return configuration;

	}

	private String getFilePathFromWorkspaceRelativePath(
			String launchFile) {
		int index = launchFile.indexOf('/');
		if (index == -1)
			index = launchFile.indexOf('\\');
		if (index == 0)
			return getFilePathFromWorkspaceRelativePath(launchFile.substring(1));
		if (index != -1) {
			return launchFile.substring(index+1);
		}
		return null;
	}

	private IProject getProjectFromWorkspaceRelativePath(String launchFile) {
		int index = launchFile.indexOf('/');
		if (index == -1)
			index = launchFile.indexOf('\\');
		if (index == 0)
			return getProjectFromWorkspaceRelativePath(launchFile.substring(1));
		if (index != -1) {
			String projectName = launchFile.substring(0, index);
			final IWorkspace workspace = ResourcesPlugin.getWorkspace();
			final IProject project = workspace.getRoot().getProject(projectName);
			return project;
		}
		return null;
	}

	/**
	 * Before the execution will return, used in normal cases and exceptional
	 * cases.
	 */
	protected void cleanup() {

		// If project is not null, cleanup by refreshing
		if (this.project != null) {

			// Ensure project is synchronized with file system
			this.refreshProject();

		}

	}

	/**
	 * Derives a new test suite model element and corresponding file
	 * representing it from a source test suite, retrieved by file identifier
	 * (test suite project relative name or absolute path)
	 * 
	 * @param project
	 *            the project of interest
	 * @param sourceTestSuite
	 *            the source test suite the created suite is deriving from
	 * @param deleteOnExit
	 *            indicates if the derived test suite file representation should
	 *            be deleted on exit of the vm
	 * @return the file representing the derived test suite model element
	 * @throws IOException
	 */
	protected File createDerivedTestSuiteFile(IProject project,
			String sourceTestSuite, boolean deleteOnExit) throws IOException {

		// Retrieve the source test suite file
		File testSuiteFile = project.getFile(sourceTestSuite).getLocation()
				.toFile();

		// Create temporary file and set to delete on exit
		String testSuiteFileName = testSuiteFile.getName();
		int index = testSuiteFileName.lastIndexOf('.');

		// Generate prefix and suffix to use
		String prefix = testSuiteFileName.substring(0, index);
		String suffix = "." + testSuiteFileName.substring(index + 1);

		// If prefix is less than three, lengthen it
		while (prefix.length() < 3) {
			prefix = prefix.concat(prefix);
		}

		// Create a derived file to use for this execution
		File derivedTestSuiteFile = File.createTempFile(prefix, suffix);
		if (deleteOnExit) {
			derivedTestSuiteFile.deleteOnExit();
		}

		// Copy test suite file into a derived test suite file
		this.copy(testSuiteFile, derivedTestSuiteFile);

		// Return file for derived test suite
		return derivedTestSuiteFile;

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service#execute()
	 */
	public final Object execute() {

		// Entire execution in throwable catch block
		try {

			// Execute common execute behavior before service specific
			Object object = super.execute();

			// If base class decides to abort this execution early (to restart)
			//NOTE: When this is changed to IApplication.EXIT_OK, we will need to change it everywhere otherwise this condition may return false since the objects are not the same:
			if (object != IPlatformRunnable.EXIT_OK) {
				return object;
			}

			// Output start of test
			this.println("Test execution started.");

			{
				// Initialize service state
				this.initialize();
				
				if (this.project != null)
					this.resultProjectDir = this.project.getLocation().toString();

				// Iterate through suite list (executing one at a time)
				for (Iterator suites = this.suiteSelector.iterator(); suites
						.hasNext();) {

					// Specifier of the next suite to execute
					String suiteSpecifier = (String) suites.next();
					this.println("Test '" + suiteSpecifier + "' started.");

					if (suiteSpecifier.endsWith(".launch"))
						this.configurationSelector = suiteSpecifier;
					
					// Invoked at beginning before launch
					ILaunchConfiguration configuration = this
							.beforeLaunch(suiteSpecifier);

					// The launch code, to initiate test execute
					ILaunch launch = this.launch(configuration,
							ILaunchManager.RUN_MODE, this.getProgressMonitor());

					// Invoked after the launch
					IProcess process = this.afterLaunch(launch);

					// Output results and other information
					EObject resultsObject = this.afterExecution(process);
					String resultPath = getPathFromProxyURI(resultsObject.toString());
					String result = null;
					
					if (this.projectSpecifier == null) {
						// A project was not specified for execution, so
						// we should return workspace relative paths.
						result = resultPath;
					}
					else {
						result = resultPath.substring(this.projectSpecifier.length() + 1);
					}

					// Set return value results URI in properties
					this.results.add(result);
					this.println("Test '" + suiteSpecifier + "' completed.");

				}

				// Set results out value to a list of results
				this.getProperties().put("results", this.results);
				if (resultProjectDir != null)
					this.getProperties().put("resultProjectDir", resultProjectDir);

			}

			// Output test completion
			this.println("Test execution completed!");

		} catch (Throwable t) {

			// Handle throwable during the template method execution
			this.handleThrowable(t);

		} finally {

			// Invoke cleanup
			this.cleanup();

		}

		// Prepare and set return state
		return this.returnResult();

	}

	private String getPathFromProxyURI(String result) {
		String match = "platform:/resource/";
		result = result.substring(result.indexOf(match)
				+ match.length());
		result = result.substring(0, result.indexOf("#"));						
		return result;
	}

	/**
	 * Retrieve process of interest from launch, the test execution process
	 * 
	 * @param launch
	 *            the launch associated with this test execution
	 * @return the test execution process of interest
	 */
	protected IProcess getProcessOfInterest(ILaunch launch) {

		// Use process object and execution context context cleanup
		IProcess[] processes = launch.getProcesses();

		// Assumes one process to specify the test launched
		if (processes.length > 0) {
			IProcess processElement = processes[0];
			if (processes != null
					&& processElement instanceof TestExecutionProcess) {
				return processElement;
			}
		}

		// Null indicates a process is not found
		return null;

	}

	/**
	 * The default method creates and offers a console progress monitor (reports task names
	 * to the console, if this service instance is not in quiet mode, that is suppressing 
	 * output/chatter to the console out), overwrite
	 * this to create a custom progress monitor
	 * 
	 * @return the progress monitor to use for the launch
	 */
	protected IProgressMonitor getProgressMonitor() {
		
		return (new NullProgressMonitor(){
						
			public void beginTask(String name, int totalWork) {
				
				if((name != null) && (name.trim().length() > 0)){
					println(name);
				}
			}

			public void subTask(String name) {

				if((name != null) && (name.trim().length() > 0)){
					println(name);
				}
			}
		});
	}

	/**
	 * The time to wait before polling the test execution process for its
	 * completion status. If not specified the default value is used.
	 * 
	 * @return
	 */
	protected int getWaitForCompletionInterval() {
		return AbstractTestExecutionService.DEFAULT_POLL_INTERVAL;
	}

	/**
	 * Default outputs throwable to standard console out
	 * 
	 * @param t
	 *            the throwable caught to be handled
	 */
	protected void handleThrowable(Throwable t) {
		t.printStackTrace();
	}

	/**
	 * Invoked at the beginning before any other template method, used to
	 * initialize instance variable state and any other actions that are
	 * required before anything else is begun.
	 */
	protected void initialize() {

		// Ensure required EMF test models are initialized and associated
		this.initializeRequisiteModels();

		// Extract primary test required properties
		this.deploymentSelector = this.getProperty("deployment");
		this.configurationSelector = this.getProperty("configuration");
		this.resultsSpecifier = this.getProperty("results");
		this.overwriteSpecifier = Boolean
				.valueOf(this.getProperty("overwrite")).booleanValue();
		this.classpathSpecifier = this.getProperty("classpath");
		this.connectionSpecifier = this.getProperty("connection");

		// Extract test suite (string or list of strings specified)
		Object suiteSelector = this.getProperties().get("suite");
		if (suiteSelector == null && configurationSelector != null) {
			this.suiteSelector = new ArrayList();
			this.suiteSelector.add(configurationSelector);
		}
		else if (suiteSelector instanceof String) {
			this.suiteSelector = new ArrayList();
			this.suiteSelector.add(suiteSelector);
		} else {
			this.suiteSelector = (List) suiteSelector;
		}

		// Allocate a return list to store result strings
		this.results = new ArrayList();//this.suiteSelector.size());

	}

	/**
	 * Models associated and initialized (in the future this might not be needed
	 * and will be done in the models plug-in intialization code or something to
	 * that effect)
	 */
	private void initializeRequisiteModels() {
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				ICommonConstants.TEST_SUITE_FILE_EXTENSION,
				new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				ICommonConstants.DEPLOYMENT_FILE_EXTENSION,
				new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				ICommonConstants.EXECUTION_FILE_EXTENSION,
				new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				ICommonConstants.ARTIFACT_FILE_EXTENSION,
				new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				ICommonConstants.LOCATION_FILE_EXTENSION,
				new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				ICommonConstants.DATAPOOL_FILE_EXTENSION,
				new DatapoolFacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap()
				.put(ICommonConstants.SUT_EXTENSION,
						new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				ICommonConstants.TEST_COMPONENT_EXTENSION,
				new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				"xmi", new FacadeResourceFactoryImpl()); //$NON-NLS-1$
		if(ModelDebugger.INSTANCE.debugDatabaseResourcePostfix!=null && ModelDebugger.INSTANCE.debugDatabaseResourcePostfix.length()>0)
		{
			//add support for RDB backed execution resources
			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("executiondb", new HyadesExtendedResourceFactory());//$NON-NLS-1$
		}

		Common_TestprofilePackageImpl.init();
	}

	/**
	 * Launches a given configuration, using the specified mode (typically
	 * ILaunchManager.RUN_MODE) and an optional progress monitor.
	 * 
	 * @param configuration
	 *            the launch configuration to launch
	 * @param mode
	 *            the mode to launch in
	 * @param monitor
	 *            ability to cancel and report status
	 * @return the launch object associated with this particular launch
	 * @throws CoreException
	 */
	protected ILaunch launch(ILaunchConfiguration configuration, String mode,
			IProgressMonitor monitor) throws CoreException {

		// Launch test (will polymorphically use the appropriate delegate)
		return configuration.launch(mode, monitor);

	}

	/**
	 * Load deployment model element given the project and deployment name
	 * 
	 * @param project
	 *            the project containing (or that should be containing) the
	 *            deployment
	 * @param deploymentName
	 *            the deployment identifier
	 * @return the loaded model element
	 */
	protected TPFDeployment loadDeployment(IProject project,
			String deploymentName) {
		return (TPFDeployment) this.loadModelElement(project, deploymentName);
	}

	/**
	 * Loads a model element, given a project and model element name
	 * 
	 * @param project
	 *            the project the model element is found in
	 * @param modelElementName
	 *            the model element within the project to load
	 * @return the named element loaded
	 */
	protected CMNNamedElement loadModelElement(IProject project,
			String modelElementName) {

		// Create URI to model element file
		IFile platformFile = project.getFile(modelElementName);

		// Create URI to point to model element
		URI uri = URI.createPlatformResourceURI(platformFile.getFullPath().toString(), false);
		ResourceSet set = new ResourceSetImpl();
		Resource modelElementResource = set.getResource(uri, true);

		// Load the requested model element from resource
		return (CMNNamedElement) modelElementResource.getContents().get(0);

	}

	/**
	 * Load test suite model element given the project and test suite name
	 * 
	 * @param project
	 *            the project the test suite is contained in (or should be
	 *            contained in)
	 * @param testSuiteName
	 *            the test suite name identifying the test suite
	 * @return the loaded model element
	 */
	protected TPFTestSuite loadTestSuite(IProject project, String testSuiteName) {
		return testSuiteName != null ? (TPFTestSuite) this.loadModelElement(
				project, testSuiteName) : null;

	}

	/**
	 * The result to return from this service
	 * 
	 * @return returns an instance of self to avoid sending back a null object
	 *         (not used for anything)
	 */
	protected Object returnResult() {
		return this;
	}

	/**
	 * Waits for the test to be fully complete, then returns
	 * 
	 * @param launch
	 *            the launch of interest with regards to completion
	 * @param interval
	 *            the amount of time to wait before checking time completion
	 *            status
	 */
	protected void waitForCompletion(IProcess process, int interval) {

		// Wait for test execution process to be terminated
		this.print("Waiting...");
		while (process != null && !process.isTerminated()) {
			synchronized (process) {
				try {
					process
							.wait(AbstractTestExecutionService.DEFAULT_POLL_INTERVAL);
					this.print(".");
				} catch (InterruptedException e) {

				}
			}
		}
		
		if (process == null) {
			System.out.println("Process was null -- test completed too fast?");
		}

		// Output extra line for end of waiting display
		this.println();

	}

}
