/********************************************************************** 
 * Copyright (c) 2005, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AbstractTestResultsInterrogationService.java,v 1.16 2009/11/26 12:43:55 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.core.services;

import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IPlatformRunnable;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.automation.server.AbstractProjectSensitiveService;
import org.eclipse.hyades.models.common.facades.behavioral.Verdict;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFVerdict;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;

/**
 * <p>Abstract test results interrogation service that provides a base 
 * implementation of the test results interrogation service and may be 
 * extended by more specialized interrogation services.</p>
 * 
 * <p>The following input properties are defined:</p> 
 * 
 * <ul>
 * <li><code>workspace</code></li>
 * <li><code>project</code></li>
 * <li><code>results (a single test execution result file or list of test execution result files)</code></li>
 * </ul>
 * 
 * <p>The following output properties are defined:</p> 
 * 
 * <ul>
 * <li><code>verdict (pass (see {@link Verdict#PASS_LITERAL}), fail (see {@link Verdict#FAIL_LITERAL}), error (see {@link Verdict#ERROR_LITERAL}), or inconclusive (see {@link Verdict#INCONCLUSIVE_LITERAL}))</code></li>
 * </ul>
 * 
 * The service identity is <code>org.eclipse.hyades.test.core.interrogate</code>.
 * 
 * 
 * @author  Scott E. Schneider
 * @author  Paul E. Slauenwhite
 * @version November 26, 2009
 * @since   March 11, 2006
 */
public abstract class AbstractTestResultsInterrogationService extends
		AbstractProjectSensitiveService {

	private static final long serialVersionUID = -3337136429895049654L;

	/**
	 * The results, to interrogate, will be stored as a list, even if only one
	 * result is specified
	 */
	private List results;

	/**
	 * The verdict out property that takes the value of any valid verdicts
	 */
	private String verdict;

	/**
	 * Default constructor invokes up hierarchy for any initialization required
	 */
	protected AbstractTestResultsInterrogationService() {
		super();
	}

	/**
	 * Cleanup after service executes
	 */
	protected void cleanup() {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service#execute()
	 */
	public final Object execute() {

		// Entire execution in throwable catch block
		try {

			// Execute common execute behavior before service specific
			Object object = super.execute();

			// If base class decides to abort this execution early (to restart)
			//NOTE: When this is changed to IApplication.EXIT_OK, we will need to change it everywhere otherwise this condition may return false since the objects are not the same:
			if (object != IPlatformRunnable.EXIT_OK) {
				return object;
			}

			// Interrogation started
			this.println("Test results interrogation started.");

			// Initialize service state (fill instance variables)
			this.initialize();

			// Interrogate test results as instructed
			this.verdict = this.interrogate();

			// Allocate a property for the verdict
			this.getProperties().setProperty("verdict", this.verdict);

			// Interrogation completed
			this.println("Test results interrogation completed.");

			// Set return value verdict in properties

		} catch (Throwable t) {

			// Handle throwable during execution
			this.handleThrowable(t);

			// Use error verdict in case of problems
			this.getProperties().setProperty("verdict",
					TPFVerdict.ERROR_LITERAL.getLabel());

		} finally {

			// Invoke cleanup
			this.cleanup();

		}

		// Prepare and set return state
		return this.returnResult();

	}

	/**
	 * Default outputs throwable to standard console out
	 * 
	 * @param t
	 *            the throwable caught to be handled
	 */
	protected void handleThrowable(Throwable t) {
		t.printStackTrace();
	}

	/**
	 * Initialize the state of this service.
	 */
	protected void initialize() {

		Object resultsProperty = this.getProperties().get("results"); //$NON-NLS-1$
		
		if (resultsProperty instanceof List) {
			results = ((List)(resultsProperty));
		} 
		else {
			
			results = new ArrayList();
			results.add(String.valueOf(resultsProperty));			
		}				
	}

	/**
	 * Interrogate the test results, seeking the verdict of the test suite
	 * 
	 * @return return the results roll-up verdicts (pass, fail, inconclusive,
	 *         error)
	 */
	protected String interrogate() {

		// Keep track of accumulated verdict
		TPFVerdict overallVerdict = TPFVerdict.PASS_LITERAL;

		// Iterate through all results to interrogate
		for (Iterator results = this.results.iterator(); results.hasNext();) {

			// Retrieve next result operand
			String result = (String) results.next();
			
			// The required extension for results
			final String extension = "." + ICommonConstants.EXECUTION_FILE_EXTENSION;

			if (!result.trim().endsWith(extension)) {
				result += extension;
			}
			
			// Handle absolute paths as filesystem URIs and relative
			// paths as platform URIs relative to the specified project.
			File resFile = new File(result);
			URI uri = null;
			String resultsPath = null;
			if (resFile.isAbsolute()) {
				resultsPath = result;
				uri = URI.createFileURI(result);
			}
			else {
				resultsPath = this.project.getName() + "/" + result;
				uri = SaveUtil.createURI(resultsPath);
			}
				
			ResourceSet resourceSet = new ResourceSetImpl();
			Resource resource = resourceSet.getResource(uri, true);
			TPFExecutionResult executionResult = (TPFExecutionResult) resource
					.getContents().get(0);

			/*
			 * Accumulate verdict into roll-up verdict, error breaks out
			 * immediately with the most severe verdict used as the final value
			 */
			TPFVerdict verdict = executionResult.getVerdict();
			this.println("Verdict for test result '" + executionResult.getName() + ".execution' is '"
					+ verdict.getLabel() + "'.");
			if (outweighs(verdict, overallVerdict))
				overallVerdict = verdict;
		}

		// Output overall verdict
		this.println("Overall verdict is '" + overallVerdict.getLabel() + "'.");

		// Return the accumulated overall verdict
		return overallVerdict.getLabel();

	}

	private boolean outweighs(TPFVerdict verdict, TPFVerdict overallVerdict) {
		if (overallVerdict.equals(TPFVerdict.ERROR_LITERAL))
			return false;
		if (overallVerdict.equals(TPFVerdict.FAIL_LITERAL) &&
				(verdict.equals(TPFVerdict.PASS_LITERAL) || verdict.equals(TPFVerdict.INCONCLUSIVE_LITERAL)))
			return false;
		if (overallVerdict.equals(TPFVerdict.INCONCLUSIVE_LITERAL) &&
				verdict.equals(TPFVerdict.PASS_LITERAL))
			return false;

		return true;
	}

	/**
	 * Derive result to return
	 * 
	 * @return the result to return, service-level not semantic-level per
	 *         service domain
	 */
	protected Object returnResult() {
		return this;
	}

}
