/********************************************************************** 
 * Copyright (c) 2005, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AbstractTestResultsPublicationService.java,v 1.14 2009/11/26 12:43:55 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.core.services;

import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IPlatformRunnable;
import org.eclipse.hyades.automation.server.AbstractProjectSensitiveService;

/**
 * <p>Abstract test results publication service that provides a base 
 * implementation of the test results publication service and may be 
 * extended by more specialized publication services.</p>
 * 
 * <p>The following input properties are defined:</p> 
 * 
 * <ul>
 * <li><code>workspace</code></li>
 * <li><code>project</code></li>
 * <li><code>results (a single test execution result file or list of test execution result files)</code></li>
 * <li><code>report</code></li>
 * </ul>
 * 
 * <p>No output properties are defined.</p>
 * 
 * The service identity is <code>org.eclipse.hyades.test.core.publish</code>.
 * 
 * 
 * @author  Scott E. Schneider
 * @author  Joe Toomey
 * @author  Paul Slauenwhite
 * @version November 26, 2009
 * @since   March 11, 2006
 */
public abstract class AbstractTestResultsPublicationService extends
		AbstractProjectSensitiveService {

	private static final long serialVersionUID = 2182481468074056364L;

	/**
	 * List of results on which to report
	 */
	protected List results;
	
	/**
	 * Path to file where report should be written
	 */
	protected String report;
	
	protected String[] resultPaths;
	
	/**
	 * optional start time range for report
	 */
	protected long startDate;
	
	/**
	 * optional end time range for report
	 */
	protected long endDate;
	
	/**
	 * optional location of BIRT report template
	 */
	protected String reportTemplateLocation; 
	
	/**
	 * Default constructor invokes up hierarchy for any initialization required
	 */
	protected AbstractTestResultsPublicationService() {
		super();
	}

	/**
	 * Cleanup after service executes
	 */
	protected void cleanup() {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service#execute()
	 */
	public final Object execute() {

		// Entire execution in throwable catch block
		try {
			initialize();
			
			// Execute common execute behavior before service specific
			Object object = super.execute();

			// If base class decides to abort this execution early (to restart)
			//NOTE: When this is changed to IApplication.EXIT_OK, we will need to change it everywhere otherwise this condition may return false since the objects are not the same:
			if (object != IPlatformRunnable.EXIT_OK) {
				return object;
			}

			// Populate the resultPaths array with paths by resolving the
			// project relative input list against the specified workspace
			// and project.
			resolvePaths();
			
			// Publish test results as instructed (execute a report)
			this.publish();

		} catch (Throwable t) {

			// Handle throwable during execution
			this.handleThrowable(t);

		} finally {

			// Invoke cleanup
			this.cleanup();

		}

		// Prepare and set return state
		return this.returnResult();

	}

	/**
	 * Initialize the state of this service.
	 */
	protected void initialize() {

		//Extract required properties:
		report = getProperty("report"); //$NON-NLS-1$
		reportTemplateLocation = getProperty("reportTemplateLocation"); //$NON-NLS-1$

		Object resultsProperty = this.getProperties().get("results"); //$NON-NLS-1$
		
		if (resultsProperty instanceof List) {
			results = ((List)(resultsProperty));
		} 
		else {
			
			results = new ArrayList();
			results.add(String.valueOf(resultsProperty));			
		}		
		
		String startDate = getProperty("startDate"); //$NON-NLS-1$

		if (startDate != null){
			this.startDate = Long.parseLong(startDate);
		}
		
		String endDate = getProperty("endDate"); //$NON-NLS-1$

		if (endDate != null){
			this.endDate = Long.parseLong(endDate);
		}

		processOutputFolder();
	}
	
	/**
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	protected void processOutputFolder(){
		//if the output folder does not exist create it
		File outFile = new File(report);
		if (!outFile.exists()){
			outFile.mkdirs();
		}
	}
	
	
	/**
	 * The pathnames specified in the results list are treated in the following
	 * manner:  If the path is absolute, it will be converted into a path
	 * relative to the project, by searching the workspace and project passed 
	 * into the service.  If it is relative, it will be unmodified under the 
	 * presumption that it is relative to the specified workspace and project.  
	 * Extending classes may override this behavior.
	 */
	protected void resolvePaths() {
		List resolvedPaths = new ArrayList();
		Iterator it = results.iterator();
		if (project != null) {
			String projectPath = project.getLocation().toString(); 
			while (it.hasNext()) {
				String tempPath = (String) it.next();
				File file = new File(tempPath);
				if (file.isAbsolute()) {
					if (tempPath.startsWith(projectPath)) {
						tempPath = tempPath.substring(projectPath.length());
					}
				}
				if (project.findMember(tempPath) != null) {
					resolvedPaths.add(project.getName() + "/" + tempPath);
				}
				else {
					System.out.println("Can not find " + tempPath + " in workspace.");
				}
			}
		}
		resultPaths = (String[]) resolvedPaths.toArray(new String[0]);		
	}

	/**
	 * Default outputs throwable to standard console out
	 * 
	 * @param t
	 *            the throwable caught to be handled
	 */
	protected void handleThrowable(Throwable t) {
		t.printStackTrace();
	}

	/**
	 * Publish the test results report
	 */
	protected abstract void publish();

	/**
	 * Derive result to return
	 * 
	 * @return the result to return, service-level not semantic-level per
	 *         service domain
	 */
	protected Object returnResult() {
		return this;
	}

}
