/********************************************************************** 
 * Copyright (c) 2009, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: ServiceUtil.java,v 1.2 2010/01/06 18:15:07 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.core.services.internal.util;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;

/**
 * <p>Service utilities.</p>
 * 
 * 
 * @author  Paul Slauenwhite
 * @version January 6, 2010
 * @since   November 27, 2009
 */
public class ServiceUtil{

	/**
	 * <p>Resolves the lowest-level container from the non-<code>null</code> path./p>
	 * 
	 * <p>The path is workspace-relative (prefixed with the project name) 
	 * or project-relative (relative to the non-<code>null</code> project) to a 
	 * directory with a trailing path separator character.  Possible path 
	 * values include:</p>
	 * 
	 * <ul>
	 * <li><code>[\]&lt;empty&gt;</code></li>
	 * <li><code>[\]&lt;file name&gt;[.&lt;file extension&gt;]</code></li>
	 * <li><code>[\]&lt;directory&gt;\[&lt;file name&gt;[.&lt;file extension&gt;]]</code></li>
	 * <li><code>[\]&lt;project name&gt;\[&lt;file name&gt;[.&lt;file extension&gt;]]</code></li>
	 * <li><code>[\]&lt;project name&gt;\&lt;directory&gt;\[&lt;file name&gt;[.&lt;file extension&gt;]]</code></li>
	 * </ul>
	 * 
	 * <p><b>Notes:</b></p>
	 * 
	 * <ol>
	 * <li>The project must exist in the workspace.</li>
	 * <li>The project-relative directory structure must exist in the workspace.</li>
	 * <li>Path separator characters are platform-independent (slash ('/') or backslash ('\') characters are interchangeable).</li>
	 * <li>The path may contain an optional leading path separator character.</li>
	 * </ol>
	 * 
	 * @param project The non-<code>null</code> project.
	 * @param path The non-<code>null</code> workspace-relative or project-relative path.
	 * @return The lowest level container from the non-<code>null</code> workspace-relative or project-relative path.
	 */
	public static IContainer getContainer(IProject project, String path){

		if((project != null) && (project.exists()) && (path != null)){

			try {

				//Resolve the project-relative path:
				IPath projectRelativePath = new Path(path.trim());

				//Remove the project name from the project-relative path:
				if(project.getName().equals(projectRelativePath.segment(0))){
					projectRelativePath = projectRelativePath.removeFirstSegments(1);					
				}
				
				//The project-relative path represents the project root (empty or root):
				if((projectRelativePath.isEmpty()) || (projectRelativePath.isRoot())){
					return project;
				}
				
				//The project-relative path represents a folder or file:
				else{
										
					IContainer folder = null;
					
					//The project-relative path represents a folder:
					if(projectRelativePath.hasTrailingSeparator()){						
						folder = project.getFolder(projectRelativePath);
					}

					//The project-relative path represents a file name:
					else{
						folder = project.getFile(projectRelativePath).getParent();
					}
					
					//The folder is the project:
					if(project.equals(folder)){
						return project;
					}
					else if (folder.exists()) {
						return folder;
					}
				}
			} 
			catch (Exception e) {
				//Ignore and return null.
			}
		}

		return null;
	}

	/**
	 * <p>Resolves the file name (excluding the file extension) from the 
	 * non-<code>null</code> and non-empty path./p>
	 * 
	 * <p>The path is workspace-relative (prefixed with the project name) 
	 * or project-relative (relative to a project) to a file with an optional 
	 * file extension (no trailing path separator character).  Possible path 
	 * values include:</p>
	 * 
	 * <ul>
	 * <li><code>[\]&lt;file name&gt;[.&lt;file extension&gt;]</code></li>
	 * <li><code>[\]&lt;directory&gt;\&lt;file name&gt;[.&lt;file extension&gt;]</code></li>
	 * <li><code>[\]&lt;project name&gt;\&lt;file name&gt;[.&lt;file extension&gt;]</code></li>
	 * <li><code>[\]&lt;project name&gt;\&lt;directory&gt;\&lt;file name&gt;[.&lt;file extension&gt;]</code></li>
	 * </ul>
	 * 
	 * <p><b>Notes:</b></p>
	 * 
	 * <ol>
	 * <li>Path separator characters are platform-independent (slash ('/') or backslash ('\') characters are interchangeable).</li>
	 * <li>The path may contain an optional leading path separator character.</li>
	 * </ol>
	 * 
	 * @param path The non-<code>null</code> and non-empty workspace-relative or project-relative path.
	 * @return The file name (excluding the file extension) from the non-<code>null</code> and non-empty workspace-relative or project-relative path.
	 */
	public static String getFileName(String path){

		if((path != null) && (path.trim().length() > 0)){

			IPath filePath = new Path(path);

			//Path represents a file name:
			if(!filePath.hasTrailingSeparator()){

				//Resolve the file name:
				String fileName = filePath.lastSegment();

				if(fileName != null){

					//Remove the file extension:
					int lastDotIndex = fileName.lastIndexOf('.');

					if(lastDotIndex != -1){
						fileName = fileName.substring(0, lastDotIndex);
					}

					//Return null if the file name is empty:
					if(fileName.trim().length() > 0){
						return fileName;
					}
				}
			}			
		}

		return null;
	}
}
