/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestGeneratorFactory.java,v 1.3 2009/12/11 20:22:09 paules Exp $
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.core.testgen;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;

/**
 * <p>Factory for managing Test Generators contributed by the 
 * <code>org.eclipse.hyades.execution.testgen.TestGenerator</code> 
 * extension point.</p>
 * 
 * 
 * @author  Ernest Jessee
 * @author  Paul Slauenwhite
 * @version December 11, 2009
 * @since   March 8, 2005
 */
public class TestGeneratorFactory {
		
	/**
	 * List of test generator configuration elements.
	 * 
	 * <p>The <i>i<sup>th</sup></i> test generator configuration element 
	 * corresponds to the <i>i<sup>th</sup></i> test generator configuration 
	 * element ID in the {@link #configurationElementIDs} array.</p>
	 */
	private List<IConfigurationElement> configurationElements = null;

	/**
	 * <p>Array of test generator configuration element IDs.</p>
	 * 
	 * <p>The <i>i<sup>th</sup></i> test generator configuration element ID 
	 * corresponds to the <i>i<sup>th</sup></i> test generator configuration element
	 * in the {@link #configurationElements} list.</p>
	 */
	private String[] configurationElementIDs = null;

	/**
	 * Singleton instance.
	 */
	private static TestGeneratorFactory instance = null;

	/**
	 * ID of the <code>org.eclipse.hyades.execution.testgen.TestGenerator</code> extension point.
	 */
	private final String TEST_GENERATOR_EXTENSION_POINT_ID = "org.eclipse.hyades.test.core.TestGenerator"; //$NON-NLS-1$

	/**
	 * Private constructor.
	 */
	private TestGeneratorFactory(){
		
		//Load the configuration elements:
		configurationElements = Arrays.asList(Platform.getExtensionRegistry().getConfigurationElementsFor(TEST_GENERATOR_EXTENSION_POINT_ID));
		
		//Sort the configuration elements by priority:
		Collections.sort(configurationElements, new Comparator<IConfigurationElement>(){

			public int compare(IConfigurationElement configurationElementOne, IConfigurationElement configurationElementTwo) {
				return (getPriority(configurationElementTwo) - getPriority(configurationElementOne));
			}
		});
		
		//Load the configuration element IDs:
		configurationElementIDs = new String[configurationElements.size()];
		
		for (int counter = 0; counter < configurationElements.size(); counter++) {
			configurationElementIDs[counter] = configurationElements.get(counter).getAttribute("id"); //$NON-NLS-1$
		}
	}
	
	
	/**
	 * Resolves the singleton instance of this test generator factory.
	 * 
	 * @return The singleton instance of this test generator factory.
	 */
	public static TestGeneratorFactory getInstance(){
		
		if(instance == null){
			instance = new TestGeneratorFactory();
		}
		
		return instance;
	}
	
	/**
	 * returns the name of the generator with the specified ID
	 * @param id
	 * @return
	 */
	public String getGeneratorName(String id){

		IConfigurationElement configurationElement = getGeneratorConfigElement(id);
		
		if(configurationElement != null){
			return (configurationElement.getAttribute("name")); //$NON-NLS-1$
		}

		return null;
	}
	
	/**
	 * returns the full classpath of the generator with the specified ID
	 * @param id
	 * @return
	 */
	public String getGeneratorClassPath(String id){
		
		IConfigurationElement configurationElement = getGeneratorConfigElement(id);
		
		if(configurationElement != null){
			return (configurationElement.getAttribute("class")); //$NON-NLS-1$
		}

		return null;
	}
	
	public int getGeneratorPriority(String id){
		return (getPriority(getGeneratorConfigElement(id)));
	}
	
	/**
	 * returns a String array of all registered generator ID's
	 * @return
	 */
	public String[] getGeneratorIDs(){
		return configurationElementIDs;
	}

	/**
	 * returns a generator ID by index
	 * @param int index
	 * @return String
	 */
	public String getGeneratorIDByIndex(int index){
		
		if((index > -1) && (index < configurationElementIDs.length)){
			return configurationElementIDs[index];
		}
		
		return null;
	}
	
	/**
	 * returns the IConfigurationElement for the generator with the specified ID
	 * @param id
	 * @return
	 */
	public IConfigurationElement getGeneratorConfigElement(String id){
		
		if((id != null) && (id.trim().length() > 0)){
		
			for (int counter = 0; counter < configurationElementIDs.length; counter++) {
				
				if(id.equals(configurationElementIDs[counter])){
					return (configurationElements.get(counter));
				}
			}
		}
		
		return null;
	}
	
	public List<IConfigurationElement> getGeneratorConfigurationElementsByRecorderID(String recorderID){

		List<IConfigurationElement> configurationElementsByRecorderID = new ArrayList<IConfigurationElement>();

		if((recorderID != null) && (recorderID.trim().length() > 0)){
			
			for (int counter = 0; counter < configurationElements.size(); counter++) {
				
				IConfigurationElement configurationElement = configurationElements.get(counter);
				String recorderIDAttribute = configurationElement.getAttribute("recorderID"); //$NON-NLS-1$
				
				//Note: If the recorder ID does not exist the test generator applies to all recorders. 
				if((recorderIDAttribute == null) || (recorderID.equals(recorderIDAttribute))){ //$NON-NLS-1$
					configurationElementsByRecorderID.add(configurationElement);
				}
			}
			
			return configurationElementsByRecorderID;
		}
		
		return configurationElementsByRecorderID;
	}
	
	private int getPriority(IConfigurationElement configurationElement){
		
		if(configurationElement != null){

			String priorityAttribute = configurationElement.getAttribute("priority"); //$NON-NLS-1$

			if (priorityAttribute != null){
	
				try{
					
					int priority = Integer.parseInt(priorityAttribute);
					
					if ((priority >= -1) && (priority <= 99)){
						return priority;
					}
				}
				catch (NumberFormatException e){
					//Ignore and return 0.
				}
			}
		}

		return 0; 
	}
}
