/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestGenFileUtil.java,v 1.10 2010/04/12 12:38:44 paules Exp $
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.core.testgen.util;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.hyades.models.trace.TRCSourceInfo;
import org.eclipse.hyades.models.trace.impl.TraceFactoryImpl;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle;
import org.eclipse.osgi.util.NLS;

/**
 * <p>Utilities for test generator files.</p>
 * 
 * 
 * @author      Ashish Mathur
 * @author      Paul E. Slauenwhite
 * @author		Jerome Bozier
 * @version     April 12, 2010
 * @since       June 8, 2005
*/
public class TestGenFileUtil {
	
	/**
	 * Creates a file with the given path.  If the path is not of ".rec" extension,
	 * it will add ".rec"
	 * @param filepath
	 * @return
	 */
	public static IFile createTraceFile(String filepath) {
		
		String recFilePath = filepath;
		IFile file = null;
		try {
			if (!recFilePath.endsWith(".rec")) { //$NON-NLS-1$
				recFilePath += ".rec"; //$NON-NLS-1$
			}
			
			file =
				ResourcesPlugin.getWorkspace().getRoot().getFile(
					new Path(recFilePath));
			if (!file.exists()) {
				try {
					file.create(new ByteArrayInputStream(new byte[] {}), true, new NullProgressMonitor());
				}
				catch (CoreException e1) {
					e1.printStackTrace();
				}
			}
		}
		catch (Exception e) {
			TestCorePlugin.getDefault().logError(e);
		}

		return file;
	}
	
	/**
	 * Creates a simple trace EMF model with a default option set
	 * @param recModelPath - a file path, ending in ".recmodel"
	 * @param recFilePath - sets the location of the rec model to this. (.rec file)
	 * @return
	 */
	public static IFile createTraceModel(String recModelPath, String recFilePath) {
		Map RESOURCE_OPTIONS = new HashMap();
		RESOURCE_OPTIONS.put(XMIResource.OPTION_DECLARE_XML, Boolean.TRUE);
		RESOURCE_OPTIONS.put(XMIResource.OPTION_SKIP_ESCAPE, Boolean.FALSE);
		RESOURCE_OPTIONS.put(XMIResource.OPTION_ENCODING, "UTF-8");
		TRCSourceInfo si = TraceFactoryImpl.eINSTANCE.createTRCSourceInfo();
		//URI testSuiteURI = EcoreUtil.getURI(tpfSuite);
		//URI uri = tpfSuite.eResource().getURI().trimFileExtension();
		
		if (recModelPath.endsWith(".rec")) { //$NON-NLS-1$
			recModelPath += "model"; //$NON-NLS-1$
		}
		else if (!recModelPath.endsWith(".recmodel")) { //$NON-NLS-1$
			recModelPath += ".recmodel"; //$NON-NLS-1$
		}
		

		URI modelURI = URI.createPlatformResourceURI(recModelPath, false);
		si.setLocation(recFilePath);
		ResourceSet rs = new ResourceSetImpl();
		Resource resource = rs.createResource(modelURI);
		resource.getContents().add(si);
		try {
			resource.save(RESOURCE_OPTIONS);
		}
		catch (IOException e) {
			e.printStackTrace();
		}
		
		return ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(recModelPath));

		
	}
	
	/**
	 * Loads an EObject[] given an IFile
	 * @param file
	 * @return
	 */
	public static EObject[] loadResource(IFile file) {
		
		try {

			//Create the resource:
			Resource resource =
				new ResourceSetImpl().getResource(
						URI.createPlatformResourceURI(file.getFullPath().toString(), false),
						true);

			//Return the array of root elements in this file:
			return ((EObject[]) resource.getContents().toArray(
						new EObject[resource.getContents().size()]));
		} 
		catch (Exception e) {
			TestCorePlugin.getDefault().logError(NLS.bind(TestCorePluginResourceBundle.TestGenFileUtil_FAILED_TO_GET_RESOURCE_ERROR_MESSAGE, file.getFullPath().toString())); 
		}
		
		return null;
	}
	
	/**
	 * @param file
	 */
	public static IFile getTraceFile(IFile file) {
		
		EObject[] elements = TestGenFileUtil.loadResource(file);
		
		if((elements != null) && (elements.length > 0)){
			
			// Find and open the trace file
			return ((IFile) ResourcesPlugin.getWorkspace().getRoot().findMember(
					((TRCSourceInfo)(elements[0])).getLocation()));
		}
		
		return null;
	}
	
}
