/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: FilesystemResourceProviderService.java,v 1.11 2008/01/21 19:09:25 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.core.testservices.resources;

import java.io.File;
import java.io.IOException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.hyades.execution.core.file.IFileManager;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList;
import org.eclipse.hyades.execution.local.file.FileManagerFactory;
import org.eclipse.hyades.execution.local.testservices.AbstractTestService;
import org.eclipse.hyades.execution.local.testservices.ITestService;
import org.eclipse.hyades.internal.execution.core.file.ServerNotAvailableException;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentImpl;
import org.eclipse.hyades.internal.execution.local.control.Connection;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.osgi.util.NLS;

/**
 * This class runs on the workbench, and provides the workbench side 
 * implementation of the WorkspaceResourceProvider service.
 * 
 * @author  Joseph Toomey
 * @author  Paul E. Slauenwhite
 * @version January 21, 2008
 * @since   March 9, 2006
 */
public class FilesystemResourceProviderService extends AbstractTestService implements ITestService {

	public static final String ROOT = "##ROOT##";
	
	/**
	 * @param error
	 * @param fileManager
	 * @param methodArgs
	 * @return
	 */
	public String getFile(Agent agent, String methodArgs, StringBuffer errBuf) {
		String retVal = null;

		Pattern pattern = Pattern.compile("source=(.*),dest=(.*)"); //$NON-NLS-1$
		Matcher matcher = pattern.matcher(methodArgs);
		
		if (matcher.find()) {
			String source = matcher.group(1);
			String dest = matcher.group(2);

			String fullSource = resolvePath(source);
			if (fullSource != null){
				
				if(new File(fullSource).isFile()) {

					IFileManager fileManager = getFileManager(agent);
					if (fileManager == null){
						errBuf.append(Messages.WorkspaceResourceProviderService_NO_FILEMANAGER); 
					}
					else {
						try {
							fileManager.putFile(fullSource, dest);
							retVal = Boolean.toString(true);
						} catch (IOException e) {
							Object[] messageElements = {source, dest}; //$NON-NLS-1$
							errBuf.append(NLS.bind(Messages.WorkspaceResourceProviderService_COPY_FAILED, messageElements)); 
							TestCorePlugin.getDefault().logError(e);
						}
					}
				}
				else {
					Object[] messageElements = {source}; //$NON-NLS-1$
					errBuf.append(NLS.bind(Messages.WorkspaceResourceProviderService_SOURCE_FILE_DIRECTORY, messageElements)); 
				}	
			}
			else {
				Object[] messageElements = {source}; //$NON-NLS-1$
				errBuf.append(NLS.bind(Messages.WorkspaceResourceProviderService_SOURCE_FILE_DNE, messageElements)); 
			}
		}
		else {
			errBuf.append(NLS.bind(Messages.WorkspaceResourceProviderService_BAD_PARAMS, methodArgs)); //$NON-NLS-1$
		}
		return retVal;
	}


	/**
	 * @param fileManager
	 * @param methodArgs
	 * @return
	 */
	public String getTempFile(Agent agent, String methodArgs, StringBuffer errBuf) {
		String retVal = null;

		Pattern pattern = Pattern.compile("source=(.*)"); //$NON-NLS-1$
		Matcher matcher = pattern.matcher(methodArgs);
		
		if (matcher.find()) {
			String source = matcher.group(1);

			String fullSource = resolvePath(source);
			if (fullSource != null) {
				
				if(new File(fullSource).isFile()){

					IFileManager fileManager = getFileManager(agent);
					if (fileManager == null){
						errBuf.append(Messages.WorkspaceResourceProviderService_NO_FILEMANAGER); 
					}
					else {
						try {
							if (fileManager instanceof IFileManagerExtended) {
								IFileManagerExtended.FileIdentifierList sourceList = FileIdentifierList.create(fullSource);
								File temp = new File(fullSource);
								// Calling putFile() with a relative target location tells 
								// the file transfer service to create the file in a temp 
								// location and return the location as a modified argument
								// for the target location.
								IFileManagerExtended.FileIdentifierList targetList = FileIdentifierList.create(temp.getName());
								((IFileManagerExtended)fileManager).putFile(sourceList, targetList);
								retVal = targetList.get(0);
							}
							else {
								retVal = Boolean.toString(false);
							}
						} catch (Exception e) {
							Object[] messageElements = {source}; //$NON-NLS-1$
							errBuf.append(NLS.bind(Messages.WorkspaceResourceProviderService_TEMP_COPY_FAILED, messageElements)); 
							TestCorePlugin.getDefault().logError(e);
						}
					}
				}
				else {
					Object[] messageElements = {source}; //$NON-NLS-1$
					errBuf.append(NLS.bind(Messages.WorkspaceResourceProviderService_SOURCE_FILE_DIRECTORY, messageElements)); 
				}				
			}
			else {
				Object[] messageElements = {source}; //$NON-NLS-1$
				errBuf.append(NLS.bind(Messages.WorkspaceResourceProviderService_SOURCE_FILE_DNE, messageElements)); 
			}
		}
		else {
			errBuf.append(NLS.bind(Messages.WorkspaceResourceProviderService_BAD_PARAMS, methodArgs)); 
		}
		return retVal;
	}

	/**
	 * @param methodArgs
	 * @return
	 */
	public String listChildren(Agent agent, String methodArgs, StringBuffer errBuf) {
		String result = null;
		Pattern pattern = Pattern.compile("parent=(.*)"); //$NON-NLS-1$
		Matcher matcher = pattern.matcher(methodArgs);
		
		if (matcher.find()) {
			String parent = matcher.group(1);
			String[] children = getChildren(parent);

			if (children != null && children.length > 0) {
				StringBuffer returnVal = new StringBuffer("Children:"); //$NON-NLS-1$
				for (int i = 0; i < children.length; i++) {
					if (i > 0)
						returnVal.append(",");
					returnVal.append(children[i]);
				}
				result = returnVal.toString();
			}
		}
		else {
			errBuf.append(NLS.bind(Messages.WorkspaceResourceProviderService_BAD_PARAMS, methodArgs)); 
		}
		
		if (result == null && errBuf.length() == 0)
			// Indicates empty child set
			result = "Children=";
		return result;			
	}
	

	/**
	 * @param methodArgs
	 * @return
	 */
	public String fileExists(Agent agent, String methodArgs, StringBuffer errBuf) {
		String result = Boolean.toString(false);
		Pattern pattern = Pattern.compile("file=(.*)"); //$NON-NLS-1$
		Matcher matcher = pattern.matcher(methodArgs);
		
		if (matcher.find()) {
			String file = matcher.group(1);
			result = Boolean.toString(exists(file));
		}
		else {
			result = NLS.bind(Messages.WorkspaceResourceProviderService_BAD_PARAMS, methodArgs); //$NON-NLS-1$
		}
		return result;			
	}

	/**
	 * Returns the fully resolved path.  
	 * <p/>
	 * Subclasses may override this method for resolving paths.
	 * 
	 * @param source Source path.
	 * @return The fully resolved path.
	 */
	protected String resolvePath(String source) {
		return source;
	}

	/**
	 * Returns true if the file denoted by the parameter file path exists.  
	 * <p/>
	 * Subclasses may override this method for testing file existence.
	 * 
	 * @param source The file path.
	 * @return True if the file denoted by the parameter file path exists, otherwise false.
	 */
	protected boolean exists(String file) {
		return new File(file).exists();
	}

	/**
	 * @param agent
	 * @return
	 */
	protected IFileManager getFileManager(Agent agent) {
		IFileManager fileManager = null;
		AgentImpl agentImpl = (AgentImpl)agent;
		Process process = agentImpl.getProcess();
		if (process != null) {
			try {
				Connection conn = process.getNode().getConnection();
				try {
					fileManager = FileManagerFactory.getInstance().create(conn);
				} catch (ServerNotAvailableException e) {
					// If we fail here, we'll return null and fail the service call
				}
				
			} catch (InactiveProcessException e) {
				// If we fail here, we'll return null and fail the service call
			}
		}
		return fileManager;
	}

	/**
	 * @param parent
	 * @return all roots if parent is ##ROOT##, null if parent does not exist 
	 * or is a directory, children of parent otherwise
	 */
	protected String[] getChildren(String parent) {
		String[] retVal = null;
		if (parent.equals(ROOT)) {
			File[] roots = File.listRoots();
			if (roots.length > 0) {
				retVal = new String[roots.length];
				for (int i = 0; i < roots.length; i++) {
					retVal[i] = roots[i].getAbsolutePath();
				}
			}			
		}
		else {
			File file = new File(parent);
			if (file.exists() && file.isDirectory()) {
				retVal = file.list();
			}
		}
		return retVal;
	}


}
