/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ArtifactUtil.java,v 1.13 2009/12/18 02:54:00 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.util;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.models.common.configuration.CFGArtifact;
import org.eclipse.hyades.models.common.configuration.CFGArtifactLocationPair;
import org.eclipse.hyades.models.common.configuration.CFGClass;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationFactory;
import org.eclipse.hyades.models.common.configuration.util.ConfigurationUtil;
import org.eclipse.hyades.models.common.interactions.BVRProperty;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.testprofile.impl.TPFBehaviorImpl;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.launching.JavaRuntime;

/**
 * <p>Utilities for {@link CFGArtifact}s.</p>
 * 
 * 
 * @author  Bianca Xue Jiang
 * @author  Julien Canches
 * @author  Paul E. Slauenwhite
 * @version December 17, 2009
 * @since   February 8, 2005
 */
public class ArtifactUtil {
	
	public static final String PROPERTY_NAME_CLASSPATH = Common_ConfigurationFactory.eINSTANCE.createHyadesClasspathCategory().getName(); //"CLASSPATH"
	
	private static final String PDE_CLASSPATH_CONTAINER_ID = "org.eclipse.pde.core.requiredPlugins";
	
	/**
	 * <p>Jars that do not require a classpath entry since they are configured 
	 * on the <code>HyadesTestingSession</code> application in the Agent Controller 
	 * (see <code>&lt;Agent Controller installation directory&gt;\plugins\org.eclipse.hyades.test.tools.core_&lt;version&gt;\config\pluginconfig.xml</code>).</p>
	 * 
	 * <p><b>Note:</b> The Jars in this list MUST be configured on the 
	 * <code>HyadesTestingSession</code> application in the Agent Controller.</p>
	 * 
	 * <p><b>Note:</b> The Jars in this list MUST be upper case for comparison 
	 * (see {@link #addClasspathEntry(String)}).</p>
	 */
	//See Bugzilla #74120.
	private static final String[] EXCLUDED_LIBRARY_JARS = new String[] {

		//org.eclipse.hyades.execution:
		"HEXCORE.JAR",
		"HEXL.JAR", 
		"HEXR.JAR",

		//org.eclipse.hyades.logging.core:
		"HL14.JAR",
		"HLCOMMONS.JAR",
		"HLCORE.JAR",
		"HLLOG4J.JAR", 

		//org.eclipse.tptp.platform.logging.events:
		"TLCORE.JAR",
		"HLCBE101.JAR",
		"TLCBE101.JAR",

		//org.eclipse.hyades.test.tools.core:
		"COMMON.RUNNER.JAR",
		"TOOLS-CORE.JAR",
		"HTTP.RUNNER.JAR",
		"JAVA.RUNNER.JAR",

		//org.junit:
		"JUNIT.JAR"
	};

	public static void associateTestAsset(CFGClass testAsset, CFGArtifact artifact)
	{
		if(!ConfigurationUtil.containsById(artifact.getDeployableInstances(), testAsset))
		{
			artifact.getDeployableInstances().add(testAsset);
			//testAsset.getArtifacts().add(artifact);
		}
		
		if(testAsset instanceof TPFTestSuite)
			calculateClasspath((TPFTestSuite)testAsset, artifact);
	}
	
	public static void calculateClasspath(TPFTestSuite testSuite, CFGArtifact artifact)
	{		
		CFGPropertyGroup propertyGroup = ConfigurationUtil.searchPropertyGroupById(artifact.getPropertyGroups(), ConfigurationUtil.ATTRS_PROP_GROUP_ID_ARTIFACT);
		if(propertyGroup == null)
		{
		    propertyGroup = Common_ConfigurationFactory.eINSTANCE.createCFGPropertyGroup();
		    propertyGroup.setPropertyGroupID(ConfigurationUtil.ATTRS_PROP_GROUP_ID_ARTIFACT);
		    artifact.getPropertyGroups().add(propertyGroup);
		}
		
		//CFGComparableProperty[] properties = findPropertiesByName(propertyGroup.getProperties(), PROPERTY_NAME_CLASSPATH, false);
		//for (int i = 0, maxi = properties.length; i < maxi; i++)
			//propertyGroup.getProperties().remove(properties[i]);
			
		if((testSuite.getBehavior() == null) || (((TPFBehaviorImpl)testSuite.getBehavior()).getProject() == null))
		{
			// Test execution should detect this upon test suite selection.
			// This is not the right place to do this.
			//MessageDialog.openError(Display.getCurrent().getActiveShell(), 
			//						UiPlugin.getString("DLG_GENERATE_TEST"),  //$NON-NLS-1$
			//						UiPlugin.getString("DLG_GENERATE_TEST_DSC")); //$NON-NLS-1$
			return;
		}
			
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(((TPFBehaviorImpl)testSuite.getBehavior()).getProject());
		if((project == null) || (!project.exists()))
		    return; 
		
		boolean workbenchMode = "org.eclipse.hyades.test.plugin.junit.testSuite".equals(testSuite.getType()) || "org.eclipse.hyades.test.auto.gui.testSuite".equals(testSuite.getType());
		String[] classpathEntries = getProjectClasspath(project, false, workbenchMode);
		for(int i = 0, maxi = classpathEntries.length; i < maxi; i++)
		{
			if(addClasspathEntry(classpathEntries[i]))
			{
			    BVRProperty[] existingProps = ConfigurationUtil.searchPropertiesByName(propertyGroup.getProperties(), PROPERTY_NAME_CLASSPATH, false);
				boolean existed = false;
			    for(int propIndex = 0; propIndex < existingProps.length; propIndex++)
			    {
			        if(classpathEntries[i].equals(existingProps[propIndex].getValue()))
			        {
			            existed = true;
			            break;
			        }
			    }
			    
			    if(existed == false)
			    {
				    CFGComparableProperty property = Common_ConfigurationFactory.eINSTANCE.createCFGComparableProperty();
					property.setName(PROPERTY_NAME_CLASSPATH);
					property.setOperator("=");
					property.setValue(classpathEntries[i]);
					propertyGroup.getProperties().add(property);
			    }
			}
		}
	}
	
	/**
	 * <p>Determines if the library should be added as a classpath entry.</p>
	 * 
	 * <p>If the library is <code>null</code>, an empty string, or configured on the 
	 * <code>HyadesTestingSession</code> application in the Agent Controller 
	 * (see <code>&lt;Agent Controller installation directory&gt;\plugins\org.eclipse.hyades.test.tools.core_&lt;version&gt;\config\pluginconfig.xml</code>),
	 * it is not added as a classpath entry.</p>
	 * 
	 * @param library The name of a library.
	 * @return <code>true</code> if the library should be added as a classpath entry, otherwise <code>false</code>.
	 * @see #EXCLUDED_LIBRARY_JARS
	 */
	protected static boolean addClasspathEntry(String library){
	    
		if((library != null) && (library.trim().length() > 0)){
		    
			String libraryUpperCase = library.toUpperCase(Locale.ENGLISH);
			
			//Check if a Jar file library:
		    if(libraryUpperCase.contains(".JAR")) { 
		        
		        //Check if an excluded library:
		        for (int counter = 0; counter < EXCLUDED_LIBRARY_JARS.length; counter++) {
	
		        	if (libraryUpperCase.contains(EXCLUDED_LIBRARY_JARS[counter])) {
		                return false;
		            }
		        }
		    } 
		    
		    return true;
		}
		
	    return false;	    
	}
		
	public static String[] getProjectClasspath(IProject project, boolean exportedOnly) {
		return getProjectClasspath(project, exportedOnly, false);
	}
	
	/**
	 * Returns a Project's classpath.  If the project has a dependency on another 
	 * project the exported entries of the last one will be added to the classpath
	 * of the former. 
	 * 
	 * @param project
	 * @param exportedOnly
	 * @param workbenchMode Whether the returned classpath will be used when launching an
	 * eclipse workbench. In this case, the classpath does not include libraries that are
	 * supposed to be already on the target platform.
	 * @return String[]
	 */
	public static String[] getProjectClasspath(IProject project, boolean exportedOnly, boolean workbenchMode)
	{
		if(project == null)
			return new String[0];
			
		IJavaProject javaProject = JavaCore.create(project);
		if(javaProject == null)
			return new String[0];

		IClasspathEntry[] cpEntries = null;
		try
		{
			cpEntries = javaProject.getRawClasspath();
		}
		catch(JavaModelException e)
		{
		}
		if((cpEntries == null) || cpEntries.length == 0)
			return new String[0];
		
		Set classpath = new HashSet(cpEntries.length);
		try
		{	
			IPath path = javaProject.getOutputLocation();
			if(path != null)
				classpath.add(getFullPath(path));
		}
		catch(JavaModelException e)
		{
		}
				
		collectClasspathEntries(classpath, cpEntries, exportedOnly, /*excludeLibraries*/false, workbenchMode, javaProject);
				
		return (String[])classpath.toArray(new String[classpath.size()]);
	}

	/**
	 * @param classpath
	 * @param cpEntries
	 * @param exportedOnly
	 * @param javaProject
	 */
	private static void collectClasspathEntries(Set classpath, IClasspathEntry[] cpEntries, boolean exportedOnly, boolean excludeLibraries, boolean workbenchMode, IJavaProject javaProject) {
		for (int i = 0;  i < cpEntries.length; i++)
		{
			IClasspathEntry entry = JavaCore.getResolvedClasspathEntry(cpEntries[i]);
			if(exportedOnly && (!entry.isExported()))
				continue;
			
			String strcp = null;
			switch(entry.getEntryKind())
			{
				case IClasspathEntry.CPE_LIBRARY:
				case IClasspathEntry.CPE_VARIABLE:
					if (!excludeLibraries) {
						strcp = getFullPath(entry.getPath());
						if(strcp != null)
							classpath.add(strcp);
					}
					break;

				case IClasspathEntry.CPE_PROJECT:
					IPath projectPath = entry.getPath().makeAbsolute();
					IProject parentProject = ResourcesPlugin.getWorkspace().getRoot().getProject(projectPath.toString());				
					classpath.addAll(Arrays.asList(getProjectClasspath(parentProject, true, workbenchMode)));
					break;
				
				case IClasspathEntry.CPE_CONTAINER:
					
					String entryPathString = entry.getPath().toString();
					
					if(entryPathString != null){

						entryPathString = entryPathString.trim();
					
						//Add the classpath container if it not the JRE classpath container 
						//since it is available in the Agent Controller:
						if (!entryPathString.startsWith(JavaRuntime.JRE_CONTAINER)){
	
							//Collect the classpath entries for the classpath container if it not the 
							//PDE classpath container since it is available in the target Eclipse instance:
							try {							
								collectClasspathEntries(classpath, JavaCore.getClasspathContainer(entry.getPath(), javaProject).getClasspathEntries(), exportedOnly, ((workbenchMode) && (entryPathString.equals(PDE_CLASSPATH_CONTAINER_ID))), workbenchMode, javaProject);
							} 
							catch (JavaModelException j) {
								TestCorePlugin.getDefault().logError(j);
							}
						}
					}
										
					break;
					
				case IClasspathEntry.CPE_SOURCE:
					strcp = getFullPath(entry.getOutputLocation());
					if(strcp != null)
						classpath.add(strcp);
			}
		}
	}
	
	
	private static String getFullPath(IPath path)
	{
		if(path == null)
			return null;
			
		IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(path.makeAbsolute());
		if(res != null)
			path = res.getLocation().makeAbsolute();
			
		return path.toString();
	}
	

    public static void adjustDefaultLocation(TPFDeployment depl, TPFTest test)
    {
		CFGLocation location = ConfigurationUtil.getDefaultLocation(depl);

		if ( location != null )
    	{
        	TPFTestSuite testSuite = null;
        	TPFTest subTest = null;
        	if (test instanceof TPFTestCase) {
        		subTest = test;
        		testSuite = ((TPFTestCase)subTest).getTestSuite();
        	} else if (test instanceof TPFTestSuite) {
        		testSuite = (TPFTestSuite)test;
        		subTest = test;
        	}
			Resource resource = depl.eResource();

        	CFGArtifact artifact = ConfigurationUtil.createArtifact(null, null);
    		ArtifactUtil.associateTestAsset(testSuite, artifact);
    		CFGArtifactLocationPair pair = ConfigurationUtil.createArtifactLocationPair(artifact, location);
			depl.getArtifactLocations().add(pair);
			resource.setModified(true);
   		}
    }
	
}
