/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EMFUtil.java,v 1.24 2010/04/12 12:38:44 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.util;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.hyades.loaders.util.RegistryReader;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle;
import org.eclipse.osgi.util.NLS;

/**
 * <p>Utility APIs for working with EMF resources and objects.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul Slauenwhite
 * @author	Jerome Bozier
 * @version April 12, 2010
 * @since   February 4, 2005
 */
public class EMFUtil {
	
	private static ResourceSet defaultResourceSet = new ResourceSetImpl();
	private static Resource resourceBeingSaved;
	private static List allCustomDeleter = new ArrayList();
	private static boolean bOk = false;

	public static final Map RESOURCE_OPTIONS = new HashMap(4);
	
	static {
		
		//NOTE: Increase RESOURCE_OPTIONS initial capacity if adding more elements.
		RESOURCE_OPTIONS.put(XMLResource.OPTION_DECLARE_XML, Boolean.TRUE);
		RESOURCE_OPTIONS.put(XMLResource.OPTION_SKIP_ESCAPE, Boolean.FALSE);
		RESOURCE_OPTIONS.put(XMLResource.OPTION_SKIP_ESCAPE_URI, Boolean.FALSE);
		RESOURCE_OPTIONS.put(XMLResource.OPTION_RECORD_UNKNOWN_FEATURE, Boolean.TRUE);
	}
	
	// in bugzilla 168544, it is the EMF loads that are causing exceptions 
	// due to java.io.tmpdir not pointing to an existing directory on Windows
	// However, this will cause a huge number of problems all over the place.
	// TMP environment variable must point to a valid directory
	private static void checkTMPDIR(){
		
		if (!bOk){  // only do this once per workbench invocation
		    String strOSName = System.getProperty("os.name");//$NON-NLS-1$
			if (strOSName.startsWith("Windows") == true){//$NON-NLS-1$		
				File tmpdir = new File(System.getProperty("java.io.tmpdir"));//$NON-NLS-1$
				if (!tmpdir.isDirectory()){	
					
					String msg = NLS.bind(TestCorePluginResourceBundle.CoreUtil_ERROR_TMPDIR_INVALID,
							tmpdir.getPath()); 
					
					IStatus status = new Status(IStatus.ERROR, ResourcesPlugin.getPlugin().getBundle().getSymbolicName(), 0, msg, null);
					RuntimeException e = new RuntimeException(msg);
					e.initCause(new CoreException(status));
					throw e;
				}else{
					//we will only check this once per workbench session to avoid
					// excessive overhead
					bOk =true;
				}
			
			}
		}
	}

	/**
	 * Saves a resource.
	 * @param resource
	 * @param progressMonitor
	 * @throws IllegalArgumentException if the resource is null
	 * @throws Exception if an exception happens during the saving attempt.
	 */
	public static void save(Resource resource) throws Exception {
		if (resource == null)
			throw new IllegalArgumentException(NLS.bind(TestCorePluginResourceBundle._ERROR_ARG_NULL, "resource")); 

		boolean isModified = resource.isModified();

		resourceBeingSaved = resource;
		final Resource theResource = resource;
			
		try
		{
			// If we're running inside the workbench, and if we're in 
			// "workspace mode" (which is an artificial designation 
			// we introduced to support non "workspace mode" RCP apps that do not use the
			// workspace, but which still have the core.resources plugin)
			// we need to wrap this save action in an IWorkspaceRunnable
			// to avoid notifying resource listeners while the save is
			// in progress.  This is run in the current thread, but the
			// action is atomic to all workspace aware resource listeners.
			if (Platform.isRunning() && RegistryReader.isWorkspaceMode()) {
				IWorkspaceRunnable runnable = new IWorkspaceRunnable(){
					public void run(IProgressMonitor arg0) throws CoreException {
						try {
							theResource.save(RESOURCE_OPTIONS);
						} catch (IOException e) {
							IStatus status = new Status(IStatus.ERROR, ResourcesPlugin.getPlugin().getBundle().getSymbolicName(), 0, e.getMessage(), e);
							throw new CoreException(status);
						}
					}
				};
				IWorkspace workspace = ResourcesPlugin.getWorkspace();
				workspace.run(runnable, new NullProgressMonitor());
			}
			else
				theResource.save(RESOURCE_OPTIONS);
		}catch (CoreException e) {
			// We know that e contains an IStatus with an IOException in it
			// because we created it in the IOException catch block above
			Exception ex = (Exception) e.getStatus().getException();
			resource.setModified(isModified);
			throw ex;
		}
		catch (Exception e)
		{
			resource.setModified(isModified);
			throw e;
		}
		finally {
			resourceBeingSaved = null;
		}
	}

	/** Unload a resource using its URI from the test navigator resource set if this resource is not one we are currently saving
	 *  This method is used in the updateContent of the resource change updater of the test navigator
	 * @param uri the URI of the resource to unload
	 * @deprecated As of TPTP 4.7.0, this method is no longer used.
	 */
	public static void unloadIfNeeded(URI uri) {
		Resource resource = getResourceSet().getResource(uri, false);
		//- we unload the resource from the resource set if this is not a resource we are currently saving
		if(resource != null && resourceBeingSaved == null) {
			resource.unload();
		}
	}

	/**
	 * <p>Resolves or loads on demand the {@link EObject} with the same URI or ID of the {@link EObject} 
	 * using the parameter {@link ResourceSet}.</p>
	 * 
	 * <p>If the parameter {@link ResourceSet} is <code>null</code>, the default {@link ResourceSet}
	 * is used.</p>
	 * 
	 * <p>If an error occurs during the resolving or loading operation, an error message is logged.</p>
	 * 
	 * @param resourceSet The {@link ResourceSet} used to load the {@link EObject}.
	 * @param eObject The {@link EObject} with the same URI or ID of the {@link EObject}.
	 * @param loadOnDemand <code>true</code> if the {@link EObject} is loaded on demand, otherwise <code>false</code>.
	 * @return The {@link EObject} from resolving or loading the {@link EObject}.
	 */
	public static EObject getEObject(ResourceSet resourceSet, EObject eObject, boolean loadOnDemand) {

		if(eObject != null){

			Resource resource = eObject.eResource();

			if (resource != null){

				ResourceSet resolvedResourceSet = resourceSet;

				if (resolvedResourceSet == null) {
					resolvedResourceSet = defaultResourceSet;
				}

				if(resolvedResourceSet.equals(resource.getResourceSet())){
					return eObject;
				}

				URI uri = resource.getURI();

				if(uri != null){

					String uriFragment = resource.getURIFragment(eObject);

					try {
						resource = resolvedResourceSet.getResource(resource.getURI(), loadOnDemand);
					} 
					catch (Exception e) {
						
						TestCorePlugin.getDefault().logError(NLS.bind(TestCorePluginResourceBundle.EMFUtil_ERROR_LOADING_RESOURCE, getWorkspaceFilePath(uri), e.getLocalizedMessage()));
						
						return null;
					}			
					
					if(resource != null){
						return (resource.getEObject(uriFragment));
					}
				}
			} 
		}

		return null;
	}

	/**
	 * <p>Resolves without loading on demand the root {@link EObject}(s) from the {@link Resource} based on its {@link IFile}
	 * using the parameter {@link ResourceSet}.</p>
	 * 
	 * <p>If the parameter {@link ResourceSet} is <code>null</code>, the default {@link ResourceSet}
	 * is used.</p>
	 * 
	 * <p>If the {@link IFile} is <code>null</code> or the {@link Resource} is empty, an empty array is returned.</p>
	 * 
	 * <p>If an error occurs during the resolving operation, an error message is logged and an empty 
	 * array is returned.</p>
	 * 
	 * @param resourceSet The {@link ResourceSet} used to resolve the {@link Resource}.
	 * @param file The {@link IFile} of the {@link Resource}.
	 * @return Root {@link EObject}(s) from resolving the {@link Resource}.
	 * @see #getEObjects(ResourceSet, IFile, boolean)
	 */
	public static EObject[] getEObjects(ResourceSet resourceSet, IFile file) {
		return (getEObjects(resourceSet, file, false));
	}
	
	/**
	 * <p>Resolves or loads on demand the root {@link EObject}(s) from the {@link Resource} based on its {@link IFile}
	 * using the parameter {@link ResourceSet}.</p>
	 * 
	 * <p>If the parameter {@link ResourceSet} is <code>null</code>, the default {@link ResourceSet}
	 * is used.</p>
	 * 
	 * <p>If the {@link IFile} is <code>null</code> or the {@link Resource} is empty, an empty array is returned.</p>
	 * 
	 * <p>If an error occurs during the resolving or loading operation, an error message is logged and an empty 
	 * array is returned.</p>
	 * 
	 * @param resourceSet The {@link ResourceSet} used to resolve or load the {@link Resource}.
	 * @param file The {@link IFile} of the {@link Resource}.
	 * @param loadOnDemand <code>true</code> if the {@link EObject}(s) are loaded on demand, otherwise <code>false</code>.
	 * @return Root {@link EObject}(s) from resolving or loading the {@link Resource}.
	 * @see #getEObjects(ResourceSet, String, boolean)
	 */
	public static EObject[] getEObjects(ResourceSet resourceSet, IFile file, boolean loadOnDemand) {

		if (file == null){
			return (new EObject[0]);
		}
		
		return (getEObjects(resourceSet, file.getFullPath().toString(), loadOnDemand));
	}

	/**
	 * <p>Resolves without loading on demand the root {@link EObject}(s) from the {@link Resource} based on its path
	 * using the parameter {@link ResourceSet}.</p>
	 * 
	 * <p>If the parameter {@link ResourceSet} is <code>null</code>, the default {@link ResourceSet}
	 * is used.</p>
	 * 
	 * <p>If the path is <code>null</code> or the {@link Resource} is empty, an empty array is returned.</p>
	 * 
	 * <p>If an error occurs during the resolving operation, an error message is logged and an empty 
	 * array is returned.</p>
	 * 
	 * @param resourceSet The {@link ResourceSet} used to resolve the {@link Resource}.
	 * @param path The path of the {@link Resource}.
	 * @return Root {@link EObject}(s) from resolving the {@link Resource}.
	 * @see #getEObjects(ResourceSet, String, boolean)
	 */
	public static EObject[] getEObjects(ResourceSet resourceSet, String path) {
		return (getEObjects(resourceSet, path, false));
	}
	
	/**
	 * <p>Resolves or loads on demand the root {@link EObject}(s) from the {@link Resource} based on its path
	 * using the parameter {@link ResourceSet}.</p>
	 * 
	 * <p>If the parameter {@link ResourceSet} is <code>null</code>, the default {@link ResourceSet}
	 * is used.</p>
	 * 
	 * <p>If the path is <code>null</code> or the {@link Resource} is empty, an empty array is returned.</p>
	 * 
	 * <p>If an error occurs during the resolving or loading operation, an error message is logged and an empty 
	 * array is returned.</p>
	 * 
	 * @param resourceSet The {@link ResourceSet} used to resolve or load the {@link Resource}.
	 * @param path The path of the {@link Resource}.
	 * @param loadOnDemand <code>true</code> if the {@link EObject}(s) are loaded on demand, otherwise <code>false</code>.
	 * @return Root {@link EObject}(s) from resolving or loading the {@link Resource}.
	 * @see #getEObjects(ResourceSet, URI, boolean)
	 */
	public static EObject[] getEObjects(ResourceSet resourceSet, String path, boolean loadOnDemand) {
		
		if(path == null){
			return (new EObject[0]);
		}
			
		return (getEObjects(resourceSet, URI.createPlatformResourceURI(path, false), loadOnDemand)); 
	}
	
	/**
	 * <p>Resolves or loads on demand the root {@link EObject}(s) from the {@link Resource} based on its {@link URI} 
	 * using a new {@link ResourceSet}.</p>
	 * 
	 * <p>If the {@link Resource} is empty, an empty array is returned.</p>
	 * 
	 * <p>If an error occurs during the resolving or loading operation, an error message is logged and an empty 
	 * array is returned.</p>
	 * 
	 * @param uri The {@link URI} of the {@link Resource}.
	 * @param loadOnDemand <code>true</code> if the {@link EObject}(s) are loaded on demand, otherwise <code>false</code>.
	 * @return Root {@link EObject}(s) from resolving or loading the {@link Resource}.
	 * @see #getEObjects(ResourceSet, URI, boolean)
	 */
	public static EObject[] getEObjects(URI uri, boolean loadOnDemand){
		return (getEObjects(new ResourceSetImpl(), uri, loadOnDemand));
	}	

	/**
	 * <p>Resolves or loads on demand the root {@link EObject}(s) from the {@link Resource} based on its {@link URI} 
	 * using the parameter {@link ResourceSet}.</p>
	 * 
	 * <p>If the parameter {@link ResourceSet} is <code>null</code>, the default {@link ResourceSet}
	 * is used.</p>
	 * 
	 * <p>If the {@link Resource} is empty, an empty array is returned.</p>
	 * 
	 * <p>If an error occurs during the resolving or loading operation, an error message is logged and an empty 
	 * array is returned.</p>
	 * 
	 * @param resourceSet The {@link ResourceSet} used to resolve or load the {@link Resource}.
	 * @param uri The {@link URI} of the {@link Resource}.
	 * @param loadOnDemand <code>true</code> if the {@link EObject}(s) are loaded on demand, otherwise <code>false</code>.
	 * @return Root {@link EObject}(s) from resolving or loading the {@link Resource}.
	 */
	public static EObject[] getEObjects(ResourceSet resourceSet, URI uri, boolean loadOnDemand) {
		
		Resource resource = null;
		
		if(uri != null){

			try {

				if (resourceSet != null) {
					resource = resourceSet.getResource(uri, loadOnDemand);
				} 
				else {					
					resource = defaultResourceSet.getResource(uri, loadOnDemand);
				}
			} 
			catch (Exception e) {
				
				IFile workspaceFile = getWorkspaceFile(uri);

				//Verify if the file is empty: 
				if ((workspaceFile != null) && (workspaceFile.getLocation().toFile().length() == 0)){
					TestCorePlugin.getDefault().logError(NLS.bind(TestCorePluginResourceBundle.EMFUtil_ERROR_LOADING_EMPTY_RESOURCE, workspaceFile.getFullPath().toString()));
				}
				else{
					TestCorePlugin.getDefault().logError(NLS.bind(TestCorePluginResourceBundle.EMFUtil_ERROR_LOADING_RESOURCE, getWorkspaceFilePath(uri), e.getLocalizedMessage()));
				}
			}
		}

		if (resource == null){
			return (new EObject[0]);
		}

		return ((EObject[])(resource.getContents().toArray(new EObject[resource.getContents().size()])));
	}

	/**
	 * <p>Loads the root {@link EObject}(s) from the {@link Resource} based on its {@link IFile}
	 * using the parameter {@link ResourceSet}.</p>
	 * 
	 * <p>If the parameter {@link ResourceSet} is <code>null</code>, the default {@link ResourceSet}
	 * is used.</p>
	 * 
	 * <p>If the {@link IFile} is <code>null</code> or the {@link Resource} is empty, an empty array is returned.</p>
	 * 
	 * <p>If an error occurs during the loading operation, an error message is logged and an empty 
	 * array is returned.</p>
	 * 
	 * @param resourceSet The {@link ResourceSet} used to load the {@link Resource}.
	 * @param file The {@link IFile} of the {@link Resource}.
	 * @return Root {@link EObject}(s) from loading the {@link Resource}.
	 * @see #load(ResourceSet, String)
	 */
	public static EObject[] load(ResourceSet resourceSet, IFile file) {
	
		if (file == null){
			return (new EObject[0]);
		}
		
		return (load(resourceSet, file.getFullPath().toString()));
	}

	/**
	 * <p>Loads the root {@link EObject}(s) from the {@link Resource} based on its path
	 * using the parameter {@link ResourceSet}.</p>
	 * 
	 * <p>If the parameter {@link ResourceSet} is <code>null</code>, the default {@link ResourceSet}
	 * is used.</p>
	 * 
	 * <p>If the path is <code>null</code> or the {@link Resource} is empty, an empty array is returned.</p>
	 * 
	 * <p>If an error occurs during the loading operation, an error message is logged and an empty 
	 * array is returned.</p>
	 * 
	 * @param resourceSet The {@link ResourceSet} used to load the {@link Resource}.
	 * @param path The path of the {@link Resource}.
	 * @return Root {@link EObject}(s) from loading the {@link Resource}.
	 * @see #load(ResourceSet, URI)
	 */
	public static EObject[] load(ResourceSet resourceSet, String path) {
		
		if (path == null){
			return (new EObject[0]);
		}
		
		return (load(resourceSet, URI.createPlatformResourceURI(path, false)));
	}

	/**
	 * <p>Loads the root {@link EObject}(s) from the {@link Resource} based on its {@link URI} 
	 * using the parameter {@link ResourceSet}.</p>
	 * 
	 * <p>If the parameter {@link ResourceSet} is <code>null</code>, the default {@link ResourceSet}
	 * is used.</p>
	 * 
	 * <p>If the {@link Resource} is empty, an empty array is returned.</p>
	 * 
	 * <p>If an error occurs during the loading operation, an error message is logged and an empty 
	 * array is returned.</p>
	 * 
	 * @param resourceSet The {@link ResourceSet} used to load the {@link Resource}.
	 * @param uri The {@link URI} of the {@link Resource}.
	 * @return Root {@link EObject}(s) from loading the {@link Resource}.
	 * @see #getEObjects(ResourceSet, URI, boolean)
	 */
	public static EObject[] load(ResourceSet resourceSet, URI uri) {
		
		checkTMPDIR();

		return (getEObjects(resourceSet, uri, true));
	}

	/**
	 * Returns the resource factory associated with a given file extension.
	 * If the file extension is <code>null</code> or if there isn't a specifc
	 * factory for the file extension, the this method returns the generic
	 * factory.
	 * @param fileExtension
	 * @return the Resource.Factory or <code>null</code> if none. 
	 */
	public static Resource.Factory getResourceFactory(String fileExtension) {
		if (fileExtension != null) {
			Resource.Factory factory = (Resource.Factory) Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().get(fileExtension);
			if (factory != null)
				return factory;
		}

		return (Resource.Factory) Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().get(Resource.Factory.Registry.DEFAULT_EXTENSION);
	}

	/**
	 * Returns the workspace file that is storing a given eObject or
	 * <code>null</code> if the object is not persisted. 
	 * @param eObject
	 * @return IFile
	 */
	public static IFile getWorkspaceFile(EObject eObject) {
		if (eObject == null)
			return null;
		return getWorkspaceFile(eObject.eResource());
	}

	public static IFile getWorkspaceFile(URI uri) {
        if(uri == null) return null;
		String filePath = getWorkspaceFilePath(uri.trimFragment());
		IWorkspaceRoot workspaceRoot= ResourcesPlugin.getWorkspace().getRoot();
		if (filePath == null)
		{
			String localPath = uri.toFileString();
			if (localPath == null) {
				// Try to extract a workspace path
				IPath path = new Path(uri.path());
				IResource res = workspaceRoot.findMember(path);
				if (res != null && res.getType() == IResource.FILE) {
					return (IFile)res;
				}
				return null;
			} else {
				IPath location= Path.fromOSString(localPath);
				IFile[] files= workspaceRoot.findFilesForLocation(location);
				if (files == null || files.length == 0)
					return null;
				return files[0];
			}
		}
		else
		{
			IResource workspaceResource = workspaceRoot.findMember(filePath);
			if ((workspaceResource == null) || (workspaceResource.getType() != IResource.FILE)) 
			{
				return null;
			}
			return (IFile) workspaceResource;
		}
	}
	
	/**
	 * Returns the workspace file that corresponds to a given resource or
	 * <code>null</code> if the resource is not available in the 
	 * workspace.
	 * @param resource
	 * @return IFile
	 */
	public static IFile getWorkspaceFile(Resource resource) {
		if (resource == null) return null;
		return getWorkspaceFile(resource.getURI());
	}
	
	public static IContainer getParent(Resource resource)
	{
		IFile file = EMFUtil.getWorkspaceFile(resource);
		if(file != null)
		{
			return file.getParent();
		}
		return null;
	}
	/**
	 * Returns file path for resource or empty string if not found
	 * @param eObject
	 * @return String file path
	 */
	public static String getFilePath(Resource resource)
	{
		if (resource == null)
		{
			return "";
		}
		
		IFile file = getWorkspaceFile(resource);
		String filePath = null;
		if (file == null)
		{
			URI uri = resource.getURI();
            filePath = uri != null ? uri.toFileString(): ""; // $NON-NLS-1$
		}
		else
		{
			filePath = file.getFullPath().toString();
		}
		if (filePath == null)
		{
			return "";
		}
		return filePath;
	}

	/**
	 * Returns file path for EObject or empty string if not found
	 * @param eObject
	 * @return String file path
	 */
	public static String getFilePath(EObject eObject)
	{
		return getFilePath(eObject.eResource());
	}
	
	/**
	 * <p>Resolves the workspace file path from a {@link Resource} containing
	 * a platform resource {@link URI}.</p>
	 * 
	 * <p>This method returns <code>null</code> if the {@link Resource} is 
	 * <code>null</code> or the contained {@link URI} does not represent a 
	 * platform resource.</p>
	 * 
	 * @param resource A {@link Resource} containing a platform resource {@link URI}.
	 * @return The decoded workspace file path, otherwise <code>null</code>.
	 */
	public static String getWorkspaceFilePath(Resource resource) {
		
		if (resource != null){
			return (getWorkspaceFilePath(resource.getURI()));
		}
		
		return null;
	}

	/**
	 * <p>Resolves the workspace file path from a platform
	 * resource {@link URI}.</p>
	 * 
	 * <p>This method returns <code>null</code> if the {@link URI} is 
	 * <code>null</code> or does not represent a platform resource.</p>
	 * 
	 * @param uri A platform resource {@link URI}.
	 * @return The decoded workspace file path, otherwise <code>null</code>.
	 */
	private static String getWorkspaceFilePath(URI uri) {
		
		if(uri != null){

			String resourceURI = uri.toString();
			
			if (resourceURI.startsWith("platform:/resource")){ //$NON-NLS-1$
				return (resourceURI.substring(18)); 
			}
		}
		
		return null;
	}

	/**
	 * Returns a resource without loading it.  If the resource is <code>null</code>,
	 * a new resource is created if a resource factory is available for the file
	 * extension.
	 * @param rs
	 * @param file
	 * @return Resource
	 */
	public static Resource getResource(ResourceSet rs, IFile file) {
		URI uri = URI.createPlatformResourceURI(file.getFullPath().toString(), false);
		return getResource(rs, uri, false);

		//		Resource.Factory factory = getResourceFactory(file.getFileExtension());
		//		if(factory != null)
		//			return factory.createResource(uri);
		//			
		//		return null;
	}

	public static Resource getResource(ResourceSet rs, URI uri, boolean load) {
		ResourceSet localRS = rs;

		if (localRS == null)
			localRS = defaultResourceSet;
		return localRS.getResource(uri, load);
	}

	/**
	 * Reloads a given eObject returning a <b>new</b> instance from the
	 * same resource.
	 * 
	 * <p>This method unloads and reloads the entire eObject's resource
	 * and tries to find the object in the new resource content.
	 * 
	 * <p>Important: after reloading the resource all the objects
	 * in that resource are reinstantiated.  This has a big impact if an object
	 * is being cached somewhere.
	 * 
	 * @param eObject
	 * @return the reloaded eObject
	 */
	public static EObject reload(EObject eObject) {
		
		if (eObject != null){

			Resource resource = eObject.eResource();
			
			if (resource != null){

				String uriFragment = resource.getURIFragment(eObject);
				
				if (resource.isLoaded()){
					resource.unload();
				}

				try {
					resource.load(RESOURCE_OPTIONS);
				} 
				catch (IOException e) {

					TestCorePlugin.getDefault().logError(NLS.bind(TestCorePluginResourceBundle.EMFUtil_ERROR_RELOADING_RESOURCE, getWorkspaceFilePath(resource), e.getLocalizedMessage()));

					return null;
				}

				return (resource.getEObject(uriFragment));
			}
		}
		
		return eObject;
	}

	/**
	 * Removes the specified eObject from its container and breaks
	 * all the references to this object within the context of its resource
	 * set. 
	 * @param eObject
	 * @return the changed resources including the eObject's resource.
	 */
	public static Resource[] remove(EObject eObject) {
		if (eObject == null)
			return new Resource[0];

		if (eObject.eResource() == null) {
			EcoreUtil.remove(eObject);
			return new Resource[0];
		}

		Resource resource = eObject.eResource();
		Set changedResources;
		for (Iterator it = allCustomDeleter.iterator(); it.hasNext(); ) {
			IEMFCustomDeleter deleter = (IEMFCustomDeleter)it.next();
			deleter.customDelete(eObject);
		}		
        try {
            changedResources = org.eclipse.hyades.models.hierarchy.util.EMFUtil.delete(eObject, resource.getResourceSet());
        } catch (ConcurrentModificationException e) {
            changedResources = new HashSet();
        }        
		changedResources.add(resource);

		if (eObject.eContainer() != null)
			EcoreUtil.remove(eObject);

		return (Resource[]) changedResources.toArray(new Resource[changedResources.size()]);
	}

	/**
	 * register a custom deleter that will be called before real EMF delete
	 * @param deleter
	 */
	public static void registerCustomDeleter(IEMFCustomDeleter deleter) {
		allCustomDeleter.add(deleter);
	}
	
	/**
	 * unregister a previously registered custom deleter
	 * @param deleter
	 * @return true if found, false else
	 */
	public static boolean unregisterCustomDeleter(IEMFCustomDeleter deleter) {
		return allCustomDeleter.remove(deleter);
	}
	
	/**
	 * Removes the specified resource from its resource set and breaks
	 * all the references to objects inside this resource within the context 
	 * of the resource set. 
	 * @param resource
	 * @return the changed resources excluding the specified resource. 
	 */
	public static Resource[] remove(Resource resource) {
		if (resource == null)
			return new Resource[0];

		Set changedResources = org.eclipse.hyades.models.hierarchy.util.EMFUtil.delete(resource);
		return (Resource[]) changedResources.toArray(new Resource[changedResources.size()]);
	}

	/**
	 * Adds the specified adapter to the notifiers in the given iterator.
	 * @param objectIterator
	 * @param adapter
	 */
	public static void addAdapter(Iterator objectIterator, Adapter adapter) {
		while (objectIterator.hasNext()) {
			Object object = objectIterator.next();
			if (object instanceof Notifier)
				 ((Notifier) object).eAdapters().add(adapter);
		}
	}

	/**
	 * Removes the specified adapter from the notifiers in the given 
	 * iterator.
	 * @param objectIterator
	 * @param adapter
	 */
	public static void removeAdapter(Iterator objectIterator, Adapter adapter) {
		while (objectIterator.hasNext()) {
			Object object = objectIterator.next();
			if (object instanceof Notifier)
				 ((Notifier) object).eAdapters().remove(adapter);
		}
	}

	/**
	 * Removes all adapters from the notifiers in the given 
	 * iterator.
	 * @param objectIterator
	 */
	public static void removeAdapters(Iterator objectIterator) {
		while (objectIterator.hasNext()) {
			Object object = objectIterator.next();
			if (object instanceof Notifier)
				 ((Notifier) object).eAdapters().clear();
		}
	}
	
	/**
	 * Removes a specific adapter from the notifiers in the given 
	 * iterator.
	 * @param objectIterator
	 */
	public static void removeAdapters(Iterator objectIterator, Adapter adapter) {
		while (objectIterator.hasNext()) {
			Object object = objectIterator.next();
			if (object instanceof Notifier)
				 ((Notifier) object).eAdapters().remove(adapter);
		}
	}
	
	/**
	 * @return the resource set cached in this class
	 */
	public static ResourceSet getResourceSet() {
		return defaultResourceSet;
	}

	/** Returns the resource which contains (at the first level) the given <code>object</code>.
	 * @param rs resource set used to search in or <code>null</code> if cached one has to be used.
	 * @param object the EMF element we are loking for its resource
	 * @return the resource which contains the looked for object
	 */
	public static Resource getResource(ResourceSet rs, EObject object) {
		ResourceSet localRS = rs;

		if (localRS == null) localRS = defaultResourceSet;
		EList resources = localRS.getResources();
		for(Iterator it1 = resources.iterator(); it1.hasNext(); ) {
			Resource res = (Resource)it1.next();
			EList resContent = res.getContents();
			for(Iterator it2 = resContent.iterator(); it2.hasNext(); ) {
				if(object.equals(it2.next())) return res;
			}
		}
		return null;
	}
	
    public static EObject getEObjectInResourceSet(EObject object, ResourceSet rs) {
    	URI uri = object.eResource().getURI();
    	String uriFragment = object.eResource().getURIFragment(object);
    	Resource res = rs.getResource(uri, /*loadOnDemand*/true);
    	EObject obj = res.getEObject(uriFragment);
    	return obj;
    }

    /** Returns the object resolved by the URI.
	 * @param rs resource set used to search in or <code>null</code> if cached one has to be used.
     * @param uri the URI to resolve.
     * @param loadOnDemand whether to create and load the resource, if it doesn't already exists.
     * @return the object resolved by the URI.
     * @see ResourceSet#getEObject(org.eclipse.emf.common.util.URI, boolean)
     * @author jgout
     */
    public static EObject getEObject(ResourceSet rs, URI uri, boolean loadOnDemand) {
		ResourceSet localRS = rs;

		if (localRS == null) localRS = defaultResourceSet;
		return localRS.getEObject(uri, loadOnDemand);
    }
    
	/**
	 * Returns the complete EMF URI which includes the resource information.
	 * @param object
	 * @return the uri of the given object
	 */
	public static URI getObjectURI(Object object) {
		if (object != null) {
			if (object instanceof EObject) {
				EObject eObject = (EObject) object;
				if (eObject.eResource() != null)
					return eObject.eResource().getURI().appendFragment(eObject.eResource().getURIFragment(eObject));
			}
		}
		return null;
	}

}
